/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#ifndef COMMON_H
#define COMMON_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <utmp.h>
#include <string.h>
#include "types.h"
#include "macros.h"
#include "hash.h"

/* User Database Format Identification */
#define ACUA_USERS_MAGIC                "ACUA304"
#define ACUA_USERS_MAGIC_LEN            8

/* ACUA Version */
#define ACUA_VERSION                    "3.04.PR11"
#define ACUA_VERSION_EXTRA		"Designed for Linux Kernel 2.4"

/* Connection Types */
#define CONN_NONE                       0    /* Unknown Type */
#define CONN_TTY                        1    /* Local/Telnet/SSH */
#define CONN_PPP                        2    /* PPP Connection */
#define CONN_PPPoE                      3    /* PPPoE Connection */
#define CONN_CIPE                       4    /* CIPE VPN Connection */
#define CONN_ETH                        5    /* Ethernet Connection */

/* Maximum number of logins a single user can have at once */
#define MAX_LOGINS                      16
/* Maximum hosts a user can be logged into */
#define MAX_HOSTS                       16
/* Maximum number of users that can be on-line at once */
#define MAX_USERS                       256
/* Maximum number of processes running */
#define MAX_PROCS                       512
/* Maximum expire warning times */
#define MAX_EXPIRE_WARN_TIMES           16
/* Maximum boot (kick) warning times */
#define MAX_BOOT_WARN_TIMES             16
/* Maximum number of time classes */
#define MAX_TIME_CLASSES                16
/* Maximum charactors in a login name (not including NULL terminator) */
#define MAX_LOGINCHARS                  16
/* Maximum PPP devices */
#define MAX_PPP_UNITS                   MAX_USERS

/* Operation codes for fLock */
/* Get an exclusive file lock */
#define LOCK_EXCLUSIVE                  0
/* Unlock the file */
#define LOCK_UNLOCK                     1
/* Block until lock is obtained */
#define LOCK_BLOCK                      2
/* Bits for UserRec.flags field */
#define FLG_PRIORITY                    0xe000
#define FLG_SMARTTIME                   0x1000
#define FLG_SSMARTTIME                  0x0800
#define FLG_TCSMARTTIME                 0x0400
#define FLG_SMARTBOOT                   0x0200
#define FLG_SSMARTBOOT                  0x0100
#define FLG_TCSMARTBOOT                 0x0080
#define FLG_ISMARTBOOT                  0x0040
#define FLG_EXPIRE                      0x0020
#define FLG_WARNEXPIRE                  0x0010
#define FLG_WARNBOOT                    0x0008
#define FLG_EXPLAINBOOT                 0x0004
#define FLG_NOUNSUB                     0x0002
#define FLG_LOCK                        0x0001
/* Macros to access fields in the UserRec.flags bitfield */
/* Get NOUNSUB flag */
#define NOUNSUB(f)                      (((f) & FLG_NOUNSUB)
/* Get priority (0-7) */
#define PRIORITY(f)                     (((f) & FLG_PRIORITY) >> 13)
/* Get SmartTime flag */
#define SMARTTIME(f)                    ((f) & FLG_SMARTTIME)
/* Get Session SmartTime flag */
#define SSMARTTIME(f)                   ((f) & FLG_SSMARTTIME)
/* Get TimeClass SmartTime flag */
#define TCSMARTTIME(f)                  ((f) & FLG_TCSMARTTIME)
/* Get SmartBoot flag */
#define SMARTBOOT(f)                    ((f) & FLG_SMARTBOOT)
/* Get Session SmartBoot flag */
#define SSMARTBOOT(f)                   ((f) & FLG_SSMARTBOOT)
/* Get TimeClass SmartBoot flag */
#define TCSMARTBOOT(f)                  ((f) & FLG_TCSMARTBOOT)
/* Get Idle SmartBoot flag */
#define ISMARTBOOT(f)                   ((f) & FLG_ISMARTBOOT)
/* Get expire flag (see EXPIRE_DELETE and EXPIRE_UNSUBSCRIBE below) */
#define EXPIRE(f)                       ((f) & FLG_EXPIRE)
/* Get WarnExpire flag */
#define WARNEXPIRE(f)                   ((f) & FLG_WARNEXPIRE)
/* Get WarnBoot flag */
#define WARNBOOT(f)                     ((f) & FLG_WARNBOOT)
/* Get ExplainBoot flag */
#define EXPLAINBOOT(f)                  ((f) & FLG_EXPLAINBOOT)
/* Get Lock flag */
#define LOCK(f)                         ((f) & FLG_LOCK)
/* Values for EXPIRE(f) */
#define EXPIRE_DELETE                   0x0000
#define EXPIRE_UNSUBSCRIBE              0x0020

typedef struct
{
    /* Connection type: CONN_TTY/CONN_PPP/CONN_PPPoE */
    int           type[MAX_LOGINS];
    /* PID to kill to kick connection */
    pid_t      	  pid[MAX_LOGINS];
    /* Login time of connection */
    time_t        loginTime[MAX_LOGINS];
    /* Interface to record stats (or terminal if TTY connection) */
    char       	  if_name[MAX_LOGINS][10];
    /* Host this record applys to */
    word          host[MAX_LOGINS];
} _conninfo;


typedef struct
{
    /* UNIX user id */
    uid_t         uid                             __attribute__((packed));
    /* Flags (see FLG_* and *(f) macros) */
    word          flags                           __attribute__((packed));
    /* Area code */
    word          phNoArea                        __attribute__((packed));
    /* Local phone number */
    word          phNoLocal                       __attribute__((packed));
    /* Number of hosts user is logged into */
    byte          nHosts                          __attribute__((packed));
    /* Hosts user is logged into */
    word          host[MAX_HOSTS]                 __attribute__((packed));
    /* Number of logins on each host */
    byte          nLogins[MAX_HOSTS]              __attribute__((packed));
    /* Last login on each host */
    time_t        lastLogin[MAX_HOSTS]            __attribute__((packed));
    /* Last time online for each host */
    time_t        lastOnline[MAX_HOSTS]           __attribute__((packed));
    /* Maximum logins permitted */
    word          maxLogins                       __attribute__((packed));
    /* Maximum number of minutes to deduct per minute */
    word          maxDeduct                       __attribute__((packed));
    /* Idle limit in minutes */
    int           idleLimit                       __attribute__((packed));
    /* PPP user must xfer at least PPPidleBytes in last PPPidleMinutes
     * or be considered idle
     */
    int           PPPidleMinutes                  __attribute__((packed));
    word          PPPidleBytes                    __attribute__((packed));
    /* overall time left */
    int           tLeft                           __attribute__((packed));
    /* overall time limit */
    int           tLimit                          __attribute__((packed));
    /* how much credit to extend the user (how far beyond their time
     * limit they can go)
     */
    int           credit                          __attribute__((packed));
    /* session time left */
    int           sLeft                           __attribute__((packed));
    /* session time limit */
    int           sLimit                          __attribute__((packed));
    /* class time left for each time class */
    int           cLeft[MAX_TIME_CLASSES]         __attribute__((packed));
    /* class time limit for each time class */
    int           cLimit[MAX_TIME_CLASSES]        __attribute__((packed));
    /* overall bytes transmitted (e.g. uploaded) */
    dword         bTx                             __attribute__((packed));
    /* overall bytes received (e.g. downloaded) */
    dword         bRx                             __attribute__((packed));
    /* bytes transmitted limit */
    dword         bTxLimit                        __attribute__((packed));
    /* bytes received limit */
    dword         bRxLimit                        __attribute__((packed));
    /* bytes transferred (xmit + recv) limit */
    dword         bLimit                          __attribute__((packed));
    /* session bytes transmitted */
    dword         bStx                            __attribute__((packed));
    /* session bytes received */
    dword         bSrx                            __attribute__((packed));
    /* session bytes transmitted limit */
    dword         bStxLimit                       __attribute__((packed));
    /* session bytes received limit */
    dword         bSrxLimit                       __attribute__((packed));
    /* session bytes transferred (xmit + recv) limit */
    dword         bSlimit                         __attribute__((packed));
    /* bytes xferred in each of last 60 minutes
     * (used by acua_updated to determine idleness)
     */
    word          nBytes[60]                      __attribute__((packed));
    /* when user was locked */
    time_t        lockDate                        __attribute__((packed));
    /* when user was subscribed */
    time_t        subscrDate                      __attribute__((packed));
    /* When a user is subscribed, data is copied into these fields so that
     * it can be retrieved when the user is unsubscribed.  This allows you
     * to make modifications to a user that will be undone when they are
     * unsubscribed.  e.g. subFlags is a copy of flags, subMaxLogins is
     * a copy of maxLogins, etc...  see acua_subscribe, acua_unSubscribe
     */
    time_t        subExpire                       __attribute__((packed));
    word          subFlags                        __attribute__((packed));
    word          subMaxLogins                    __attribute__((packed));
    word          subMaxDeduct                    __attribute__((packed));
    int           subIdleLimit                    __attribute__((packed));
    int           subPPPidleMinutes               __attribute__((packed));
    word          subPPPidleBytes                 __attribute__((packed));
    int           subTlimit                       __attribute__((packed));
    int           subCredit                       __attribute__((packed));
    int           subSlimit                       __attribute__((packed));
    int           subClimit[MAX_TIME_CLASSES]     __attribute__((packed));
    dword         subBtxLimit                     __attribute__((packed));
    dword         subBrxLimit                     __attribute__((packed));
    dword         subBlimit                       __attribute__((packed));
    dword         subBStxLimit                    __attribute__((packed));
    dword         subBSrxLimit                    __attribute__((packed));
    dword         subBSlimit                      __attribute__((packed));
    /* total minutes online */
    word          tMinutes                        __attribute__((packed));
    /* when account was created */
    time_t        creation                        __attribute__((packed));
    /* expiry date */
    time_t        expire                          __attribute__((packed));
    /* time of last login */
    time_t        loginTime                       __attribute__((packed));
    /* connection info */
    _conninfo	  conninfo                        __attribute__((packed));

} UserRec;      /* sizeof(UserRec) = 885 */

typedef struct
{
    time_t        startTime;
    time_t        expire;
    word          nMinutes[24 * 60];
    word          linesBusy[24 * 60];
    word          sessionLength[24 * 60];
} UsageRec;

typedef struct
{ 
    byte          flags                           __attribute__((packed));
    word          phNoArea                        __attribute__((packed));
    word          phNoLocal                       __attribute__((packed));
    time_t        expire                          __attribute__((packed));
} BanRec;

typedef struct
{
    pid_t         pid;
    uid_t         uid;
    uid_t         euid;
    dev_t         tty;
    char          name[16];
} ProcRec;

typedef struct
{
    uid_t         uid;
    dev_t         tty;
    char          ttyName[13];
    time_t        time;
} LoginRec;

typedef struct
{
    int           startDay;
    int           endDay;
    int           startHour;
    int           startMin;
    int           endHour;
    int           endMin;
} TimeClass;

typedef struct PPPuserRec
{
  uid_t         uid;
  Boolean       root;
  pid_t         pid;
  int           unit;
  dev_t         tty;
  time_t        loginTime;
} PPPuserRec;

typedef struct devStats {
    unsigned long long int  ibytes;
    unsigned long long int  obytes;
} devStats;

/* These are interfaces for most of the acua commands (see acua.cc) */
extern int               acua_addRec(int argc, char **argv);
extern int               acua_ban(int argc, char **argv);
extern int               acua_daysLeft(int argc, char **argv);
extern int               acua_delRec(int argc, char **argv);
extern int               acua_dump(int argc, char **argv);
extern int               acua_dumpStats(int argc, char **argv);
extern int               acua_expire(int argc, char **argv);
extern int               acua_forEach(int argc, char **argv);
extern int               acua_kickUser(int argc, char **argv);
extern int               acua_lock(int argc, char **argv);
extern int               acua_modRec(int argc, char **argv);
extern int               acua_purge(int argc, char **argv);
extern int               acua_renew(int argc, char **argv);
extern int               acua_subscribe(int argc, char **argv);
extern int               acua_sync(int argc, char **argv);
extern int               acua_touch(int argc, char **argv);
extern int               acua_unBan(int argc, char **argv);
extern int               acua_unLock(int argc, char **argv);
extern int               acua_unSubscribe(int argc, char **argv);
extern int               acua_pp(int argc, char **argv);
extern int               acua_timeLeft(int argc, char **argv);

/* Quit with an error message */
extern void              errQuit(char *format,...);
/* Quit with an error message - with syslog support */
extern void              errlQuit(char *format,...);
/* Quit with an OS error message */
extern void              perrQuit(char *format,...);
/* Quit with an OS error message - with syslog support */
extern void              perrlQuit(char *format,...);
/* Get a device number from a device name */
extern dev_t             devNumFromName(char *devName);
/* Get an acua line# from a device number */
extern int               lineNo(dev_t dev);
/* Run a command */
extern int               runCommand(char *path,...);
extern int               runCommandV(char *path, char **arg);
/* Become a daemon */
extern void              daemonInit();
/* Add time to the given time */
extern time_t            addTime(time_t t, char *s);
/* Get a process list */
extern void              procList(int *nProcs, ProcRec *procRec);
/* Determine the maximum number of users to kick off now
 * (in order to return system to non-busy state)
 */
extern int               nKick(int nProcs, ProcRec *procRec);
/* Get a list of logged-in users */
extern void              userList(int *nLogins, LoginRec *loginRec);
/* Last login on *any* host */
extern time_t            lastLogin(UserRec *ur);
/* Last online time on *any* host */
extern time_t            lastOnline(UserRec *ur);
/* Add a host to the list of hosts user is logged into */
extern void              addHost(UserRec *ur, word host);
/* Find index of host in list of hosts user is logged into */
extern int               findHost(UserRec *ur, word host);
/* Remove a host from the list of hosts user is logged into */
extern void              removeHost(UserRec *ur, word host);
/* Get total number of logins on *all* hosts */
extern word              nLogins(UserRec *ur);
/* Get total number of logins on a specific host */
extern word		 nLoginsHost(UserRec *ur, word host);
/* Determine whether a user is logged in */
extern int               userLoggedIn(int nLogins,
                                      LoginRec *loginRec,
                                      uid_t uid);
/* Tokenizer */
extern int               getToken(char *d, char *s, int *idx);
/* Determine if the test string given is all numbers or not */
extern int               isNumber (char *test);
/* Convert a login name to a UNIX user id */
extern uid_t             UIDfromLogin(char *login);
/* Find the group the login name belongs to */
extern gid_t             GIDfromLogin(char *login);
/* Convert a UID into a login name */
extern char             *loginFromUID(char *login, uid_t uid);
/* Open the user file */
extern void              userFileOpen();
/* Close the user file */
extern void              userFileClose();
/* Read a record from the user file */
extern int               userFileRead(UserRec *ur);
/* Write a record to the user file (at current file position) */
extern int               userFileWrite(UserRec *ur);
/* Edit a record in the user file (rewind one record, the write a record) */
extern void              userFileEdit(UserRec *ur);
/* Search the user file for the given uid */
extern int               userFileSearch(UserRec *ur, uid_t uid);
/* Rewind the user file */
extern void              userFileRewind();
/* Open the ban file */
extern void              banFileOpen();
/* Close the ban file */
extern void              banFileClose();
/* Read a ban record */
extern int               banFileRead(BanRec *br);
/* Write a ban record */
extern int               banFileWrite(BanRec *br);
/* Search for a ban on an area/local phone number */
extern int               banFileSearch(BanRec *br,
                                       word phNoArea,
                                       word phNoLocal);
/* Convert a phone number to integers */
extern int               phNo2Words(word *phNoArea,
                                    word *phNoLocal,
                                    char *phNo,
                                    int phNoDigits);
/* Convert integers into a phone number */
extern char             *words2PhNo(char *phNo,
                                    word phNoArea,
                                    word phNoLocal);
/* Determine sum of integers in a string */
extern int               stringSum(char *s);
/* Convert byte total to string (e.g. 1048576 ==> "1 MB") */
extern void              bytes2ASCII(char *str, dword bytes);
/* Determine whether current time falls within the given time class */
extern int               inTimeClass(int tc);
/* Determine the current time class */
extern int               curTimeClass(UserRec *ur);
/* Lock a file */
extern int               fLock(char *path, int operation);
/* Do ACUA pre-processing on a file */
extern void              preprocessFile(UserRec *ur,
                                        char *inPath,
                                        char *outPath);
/* Convert minutes to "hh:mm" string */
extern void              hhmm(char *s, int nMinutes);
/* Find currently active PPP units */
extern void              PPPfindUnits(HashTable *PPPuser,
                                      HashTable *loginRecTable,
                                      HashTable *procRecTable);
/* Get an INET host address from a hostname (e.g. resolve the name) */
extern word              hostAddr(char *hostname);
/* Hash a user ID */
extern word              UIDhash(word size, void *key);
/* Compare user IDs */
extern int               UIDcomp(void *p1, void *p2);
/* Hash a process ID */
extern word              PIDhash(word size, void *key);
/* Compare process IDs */
extern int               PIDcomp(void *p1, void *p2);
/* Hash a TTY */
extern word              TTYhash(word size, void *key);
/* Compare TTYs */
extern int               TTYcomp(void *p1, void *p2);
/* Read the ACUA configuration (acua.config) and use it to set the
 * opt* variables below
 */
extern void              readConfig();
/* Report how many PPP/PPPoE connections this user has. */
extern int		 PPPUserOnline(UserRec *ur);
/* Report how many TTY connections this user has. */
extern int		 TTYUserOnline(UserRec *ur);
/* Connection information search for empty slot */
extern int               conninfoSearchEmpty(UserRec *ur);
/* Clear the conninfo struct, but only the desired slot */
extern void		 conninfoReset(UserRec *ur, int idx);
/* Searches for changes like pppd's gone missing and so on. */
extern int 		 conninfoUpdate(UserRec *ur, word myHostAddr);
/* Reset all entrys in conninfo struct for the specified host. */
extern void		 conninfoResetHost(UserRec *ur, word myHostAddr);
/* Kill a process - sends SIGHUP, SIGTERM, and SIGKILL.  Also has timeouts. */
extern void		 killProcess(pid_t pid);
/* Determine what type of login this is (TTY, PPP, PPPoE) */
extern int		 loginType(void);
/* Figure out the login name from the environment */
extern char             *getLoginName(void);
/* Security checks */
extern int		 securityCheck(int CONN_TYPE, char *login);
/* Get device statistics for ANY type of networking device */
extern int		getDeviceStats(char *ifname, devStats *ds, int dev_pos, int ibytes_pos, int obytes_pos);

extern const int         days[12];
extern char             *configFilePath;
extern char             *userFilePath;
extern FILE             *userFile;
extern char             *banFilePath;
extern FILE             *banFile;
extern int               optPurgeDays;
extern int               optMinDeduct;
extern int               optBusyThreshold;
extern int               optMaxKick;
extern int               optLowCPUpriority,
                         optHighCPUpriority;
extern int               optReturnDelay;
extern int               optGuestTime;
extern int               optGuestPriority;
extern char              optMailHost[64];
extern char              optMailProg[64];
extern int               optMailWait;
extern int               optMailUser;
extern int               optMailGroup;
extern int               optExplainBoot;
extern int               nBootWarnTimes;
extern int               bootWarnTime[MAX_BOOT_WARN_TIMES];
extern int               optPPPWarnBoot;
extern int               nExpireWarnTimes;
extern int               expireWarnTime[MAX_EXPIRE_WARN_TIMES];
extern char              optWarnExpireCC[64];
extern int               optIdleLimit;
extern int               optPPPidleMinutes;
extern int               optPPPidleBytes;
extern int               optSmartTime;
extern int               optSessionSmartTime;
extern int               optTimeClassSmartTime;
extern int               optSmartBoot;
extern int               optSessionSmartBoot;
extern int               optTimeClassSmartBoot;
extern int               optIdleSmartBoot;
extern int               nTimeClasses;
extern TimeClass         timeClass[MAX_TIME_CLASSES];
extern int               nExcluded;
extern uid_t             excluded[64];
extern char              optModemDial[64];
extern char              optPhNoAreaFormat[16];
extern char              optPhNoLocalFormat[16];
extern int               optPhNoDigits;
extern int               optNoRecLogin;
extern int               userFileFD,
			 banFileFD;
extern int               __acua_internal_UserFileOpen;
extern int               nLines;
extern dev_t             lineDev[MAX_USERS];

#endif
