/* aide, Advanced Intrusion Detection Environment
 *
 * Copyright (C) 1999 Rami Lehti
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include "error.h"
#include "db_config.h"
#include "aide.h"
#include "list.h"
#include "db.h"
#include "cipher.h"

list* find_line_match(db_line* line,list* l)
{
  list*r=NULL;
  
  for(r=l;r;r=r->next){
    if(strcmp(line->filename,((db_line*)r->data)->filename)==0){
      return r;
    }
  }
  if(l!=NULL){
    for(r=l->prev;r;r=r->prev){
      if(strcmp(line->filename,((db_line*)r->data)->filename)==0){
	return r;
      }
    }
  }

  return NULL;
}

int compare_md_entries(byte* e1,byte* e2,int len)
{
  if(e1!=NULL && e2!=NULL){
    if(strncmp(e1,e2,len)!=0){
      return RETFAIL;
    }else{
      return RETOK;
    }
  } else {
    /* At least the other is NULL */
    if(e1==NULL && e2==NULL){
      return RETOK;
    }else{
      return RETFAIL;
    }
  }
  return RETFAIL;
}


int compare_lines(db_line* l1,db_line* l2)
{
  
  if ( (DB_SIZEG&l2->attr) && !(DB_SIZE&l2->attr) ){
    if(l1->size>l2->size){
      return RETFAIL;
    }
  } else {
    if(l1->size!=l2->size){
      return RETFAIL;
    }
  }
  if(l1->perm!=l2->perm){
    return RETFAIL;
  }
  if(l1->uid!=l2->uid){
    return RETFAIL;
  }
  if(l1->gid!=l2->gid){
    return RETFAIL;
  }
  if(l1->atime!=l2->atime){
    return RETFAIL;
  }
  if(l1->mtime!=l2->mtime){
    return RETFAIL;
  }
  if(l1->ctime!=l2->ctime){
    return RETFAIL;
  }
  if(l1->inode!=l2->inode){
    return RETFAIL;
  }
  if(l1->nlink!=l2->nlink){
    return RETFAIL;
  }

  if(compare_md_entries(l1->md5,l2->md5,
			md_digest_length(DIGEST_ALGO_MD5))==RETFAIL){
    return RETFAIL;
  }
  if(compare_md_entries(l1->sha1,l2->sha1,
			md_digest_length(DIGEST_ALGO_SHA1))==RETFAIL){
    return RETFAIL;
  }
  if(compare_md_entries(l1->rmd160,l2->rmd160,
			md_digest_length(DIGEST_ALGO_RMD160))==RETFAIL){
    return RETFAIL;
  }
  if(compare_md_entries(l1->tiger,l2->tiger,
			md_digest_length(DIGEST_ALGO_TIGER))==RETFAIL){
    return RETFAIL;
  }

  return RETOK;
}

void print_md_changes(byte*old,byte*new,int len,char* fnam)
{
  if(old!=NULL && new!=NULL){
    if(strncmp(old,new,len)!=0){
      error(20,"%s: old = %s , new = %s\n",
	    fnam,
	    encode_base64(old,len),
	    encode_base64(new,len));
    }
  }else {
    if(old == NULL && new == NULL){
      return;
    }
    error(20,"%s: old = ",fnam);
    if(old==NULL){
      error(20,"0");
    } else {
      error(20,"%s",encode_base64(old,len));
    }
    error(20," new = ");
    if(new==NULL){
      error(20,"0");
    }else {
      error(20,"%s",encode_base64(new,len));
    }
    error(20,"\n");
  }
}

void print_dbline_changes(db_line* old,db_line* new)
{
  error(20,"file: %s\n",old->filename);

  if(old->size!=new->size){
    error(20,"Size: old = %i , new = %i\n",
	  old->size,new->size);
  }
  if(old->perm!=new->perm){
    error(20,"Permissions: old = %i , new = %i\n",
	  old->perm,new->perm);
  }
  if(old->uid!=new->uid){
    error(20,"Uid: old = %i , new = %i\n",
	  old->uid,new->uid);
  }
  if(old->gid!=new->gid){
    error(20,"Gid: old = %i , new = %i\n",
	  old->gid,new->gid);
  }
  if(old->atime!=new->atime){
    error(20,"Atime: old = %i , new = %i\n",
	  old->atime,new->atime);
  }
  if(old->mtime!=new->mtime){
    error(20,"Mtime: old = %i , new = %i\n",
	  old->mtime,new->mtime);
  }
  if(old->ctime!=new->ctime){
    error(20,"Ctime: old = %i , new = %i\n",
	  old->ctime,new->ctime);
  }
  if(old->inode!=new->inode){
    error(20,"Inode: old = %i , new = %i\n",
	  old->inode,new->inode);
  }
  if(old->nlink!=new->nlink){
    error(20,"Linkcount: old = %i , new = %i\n",
	  old->nlink,new->nlink);
  }

  print_md_changes(old->md5,new->md5,
		   md_digest_length(DIGEST_ALGO_MD5),"MD5");

  print_md_changes(old->sha1,new->sha1,
		   md_digest_length(DIGEST_ALGO_SHA1),"SHA1");

  print_md_changes(old->rmd160,new->rmd160,
		   md_digest_length(DIGEST_ALGO_RMD160),"RMD160");

  print_md_changes(old->tiger,new->tiger,
		   md_digest_length(DIGEST_ALGO_TIGER),"TIGER");
  

}

void free_db_line(db_line* dl)
{
  free(dl->md5);
  free(dl->sha1);
  free(dl->rmd160);
  free(dl->tiger);
  free(dl->filename);

}

int compare_db(list* new)
{
  db_line* old=NULL;
  list* l=new;
  list* r=NULL;
  list* removed=NULL;
  list* changednew=NULL;
  list* changedold=NULL;
  list* added=NULL;
  long nrem=0;
  long nchg=0;
  long nadd=0;
  long nfil=0;
  
  error(200,"compare_db()\n");

  for(old=db_readline(conf);old;old=db_readline(conf)){
    nfil++;
    r=find_line_match(old,l);
    if(r==NULL){
      removed=list_append(removed,(void*)old);
      nrem++;
    }else {
      if(compare_lines(old,(db_line*)r->data)==RETFAIL){
	changednew=list_append(changednew,r->data);
	changedold=list_append(changedold,(void*)old);
	r->data=NULL;
	l=list_delete_item(r);
	nchg++;
      }else {
	/* This line was ok */
	free_db_line(old);
	free_db_line((db_line*)r->data);
	r->data=NULL;
	l=list_delete_item(r);
      }
    }
    
  }
  /* Now we have checked the old database and removed the lines *
   * that have matched. */
  if(l!=NULL){
    added=l->head;
  }else {
    added=NULL;
  }
  
  for(l=added;l;l=l->next){
    nadd++;
  }


  if(nadd!=0||nrem!=0||nchg!=0){
    error(0,"Summary:\nTotal number of files=%i,added files=%i"
	  ",removed files=%i,changed files=%i\n\n",nfil,nadd,nrem,nchg);
    if(nrem!=0){
      error(5,"Removed files:\n");
      for(r=removed;r;r=r->next){
	error(5,"removed:%s\n",((db_line*)r->data)->filename);
      }
    }
    if(nchg!=0){
      error(5,"Changed files:\n");
      for(r=changedold;r;r=r->next){
	error(5,"changed:%s\n",((db_line*)r->data)->filename);
      }
    }

    if(nadd!=0){
      error(5,"Added files:\n");
      for(r=added;r;r=r->next){
	error(5,"added:%s\n",((db_line*)r->data)->filename);
      }
    }
    
    if(conf->verbose_level>5){
      error(20,"Detailed information about changes:\n");
      for(r=changedold,l=changednew;r;r=r->next,l=l->next){
	print_dbline_changes((db_line*)r->data,(db_line*)l->data);
      }
    }
  }
}
