/* aide, Advanced Intrusion Detection Environment
 *
 * Copyright (C) 1999 Rami Lehti, Pablo Virolainen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "error.h"
#include "db_config.h"
#include "util.h"

#define URL_UNSAFE " <>\"#%{}|\\^~[]`@:\033"

static const char* url_name[] = { 
  "file", "stdin", "stdout", "stderr", "fd"};

static const int url_value[] = {
  url_file, url_stdin, url_stdout,url_stderr,url_fd };

const int url_ntypes=sizeof(url_value)/sizeof(int);

int cmpurl(url_t* u1,url_t* u2)
{
  if(u1->type!= u2->type){
    return RETFAIL;
  };
  if(strcmp(u1->value,u2->value)!=0){
    return RETFAIL;
  }

  return RETOK;
};

url_t* parse_url(char* val)
{
  url_t* u=NULL;
  char* r=NULL;
  int i=0;

  if(val==NULL)
    return NULL;

  u=(url_t*)malloc(sizeof(url_t));

  for(r=val;r[0]!=':'&&r[0]!='\0';r++);
  
  if(r[0]=='\0'){
    u->value="";
  }
  else{
    int i=0;
    r[0]='\0';
    r++;
    for(i=0;r[0]=='/'&&i<2;r++,i++);
    u->value=strdup(r);
  }
  u->type=url_unknown;
  for(i=0;i<url_ntypes;i++){
    if(strcmp(val,url_name[i])==0){
      u->type=url_value[i];
    }
  }

  if(u->type==url_unknown){
    error(20,"Unknown URL-type:%s\n",val);
  }

  return u;
}

/* Returns 1 if the string contains unsafe characters, 0 otherwise.  */
int contains_unsafe (const char *s)
{
  for (; *s; s++)
    if (strchr (URL_UNSAFE,(int) *s)||!isprint((int)*s))
      return 1;
  return 0;
}
 
/* Decodes the forms %xy in a URL to the character the hexadecimal
   code of which is xy.  xy are hexadecimal digits from
   [0123456789ABCDEF] (case-insensitive).  If x or y are not
   hex-digits or `%' precedes `\0', the sequence is inserted
   literally.  */
 
void decode_string (char* s)
{
  char *p = s;
 
  for (; *s; s++, p++)
    {
      if (*s != '%')
        *p = *s;
      else
        {
          /* Do nothing if at the end of the string, or if the chars
             are not hex-digits.  */
          if (!*(s + 1) || !*(s + 2)
              || !(ISXDIGIT (*(s + 1)) && ISXDIGIT (*(s + 2))))
            {
              *p = *s;
              continue;
            }
          *p = (ASC2HEXD (*(s + 1)) << 4) + ASC2HEXD (*(s + 2));
          s += 2;
        }
    }
  *p = '\0';
}
 
/* Encodes the unsafe characters (listed in URL_UNSAFE) in a given
   string, returning a malloc-ed %XX encoded string.  */
char* encode_string (const char* s)
{
  const char *b;
  char *p, *res;
  int i;
 
  b = s;
  for (i = 0; *s; s++, i++){
    if (strchr (URL_UNSAFE,(int) *s)||!isprint((int)*s)){
      i += 2; /* Two more characters (hex digits) */
    }
  }

  res = (char *)malloc (i + 1);
  s = b;
  for (p = res; *s; s++){
    if (strchr (URL_UNSAFE, *s)||!isprint(*s))
      {
        const unsigned char c = *s;
        *p++ = '%';
        *p++ = HEXD2ASC (c >> 4);
        *p++ = HEXD2ASC (c & 0xf);
      }
    else {
      *p++ = *s;
    }
  }
  *p = '\0';
  return res;
}
