/*-
 *
 * Audit Daemon for Linux (v1.11) 
 * Markus Wolf <klog@hert.org>, Promisc Security
 *
 * Copyright (C) 1999 Hacker Emergency Response Team
 * http://www.hert.org
 *
 * This file is part of Audit Daemon
 * 
 * Audit Daemon is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * Audit Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 * 
 */

#include <stdarg.h>

#include <asm/segment.h>
#include <asm/system.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/tty_driver.h>

#define AUDIT_BUF_LEN     1024 

static char buf[1024];
static char audit_buf[AUDIT_BUF_LEN];
static unsigned long audit_start = 0;
static unsigned long audit_chars = 0;
unsigned long audit_size = 0;                    
struct wait_queue *audit_wait = NULL;

asmlinkage int sys_audit(char *buf, int len)
{
        unsigned long i;
        char c;
        int error;

        if (!suser())
                return -EPERM;
        if (!buf || len < 0)
                return -EINVAL;
        if (!len)
                return 0;
        error = verify_area(VERIFY_WRITE,buf,len);
        if (error)
                return error;
        cli();
        while (!audit_size) {
                if (current->signal & ~current->blocked) {
                        sti();
                        return -ERESTARTSYS;
                }
                interruptible_sleep_on(&audit_wait);
        }
        i = 0;
        while (audit_size && i < len) {
        	c = *((char *) audit_buf+audit_start);
        	audit_start++;
        	audit_size--;
        	audit_start &= AUDIT_BUF_LEN-1;
        	sti();
        	put_user(c,buf);
                buf++;
                i++;
                cli();
        }
        sti();
        return i;
}

asmlinkage int audit(const char *fmt, ...)
{
        va_list args;
        int i;
        char *p, *buf_end;
        long flags;

        save_flags(flags);
        cli();
        va_start(args, fmt);
	
        i = vsprintf(buf, fmt, args); 
        buf_end = buf + i;
        va_end(args);

        for (p=buf; p < buf_end; p++) {
                audit_buf[(audit_start+audit_size) & (AUDIT_BUF_LEN-1)] = *p;
                if (audit_size < AUDIT_BUF_LEN)
                        audit_size++;
                else {
                        audit_start++;
                        audit_start &= AUDIT_BUF_LEN-1;
                }
                audit_chars++;
                if (*p == '\n')
                        break;
        }

        restore_flags(flags);
        wake_up_interruptible(&audit_wait);
        return i;
}
