/*-
 *
 * Audit Daemon for Linux (v1.11)
 *
 * Copyright (C) 1999 Hacker Emergency Response Team
 * Markus Wolf <klog@hert.org>, Promisc Security 
 *
 * This file is part of Audit Daemon
 * 
 * Audit Daemon is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * Audit Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 *
 */


#include <stdio.h> 
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <grp.h>
#include "../audit.h" 

#define ISFILE(x)  (x == AUDIT_OPEN) || \
                   (x == AUDIT_EXEC) || \
                   (x == AUDIT_MODINIT)

#define PATHLEN 1024
#define ULAND 0
#define KLAND 1

int usetime;
int debug;
int logpos;

char myself[512];

void logline(char *buf, int src)
{
	int flag, pid, uid;
	char pname[17];
	char info[1024];
	char file[PATHLEN];
	char tmp[20];
	char *date;
	time_t t;
	long temppos;
	FILE *logfd;
	char *sources[] = {"process", "kernel"};

	logpos = 0;

        flag = getunit(buf, tmp, sizeof(tmp));
        getunit(buf, pname, sizeof(pname));
	pid = getunit(buf, tmp, sizeof(tmp));
	uid = getunit(buf, tmp, sizeof(tmp));
	temppos = logpos;
	if (ISFILE(flag)) getunit(buf, file, sizeof(file));
	logpos = temppos;

        /*-
         * This definitely suck as coding behavior but we dont want
         * processes (even trusted) spoofing system call trails
         */

	if (src == ULAND) flag = AUDIT_DEBUG; 
	if ((flag < 'a')||(flag > 'z')) return; /* crapped */

	logfd = (FILE *)get_logfd(flag, pname, pid, uid, file);

	if (debug&&logfd) logfd = stdout;
	if (logfd == NULL) return;

	if (usetime) {
		t = time(NULL);
		date = ctime(&t);
		date[strlen(date)-5] = 0; /* remove the year */
		fprintf(logfd, "%s", date);
	}

	fprintf(logfd, "%s[%d]: ", pname, uid);

	switch(flag) {
	    case AUDIT_CONNECT:
		log_socket(logfd, buf, "connect", "-->", 1); break;
            case AUDIT_ACCEPT:
		log_socket(logfd, buf, "accept", "<--", 0); break;
            case AUDIT_LISTEN:
		log_socket(logfd, buf, "listen", NULL, 0); break;
	    case AUDIT_OPEN:
		log_file(logfd, buf, "open"); break;
	    case AUDIT_SETUID:
		log_setuid(logfd, buf, "setuid"); break;
	    case AUDIT_EXEC:
		log_file(logfd, buf, "exec"); break;
	    case AUDIT_MODINIT:
		log_file(logfd, buf, "modinit"); break;
	    case AUDIT_DEBUG:
		log_debug(logfd, buf, sources[src]); break;
	}

	fflush(logfd);
}

process_trail(int kaffd, int sockfd)
{
	char buf[1200];
	fd_set entries;

	FD_ZERO(&entries);
	bzero(&buf, sizeof(buf));

	FD_SET(kaffd, &entries);
	FD_SET(sockfd, &entries);

        if (select(kaffd+1, &entries, NULL, NULL, NULL) <0)
		die("select");
 
       	if (FD_ISSET(kaffd, &entries)) {
               	getline(kaffd, buf, sizeof(buf));
		logline(buf, KLAND);
	} else {
		recv(sockfd, buf, sizeof(buf), 0);
		logline(buf, ULAND);
	}

}

int init_sock()
{
	struct sockaddr addr;
	int sockfd;
	struct group *auditgrp;

	auditgrp = getgrnam(AUDITGRP);
	if (!auditgrp) die(AUDITGRP);

        addr.sa_family = AF_UNIX;
        strncpy(&addr.sa_data, SOCKPATH, sizeof(addr.sa_data));

        sockfd = socket(AF_UNIX, SOCK_DGRAM, 0);
        if (sockfd <0) die("socket");
        if (bind(sockfd, &addr, sizeof(addr)) < 0) die(SOCKPATH);

        chown(SOCKPATH, 0, auditgrp->gr_gid);
	chmod(SOCKPATH, 0660);

	return sockfd;
}

int init_kaf()
{
	int kaffd;

        kaffd = open(AUDIT_SRC_PATH, O_RDONLY);
        if (kaffd < 0) {
		/* TODO: modload modkaf? */
		die(AUDIT_SRC_PATH);
	}

	return kaffd;
}

help()
{
	printf("usage: auditd [-t] [-d] [-h]\n");
	printf("       -t     use timestamps\n");
	printf("       -d     debug mode\n");
	printf("       -h     help\n");
	exit(0);
}

main(int argc, char *argv[])
{
	int kaffd, sockfd;
	int opt; 
	
	strncpy(myself, argv[0], sizeof(myself));

	while(1) {
		opt = getopt(argc, argv, "tdh");
		if (opt<0) break; 
		switch(opt) {
		    case 't': usetime = 1; break;
		    case 'd': debug = 1; break;
		    case 'h':
		    default: help();
		}
	}

	init_sigs();
	init_conf();
	sockfd = init_sock();
	kaffd = init_kaf();

	if (!debug) {
		if (!fork()) while(1) process_trail(kaffd, sockfd);
	} else {
		while(1) process_trail(kaffd, sockfd);
	}

	exit(0);
}


