# 
# Case Management methods, including the windows and functions to read the 
# config files
#
# ver 2.00+
# Brian Carrier [carrier@sleuthkit.org]
# Copyright (c) 2003-2004 by Brian Carrier.  All rights reserved
#
# This file is part of the Autopsy Forensic Browser (Autopsy)
#
# Autopsy is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Autopsy is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Autopsy; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
# IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


package Caseman;

# If the order of these views are changed, then the order of the main
# function may have to be as well

# Case Views
$Caseman::CASE_NEW = 1;
$Caseman::CASE_NEW_DOIT = 2;
$Caseman::CASE_OPEN = 3;
$Caseman::CASE_OPEN_LOG = 4;
$Caseman::CASE_DETAILS = 5;
# $Caseman::CASE_DEL = 6;
my $CASE_MAX = 5;


# Host Views
$Caseman::HOST_ADD = 7;
$Caseman::HOST_ADD_DOIT = 8;
$Caseman::HOST_OPEN = 9;
$Caseman::HOST_OPEN_LOG = 10;
$Caseman::HOST_DETAILS = 11;
# $Caseman::HOST_DEL = 12;
my $HOST_MAX = 11;

# Image Views
$Caseman::IMG_ADD = 13;
$Caseman::IMG_ADD_DOIT = 14;
$Caseman::IMG_OPEN = 15;
$Caseman::IMG_OPEN_LOG = 16;
$Caseman::IMG_DETAILS = 17;
$Caseman::IMG_DEL = 18;
$Caseman::IMG_MAKESTR = 19;
$Caseman::IMG_MAKEDLS = 20;
my $IMG_MAX = 20;



# Module Variables
# %img2mnt;   # Mapping for mount point, given the image name
# %img2ftype; # Mapping for file system type, given the image name
# %mod2img;   # Mapping for image, given the strings or dls
# %img2str;   # Mapping for strings file, given  the image name
# %img2dls;   # Mapping for dls file, given  the image name


sub main {


    # By default, show the case open window
    $Args::args{'view'} = $Args::enc_args{'view'} = $Caseman::CASE_OPEN
          unless (exists $Args::args{'view'});

    Args::check_view();    
    my $view = Args::get_view();

	# The only live function is for the open img
	if ($::LIVE == 1) {
		Args::check_inv();
		if ($view == $Caseman::IMG_OPEN) {
			return img_open();	
		}

		Args::check_img('img');
		Args::check_ftype();
		Args::check_mnt();

		if ($view == $Caseman::IMG_OPEN_LOG) {
			return img_open_log();	
		}	
		else {
			Print::print_check_err("Invalid Live Analysis Case Management View");
		}
		return 0;
	}


    # Case functions
    if ($view <= $CASE_MAX) {
        if ($view == $Caseman::CASE_OPEN) {
            return case_open();	
        }
        elsif ($view == $Caseman::CASE_NEW) {
            return case_new();	
        }
    
        Args::check_case();
        $::case_dir = "$::LOCKDIR/".Args::get_case()."/";
		$::case_dir =~ s/\/\//\//g;
    
        if ($view == $Caseman::CASE_OPEN_LOG) {
            return case_open_log();	
        }
        elsif ($view == $Caseman::CASE_NEW_DOIT) {
            return case_new_doit();	
        }
        elsif ($view == $Caseman::CASE_DETAILS) {
            return case_details();	
        }
    }

    Args::check_case();
    $::case_dir = "$::LOCKDIR/".Args::get_case()."/";
	$::case_dir =~ s/\/\//\//g;
    
    # Host functions
    if ($view <= $HOST_MAX) {
        if ($view == $Caseman::HOST_OPEN) {
            return host_open();	
        }
        elsif ($view == $Caseman::HOST_ADD) {
            return host_add();	
        }

        Args::check_host();
        $::host_dir = "$::case_dir".Args::get_host()."/";
		$::host_dir =~ s/\/\//\//g;
        if ($view == $Caseman::HOST_ADD_DOIT) {
            return host_add_doit();	
        }

        Caseman::read_host_config();
        if ($view == $Caseman::HOST_OPEN_LOG) {
            return host_open_log();	
        }
        elsif ($view == $Caseman::HOST_DETAILS) {
            return host_details();	
        }
    }

    Args::check_host();
    $::host_dir = "$::case_dir".Args::get_host()."/";
	$::host_dir =~ s/\/\//\//g;
    Caseman::read_host_config();
    Args::check_inv();
    
    if ($view <= $IMG_MAX) {
        if ($view == $Caseman::IMG_OPEN) {
            return img_open();	
        }
        elsif ($view == $Caseman::IMG_ADD) {
            return img_add();	
        }
        elsif ($view == $Caseman::IMG_ADD_DOIT) {
            return img_add_doit();	
        }

    
        Args::check_img('img');
        Args::check_ftype();
        Args::check_mnt();

        if ($view == $Caseman::IMG_OPEN_LOG) {
            return img_open_log();	
        }	
        elsif ($view == $Caseman::IMG_DETAILS) {
            return img_details();	
        }
#       elsif ($view == $Caseman::IMG_DEL) {
# 		return img_del();	
# 	}
        elsif ($view == $Caseman::IMG_MAKESTR) {
            return img_makestr();	
        }
        elsif ($view == $Caseman::IMG_MAKEDLS) {
            return img_makedls();	
        }
    }
    
    Print::print_check_err("Invalid Case Management View");
}



####################################################################
# General menu Functions

sub print_menu_tabs {

	print "<b>Case:</b> $Args::args{'case'}" if (exists $Args::args{'case'});
	print "<br>\n";
	print "<b>Host:</b> $Args::args{'host'}" if (exists $Args::args{'host'});
	print "<br>\n";

	print "<center>\n";

	if ($::LIVE == 1) {
		print "<h2>Live Analysis Mode</h2>\n";
	}

	print "<table width=\"600\" height=\"60\" background=\"$::YEL_PIX\" ".
	  "border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n<tr>\n".
	  "<td align=\"center\" width=\"200\">";
        
        my $view = Args::get_view();
        
	# Case Gallery Tab
	if ($view == $Caseman::CASE_OPEN) {
		print "<img border=0 src=\"pict/menu_t_cg_cur.jpg\" ".
		  "width=200 height=65 alt=\"Case Gallery (Current Mode)\">\n";
	}
	elsif ($::LIVE == 1) {
		print "<img border=0 src=\"pict/menu_t_cg_org.jpg\" ".
			"width=200 height=65 alt=\"Case Gallery\">\n";
	}
	else {
		print "<a href=\"$::PROGNAME?".
		  "mod=$::MOD_CASEMAN&view=$Caseman::CASE_OPEN\">".
		  "<img border=0 src=\"pict/menu_t_cg_link.jpg\" ".
		  "width=200 height=65 alt=\"Case Gallery\"></a>\n";
	}
	print "</td>\n".
	  "<td align=\"center\" width=\"200\">";

	# Host Gallery Tab
	# Current
	if ($view == $Caseman::HOST_OPEN) {
		print "<img border=0 src=\"pict/menu_t_hg_cur.jpg\" ".
		  "width=200 height=65 alt=\"Host Gallery (Current Mode)\">\n";
	}
	# Link
	elsif (($view == $Caseman::IMG_OPEN) && ($::LIVE == 0)) {
		print "<a href=\"$::PROGNAME?".
		  "mod=$::MOD_CASEMAN&view=$Caseman::HOST_OPEN".
		  "&case=$Args::args{'case'}\">".
		  "<img border=0 src=\"pict/menu_t_hg_link.jpg\" ".
		  "width=200 height=65 alt=\"Host Gallery\"></a>\n";
	}
	# Non-link
	else {
		print "<img border=0 src=\"pict/menu_t_hg_org.jpg\" ".
		  "width=200 height=65 alt=\"Host Gallery (Not Available)\">\n";
	}

	print "</td>\n".
	  "<td align=\"center\" width=\"200\">";

	# Host Manager Tab
	# Current
	if ($view == $Caseman::IMG_OPEN) {
            print "<img border=0 src=\"pict/menu_t_hm_cur.jpg\" ".
              "width=200 height=65 alt=\"Host Manager (Current Mode)\">\n";
	}
	# non-link
	else {
            print "<img border=0 src=\"pict/menu_t_hm_org.jpg\" ".
              "width=200 height=65 alt=\"Host Manager (Not Available)\">\n";
	}

	print "</td>\n</tr>\n".
	  "</table>\n";
};





####################################################################
# Case Functions

# if no args are passed, return case config using args{'case'},
# else use the case value passed
#
# Case config:
# In case directory with case_name.case       
sub case_config_fname {
    if (scalar (@_) == 1) {
        my $c = shift; 
        return "$::LOCKDIR/"."$c/case.aut";
    } else {
        return "$::LOCKDIR/"."$Args::args{'case'}/case.aut";
    }
}


# Read case config and save it to $Caseman::cvals
sub read_case_config {
	return if ($::LIVE == 1);

	my $case;

	if (scalar (@_) == 1) {
		$case = shift;
	} else {
		$case = Args::get_case();
	}

	my $fname = case_config_fname($case);
        
	%Caseman::cvals = ();
        
	open CONFIG, "<$fname" 
	  or die "Can't open case config file ($fname)";

	while (<CONFIG>) {
        next if ((/^\#/) || (/^\s+$/));
		s/^\s+//;
		s/\s+$//;
		$Caseman::cvals{$1} = $2 if (/^(\S+)\s+(.*)$/);
	}
	close (CONFIG);

	$Caseman::cvals{'desc'} = "None Provided" 
	  unless (exists $Caseman::cvals{'desc'});

	$Caseman::cvals{'created'} = "unknown" 
	  unless (exists $Caseman::cvals{'created'});
};


sub case_new {
    Print::print_html_header("Create A New Case");

print <<EOF;
<br>
<br>
<center>
<img src=\"pict/menu_h_cnew.jpg\" alt=\"New Case\">
<br><br><br>

<table width=\"600\" background=\"$::YEL_PIX\" cellspacing=\"0\"
  cellpadding=\"2\" border=0>
<form action=\"$::PROGNAME\" method=\"get\">
<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">
<input type=\"hidden\" name=\"view\" value=\"$Caseman::CASE_NEW_DOIT\">
<tr>
  <td colspan=3 align=left>
    1.  <b>Case Name:</b> The name of this investigation.  It can only contain letters, numbers, and symbols.
  </td>
</tr>
<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=2><input type=\"text\" name=\"case\"></td>
</tr>

<tr><td colspan=3>&nbsp;</td></tr>

<tr>
  <td colspan=3 align=left>
    2.  <b>Description:</b> An optional, one line description of this case.
  </td>
</tr>
<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=2><input type=\"text\" name=\"desc\" size=32 maxlength=32></td>
</tr>

<tr><td colspan=3>&nbsp;</td></tr>

<tr>
  <td colspan=3 align=left>
    3.  <b>Investigator Names:</b> The optional names (with no spaces) of the investigators for this case.
  </td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td align=left><tt>a.</tt> <input type=\"text\" name=\"inv1\"></td>
  <td align=left><tt>b.</tt> <input type=\"text\" name=\"inv2\"></td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td align=left><tt>c.</tt> <input type=\"text\" name=\"inv3\"></td>
  <td align=left><tt>d.</tt> <input type=\"text\" name=\"inv4\"></td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td align=left><tt>e.</tt> <input type=\"text\" name=\"inv5\"></td>
  <td align=left><tt>f.</tt> <input type=\"text\" name=\"inv6\"></td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td align=left><tt>g.</tt> <input type=\"text\" name=\"inv7\"></td>
  <td align=left><tt>h.</tt> <input type=\"text\" name=\"inv8\"></td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td align=left><tt>i.</tt> <input type=\"text\" name=\"inv9\"></td>
  <td align=left><tt>j.</tt> <input type=\"text\" name=\"inv10\"></td>
</tr>
</table>

<br><br>
<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">
<tr>
  <td align=center>
    <input type=\"image\" src=\"pict/menu_b_cnew.jpg\" 
      alt=\"Create Case\" width=\"176\" height=20 border=0>
  </td>
</form>
  <td align=center>
    <form action=\"$::PROGNAME\" method=\"get\">
    <input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">
    <input type=\"hidden\" name=\"view\" value=\"$Caseman::CASE_OPEN\">
    <input type=\"image\" src=\"pict/menu_b_cancel.jpg\" 
    alt=\"Cancel\" width=\"167\" height=20 border=0>
    </form>
  </td>
  <td align=center><a href=\"$::HELP_URL\" 
    target=\"_blank\">
    <img src=\"pict/menu_b_help.jpg\" alt=\"Help\" 
    width=\"167\" height=20 border=0></a>
  </td>
</tr>
</table>
EOF

    Print::print_html_footer();
	return;
};


# Create the directory and case configuration file
# Gets the input from CASE_NEW
sub case_new_doit {
	Print::print_html_header("Creating Case: $Args::args{'case'}");

	print "<h3>Creating Case: <tt>$Args::args{'case'}</tt></h3>\n";
	# Make the directory
	if (-d "$::case_dir") {
                Print::print_err("Error: $::case_dir already exists<br>".
                  "Please remove the directory and its contents and try again".        
                  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                  "view=$Caseman::CASE_OPEN\">".
                  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
                  "width=\"43\" height=20 border=\"0\"></a>\n");	
	}

	unless (mkdir "$::case_dir", $::MKDIR_MASK) {
	  Print::print_err ("Error making directory $::case_dir: $!");
	}

	print "Case directory (<tt>$::case_dir</tt>) created<br>\n";
	Print::log_case_info ("Case $Args::args{'case'} created");


	my $fname = Caseman::case_config_fname();

	open CASE_CONFIG, ">$fname" or die "Can't open case config: $fname";

	print CASE_CONFIG "# Autopsy case config file\n".
	  "# Case: $Args::args{'case'}\n\n".
	  "created ".localtime()."\n";

	if ((exists $Args::args{'desc'}) && ($Args::args{'desc'} ne "")){
		Print::print_err ("Invalid Description<br>\n".
                  "Use the browser's back button to fix") 
                  if ($Args::args{'desc'} =~ /\n/);

		print CASE_CONFIG "desc $Args::args{'desc'}\n";
	}
	print CASE_CONFIG "images	$::IMGDIR\n";
	print CASE_CONFIG "data		$::DATADIR\n";
	print CASE_CONFIG "log		$::LOGDIR\n";
	print CASE_CONFIG "reports	$::REPDIR\n";

	close CASE_CONFIG;
	print "Configuration file (<tt>$fname</tt>) created<br>\n";

	write_invest ($Args::args{'inv1'}) 
	  if ((exists $Args::args{'inv1'}) && ($Args::args{'inv1'} ne ""));
	write_invest ($Args::args{'inv2'}) 
	  if ((exists $Args::args{'inv2'}) && ($Args::args{'inv2'} ne ""));
	write_invest ($Args::args{'inv3'}) 
	  if ((exists $Args::args{'inv3'}) && ($Args::args{'inv3'} ne ""));
	write_invest ($Args::args{'inv4'}) 
	  if ((exists $Args::args{'inv4'}) && ($Args::args{'inv4'} ne ""));
	write_invest ($Args::args{'inv5'}) 
	  if ((exists $Args::args{'inv5'}) && ($Args::args{'inv5'} ne ""));
	write_invest ($Args::args{'inv6'}) 
	  if ((exists $Args::args{'inv6'}) && ($Args::args{'inv6'} ne ""));
	write_invest ($Args::args{'inv7'}) 
	  if ((exists $Args::args{'inv7'}) && ($Args::args{'inv7'} ne ""));
	write_invest ($Args::args{'inv8'}) 
	  if ((exists $Args::args{'inv8'}) && ($Args::args{'inv8'} ne ""));
	write_invest ($Args::args{'inv9'}) 
	  if ((exists $Args::args{'inv9'}) && ($Args::args{'inv9'} ne ""));
	write_invest ($Args::args{'inv10'}) 
	  if ((exists $Args::args{'inv10'}) && ($Args::args{'inv10'} ne ""));

	print "Investigators added<br>\n";

	Print::log_session_info ("Case $Args::args{'case'} created");

	print "<br><br><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
          "view=$Caseman::CASE_OPEN\">".
	  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
	  "width=\"43\" height=20 border=\"0\"></a>\n";

    Print::print_html_footer();
	return;
};


# Open a Case
# This provides a form with a list of options
sub case_open {
	Print::print_html_header("Open A Case");

	print_menu_tabs();

	print "<table width=\"600\" background=\"$::YEL_PIX\" ".
	  " cellspacing=\"0\" cellpadding=\"2\" border=0>\n";


	# Read the directories of the Evidence Locker into an array
	# Verify that there is a config file in the directory
	my @cases;
    opendir CASES, $::LOCKDIR or die "Can't open $::LOCKDIR directory: $!";
	foreach my $c (readdir CASES) {
        next if (($c eq '.') || ($c eq '..'));
		my $cfile = Caseman::case_config_fname($c);

		push @cases, $c 
		  if ((-d "$::LOCKDIR/$c") && (-e "$cfile"));
	}

	closedir CASES;

	# Were there any cases?  
	if (scalar @cases == 0) {
            print "<tr><td align=\"center\">".
              "No cases exist in <tt>$::LOCKDIR</tt>".
              "</td></tr>\n";
	}
	else {
            print "<form action=\"$::PROGNAME\" method=\"get\">\n".
              "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
              "<input type=\"hidden\" name=\"view\" value=\"$Caseman::CASE_OPEN_LOG\">\n".
              Args::make_hidden().
              "<tr><th>Name</th>".
              "<th>Description</th>".
              "<td>&nbsp;</td></tr>\n";

            my $first = 0;
            foreach my $c (@cases) {

                print "<tr><td align=\"left\">".
                    "<input type=\"radio\" name=\"case\" value=$c";
                if ($first == 0) {
                    print " CHECKED";
                    $first = 1;
                }
                print ">$c</td>";

                Caseman::read_case_config($c);

                print "<td>$Caseman::cvals{'desc'}</td>".
                  "<td align=center>".
                  "<a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                  "view=$Caseman::CASE_DETAILS&case=$c\">".
                  "details</a></td>".
                  "</tr>\n";
            }
	}
	print "</table>\n";

	print "<br><br>".
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">\n".
	  "<tr>\n";

	# Print the OK button if there were cases
	if (scalar @cases != 0) {
		  print "<td align=center>".
			"<input type=\"image\" src=\"pict/menu_b_ok.jpg\" ".
			"width=167 height=20 alt=\"Ok\" border=0>".
			"</form></td>\n\n";
	}

	# Print a 'New Case' Button
	print "<td align=center>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::CASE_NEW\">\n".
	  "<input type=\"image\" src=\"pict/menu_b_cnew.jpg\" ".
	  "width=167 height=20 alt=\"New Case\" border=0>\n".
	  "</form></td>".

	# Print a Menu Button
	  "<td align=center>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
	  "<input type=\"image\" src=\"pict/menu_b_menu.jpg\" ".
	  "width=167 height=20 alt=\"Main Menu\" border=0>\n".
	  "</form></td></tr></table>\n";

	print 
	  "<table width=600 cellspacing=\"0\" cellpadding=\"2\">\n<tr>".
	  "<td>&nbsp;</td>\n".
	  "<td align=center width=200><a href=\"$::HELP_URL\" ".
	  " target=\"_blank\">".
	  "<img src=\"pict/menu_b_help.jpg\" alt=\"Help\" ".
	  "width=\"167\" height=20 border=0>".
	  "</a></td>".
	  "<td>&nbsp;</td>\n".
	  "</tr>\n".
	  "</table>";

    Print::print_html_footer();
    return;
};

# Log that a given case was opened and then proceed to open a host
sub case_open_log {
    Print::log_session_info ("Case $Args::args{'case'} opened");
    Print::log_case_info ("Case $Args::args{'case'} opened");
    $Args::args{'view'} = $Args::enc_args{'view'} = $Caseman::HOST_OPEN;
    host_open();
};

# Display Case Details
sub case_details {

	Print::print_html_header("Details of $Args::args{'case'}");

	read_case_config();

	$Caseman::cvals{'desc'} = "&nbsp;" unless (exists $Caseman::cvals{'desc'});
	$Caseman::cvals{'created'} = "&nbsp;" unless (exists $Caseman::cvals{'created'});

	print "<br><br>".
	  "<center>".
	  "<img src=\"pict/menu_h_cdet.jpg\" alt=\"Case Details\">".
	  "<br><br><br>\n".
	  "<table width=\"600\" cellspacing=\"0\" background=\"$::YEL_PIX\" ".
	  "cellpadding=\"2\" border=0>\n".
	  "  <tr><td align=\"right\" width=300><b>Name:</b></td>".
	  "<td align=\"left\" width=300><tt>$Args::args{'case'}</tt></td></tr>\n".

	  # Description
	  "  <tr><td align=\"right\"><b>Description:</b></td>".
	  "<td align=\"left\"><tt>$Caseman::cvals{'desc'}</tt></td></tr>\n".
	  "  <tr><td align=\"right\"><b>Created:</b></td>".
	  "<td align=\"left\"><tt>$Caseman::cvals{'created'}</tt></td></tr>\n";

	# These directories are static, so there is little point in printing
	# them out.
if (0) {
	  # Directories
	  "  <tr><td align=\"right\"><b>Directory:</b></td>".
	  "<td align=\"left\"><tt>$::case_dir</tt></td></tr>\n".
	  "  <tr><td colspan=\"2\">&nbsp;</td></tr>\n".
	  "  <tr><td align=\"right\"><b>Images Dir:</b></td>".
	  "<td align=\"left\"><tt>$::IMGDIR</tt></td></tr>\n".
	  "  <tr><td align=\"right\"><b>Output Dir:</b></td>".
	  "<td align=\"left\"><tt>$::DATADIR</tt></td></tr>\n".
	  "  <tr><td align=\"right\"><b>Report Dir:</b></td>".
	  "<td align=\"left\"><tt>$::REPDIR</tt></td></tr>\n".
	  "  <tr><td align=\"right\"><b>Log Dir:</b></td>".
	  "<td align=\"left\"><tt>$::LOGDIR</tt></td></tr>\n";
}

	# Display the valid investigators
	my @invs = read_invest();
	my $cnt = 0;
	print "  <tr><td colspan=\"2\">&nbsp;</td></tr>\n"
	  if (scalar @invs > 0);

	foreach my $i (@invs) {
		if ($cnt == 0) {
			print "  <tr><td align=\"right\"><b>Investigators:</b></td>";
			$cnt++;
		} else {
			print "  <tr><td>&nbsp;</td>";
		}
		print "<td align=\"left\"><tt>$i</tt></td></tr>\n";
	}

	print "</table>\n".
	  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&view=$Caseman::CASE_OPEN\">".
	  "<img src=\"pict/menu_b_ok.jpg\" alt=\"Ok\" ".
	  "width=\"167\" height=20 border=\"0\"></a>";

    Print::print_html_footer();
	return 0;
};










####################################################################
# Host Functions


# if no args are passed, return host config using args{'host'},
# else use the host value passed
sub host_config_fname {
    if (scalar (@_) == 1) {
        my $h = shift;
        return "$::case_dir"."$h/host.aut";
    } else {
        return "$::host_dir"."host.aut";
    }
}

# reads host config file and sets global hash values for images and other
sub read_host_config {

	if ($::LIVE == 1) {
		%Caseman::mod2img = ();
		%Caseman::img2str = ();
		%Caseman::img2dls = ();
	
		$Caseman::tz = "";
		$Caseman::ts = 0;
		$Caseman::host_desc = "";
		$Caseman::exclude_db = "";
		$Caseman::alert_db = "";
		return;
	}

	my $host = Args::get_host();

	my $cfile = host_config_fname();

	unless (open (FILE, $cfile)) { 
		Print::print_check_err("Error opening $cfile");
	}

	%Caseman::img2mnt = ();
	%Caseman::img2ftype = ();
	%Caseman::mod2img = ();
	%Caseman::img2str = ();
	%Caseman::img2dls = ();
	
	$Caseman::tz = "";
	$Caseman::ts = 0;
	$Caseman::host_desc = "";
	$Caseman::exclude_db = "";
	$Caseman::alert_db = "";

	while (<FILE>) {
		next if ((/^\#/) || (/^\s+$/));

		# remove whitespace
		s/^\s+//;      
		s/\s+$//;

		# normal file system image entry
		#
		# 'image	images/hda1.dd		openbsd		/usr
		if (/^image\s+($::REG_IMG)\s+([\w\-]+)\s+([\w\-\_\.\/:\\]+)$/o) {
			my $i = $1;
			my $t = $2;
			my $r = $3;

			unless (defined $Fs::root_meta{$t}) {
				Print::print_check_err(
				  "Error: unknown type: $t in host config: $.".
				  "<br>Edit the file and refresh your browser");
			}

			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err(
				  "Error: image in ${host}.host:$. not found: ".
				  "$::host_dir"."$i <br>Edit the config file and refresh your browser".
				  "<br>(Or your version of Perl does not support large files)");
			}

			$Caseman::img2ftype{$i} = $t;

			# Add trailing / to original mount point if needed
			if (($r !~ /.*?\/$/) && ($r !~ /.*?\\$/)) {
				$r .= '/';
			}
			$Caseman::img2mnt{$i} = $r;
		}
		# entry for a strings or dls file
		#
		# strings 	data/hda1.str		images/hda1.dd
		elsif (/^strings\s+($::REG_IMG)\s+($::REG_IMG)$/o) {
			my $i = $1;
			my $o = $2;

			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err(
				  "Error: strings file not found: ".
				  "$::host_dir$i<br>Edit host config in $::host_dir and refresh your browser");
			}

			unless ((-e "$::host_dir$o") || 
			  ((-l "$::host_dir$o") && (-e readlink "$::host_dir$o"))) {
				Print::print_check_err(
				  "Error: strings original not found: ".
				  "$::host_dir$o<br>Edit host config in $::host_dir and refresh your browser");
			}

			$Caseman::img2ftype{$i} = "strings";
			$Caseman::mod2img{$i} = $o;
			$Caseman::img2str{$o} = $i;
		}

		# dls entry
		# dls data/image.dls	[images/image.dd]
		elsif (/^dls\s+($::REG_IMG)\s*($::REG_IMG)?$/o) {
			my $dls = $1;
 
			unless ((-e "$::host_dir$dls") || 
			  ((-l "$::host_dir$dls") && (-e readlink "$::host_dir$dls"))) {
				Print::print_check_err(
				  "Error: dls file not found: ".
				  "$::host_dir$dls <br>Edit host config in $::host_dir and refresh your browser");
			}
     
			$Caseman::img2ftype{$dls} = "dls";
    
			if (defined $2) {
				my $dd = $2;
				
				unless ((-e "$::host_dir$dd") || 
			  	  ((-l "$::host_dir$dd") && (-e readlink "$::host_dir$dd"))) {
					Print::print_check_err(
					  "Error: dls original not found: ".
					  "$::host_dir$dd <br>Edit host config in $::host_dir and refresh your browser");
				}

				$Caseman::mod2img{$dls} = $dd;
				$Caseman::img2dls{$dd} = $dls;
			}

		}
		# swap
		# swap		images/hda3.dd		
		elsif (/^swap\s+($::REG_IMG)\s*$/o) {
			my $i = $1;

			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err("Error: swap in ${host}.host:$. not found: ".
				  "$::host_dir$i <br>Edit the config file and refresh your browser".
				  "<br>(Or your version of Perl does not support large files)");
			}

			$Caseman::img2ftype{$i} = "swap";
		}
		# raw
		# raw	images/hda3.dd		
		elsif (/^raw\s+($::REG_IMG)\s*$/o) {
			my $i = $1;

			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err("Error: raw in ${host}.host:$. not found: ".
				  "$::host_dir$i<br>Edit the config file and refresh your browser".
				  "<br>(Or your version of Perl does not support large files)");
			}

			$Caseman::img2ftype{$i} = "raw";
		}

		# body data/body.txt
		elsif (/^body\s+($::REG_IMG)$/o) {
			my $i = $1;
 
			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err("Error: body file not found: ".
				  "$::host_dir$i <br>Edit host config in $::host_dir and refresh your browser");
			}

			$Caseman::img2ftype{$i} = "body";
		}
		# timeline data/timeline.txt
		elsif (/^timeline\s+($::REG_IMG)$/o) {
			my $i = $1;

			unless ((-e "$::host_dir$i") || 
			  ((-l "$::host_dir$i") && (-e readlink "$::host_dir$i"))) {
				Print::print_check_err("Error: timeline file not found: ".
				  "$::host_dir$i <br>Edit host config in $::host_dir and refresh your browser");
			}
 
			$Caseman::img2ftype{$i} = "timeline";
		}
		# timezone XYZ
		elsif (/^timezone\s+($::REG_ZONE_ARGS)$/o) {
			$Caseman::tz = $1;	
		}
		# timeskew XYZ
		elsif (/^timeskew\s+($::REG_SKEW)$/o) {
			$Caseman::ts = $1;	
		}
		# desc XYZ
		elsif (/^desc\s+(.*)$/) {
			$Caseman::host_desc = $1;	
		}
		# hash databases
		elsif (/^alert_db\s+'(.*)'$/) {
			$Caseman::alert_db = $1;
		}
		elsif (/^exclude_db\s+'(.*)'$/) {
			$Caseman::exclude_db = $1;
		}
		else {
			my $msg = "Error: invalid entry in $cfile:$.".
			  "<br>\n".
			  "image	path	fs_type		mnt_point<br>\n".
			  "strings	path	orig_img<br>\n".
			  "dls		path	[orig_img]<br>\n".
			  "body		path<br>\n".
			  "timeline	path<br>\n".
			  "timezone	TZ<br>\n".
			  "desc		DESCRIPTION<br>\n";
			Print::print_check_err($msg);
		}
	}
	close (FILE);
}


# Remove any entries with the given image name from the host config
# and add a new one
#
# If type is "", then nothing is written, just removed
sub update_host_config {
	my $type = shift;
	my $img = shift;
	my $ref = shift;

	my $read = host_config_fname();
	my $write = $read."-".rand();

	unless (open (READ, "<$read")) {
		Print::check_err("Error opening $read");
	}

	unless (open (WRITE, ">$write")) {
		Print::check_err("Error opening $write");
	}

	while (<READ>) {
		s/^\s+//;
		s/\s+$//;
		print WRITE "$_\n" unless ((/^(\w+)\s+($::REG_IMG)/o) && ($2 eq $img)
		  && ($1 ne 'desc') && ($1 ne 'timezone'));
	}

	if ($type ne "") {
		if (defined $ref) {
			print WRITE "$type  $img    $ref\n";
		} else {
			print WRITE "$type  $img\n";
		}
	}

	close (READ);
	close (WRITE);
	rename $write, $read;

	return;
};



# Given the image and md5, it is saved to the MD5.txt file and any other
# references to the image are removed
#
# if $md5 is "", then nothing is written
sub update_md5 {
	my $img = shift;
	my $md5 = shift;
	$md5 =~ tr/[a-f]/[A-F]/;

	(my $img_dir) = split (/\//,$img);
	my $read = "$::host_dir"."$img_dir/md5.txt";
	my $write = $read."-".rand();
    
	unless (open (WRITE, ">$write")) {
		Print::check_err("Error opening $write");
	}

	if (-e "$read") {
		unless (open (READ, "<$read")) {
			Print::check_err("Error opening $read");
		}

		while (<READ>) {
			s/^\s+//;
			s/\s+$//;
			print WRITE "$_\n" unless ((/^$::REG_MD5\s+($::REG_IMG)/o) && ($1 eq $img));
		}
		close (READ);
	}

	print WRITE "$md5   $img\n" if ($md5 ne "");

	close (WRITE);

	rename $write, $read;
	return;
}



sub host_add {
	Print::print_html_header("Add A New Host To $Args::args{'case'}");

	print "<b>Case: </b> $Args::args{'case'}<br><br>\n";

        print "<center>".
	  "<img src=\"pict/menu_h_hnew.jpg\" alt=\"Add Host\">".
	  "<br><br><br>\n";

	print "<table width=\"600\" cellspacing=\"0\" background=\"$::YEL_PIX\" ".
	  "cellpadding=\"2\" border=0>\n".
          "<form action=\"$::PROGNAME\" method=\"get\">\n".
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_ADD_DOIT\">\n".
	  Args::make_hidden().
	  "<tr><td colspan=\"2\">&nbsp;</td></tr>".
	
	# Host name
	  "<tr><td align=\"left\" colspan=2>1.  <b>Host Name:</b>  The name of the computer being investigated.  It can only have letters, numbers, and symbols.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\"><input type=\"text\" name=\"host\" size=32 maxlength=32 value=\"host1\"></td></tr>\n".

	# Description
	  "<tr><td align=\"left\" colspan=2>2.  <b>Description:</b>  An optional one-line description or note about this computer.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\">".
	  "<input type=\"text\" name=\"desc\" size=32 maxlength=32></td></tr>\n".

	# Empty line
	  "<tr><td colspan=\"2\">&nbsp;</td></tr>".

	# Timezone
	  "<tr><td align=\"left\" colspan=2>3.  <b>Timezone:</b> An optional timezone value (i.e. EST5EDT).  If not given, it defaults to the local setting.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\">".
	  "<input type=\"text\" name=\"tz\" size=10 maxlength=16></td></tr>\n".

	# Timeskew
	  "<tr><td align=\"left\" colspan=2>4.  <b>Timeskew Adjustment:</b> An optional value to describe how many seconds this computer's clock was out of sync.  For example, if the computer was 10 seconds fast, then enter -10 to compensate.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\">".
	  "<input type=\"text\" name=\"ts\" size=8 maxlength=16 value=\"0\">".
	  "</td></tr>\n".

	# Spacer
	  "<tr><td colspan=\"2\">&nbsp;</td></tr>".

	# Alert
	  "<tr><td align=\"left\" colspan=2>5.  <b>Path of Alert Hash Database:</b> An optional hash database of known bad files.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\">".
	  "<input type=\"text\" name=\"alert_db\" size=32 maxlength=512>".
	  "</td></tr>\n".

	# Ignore
	  "<tr><td align=\"left\" colspan=2>6.  <b>Path of Ignore Hash Database:</b> An optional hash database of known good files.</td></tr>".
	  "<tr><td align=\"left\">&nbsp;&nbsp;&nbsp;</td>".
	  "<td align=\"left\">".
	  "<input type=\"text\" name=\"exclude_db\" size=32 maxlength=512>".
	  "</td></tr>\n".

	# Spacer
	  "<tr><td colspan=\"2\">&nbsp;</td></tr>".
	  "</table>\n";

	# Ok Button
	print "<br><br><table width=\"600\" cellspacing=\"8\" cellpadding=\"2\">\n".
	  "<tr><td align=center>".
	  "<input type=\"image\" src=\"pict/menu_b_hnew.jpg\" ".
	  "width=167 height=20 alt=\"Add Host\" border=0>\n".
	  "</form></td>\n".

	# Cancel Button
	  "<td align=center>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_OPEN\">\n".
          "<input type=\"hidden\" name=\"case\" value=\"$Args::args{'case'}\">\n".
	  "<input type=\"image\" src=\"pict/menu_b_cancel.jpg\" ".
	  "alt=\"Cancel\" width=\"167\" height=20 border=0>\n".
	  "</form></td>\n".

	# Help Button
	  "<td align=center><a href=\"$::HELP_URL\" ".
	  "target=\"_blank\">".
	  "<img src=\"pict/menu_b_help.jpg\" alt=\"Help\" ".
	  "width=\"167\" height=20 border=0></a>".
	  "</td></tr>\n".
	  "</table>";

    Print::print_html_footer();

    return 0;
};


# Make the directories and config files for a host
sub host_add_doit {
    Args::check_tz() if ((exists $Args::args{'tz'}) && ($Args::args{'tz'} ne ""));
    Args::check_ts();
    Print::print_html_header("Adding Host $Args::args{'host'} to $Args::args{'case'}");

    print "<h3>Adding host: <tt>$Args::args{'host'}</tt> to ".
	  "case <tt>$Args::args{'case'}</tt></h3>\n";

	# Do some sanity checks before we start making the directories and such
	if ((exists $Args::args{'alert_db'}) && ($Args::args{'alert_db'} ne "")) {

		unless ($Args::args{'alert_db'} =~ /^$::REG_HASHDB$/o) {
			print "Invalid Alert Database path\n". 
			  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::HOST_ADD&case=$Args::args{'case'}\">".
			  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
			  "width=\"43\" height=20 border=\"0\"></a>\n";
			return 1;
		}

		unless (-e "$Args::args{'alert_db'}") {
			print "Alert Database Not Found: $Args::args{'alert_db'}".
			  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::HOST_ADD&case=$Args::args{'case'}\">".
			  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
			  "width=\"43\" height=20 border=\"0\"></a>\n";
			return 1;
		}
	}

	if ((exists $Args::args{'exclude_db'}) && ($Args::args{'exclude_db'} ne "")) {
		unless ($Args::args{'exclude_db'} =~ /^$::REG_HASHDB$/o) {
			print "Invalid Exclude Database path\n". 
			  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::HOST_ADD&case=$Args::args{'case'}\">".
			  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
			  "width=\"43\" height=20 border=\"0\"></a>\n";
			return 1;
		}

		unless (-e "$Args::args{'exclude_db'}") {
			print "Exclude Database Not Found: $Args::args{'exclude_db'}".
			  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::HOST_ADD&case=$Args::args{'case'}\">".
			  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
			  "width=\"43\" height=20 border=\"0\"></a>\n";
			return 1;
		}
	}

    # Make the directory
    if (-d "$::host_dir") {
        Print::print_err("Error: $::host_dir already exists<br>".
            "Please remove the directory and its contents and try again".        
            "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
            "view=$Caseman::HOST_OPEN&case=$Args::enc_args{'case'}\">".
            "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
            "width=\"43\" height=20 border=\"0\"></a>\n");	
    }
    unless (mkdir "$::host_dir", $::MKDIR_MASK) {
        Print::print_err ("Error making directory $::host_dir: $!");
    }

    print "Host Directory (<tt>$::host_dir</tt>) created<p>\n";
    Print::log_case_info ("Host $Args::args{'host'} added to case");

    # Images directory
    unless (mkdir "$::host_dir"."$::IMGDIR", $::MKDIR_MASK) {
		rmdir "$::host_dir";
		Print::print_err ("Error making $::host_dir"."$::IMGDIR");
    }
    # Output Directory
    unless (mkdir "$::host_dir"."$::DATADIR", $::MKDIR_MASK) {
		rmdir "$::host_dir"."$::IMGDIR";
		rmdir "$::host_dir";
		Print::print_err ("Error making $::host_dir"."$::DATADIR");
    }
    # Log Directory
    unless (mkdir "$::host_dir"."$::LOGDIR", $::MKDIR_MASK) {
		rmdir "$::host_dir"."$::DATADIR";
		rmdir "$::host_dir"."$::IMGDIR";
		rmdir "$::host_dir";
		Print::print_err ("Error making $::host_dir"."$::LOGDIR");
    }
    # Reports directory
    unless (mkdir "$::host_dir"."$::REPDIR", $::MKDIR_MASK) {
		rmdir "$::host_dir"."$::LOGDIR";
		rmdir "$::host_dir"."$::DATADIR";
		rmdir "$::host_dir"."$::IMGDIR";
		rmdir "$::host_dir";
		Print::print_err ("Error making $::host_dir"."$::REPDIR");
    }
    # Make a directory for mounting the image in loopback
    unless (mkdir "$::host_dir"."mnt", $::MKDIR_MASK) {
		rmdir "$::host_dir"."$::REPDIR";
		rmdir "$::host_dir"."$::LOGDIR";
		rmdir "$::host_dir"."$::DATADIR";
		rmdir "$::host_dir"."$::IMGDIR";
		rmdir "$::host_dir";
		Print::print_err ("Error making $::host_dir"."mnt");
    }

    Print::log_host_info ("Host $Args::args{'host'} added to case $Args::args{'case'}");

    # Create config file
    my $fname = Caseman::host_config_fname();
    open HOST_CONFIG, ">$fname" or die "Can't open host config: $fname";

    print HOST_CONFIG "# Autopsy host config file\n".
	  "# Case: $Args::args{'case'}		Host: $Args::args{'host'}\n".
	  "# Created: ".localtime()."\n\n";

    if ((exists $Args::args{'desc'}) && ($Args::args{'desc'} ne "")) {
        Print::print_err ("Invalid Description<br>\n".
                  "Use the browser's back button to fix") 
                  if ($Args::args{'desc'} =~ /\n/);

        print HOST_CONFIG "desc $Args::args{'desc'}\n";
    }

    print HOST_CONFIG "timezone  ".Args::get_tz()."\n"
	  if ((exists $Args::args{'tz'}) && ($Args::args{'tz'} ne ""));
    print HOST_CONFIG "timeskew  ".Args::get_ts()."\n";


	if ((exists $Args::args{'alert_db'}) && ($Args::args{'alert_db'} ne "")) {

		# Index it if it is not
		unless (-e "$Args::args{'alert_db'}-md5.idx") {
			print "Alert Database has not been indexed - it will be as an md5sum file<br>\n";

			print "<hr>\n";
			Hash::index_md5sum ($Args::args{'alert_db'});
			print "<hr>\n";
		}

		# only print it if it was successful
		print HOST_CONFIG "alert_db \'$Args::args{'alert_db'}\'\n"
		  if (-e "$Args::args{'alert_db'}-md5.idx");
	} 

	if ((exists $Args::args{'exclude_db'}) && ($Args::args{'exclude_db'} ne "")) {
		# Index it if it is not
		unless (-e "$Args::args{'exclude_db'}-md5.idx") {
			print "Exclude Database has not been indexed - it will be as an md5sum file<br>\n";

			print "<hr>\n";
                        Hash::index_md5sum ($Args::args{'exclude_db'});
			print "<hr>\n";
		}

		# only print it if it was successful
		print HOST_CONFIG "exclude_db \'$Args::args{'exclude_db'}\'\n"
		  if (-e "$Args::args{'exclude_db'}-md5.idx");
	}

    close HOST_CONFIG;

    print "Configuration file (<tt>$fname</tt>) created<br><br>\n";

	print "<a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
          "view=$Caseman::HOST_OPEN&case=$Args::args{'case'}\">".
	  "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
	  "width=\"43\" height=20 border=\"0\"></a>\n";

    Print::print_html_footer();
	return 0;
};

# Open a host in the given case
sub host_open {
	Print::print_html_header("Open Host In $Args::args{'case'}");

	print_menu_tabs();

	print "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\" ".
	  "background=\"$::YEL_PIX\" border=0>\n";

	# Create an array of directories in the case, verifying that there is
	# a config file
	my @hosts;
	opendir HOSTS, $::case_dir or die "Can't open $::case_dir directory: $!";
	foreach my $h (readdir HOSTS) {
		next if (($h eq '.') || ($h eq '..'));

		my $hfile = Caseman::host_config_fname($h);
		push @hosts, $h
		  if ((-d "$::case_dir"."$h") && (-e "$hfile"));
	}
	closedir HOSTS;

	if (scalar @hosts == 0) {
        print "<tr><td colspan=\"4\" align=center>".
		  "No hosts have been added to case yet".
		  "</td></tr></table>\n";
	}
	else {
		print "<form action=\"$::PROGNAME\" method=\"get\">\n".
                  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
                  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_OPEN_LOG\">\n".
		  Args::make_hidden().
		  "<tr><th>Name</th>".
		  "<th>Description</th><th>&nbsp;</th></tr>\n";


		my $first = 0;
		foreach my $h (@hosts) {

			print "<tr><td align=\"left\">".
			  "<input type=\"radio\" name=\"host\" value=$h";
			if ($first == 0) {
				print " CHECKED";
				$first = 1;
			}
                        print "> $h </td>";

			my $fname = Caseman::host_config_fname($h);
                        open CONFIG, "<$fname" or die "Can't open host config file ($fname)";

                        my $desc = "None Provided";
                        while (<CONFIG>) {
				s/^\s+//;
				s/\s+$//;

                                if (/^desc\s+(.*)$/) {
                                $desc = $1; 
                                last;
                            }
                    }
                    close CONFIG;
                    
                    print "<td align=left>$desc</td>".
			  "<td align=center>".
			  "<a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::HOST_DETAILS&$Args::baseargs&".
			  "host=$h\">details</a></td></tr>\n";
                    }
                    print "</table>\n";


		# Display pulldown of investigators
                print "<br>Investigator (for reports only): ";
		my @invs = read_invest();
		if (scalar @invs == 0) {
	  		print "None Provided<br>\n".
			  "<input type=\"hidden\" name=\"inv\" value=\"unknown\">\n";
		}
		else {
			my $cur_inv = "";
			$cur_inv = $Args::args{'inv'} if (exists $Args::args{'inv'});

		  	print "<select name=\"inv\" size=\"1\">\n";

			if (($cur_inv eq "") && (scalar @invs != 1)) {
				print "<option value=\"\" selected>Select One".
				  "</option>\n";
			}
			foreach my $i (@invs) {
				print "<option value=\"$i\"";
				print " selected" if ($cur_inv eq $i);
				print ">$i</option>\n";
			}
    		print "</select>\n";
		}

	}
	print "<br><br><table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">\n".
	  "<tr>\n";

	# Make a table for the buttons.  The table will either be 3 or 2 
	# entries wide, depending on if there is an 'Ok' button or not

	unless (scalar @hosts == 0) {
		print "<td align=center>".
	  	  "<input type=\"image\" src=\"pict/menu_b_ok.jpg\" ".
	  	  "alt=\"Ok\" width=\"167\" height=20 border=0>\n".
		  "</form>\n</td>\n";
	}

	# Add Host
	print "<td align=center>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_ADD\">\n".
	  "<input type=\"hidden\" name=\"case\" value=\"$Args::args{'case'}\">\n".
	  "<input type=\"image\" src=\"pict/menu_b_hnew.jpg\" ".
  	  "alt=\"Add Host\" width=\"176\" height=20 border=0>\n".
	  "</form></td>".

	# Close Button
	  "<td align=center>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::CASE_OPEN\">\n".
	  "<input type=\"image\" src=\"pict/menu_b_ccls.jpg\" ".
  	  "alt=\"Close Case\" width=\"176\" height=20 border=0>\n".
	  "</form></td></tr></table>\n";

	print 
	  "<table width=\"600\"  cellspacing=\"0\" cellpadding=\"2\">\n".
	  "<tr><td>&nbsp;</td>".
	  "<td align=center><a href=\"$::HELP_URL\" ".
	  " target=\"_blank\">".
	  "<img src=\"pict/menu_b_help.jpg\" alt=\"Help\" ".
	  "width=\"167\" height=20 border=0>".
	  "</a></td><td>&nbsp;</td></tr>\n".
	  "</table>\n";

    Print::print_html_footer();
	return 0;
};

# Log that a given host was opened and then proceed to open an image
sub host_open_log {
    unless ((exists $Args::args{'inv'}) && ($Args::args{'inv'} ne "")) {
        Print::print_html_header("Missing Investigator");
        print "<br>An investigator must be selected<p>\n".
          "<form action=\"$::PROGNAME\" method=\"get\">\n".
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_OPEN_LOG\">\n".
          Args::make_hidden();

        print "Select one of the following:";       
        my @invs = read_invest();
        print "<select name=\"inv\" size=\"1\">\n";

        print "<option value=\"\" selected>Select One".
          "</option>\n";

        foreach my $i (@invs) {
            print "<option value=\"$i\">$i</option>\n";
        }
        print "</select><p>\n".
          "<input type=\"image\" src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
          "width=43 height=20 border=\"0\">\n".
          "</form>\n";

    	Print::print_html_footer();
        return 0;
    }

    Args::check_inv();
    Print::log_case_info ("Host $Args::args{'host'} opened by $Args::args{'inv'}");
    Print::log_host_info ("Host $Args::args{'host'} opened by $Args::args{'inv'}");
    Print::log_host_inv  ("Host $Args::args{'host'} opened");

    $Args::args{'view'} = $Args::enc_args{'view'} = $Caseman::IMG_OPEN;
    img_open();
};

# Provide details about the configuration of a host.  This window is
# a link from the HOST_OPEN window
sub host_details {
	Print::print_html_header("Details of $Args::args{'case'}:$Args::args{'host'}");

	print "<b>Case: </b>$Args::args{'case'}<br><br>".
	  "<center>".
	  "<img src=\"pict/menu_h_hdet.jpg\" alt=\"Host Details\">".
	  "<br><br><br>\n".
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\" ".
	  "background=\"$::YEL_PIX\" border=0>\n".
	  # Name
	  "<tr><td align=\"right\" width=300><b>Name:</b></td>".
	  "<td align=\"left\" width=300><tt>$Args::args{'host'}</tt></td></tr>\n".
	  # Description
	  "<tr><td align=\"right\"><b>Description:</b></td>".
	  "<td align=\"left\"><tt>".
	  (($Caseman::host_desc ne "") ? $Caseman::host_desc : "&nbsp;").
	  "</tt></td></tr>\n".
	  # Timezone
	  "<tr><td align=\"right\"><b>Timezone: </b></td>".
	  "<td align=\"left\"><tt>$Caseman::tz</tt></td></tr>\n".
	  # Timeskew
	  "<tr><td align=\"right\"><b>Timeskew:</b></td>".
	  "<td align=\"left\"><tt>$Caseman::ts</tt></td></tr>\n".
	  "<tr><td colspan=2>&nbsp;</td></tr>\n".
	  # Actual Directory
	  "<tr><td align=\"right\"><b>Directory:</b></td>".
	  "<td align=\"left\"><tt>$::host_dir</tt></td></tr>\n".
	  "<tr><td colspan=2>&nbsp;</td></tr>\n".
	  # Alert Database
	  "<tr><td align=\"right\"><b>Alert Hash Database:</b></td>".
	  "<td align=\"left\"><tt>".
	  (($Caseman::alert_db ne "") ? $Caseman::alert_db : "&nbsp;").
	  "</tt></td></tr>\n".
	  # Exclude Database
	  "<tr><td align=\"right\"><b>Exclude Hash Database:</b></td>".
	  "<td align=\"left\"><tt>".
	  (($Caseman::exclude_db ne "") ? $Caseman::exclude_db : "&nbsp;").
	  "</tt></td></tr>\n".
	  "</table>\n";

	# Final Button
	print "<br><br><form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_OPEN\">\n".
	  Args::make_hidden().
	  "<input type=\"image\" src=\"pict/menu_b_ok.jpg\" ".
	  "alt=\"Ok\" width=\"167\" height=20 border=\"0\">\n</form>";

    Print::print_html_footer();

	return;
};

# Write the login name to the investigators file
sub write_invest {
    my $name = shift;
    my $fname = investig_fname();
    open INVES, ">>$fname" or die "Can't open investigators file: $fname";
    if ($name =~ /^$::REG_INVESTIG$/o) {
        print INVES "$name\n";
    } else {
        print_err ("Invalid investigator: $name<br>\n".
          "Use the browser's back button to fix");
    }
    close (INVES);
}


# Read the investigators file and return a sorted list
sub read_invest {
    my $fname = investig_fname();
    open INVES, "<$fname" or return;

    my @investigs;
    while (<INVES>) {
        chomp;
        s/^\s+//;
        s/\s+$//;
        push @investigs, $_
          if (/^$::REG_INVESTIG$/o);
    }
    close (INVES);
    sort {lc($a) cmp lc($b)}  @investigs;
}


# File of investigators name in list
sub investig_fname {
    return "$::case_dir"."investigators.txt";
}


####################################################################
# Image Functions


# Types of modes for fname (i.e. can we overwrite it if it exists)
my $FNAME_MODE_INIT=0;
my $FNAME_MODE_OVER=1;


my $MD5_NOTHING = 1;
my $MD5_CALC = 2;
my $MD5_ADD = 3;

my $IMG_ADD_SYM = 1;
my $IMG_ADD_COPY = 2;
my $IMG_ADD_MOVE = 3;


# Open an image that has been configured
sub img_open {
	Print::print_html_header("Open Image In $Args::args{'case'}:$Args::args{'host'}");

	print_menu_tabs();

	print "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\" ".
	  "background=\"$::YEL_PIX\" border=0>\n";

	# the images have been loaded from reading the host config file in
	# autopsy_main
	if (scalar (keys %Caseman::img2ftype) == 0) {
		print "</table>No images have been added to this host yet<br>\n";
		goto EGRESS;
	}

	# We want to sort by mounting point, so rearrange the hash
	my %mnt2img;

	# Cycle through each image we read from the host config
	foreach my $i (keys %Caseman::img2ftype) {
		next if (($Caseman::img2ftype{$i} eq "dls") || 
		  ($Caseman::img2ftype{$i} eq "strings") ||
		  ($Caseman::img2ftype{$i} eq "body") || 
		  ($Caseman::img2ftype{$i} eq "timeline") );

		if (($Caseman::img2ftype{$i} eq "raw") || ($Caseman::img2ftype{$i} eq "swap")) {
			$mnt2img{"$Caseman::img2ftype{$i}--AUTOPSY--$i"} = $i;
		}
		else {
			$mnt2img{"$Caseman::img2mnt{$i}--AUTOPSY--$i"} = $i;
		}
	}

	# sort via mount point (which includes the name)
	my @mnt = sort {
		lc($a) cmp lc($b)
	} keys %mnt2img;


	# It is possible to have only the dls image and not the original
	# We need to search for those now because they will not be in the
	# list that we just made (which are arranged via mount point)
	my @orphan_dls;

	# cycle through each image and check its type and original
	foreach my $k (keys %Caseman::img2ftype) {
		if (($Caseman::img2ftype{$k} eq "dls") && (!defined $Caseman::mod2img{$k})) {
			push @orphan_dls, $k;
		} 
	}


	print "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_OPEN_LOG\">\n".
	  Args::make_hidden().
	  "<tr><th align=left>mount</th>".					# mnt
	  "<th>&nbsp;</th>".	# img button and unalloc
	  "<th align=left>name</th><th>&nbsp;</th></tr>\n";		# img name

	for (my $i = 0; $i <= $#mnt; $i++) {
		print "<tr>";

		my $img = $mnt2img{$mnt[$i]};

		# Mount Point
		# If we have the dummy string at the end of the duplicate
		# entry, then take it off and print the original
		$mnt[$i] = $1 if ($mnt[$i] =~ /^(.*?)--AUTOPSY--$::REG_IMG$/o);
		print "<td><tt>$mnt[$i]</tt></td>";

		# Image button - only print if the image still exists
		print "<td>";

		if ((-e "$::host_dir"."$img") || 
		  ((-l "$::host_dir"."$img") && (-e readlink "$::host_dir"."$img"))) {
			print "<input type=\"radio\" ".
			  "name=\"img\" value=$img";
			print " CHECKED" if ($i == 0);	
			print ">";
		}

                
		# if there is a corresponding dls entry, make an unalloc button
		if (defined $Caseman::img2dls{$img}) {
			if ((-e "$::host_dir"."$Caseman::img2dls{$img}") || 
			  (-l "$::host_dir"."$Caseman::img2dls{$img}")) {
				print " (<input type=\"radio\" name=\"img\" ".
				  "value=$Caseman::img2dls{$img}> unalloc)</td>";
			}
		} 
			
		print "</td>";

		# image name and ftype
		print "<td><tt>$img</tt></td>";
		if ($::LIVE == 0) {
			print "<td align=center><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                  "view=$Caseman::IMG_DETAILS&$Args::baseargs&".
			  "img=$img&mnt=$mnt[$i]\">details</a></td>".
			  "</tr>\n";
		}
		else {
			print "<td>&nbsp;</td></tr>\n";
		}
	}
		
	# If we are done with the regular images and have some orphan
	# dls images, print them
	my @sort  = sort @orphan_dls;
	for (my $i = 0; $i <= $#sort; $i++) {
		print "<tr><td>&nbsp;</td><td>&nbsp;</td><td>(<input type=\"radio\" name=\"img\" ".
			  "value=$sort[$i]";
		print " CHECKED" if ($#mnt == 0);	
		print "> unalloc)</td><td><tt>$sort[$i]</tt></td></tr>\n";
	}

	# Begin Button
	print "</table>\n";

EGRESS:

	print "<br><br>".
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\"><tr>\n";

	my $width = 300;

	# Ok Button
	unless (scalar (keys %Caseman::img2ftype) == 0) {
		$width = 200;
		print "<td align=center width=$width>".
		  "<input type=\"image\" src=\"pict/menu_b_ok.jpg\" ".
		  "alt=\"Ok\" width=\"167\" height=20 border=0>\n".
		  "</form></td>\n";
	}

	# Image Add Button
	if ($::LIVE == 0) {
	print "<td align=center width=$width>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_ADD\">\n".
	  Args::make_hidden().
	  "<input type=\"image\" src=\"pict/menu_b_inew.jpg\" ".
  	  "alt=\"Add Image\" width=167 height=20 border=0></form></td>\n".

	# Cancel Button
	  "<td align=center width=$width>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::HOST_OPEN\">\n".
	  "<input type=\"hidden\" name=\"case\" value=\"$Args::args{'case'}\">\n".
	  "<input type=\"image\" src=\"pict/menu_b_hcls.jpg\" ".
  	  "width=167 height=20 alt=\"Close Host\" border=0>\n".
	  "</form></td>";
	}

	print "</tr>\n</table>\n";

	# HElp Button
	print 
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">\n".
	  "<tr><td>&nbsp;</td>\n".
	  "<td align=center width=200><a href=\"$::HELP_URL\" ".
	  " target=\"_blank\">".
	  "<img src=\"pict/menu_b_help.jpg\" alt=\"Help\" ".
	  "width=\"167\" height=20 border=0>".
	  "</a></td><td>&nbsp;</td></tr>\n".
	  "</table>\n";

	# Other features that can be done on a host

	if ($::LIVE == 0) {
	print "<hr><p>".
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">\n".
	  "<tr>\n";


	# Timeline of file activity
	print "<td align=\"center\" width=200>".
	  "<a href=\"$::PROGNAME?${Args::baseargs_noimg}&".
          "mod=$::MOD_TL&view=$Timeline::FRAME\">".
	  "<img border=0 ".
	  "src=\"pict/menu_b_tl.jpg\" ".
	  "width=\"167\" height=20 ".
	  "alt=\"File Activity Timelines\"></a></td>\n";

	# verify the integrity of the images
	print "<td align=\"center\" width=200>".
	  "<a href=\"$::PROGNAME?${Args::baseargs_noimg}&".
          "mod=$::MOD_HASH&view=$Hash::IMG_LIST_FR\">".
	  "<img border=0 ".
	  "src=\"pict/menu_b_int.jpg\" ".
	  "width=\"167\" height=20 ".
	  "alt=\"Image Integrity\"></a></td>\n".

	# Hashdatabases
	  "<td align=\"center\" width=200>".
	  "<a href=\"$::PROGNAME?${Args::baseargs_noimg}&".
          "mod=$::MOD_HASH&view=$Hash::DB_MANAGER\">".
	  "<img border=0 ".
	  "src=\"pict/menu_b_hashdb.jpg\" ".
	  "width=\"167\" height=20 ".
	  "alt=\"Hash Databases\"></a></td>\n".
	  "</tr></table>\n";

	# Notes 
	if ($::USE_NOTES == 1) {
		print 
		  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">\n".
		  "<tr>\n".
		  "<td align=\"center\" width=300>".
		  "<a href=\"$::PROGNAME?${Args::baseargs_noimg}&mod=$::MOD_NOTES&view=$Notes::READ_NORM\">".
		  "<img border=0 ".
		  "src=\"pict/menu_b_note.jpg\" ".
		  "width=\"167\" height=20 ".
		  "alt=\"View Notes\"></a></td>\n".

		  "<td width=300 align=\"center\">".
		  "<a href=\"$::PROGNAME?${Args::baseargs_noimg}&mod=$::MOD_NOTES&view=$Notes::READ_SEQ\">".
		  "<img border=0 ".
		  "src=\"pict/menu_b_seq.jpg\" ".
		  "width=\"167\" height=20 ".
		  "alt=\"Event Sequencer\"></a></td>\n".

		  "</tr>\n".
		  "</table>\n";
	}


	# If LIVE
	} else {
		print "<a href=\"./about\"><br>\n".
		  "<img src=\"pict/logo.jpg\" border=0 alt=\"Logo\"></a><br>\n";

	}

	# check if the permissions are ok
	# if (((stat("$::host_dir"."$::IMGDIR"))[2]) & 0000222) {
	#	print "<br><br><br>".
	#	  "<font color=\"$::DEL_COLOR[0]\">".
	#	  "NOTE: The images directory allows for writes.<br> It is ".
	#	  "recommended that these permissions be turned off</font><br>\n"; 
	#}

	Print::print_html_footer();

    return 0;
};

# Log in the host log that a given image was opened by what user
# then open the main window
sub img_open_log {
	
	# These will be stopped in the func during  LIVE
	Print::log_host_info ("Image $Args::args{'img'} opened by $Args::args{'inv'}");
	Print::log_host_inv ("$Args::args{'img'}: Image opened");

	$Args::args{'mod'} = $Args::enc_args{'mod'} = $::MOD_FRAME;
	$Args::args{'view'} = $Args::enc_args{'view'} = $Frame::IMG_FRAME;
	Frame::main();
};




# Menu to add a new image to the host
#
# The list of new images is determined by looking in the images directory
# and seeing which is not yet configured
#
sub img_add {
	Print::print_html_header("Add Image To $Args::args{'case'}:$Args::args{'host'}");

	print "<b>Case:</b> $Args::args{'case'}<br>\n".
	  "<b>Host:</b> $Args::args{'host'}<br>\n";

	print "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_ADD_DOIT\">\n".

	  Args::make_hidden();

print <<EOF1;
<center>
<img src=\"pict/menu_h_inew.jpg\" alt=\"Add Image\">
<br><br><br>

<table width=\"600\" cellpadding=\"2\" cellspacing=\"0\" background=\"$::YEL_PIX\" border=0>
<tr>
  <td colspan=4>&nbsp;</td>
</tr>

<tr>
  <td align=left colspan=4>1. <b>Location:</b> The full path (starting with <tt>/</tt>) to the raw file system image.</td>
</tr>
<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>
    <input type=\"text\" name=\"img_path\" size=36 maxlength=256>
  </td>
</tr>
<tr>
  <td align=left colspan=4>2. <b>Import Method:</b>  The image can be imported into the Autopsy Evidence Locker from its current location by making a symbolic link, by copying it, or by moving it.  Note that if a system failure occurs during the move, then the image could become corrupt.</td>
</tr>
<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left>
    <input type=\"radio\" name=\"sort\" value=$IMG_ADD_SYM CHECKED>
      Symlink
  </td>
  <td align=left>
    <input type=\"radio\" name=\"sort\" value=$IMG_ADD_COPY>
      Copy 
  </td>
  <td align=left>
    <input type=\"radio\" name=\"sort\" value=$IMG_ADD_MOVE>
	  Move 
  </td>
</tr>

EOF1

	# Show the file system types that we know of
	print "<tr><td align=left colspan=4>3. <b>File System Type:</b> Specify the type of file system.</td></tr>\n".
	"<tr><td>&nbsp;&nbsp;</td>".
	  "<td align=left colspan=3><select name=\"ftype\">\n";

	foreach my $fs (sort keys %Fs::root_meta) {
		print "<option value=\"$fs\">$fs</option>\n";
	}
	
	# The following do not have 'metas' but should be in the list
	print "<option value=\"\">======</option>\n";
	print "<option value=\"raw\">raw</option>\n";
	print "<option value=\"swap\">swap</option>\n";

	print "</select></td></tr>\n";


	# Enter the mounting point and MD5

print <<EOF2;

<tr>
  <td align=left colspan=4>4. <b>Mount Point:</b> The directory or drive where the file system was mounted in the original suspect system (i.e. <tt>C:\\</tt> for Windows or <tt>/usr/</tt> for UNIX).  Not needed for swap or raw file system
  types. 
  </td>
</tr>
<tr>
  <td>&nbsp;</td>
  <td>
    <select name=\"mnt\">
	  <option value=\"other\">other</option>
	  <option value=\"C:\\\">C:\\</option>
	  <option value=\"D:\\\">D:\\</option>
	  <option value=\"E:\\\">E:\\</option>
	  <option value=\"/\">/</option>
	  <option value=\"/home/\">/home/</option>
	  <option value=\"/usr/\">/usr/</option>
	  <option value=\"/var/\">/var/</option>
    </select>
  </td>
  <td align=left colspan=2>other: <input type=\"text\" name=\"mnt2\"></td>
</tr>

<tr>
  <td colspan=4>&nbsp;</td>
</tr>

<tr>
  <td align=center colspan=4>
	5.  <b>Data Integrity: </b> An MD5 hash can be used to verify the 
	integrity of the file system image.  
  </td>
</tr>

<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>
    <input type=\"radio\" name=\"do_md5\" value=\"$MD5_CALC\" CHECKED>
	<u>Calculate</u> the hash value for this image.
  </td>
</tr>

<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>
    <input type=\"radio\" name=\"do_md5\" value=\"$MD5_NOTHING\">
	<u>Ignore</u> the hash value for this image.
  </td>
</tr>

<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>
    <input type=\"radio\" name=\"do_md5\" value=\"$MD5_ADD\">
	<u>Add</u> the following hash value for this image:
  </td>
</tr>

<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>&nbsp;&nbsp;&nbsp;&nbsp;
	<input type=\"text\" name=\"md5\" size=36 maxlength=32>
  </td>
</tr>

<tr>
  <td>&nbsp;&nbsp;</td>
  <td align=left colspan=3>&nbsp;&nbsp;&nbsp;&nbsp;
	<input type=\"checkbox\" name=\"ver_md5\" value=\"1\">
	&nbsp;Verify MD5 After Importing?
  </td>
</tr>

<tr>
  <td colspan=4>&nbsp;</td>
</tr>

</table>

<br><br>
<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\">
<tr>
  <td align=center>
    <input type=\"image\" src=\"pict/menu_b_inew.jpg\" alt=\"Add Image\" width=\"167\" height=20 border=0>
  </td>

</form>

EOF2

	print 
	  "  <td align=center>\n".
	  "    <form action=\"$::PROGNAME\" method=\"get\">\n".
          "    <input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "    <input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_OPEN\">\n".


	  Args::make_hidden().
	  "    <input type=\"image\" src=\"pict/menu_b_cancel.jpg\" ".
	  "alt=\"Cancel\" width=\"167\" height=20 border=0></form>\n".
	  "  </td>\n".

	# HELP
	  "  <td align=center>\n".
	  "    <a href=\"$::HELP_URL\" target=\"_blank\">\n".
	  "    <img src=\"pict/menu_b_help.jpg\" alt=\"Help\" ".
	  "width=\"167\" height=20 border=0></a>\n".
	  "  </td>\n".
	  "</tr>\n".
	  "</table>\n";

	Print::print_html_footer();

    return 0;
};


# Add the image to the configuration by adding it to the host config file
# and the md5.txt file if that data was provided
sub img_add_doit {
	Args::check_img_path(); 
	Args::check_ftype(); 
	Args::check_sort();
	Args::check_do_md5();

	if ((exists $Args::args{'mnt'}) && ($Args::args{'mnt'} eq "other") &&
	  (exists $Args::args{'mnt2'})) {
	  	$Args::args{'mnt'} = $Args::args{'mnt2'};
	  	$Args::enc_args{'mnt'} = $Args::enc_args{'mnt2'};
	}

	Args::check_mnt();

	Print::print_html_header("Add a new image to an Autopsy Case");

	my $mnt = Args::get_mnt();
	my $fstype = Args::get_ftype();
	my $img_path = Args::get_img_path();
	my $import_type = Args::get_sort();

	# Test the File System
	if (($fstype ne 'swap') && ($fstype ne 'raw')) {
		local *OUT;
		my $out;
		# Run 'fsstat' and see if there is any output - else there was
		# an error and the data went to STDERR
		Exec::exec_pipe(*OUT, "'$::TSKDIR/fsstat' -f $fstype '$img_path'");
		unless (read (OUT, $out, 1)) {
			print "<p>The image is not a <tt>$fstype</tt> file system<br>\n".
			  "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
                          "view=$Caseman::IMG_ADD&$Args::baseargs\">".
			  "<img src=\"pict/but_ok.jpg\" border=\"0\" ".
			  "width=\"43\" height=20 alt=\"Ok\"></a>\n";
			return;
		}
		close (OUT);
	}

	# The get the images directory and file name
	my $img = "";
	if ($img_path =~ /\/($::REG_FILE)$/) {
		$img = "$::IMGDIR/$1";
	}
	else {
		Print::print_err ("Error Parsing Image Path ($img_path)<br>\n".
                  "Use the browser's back button to fix");
	}

	# Get the full path of the destination
	my $img_dst = "$::host_dir"."$img";
	if ((-e "$img_dst") || (-l "$img_dst")) {
		Print::print_err ("Image by the same name already exists in the Host directory ($img)<br>\n".
                  "Use the browser's back button to fix the name or delete the file");
	}

	my $orig_size = (stat("$img_path"))[7];

	# Copy, Move, or link it
	if ($import_type == $IMG_ADD_SYM) {

		Print::log_host_info 
		  ("Sym Linking image $img_path into $Args::args{'case'}:$Args::args{'host'}");

		Print::print_err ("ERROR: /bin/ln missing")
		  unless (-x '/bin/ln');

		print "Linking <tt>$img_path</tt> to <tt>$img_dst</tt><br>\n";

		Exec::exec_sys ("/bin/ln -s '$img_path' '$img_dst'");
		
	} elsif ($import_type == $IMG_ADD_COPY) {
		Print::log_host_info 
		  ("Copying image $img_path into $Args::args{'case'}:$Args::args{'host'}");

		Print::print_err ("ERROR: /bin/cp missing")
		  unless (-x '/bin/cp');

		print "Copying <tt>$img_path</tt> to <tt>$img_dst</tt><br>\n";
		Exec::exec_sys ("/bin/cp '$img_path' '$img_dst'");

	} elsif ($import_type == $IMG_ADD_MOVE) {
		Print::log_host_info 
		  ("Moving image $img_path into $Args::args{'case'}:$Args::args{'host'}");

		Print::print_err ("ERROR: /bin/mv missing")
		  unless (-x '/bin/mv');

		print "Moving <tt>$img_path</tt> to <tt>$img_dst</tt><br>\n";

		Exec::exec_sys ("/bin/mv '$img_path' '$img_dst'");

	} else {
		Print::print_err ("Invalid Import Type: $import_type<br>\n".
                  "Use the browser's back button to fix");
	}

	my $new_size = (stat("$img_dst"))[7];

	if ($new_size != $orig_size) {
		Print::print_err ("Original image size ($orig_size) is not the same as the destination size ($new_size)");
	}

	my $do_md5 = Args::get_do_md5();

	unless ($do_md5 == $MD5_NOTHING) {

		my $act_md5 = "";

		# Do we need to calculate an MD5?
		if (($do_md5 == $MD5_CALC) || (($do_md5 == $MD5_ADD) && 
		  (exists $Args::args{'ver_md5'}) && ($Args::args{'ver_md5'} == 1) ) ) {

			print "<p>Calculating MD5 of <tt>$img</tt> ".
			  "(this could take a while)<br>\n";
			$act_md5 = Hash::calc_md5($img_dst);
			unless ($act_md5 =~ /^$::REG_MD5$/o) {
				print "Error calculating MD5: $act_md5<br>\n";
				return 1;
			}
			print "Current MD5: <tt>$act_md5</tt><br>\n"; 
		}

		# And md5 value was given so we can add it to the md5.txt file
		if (($do_md5 == $MD5_ADD) && (exists $Args::args{'md5'})) {

			my $md5 = $Args::args{'md5'};
			unless ($md5 =~ /^$::REG_MD5$/o) {
				if ($md5 eq "") {
					print "MD5 value missing<br>\n";
				}
				else {
					print "Invalid MD5 value (32 numbers or letters a-f)<br>\n";
				}
				print "<p><a href=\"$::PROGNAME?".
                                  "mod=$::MOD_CASEMAN&view=$Caseman::IMG_ADD&".
                                  "$Args::baseargs\">".
				  "<img src=\"pict/but_ok.jpg\" border=\"0\" ".
				  "width=\"43\" height=20 alt=\"Ok\"></a>\n";
				return 1;
			}
			$md5 =~ tr/[a-f]/[A-F]/;

			# They also want us to validate the MD5
		  	if ((exists $Args::args{'ver_md5'}) && ($Args::args{'ver_md5'} == 1) ) {

				if ($act_md5 eq $md5) {
            		print "Integrity Check Passed<br>\n";
					Print::log_host_info ("Integrity check passed on new image ($img)");
				} else { 
					print "<font color=\"$::DEL_COLOR[0]\">".
					  "Integrity Check Failed<br></font><br>\n".
					  "Provided: <tt>$md5</tt><br>\n".
                                          "Image added to the <tt>images</tt> directory, but not added to the host configuration file<br>\n";

					Print::log_host_info ("Integrity check failed on new image ($img)");
					return 1;
				}
			} 
			# set the act_md5 value to what was given and verified
			$act_md5 = $md5;
		}
		Caseman::update_md5 ("$img", "$act_md5");
	}

	if ($fstype eq 'swap') {
		Caseman::update_host_config ("swap", "$img");
	} elsif ($fstype eq 'raw') {
		Caseman::update_host_config ("raw", "$img");
	} else {
		Caseman::update_host_config ("image", "$img", "$fstype    $mnt");
	}
	print "<p>Image: <tt>$img_path</tt> added to config file as <tt>$img</tt>\n";

print <<EOF;
<p>
<center>
<table width=600>
<tr>
  <td width=300 align=center>
    <a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&view=$Caseman::IMG_OPEN&${Args::baseargs_noimg}\">
    <img src=\"pict/menu_b_ok.jpg\" alt=\"Ok\" width=\"167\" height=20 border=\"0\">
    </a>
  </td>
  <td width=300 align=center>
    <a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&view=$Caseman::IMG_ADD&${Args::baseargs_noimg}\">
    <img src=\"pict/menu_b_inew.jpg\" alt=\"Ok\" width=\"167\" height=20 border=\"0\">
    </a>
  </td>
</tr>
</table>
</center>

EOF
	Print::print_html_footer();

	return 0;
};

# Display details of image based on config values
# provides links to remove the config of the image and to get the file
# system details

sub img_details {
	Print::print_html_header("Details of $Args::args{'img'}");

	Args::get_unitsize(); 

	my $mnt = Args::get_mnt();
	my $ftype = Args::get_ftype();

	print "<center>".
	  "<img src=\"pict/menu_h_idet.jpg\" alt=\"Image Details\">".
	  "<br><br><br>\n".
	  "<table width=\"600\" cellspacing=\"0\" cellpadding=\"2\" ".
	  "background=\"$::YEL_PIX\" border=0>\n".
	  "  <tr><td colspan=\"2\">&nbsp;</td></tr>\n".
	  # Name
	  "  <tr><td align=\"right\" width=\"300\"><b>Name:</b></td>".
	  "<td align=\"left\"><tt>$Args::args{'img'}</tt></td></tr>\n".
	  # Mount
	  "  <tr><td align=\"right\"><b>Mounting Point:</b></td>".
	  "<td align=\"left\"><tt>$mnt</tt></td></tr>\n".
	  # Type
	  "  <tr><td align=\"right\"><b>File System Type:</b></td>".
	  "<td align=\"left\"><tt>$ftype</tt></td></tr>\n";

	my $md5 = Hash::lookup_md5($Args::args{'img'}); 
	print "  <tr><td align=\"right\"><b>MD5:</b></td><td><tt>".
	  (($md5 =~ /^$::REG_MD5$/o)? $md5 : "&nbsp;").
	  "</tt></td></tr>\n".

	  # Host Directory
	  "  <tr><td align=\"right\"><b>Host Directory:</b></td>".
	  "<td align=\"left\"><tt>$::host_dir</tt></td></tr>\n".
	  "  <tr><td colspan=\"2\">&nbsp;</td></tr>\n".

	  # Strings File
	  "  <tr><td align=\"right\"><b>Strings File:</b></td>".
	  "<td align=\"left\"><tt>".
	  ((exists $Caseman::img2str{$Args::args{'img'}}) ? 
		$Caseman::img2str{$Args::args{'img'}} : "&nbsp;").
	  "</tt></td></tr>\n";


	if (($ftype ne "raw") && ($ftype ne "swap")) {
		# dls file
		print 
		  "  <tr><td align=\"right\"><b>Unallocated $Fs::addr_unit{$ftype}s (dls) File:</b></td>".
		  "<td align=\"left\"><tt>".
		  ((exists $Caseman::img2dls{$Args::args{'img'}}) ? $Caseman::img2dls{$Args::args{'img'}} : "&nbsp;").
		  "</tt></td></tr>\n";

		# Strings of dls
		print 
		  "  <tr><td align=\"right\"><b>Strings of Unallocated $Fs::addr_unit{$ftype}s File:</b></td>".
		  "<td align=\"left\"><tt>".
		  ( ( (exists $Caseman::img2dls{$Args::args{'img'}}) &&
		  (exists $Caseman::img2str{$Caseman::img2dls{$Args::args{'img'}}})) ? 
			$Caseman::img2str{$Caseman::img2dls{$Args::args{'img'}}} : "&nbsp;").
		  "</tt></td></tr>\n";
	}

	print "  <tr><td colspan=\"2\">&nbsp;</td></tr>\n".
	  "</table>\n";

	# Section for Strings file and 'dls' file

	if ( (!(defined $Caseman::img2str{$Args::args{'img'}})) || 
	  (!(defined $Caseman::img2dls{$Args::args{'img'}})) || 
	  ((defined $Caseman::img2dls{$Args::args{'img'}}) && 
	   (!(defined $Caseman::img2str{$Caseman::img2dls{$Args::args{'img'}}})))) { 
		print "<hr><table width=600>\n<tr>";
	}

	# Strings File
	if (!(defined $Caseman::img2str{$Args::args{'img'}})) {

		(my $img_dir, my $str_img) = split (/\//,$Args::args{'img'});

		print "<td align=\"center\" width=280><h3>Extract ASCII Strings</h3>".
		  "Extracting the ASCII strings from a file system image will ".
		  "make keyword searching faster.<br><br>\n".
		  "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "File Name: <input type=\"text\" name=\"fname\" ".
		  "value=\"$str_img.str\"><br><br>\n".
		  "Generate MD5? ".                      
		  "<input type=\"checkbox\" name=\"md5\" value=\"1\" CHECKED><br><br>".
                  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
                  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_MAKESTR\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_INIT\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/srch_b_str.jpg\" ". 
		  "alt=\"Extract Strings\" border=\"0\">\n</form></td>\n".
		  "<td width=40>&nbsp;</td>\n";
	}

	if (($ftype eq 'dls') || ($ftype eq 'swap') || ($ftype eq 'raw')) {
		# Place holder for types that have no notion of unallocated
	}
	# Unallocated Space File
	elsif (!(defined $Caseman::img2dls{$Args::args{'img'}})) { 

		print "<td align=\"center\" width=280><h3>Extract Unallocated $Fs::addr_unit{$ftype}s</h3>".
		  "Extracting the unallocated data in a file system image allows ".
		  "more focused keyword searches and data recovery.<br><br>\n".
		  "(Note: This Does Not Include Slack Space)<br>\n".
		  "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "File Name: <input type=\"text\" name=\"fname\" ";
		
		(my $img_dir, my $dls_img) = split (/\//,$Args::args{'img'});
		if ($dls_img =~ /^(.*?)\.dd$/) {
			print "value=\"$1.dls\"><br><br>\n";
		} else {
			print "value=\"${dls_img}.dls\"><br><br>\n";
		}

		print "Generate MD5? ".                      
		  "<input type=\"checkbox\" name=\"md5\" value=\"1\" CHECKED><br><br>".
                  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
                  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_MAKEDLS\">\n".

		  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_INIT\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/srch_b_un.jpg\" ". 
		  "alt=\"Extract Unallocated Data\" border=\"0\">\n<br></form>\n";
	}

	# strings of 'dls'
	elsif (!(defined $Caseman::img2str{$Caseman::img2dls{$Args::args{'img'}}})) { 
		(my $img_dir, my $str_img) = split (/\//,$Caseman::img2dls{$Args::args{'img'}});

		print "<td align=\"center\" width=280><h3>Extract ASCII Strings<br>of Unallocated $Fs::addr_unit{$ftype}s</h3>".
		  "Extracting the ASCII strings from the unallocated data will make ".
		  "keyword searching faster.<br><br>\n".
		  "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "File Name: <input type=\"text\" name=\"fname\" ".
		  "value=\"$str_img.str\"><br><br>\n".
		  "Generate MD5? ".                      
		  "<input type=\"checkbox\" name=\"md5\" value=\"1\" CHECKED><br><br>".
                  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
                  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_MAKESTR\">\n".

		  "<input type=\"hidden\" name=\"img\" value=\"$Caseman::img2dls{$Args::args{'img'}}\">\n".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_INIT\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/srch_b_str.jpg\" ". 
		  "alt=\"Extract Strings\" border=\"0\">\n</form></td>\n";
	}

	if ( (!(defined $Caseman::img2str{$Args::args{'img'}})) || 
	  (!(defined $Caseman::img2dls{$Args::args{'img'}})) || 
	  ((defined $Caseman::img2dls{$Args::args{'img'}}) && (!(defined $Caseman::img2str{$Caseman::img2dls{$Args::args{'img'}}})))) { 
		print "</tr></table><hr>\n";
	}

	print "<p>".
	  "<table width=\"400\" cellspacing=\"0\" cellpadding=\"2\">\n".
	
	# Ok
	  "<tr><td align=center width=200>".
	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_OPEN\">\n".
	  Args::make_hidden().
	  "<input type=\"image\" src=\"pict/menu_b_ok.jpg\" ".
	  "alt=\"Ok\" width=\"167\" height=20 border=0></form></td>\n";


	print "<td align=center width=200>";
	if (($ftype ne "raw") && ($ftype ne "swap")) {
		# File System Details
		print 
		  "<form action=\"$::PROGNAME\" method=\"get\" target=\"_blank\">\n". 
		  Args::make_hidden().
	   	  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_FRAME\">\n".
   		  "<input type=\"hidden\" name=\"submod\" value=\"$::MOD_FS\">\n".
	   	  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  "<input type=\"hidden\" name=\"mnt\" value=\"$Args::args{'mnt'}\">\n".
		  "<input type=\"image\" src=\"pict/menu_b_fs.jpg\" ".
		  "width=167 height=20 ".
		  "alt=\"File System\" border=0></form></td>\n";
	} else {
		print "&nbsp;</td>\n";
	}

	# Remove Image
        # THis was removed 12/03 because it causes problems because the image still
        # exists and config entries and ... it becomes a mess
#	print 
#	  "<td align=center width=200>".
#	  "<form action=\"$::PROGNAME\" method=\"get\">\n". 
#          "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
#          "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_DEL\">\n".
#	  Args::make_hidden().
#          "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
#          "<input type=\"hidden\" name=\"mnt\" value=\"$Args::args{'mnt'}\">\n".
#	  "<input type=\"image\" src=\"pict/menu_b_rem.jpg\" ".
#	  "width=167 height=20 alt=\"Remove\" border=0></form>".
#          "</td>\n".
#	  "</tr></table>\n";

	Print::print_html_footer();
	return 0;
};



# remove the config files
sub img_del {
    Args::check_img('img'); Args::check_ftype(); 
    Print::print_html_header("Removing Configuration Settings for $Args::args{'img'}");

    Caseman::update_host_config("", $Args::args{'img'}, "");
    Caseman::update_md5($Args::args{'img'}, "");

    print "Settings for <tt>$Args::args{'img'}</tt> removed from ".
      "<tt>$Args::args{'case'}:$Args::args{'host'}</tt>.\n".
      "<p>NOTE: The actual file still exists in the host directory.\n";


    print "<p><a href=\"$::PROGNAME?mod=$::MOD_CASEMAN&".
      "view=$Caseman::IMG_OPEN&${Args::baseargs_noimg}\">".
      "<img src=\"pict/but_ok.jpg\" alt=\"Ok\" ".
      "width=\"43\" height=20 border=\"0\"></a>\n";

	Print::print_html_footer();

    return 0;
};


# Make a strings -t d file for the image to decrease the search time
sub img_makestr {
	Args::check_fname(); Args::check_fname_mode(); 

	Print::print_html_header("Extracting ASCII Strings");

	my $fname = Args::get_fname();
	my $fname_rel = Args::get_fname_rel();
	my $fname_mode = $Args::args{'fname_mode'};
	my $ftype = Args::get_ftype();

	if ((-e "$fname") && ($FNAME_MODE_INIT == $fname_mode)) {
		print "File Already Exists: $fname_rel\n";

		my $hidden = 
		  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
		  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_MAKESTR\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  Args::make_hidden();

		$hidden .= "<input type=\"hidden\" name=\"md5\" value=\"$Args::args{'md5'}\">\n" 
			if (exists $Args::args{'md5'});

		print "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "New Name: <input type=\"text\" name=\"fname\">\n".
		  "<table cellspacing=\"30\" cellpadding=\"2\"><tr><td>".
		  "$hidden".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_INIT\">\n".
		  "<input type=\"image\" src=\"pict/but_new_name.jpg\" ". 
		  "width=79 height=20 alt=\"Use New Name\" border=\"0\">\n".
		  "</form></td>\n";

		print "<td><form action=\"$::PROGNAME\" method=\"get\">\n".
		  "$hidden".
		  "<input type=\"hidden\" name=\"fname\" value=\"$Args::args{'fname'}\">\n".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_OVER\">\n".
		  "<input type=\"image\" src=\"pict/but_replace.jpg\" ". 
		  "width=66 height=20 alt=\"Replace\" border=\"0\">\n".
		  "</form></td></tr></table>";

		  return 0;
	}

	my $img = Args::get_img('img');

	print "Extracting strings from <tt>$Args::args{'img'}</tt> and saving to ".
	  "<tt>$Args::args{'fname'}</tt><br><br>\n";

	Print::log_host_inv ("$Args::args{'img'}: Saving ASCII strings to $fname_rel");

	local *OUT;

    my $hit_cnt = 0;
    $SIG{ALRM} = sub {
        if (($hit_cnt++ % 5) == 0) {
            print "+";
        } else {
            print "-";
        }
        alarm(5);
    };

    alarm(5);

	Exec::exec_pipe (*OUT,
	  "'$::STRINGS_EXE' -a -t d '$img' > '$fname'");
	print $_ while ($_ = Exec::read_pipe_line(*OUT));
	close (OUT);

	$SIG{ALRM} = 'DEFAULT';

	print "<br>\n" 
      if ($hit_cnt != 0);

	# Verify that it worked
	unless (open (STR, "$fname")) { 
		print "Error opening $fname<br>\n".
		  "Either an error occurred while generating the file or ".
		  "no ASCII strings exist";
		return (1);
	}

	my $sz = 0;
	while (<STR>) {
		unless (/^\s*\d+\s+\S.*$/) {
			print ("Error making strings file: $_");
			return (1);
		}
		$sz = 1;
		last;
	}

	if ($sz == 0) {
		print "Strings file has no valid entries, there could be no ASCII".
		  " strings in the original image";

		print "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
		  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_DETAILS\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/but_ok.jpg\" ".        
		  "width=43 height=20 alt=\"Ok\" border=\"0\">\n</form>";
		return (1);
	}

	close (STR);

	# append to config
	Caseman::update_host_config("strings", $fname_rel, 
	  "$Args::args{'img'}");
	print "<tt>$fname_rel</tt> added to host config file<br><br>";


	$Caseman::img2ftype{$fname_rel} = "strings";
	$Caseman::mod2img{$fname_rel} = $Args::args{'img'};
	$Caseman::img2str{$Args::args{'img'}} = $fname_rel;

    # Calculate MD5
    if ((exists $Args::args{'md5'}) && ($Args::args{'md5'} == 1)) {
        print "Calculating MD5 Value<br><br>\n";
        my $m = Hash::int_create_wrap ($fname, $fname_rel);
        print "MD5 Value: <tt>$m</tt><br><br>\n";
    }

	my $dest_img = $Args::args{'img'};
	# We need to return with a real image to IMG_DETAILS so check the mod
	$dest_img = $Caseman::mod2img{$Args::args{'img'}}
	  if (defined $Caseman::mod2img{$Args::args{'img'}});


    print "<a href=\"$::PROGNAME?$Args::baseargs_noimg&mod=$::MOD_CASEMAN&".
      "view=$Caseman::IMG_DETAILS&img=$dest_img\" target=_top>Image Details</a><p>\n";
    
    print "<a href=\"$::PROGNAME?mod=$::MOD_FRAME&submod=$::MOD_KWSRCH&$Args::baseargs\"".
      " target=\"_top\">Keyword Search</a>\n";

	Print::print_html_footer();
    return 0;
};

sub img_makedls {
	Args::check_fname(); 
	Args::check_fname_mode(); 

	Print::print_html_header("Extracting Unallocated Space");

	my $fname = Args::get_fname();
	my $fname_rel = Args::get_fname_rel();
	my $fname_mode = $Args::args{'fname_mode'};

	if ((-e "$fname") && ($FNAME_MODE_INIT == $fname_mode)) {
		print "File Already Exists: $fname_rel\n";
		my $hidden = 
		  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
		  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_MAKEDLS\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
		  Args::make_hidden();

		$hidden .= "<input type=\"hidden\" name=\"md5\" value=\"$Args::args{'md5'}\">\n" 
			if (exists $Args::args{'md5'});

		print "<form action=\"$::PROGNAME\" method=\"get\">\n".
		  "New Name: <input type=\"text\" name=\"fname\"> ".
		  "<table cellspacing=\"30\" cellpadding=\"2\"><tr><td>".
		  "$hidden".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_INIT\">\n".
		  "<input type=\"image\" src=\"pict/but_new_name.jpg\" ".               
		  "width=79 height=20 alt=\"Use New Name\" border=\"0\">\n".
		  "</form></td>\n";

		print "<td><form action=\"$::PROGNAME\" method=\"get\">\n".
		  "<input type=\"image\" src=\"pict/but_replace.jpg\" ".               
		  "width=66 height=20 alt=\"Replace\" border=\"0\"><br>\n".
		  "$hidden".
		  "<input type=\"hidden\" name=\"fname\" value=\"$Args::args{'fname'}\">\n".
		  "<input type=\"hidden\" name=\"fname_mode\" value=\"$FNAME_MODE_OVER\">\n".
		  "</form></td></tr></form>";

		  return 0;
	}

	my $img = Args::get_img('img');
	my $ftype = Args::get_ftype();

	Print::log_host_inv ("$Args::args{'img'}: Saving unallocated data to $fname_rel");

	print "Extracting unallocated data from <tt>$Args::args{'img'}</tt> and ".
	  "saving to <tt>$fname_rel</tt><br><br>\n";

	local *OUT;

    my $hit_cnt = 0;
    $SIG{ALRM} = sub {
        if (($hit_cnt++ % 5) == 0) {
            print "+";
        } else {
            print "-";
        }
        alarm(5);
    };

    alarm(5);

	Exec::exec_pipe (*OUT, 
	  "'$::TSKDIR/dls' -f $ftype '$img' > '$fname'");
	print "$_" while ( $_ = Exec::read_pipe_line(*OUT) );
	close (OUT);

	$SIG{ALRM} = 'DEFAULT';
	print "<br>\n" 
      if ($hit_cnt != 0);

	# append to fsmorgue 
	Caseman::update_host_config("dls", $fname_rel, $Args::args{'img'}); 

	print "<tt>$fname_rel</tt> added to host config file<br><br>";

	$Caseman::img2ftype{$fname_rel} = "dls";
	$Caseman::mod2img{$fname_rel} = $Args::args{'img'};
	$Caseman::img2dls{$Args::args{'img'}} = $fname_rel;

    # Calculate MD5
    if ((exists $Args::args{'md5'}) && ($Args::args{'md5'} == 1)) {
        print "Calculating MD5 Value<br><br>\n";
        my $m = Hash::int_create_wrap ($fname, $fname_rel);
        print "MD5 Value: <tt>$m</tt><br><br>\n";
    }
    
    print "<a href=\"$::PROGNAME?$Args::baseargs&mod=$::MOD_CASEMAN&".
      "view=$Caseman::IMG_DETAILS\" target=_top>Image Details</a><p>\n";
    
    print "<a href=\"$::PROGNAME?mod=$::MOD_FRAME&submod=$::MOD_KWSRCH&$Args::baseargs_noimg&".
      "img=$fname_rel\" target=\"_top\">Keyword Search</a>\n";

	Print::print_html_footer();
    return 0;
};

1;
