# 
# Keyword search mode
#
# ver 2.00+
# Brian Carrier [carrier@sleuthkit.org]
# Copyright (c) 2003-2004 by Brian Carrier.  All rights reserved
#
# This file is part of the Autopsy Forensic Browser (Autopsy)
#
# Autopsy is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Autopsy is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Autopsy; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
# IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package Kwsrch;

require 'search.pl';

$Kwsrch::ENTER = 1;
$Kwsrch::RESULTS_FR = 2;
$Kwsrch::RUN = 3;
$Kwsrch::LOAD = 4;
$Kwsrch::BLANK = 5;


my $IMG_DETAILS = 0x80;

sub main {

    # By default, show the main frame
    $Args::args{'view'} = $Args::enc_args{'view'} = $Kwsrch::ENTER    
          unless (exists $Args::args{'view'});

    Args::check_view();    
    my $view = Args::get_view();

    if ($view == $Kwsrch::BLANK) {
        blank();	
        return 0;
    }

    # Check Basic Args
    Args::check_img('img');
    Args::check_ftype();
    Args::check_mnt();
    
    # These windows don't need the meta data address
    if ($view == $Kwsrch::ENTER) {
        return enter();	
    }
    elsif ($view == $Kwsrch::RESULTS_FR) {
        return results_fr();	
    }
    elsif ($view == $Kwsrch::RUN) {
        return run();	
    }
    elsif ($view == $Kwsrch::LOAD) {
        return load();	
    }
    else {
        Print::print_check_err("Invalid Keyword Search View");
    }
}




my $CASE_INSENS = 1;
my $CASE_SENS = 0;

my $REG_EXP = 1;
my $STRING = 0;

# Form to enter search data
sub enter {
	Print::print_html_header("Search on $Args::args{'img'}");

	print "<center><h3>Keyword Search on <tt>$Args::args{'img'}</tt></h3>\n";

	print "<form action=\"$::PROGNAME\" method=\"get\">\n".
	  "Enter String: <input type=\"text\" name=\"str\"><br><br>\n".
	  Args::make_hidden().
	  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_KWSRCH\">\n".
	  "<input type=\"hidden\" name=\"view\" value=\"$Kwsrch::RESULTS_FR\">\n".
	  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
	  "<input type=\"checkbox\" name=\"srch_case\" value=\"$CASE_INSENS\">".
	  "Case Insensitive\n".
	  "<input type=\"checkbox\" name=\"regexp\" value=\"$REG_EXP\">".
	  "Regular Expression<br><br>\n".
	  "<input type=\"image\" src=\"pict/but_search.jpg\" ".
	  "alt=\"Search\" border=\"0\">\n</form>\n";

	my $ftype = Args::get_ftype();

	if ($::LIVE == 0) {
		print "<table width=600><tr>\n";

		# If we are a non-dls image and one exists - make a button to load it
		if ( ($ftype ne 'dls') && (exists $Caseman::img2dls{$Args::args{'img'}}) ) {
		print 
		  "<td align=center width=200>".
		  "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
		  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_FRAME\">\n".
		  "<input type=\"hidden\" name=\"submod\" value=\"$::MOD_KWSRCH\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Caseman::img2dls{$Args::args{'img'}}\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/srch_b_lun.jpg\" ".
		  "alt=\"Load Unallocated Image\" border=\"0\">\n<br></form></td>\n";
		}

		# If we are a dls and the original exists - make a button to load it
		elsif ( ($ftype eq 'dls') && (exists $Caseman::mod2img{$Args::args{'img'}}) ) {
		print 
		  "<td align=center width=200>".
		  "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
		  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_FRAME\">\n".
		  "<input type=\"hidden\" name=\"submod\" value=\"$::MOD_KWSRCH\">\n".
		  "<input type=\"hidden\" name=\"img\" value=\"$Caseman::mod2img{$Args::args{'img'}}\">\n".
		  Args::make_hidden().
		  "<input type=\"image\" src=\"pict/srch_b_lorig.jpg\" ". 
		  "alt=\"Load Original Image\" border=\"0\">\n<br></form></td>\n";
		}

		# Strings Button
		if (!(exists $Caseman::img2str{$Args::args{'img'}})) {

			my $dest_img = $Args::args{'img'};
			$dest_img = $Caseman::mod2img{$Args::args{'img'}} 
			  if exists ($Caseman::mod2img{$Args::args{'img'}});

			print 
			  "<td align=center width=200>".
			  "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
			  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
			  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_DETAILS\">\n".
			  "<input type=\"hidden\" name=\"img\" value=\"$dest_img\">\n".
			  Args::make_hidden().
			  "<input type=\"image\" src=\"pict/srch_b_str.jpg\" ". 
			  "alt=\"Extract Strings\" border=\"0\">\n<br></form></td>\n";
		}

		# Unallocated Space Button
		if (($ftype ne 'dls') && ($ftype ne 'swap') && ($ftype ne 'raw') && 
		  (!(exists $Caseman::img2dls{$Args::args{'img'}}))) {
			print 
			  "<td align=center width=200>".
			  "<form action=\"$::PROGNAME\" method=\"get\" target=\"_top\">\n".
			  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_CASEMAN\">\n".
			  "<input type=\"hidden\" name=\"view\" value=\"$Caseman::IMG_DETAILS\">\n".
			  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
			  Args::make_hidden().
			  "<input type=\"image\" src=\"pict/srch_b_un.jpg\" ". 
			  "alt=\"Extract Unallocated Space\" border=\"0\">\n<br></form></td>\n";
		}

		print "</tr></table>\n";
	}

	print "<a href=\"help/grep.html\" target=\"_blank\">".
	  "grep cheat sheet</a>\n<br><br>\n";

	print "<p><font color=\"red\">NOTE:</font> The keyword search runs ".
	  "<tt>grep</tt> on the image.<br>\n".
	  "A list of what will and ".
	  "what will not be found is available ".
	  "<a href=\"help/grep_lim.html\" target=\"_blank\">here</a>.<br>\n";


	# Section for previous searches
	if ($::LIVE == 0) {
		my $srch_name = srch_fname(0);
		if (-e $srch_name) {
			print "<hr><h3>Previous Searches</h3>\n".
			  "<table width=600>\n";
			my $row_idx = 0;

			# Cycle through the files
			for (my $srch_idx = 0; ;$srch_idx++) {

				$srch_name = srch_fname($srch_idx);
			
				last unless (-e $srch_name);

				# Open the file to get the string and count
				unless (open (SRCH, "$srch_name")) {
					print "Error opening search file: $srch_name\n";
					return 1;
				}
				my $prev_str = "";
				my $prev_cnt = 0;
				while (<SRCH>) {
					unless (/^(\d+)\|(.*?)?\|(.*)$/) {
						print "Error pasing header of search file: $srch_name\n";
						return 1;
					}
					$prev_cnt = $1;
					$prev_str = $3;
					if (length ($prev_str) > 32) {
						$prev_str = substr($prev_str, 0, 32);
						$prev_str .= "...";
					}

					last;
				}
				close (SRCH);

				print "<tr>\n" if ($row_idx == 0); 

				print "  <td align=center width=150>\n".
			  	  "<form action=\"$::PROGNAME\" method=\"get\">\n".
			  	  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_KWSRCH\">\n".
			  	  "<input type=\"hidden\" name=\"view\" value=\"$Kwsrch::RESULTS_FR\">\n".
			  	  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
			  	  "<input type=\"hidden\" name=\"srchidx\" value=\"$srch_idx\">\n".
				  Args::make_hidden();

				print "<input type=\"SUBMIT\" value=\"$prev_str ($prev_cnt)\">".
				  "<br></form>\n";

				if ($row_idx == 3) {
					print "</tr>\n";
					$row_idx = 0;
				} else {
					$row_idx++;
				}
			}
			print "</table>\n";
		}
	}

	# Predefined expressions from search.pl
	print "<hr><h3>Predefined Searches</h3>\n";
	print "<table width=600>\n";
	my $row_idx = 0;
	my $r;
	foreach $r (keys %Kwsrch::auto_srch) {

		$Kwsrch::auto_srch_reg{$r} = 0 unless (defined $Kwsrch::auto_srch_reg{$r});
		$Kwsrch::auto_srch_csense{$r} = 1 unless (defined $Kwsrch::auto_srch_csense{$r});

		print "<tr>\n" if ($row_idx == 0); 

		print "  <td align=center width=150>\n".
	  	  "<form action=\"$::PROGNAME\" method=\"get\">\n".
	  	  "<input type=\"hidden\" name=\"mod\" value=\"$::MOD_KWSRCH\">\n".
	  	  "<input type=\"hidden\" name=\"view\" value=\"$Kwsrch::RESULTS_FR\">\n".
	  	  "<input type=\"hidden\" name=\"img\" value=\"$Args::args{'img'}\">\n".
	  	  "<input type=\"hidden\" name=\"str\" value=\"$Kwsrch::auto_srch{$r}\">\n".
		  Args::make_hidden();

		if ($Kwsrch::auto_srch_reg{$r} == 1) {
	  		print 
			  "<input type=\"hidden\" name=\"regexp\" value=\"$REG_EXP\">\n";
		} 
		if ($Kwsrch::auto_srch_csense{$r} == 0) {
	  		print
			  "<input type=\"hidden\" name=\"srch_case\" value=\"$CASE_INSENS\">\n";
		}
	  	print "<input type=\"SUBMIT\" value=\"$r\"><br></form>\n".
		  "  </td>\n";
		
		if ($row_idx == 3) {
			print "</tr>\n";
			$row_idx = 0;
		} else {
			$row_idx++;
		}
	}
	print "</table>\n";

	Print::print_html_footer();
	return 0;
};

# MAIN WITH RESULTS
# Page that makes frame with the results on left and data units on right
sub results_fr {
	# A string was given for a new search
	if (exists $Args::args{'str'}) {
		Args::check_str();

		Print::print_html_header_frameset("Search on $Args::args{'img'} for $Args::args{'str'}");

		print "<frameset cols=\"35%,65%\">\n";

		my $srch_case = "";
		$srch_case = "&srch_case=$Args::args{'srch_case'}" 
		  if (exists $Args::args{'srch_case'});

		my $regexp = "";
		$regexp = "&regexp=$Args::args{'regexp'}" 
		  if (exists $Args::args{'regexp'});

		# Block List
		print "<frame src=\"$::PROGNAME?".
                  "mod=$::MOD_KWSRCH&view=$Kwsrch::RUN&".
		  "$Args::baseargs$srch_case$regexp&str=$Args::enc_args{'str'}\">\n";
	}
	elsif (exists $Args::args{'srchidx'}) {
		Args::check_srchidx();

		Print::print_html_header_frameset("Search on $Args::args{'img'} for Index $Args::args{'srchidx'}");

		print "<frameset cols=\"35%,65%\">\n";

		# Block List
		print "<frame src=\"$::PROGNAME?".
                  "mod=$::MOD_KWSRCH&view=$Kwsrch::LOAD&".
		  "$Args::baseargs&srchidx=$Args::enc_args{'srchidx'}\">\n";
	}

	# Block Contents
	print "<frame src=\"$::PROGNAME?mod=$::MOD_KWSRCH&view=$Kwsrch::BLANK&".
          "$Args::baseargs\" name=\"content\">\n".
	  "</frameset>\n";

	Print::print_html_footer_frameset();
	return 0;
};



# Find an empty file to save the keyword searches to
sub find_srch_file {
    my $tmp_img = $Args::args{'img'};
    $tmp_img =~ tr/\//\-/;

    my $out_name = "$::host_dir"."$::DATADIR/$tmp_img";
    my $i;
    for ($i = 0; -e "${out_name}-${i}.srch"; $i++) { }

	return "${out_name}-${i}.srch";
}



# Pass the index and get the full path of the file returned
sub srch_fname {
    my $idx = shift;
    my $tmp_img = $Args::args{'img'};
    $tmp_img =~ tr/\//\-/;
    return "$::host_dir"."$::DATADIR"."/$tmp_img-${idx}.srch";
}



sub load {
	Args::check_srchidx(); 

	Print::print_html_header("");
	
	if ($::LIVE == 1) {
		print "Searches cannot be loaded during live analysis<br>\n";
		return 1;
	}

	my $srch_name = srch_fname($Args::args{'srchidx'});
	unless (open (SRCH, "$srch_name")) {
		print "Error opening search file: $srch_name\n";
		return 1;
	}

	my @results;
	my $prev_str = "";
	my $grep_flag = "";

	while (<SRCH>) {
		if ($. == 1) {
			unless (/^(\d+)\|(.*?)?\|(.*)$/) {
				print "Error pasing header of search file: $srch_name\n";
				return 1;
			}
			$grep_flag = $2;
			$prev_str = $3;
		} else {
			push @results, "$_";
		}
	}
	close (SRCH);
	print_srch_results($prev_str, $grep_flag, \@results, $srch_name);

	Print::print_html_footer();
	return 0;
};


# performs actual search, saves hits to file, and calls method to print
sub run {
	Args::check_str(); 

	Print::print_html_header("");

	my $img = Args::get_img('img');
	my $ftype = Args::get_ftype();

	my $orig_str = Args::get_str();
	my $grep_str = $orig_str;   # we will escape some values in the grep ver


	# Check for a search string 
	if ($orig_str eq "") {
		print "You must enter a string value to search<br>\n";
		print "<b><a href=\"$::PROGNAME?mod=$::KWSRCH&view=$Kwsrch::ENTER&".
		  "$Args::baseargs\" target=\"_parent\">New Search</a></b>\n<p>";
		return 1;
	}


	my $log = "";			# Log entry string
	my $grep_flag = "";		# Flags to pass to grep

	# Check if search is case insensitive
	my $case = 0;
	if ((exists $Args::args{'srch_case'}) && ($Args::args{'srch_case'} == $CASE_INSENS)) {
		$grep_flag = "-i";
		$case = 1;
		$log .= "Case Insensitive ";
	}

	# Check if search is a regular expression
	my $regexp = 0;
	if ((exists $Args::args{'regexp'}) && ($Args::args{'regexp'} == $REG_EXP)) {
		$grep_flag .= " -E";
		$regexp = 1;
		$log .= "Regular Expression ";
	}

	# if not a reg-exp, we need to escape some special values that
	# 'grep' will misinterpret
	else {
		$grep_str =~ s/\\/\\\\/g;	# \
		$grep_str =~ s/\./\\\./g;	# .
		$grep_str =~ s/\[/\\\[/g;	# [
		$grep_str =~ s/\^/\\\^/g;	# ^
		$grep_str =~ s/\$/\\\$/g;	# $
		$grep_str =~ s/\*/\\\*/g;	# *
		# We need to add ' to end begin and end of the search as well
		if ($grep_str =~ /\'/) {
			$grep_str =~ s/\'/\\\'/g;	# '
			$grep_str = "'$grep_str'";
		}
		$grep_str =~ s/^\-/\\\-/;	# starting with - (mistakes for an arg)
	}


	Print::log_host_inv ("$Args::args{'img'}: ${log}search for $grep_str");

	# Get the addressable unit of image
	my $bs = Args::get_unitsize();

	local *OUT;

	my $hit_cnt = 0;
	$SIG{ALRM} = sub {
		if (($hit_cnt++ % 5) == 0) {
			print "+";
		} else {
			print "-";
		}
		alarm(5);
	};

	alarm(5);
	print "<b>Searching</b>: ";

	# if the string is less than 4 chars, then it will not be in the
	# strings file so it will be searched for the slow way
	if (length($orig_str) < 4) {
		my $ltmp = length ($orig_str);
		Exec::exec_pipe (*OUT, 
		  "'$::STRINGS_EXE' -t d -n $ltmp '$img' | '$::GREP_EXE' $grep_flag '$grep_str'");
	}
	# Use the strings file if it exists
	elsif (defined $Caseman::img2str{$Args::args{'img'}}) {
		my $str_img = "$::host_dir"."$Caseman::img2str{$Args::args{'img'}}";
		Exec::exec_pipe (*OUT, 
		  "'$::GREP_EXE' $grep_flag '$grep_str' '$str_img'");
	}
	# Run strings on the image first and then grep that
	else {
		Exec::exec_pipe (*OUT, 
		  "'$::STRINGS_EXE' -a -t d '$img' | '$::GREP_EXE' $grep_flag '$grep_str'");
	} 


	# $norm_str is normalized to find the "hit" in the output
	my $norm_str = $orig_str;

	# make this lowercase if we are doing case insens 
	$norm_str =~ tr/[A-Z]/[a-z]/ if ($case == 1);

	my $norm_str_len = length ($norm_str);

	# array to pass to printing method
	my @results;

	# Cycle through the results and put them in an array
	while ( $_ = Exec::read_pipe_line(*OUT) ) {

		# Parse out the byte offset and hit string
		if (/^\s*(\d+)\s+(.+)$/)  {
			my $off = $1;
			my $hit_str_orig = $2;
			my $idx = 0;

			# Make a copy that we can modify & play with
			my $hit_str = $hit_str_orig;
			$hit_str =~ tr/[A-Z]/[a-z]/ if ($case == 1);

			# How long was the string that we hit?
			my $hit_str_len = length ($hit_str);

			# I'm not sure how to find a grep re in the hit yet, so 
			# for now we do not get the exact offset
			if ($regexp) {
				my $b = int ($off / $bs);
				my $o = int ($off % $bs);

				# $hit =~ s/\n//g;
				push @results, "${b}|${o}|";
				next;
			}


			# There could be more than one keyword in the string
			# so cylcle through all of them
			my $psize = scalar (@results);
			while (($idx = index ($hit_str, $norm_str, $idx)) > -1) {

				my $b = int (($off + $idx) / $bs);
				my $o = int (($off + $idx) % $bs);

				# The summary of the hit starts 5 chars before it
				my $sum_min = $idx - 5;
				$sum_min = 0 if ($sum_min < 0);

				# Goto 5 after, if there is still space
				my $sum_max = $idx + $norm_str_len + 5;
				$sum_max = $hit_str_len if ($sum_max > $hit_str_len);

				my $sum_hit = substr ($hit_str_orig, $sum_min, $sum_max - $sum_min);
				# remove new lines
				$sum_hit =~ s/\n/ /g;

				push @results, "${b}|${o}|$sum_hit";

				# advance index to find next hit
				$idx++;
			}
			# If we did not find a term, then just print what
			# was found-this occurs bc index does not find it
			# sometimes.
			if ($psize == scalar(@results)) {

				my $b = int ($off / $bs);
				my $o = int ($off % $bs);

				# $hit =~ s/\n//g;
				push @results, "${b}|${o}|";
				next;
			}
		}
		# A negative offset is common on FreeBSD with large images
		elsif (/^\s*(\-\d+):?\s*(.+)$/) {
			print "ERROR: Negative byte offset ($1) Your version of ".
			  "strings likely does not support large files: $2<br>\n";
		}
		else {
			print "Error parsing grep result: $_<br>\n";
		}
	}
	close (OUT);
	$SIG{ALRM} = 'DEFAULT';

	print " <b>Done</b><br>";

	my $srch_name = "";
	if ($::LIVE == 0) {
		print "<b>Saving</b>: ";
		# Find a file to save the results to
		$srch_name = find_srch_file();
		unless (open(IDX, ">$srch_name")) {
			print "Error opening $srch_name\n";
			return (1);
		}

		# Print the header 
		my $cnt = scalar(@results);
		print IDX "$cnt|${grep_flag}|${orig_str}\n";

		for (my $i = 0; $i < $cnt; $i++) {
			print IDX "$results[$i]\n";
		}
		close (IDX);
		print " <b>Done</b><hr>\n";
	}

	print_srch_results($grep_str, $grep_flag, \@results, $srch_name);

	Print::print_html_footer();
	return 0;
};


# Args are search string, grep flags, and array of hits
sub print_srch_results {

	if (scalar (@_) != 4) {
		print "Missing Args for print_srch_results()\n";
		return 1;
	}

	my $grep_str = shift();
	my $grep_flag = shift();
	my @results = @{shift()};
	my $srch_name = shift();
	my $cnt = scalar(@results);

	my $ftype = Args::get_ftype();
	my $addr_str = $Fs::addr_unit{$ftype};

	print "<b><a href=\"$::PROGNAME?mod=$::MOD_KWSRCH&view=$Kwsrch::ENTER&".
          "$Args::baseargs\" ".
	  "target=\"_parent\">New Search</a></b>\n<p>";

	my $grep_str_html = Print::html_encode($grep_str);
	if ($cnt == 0) {    
		print "<b><tt>$grep_str_html</tt> was not found</b><br>\n";
	} elsif ($cnt == 1) {
		print "<b>1 occurrence of <tt>$grep_str_html</tt> was found</b><br>\n";
	} else {
		print "<b>$cnt occurrences of <tt>$grep_str_html</tt> were found</b><br>\n";
	}

	print "Search Options:<br>\n";
	if ($grep_flag =~ /\-i/) {
		print "&nbsp;&nbsp;Case Insensitive<br>\n";
	} else {
		print "&nbsp;&nbsp;Case Sensitive<br>\n";
	}
	if ($grep_flag =~ /\-E/) {
		print "&nbsp;&nbsp;Regular Expression<br>\n";
		
	}

	print "<hr>\n";

	if ($cnt > 1000) {
		print "There were more than <U>1000</U> hits.<br>\n";
		print "Please revise the search to a managable amount.\n";
		print "<p>The $cnt hits can be found in: <tt>$srch_name</tt><br>\n";
		print "<p><b><a href=\"$::PROGNAME?mod=$::MOD_KWSRCH&".
                  "view=$Kwsrch::ENTER&$Args::baseargs\" ".
		  "target=\"_parent\">New Search</a></b>\n<p>";
		return 0;
	}

	my $prev = -1;
	for (my $i = 0; $i < $cnt; $i++) {

		unless ($results[$i] =~ /^(\d+)\|(\d+)\|(.*)?$/) {
			print "Error parsing search array: $results[$i]\n";
			return 1;
		}

		my $blk = $1;
		my $off = $2;
		my $str = $3;

		if ($blk != $prev) {
			my $url="$::PROGNAME?mod=$::MOD_DATA&view=$Data::CONT_MENU_FR&$Args::baseargs&block=$blk";
			
			print "<br>\n$addr_str $blk (<a href=\"$url&sort=$Data::SORT_HEX\" ". 
			  "target=content>Hex</a> - ".
			  "<a href=\"$url&sort=$Data::SORT_ASC\" target=content>".
			  "Ascii</a>";

			print " - <a href=\"$::PROGNAME?$Args::baseargs&mod=$::MOD_DATA&view=$Data::CONT_MENU_FR&".
			  "mnt=$Args::enc_args{'mnt'}&img=$Caseman::mod2img{$Args::args{'img'}}&".
			  "btype=$Data::ADDR_DLS&block=$blk\" target=content>Original</a>"
			  if (($ftype eq 'dls') && (exists $Caseman::mod2img{$Args::args{'img'}}));

			print ")<br>";
			$prev = $blk;
		}

		my $occ = $i + 1;
		if ($str ne "")  {
			$str = Print::html_encode($str);	
			print "$occ: $off (<tt>$str</tt>)<br>\n";
		} else {
			print "$occ: $off<br>\n";
		}
	}

	return 0;
};

# Blank Page
sub blank { 
	Print::print_html_header("");
	print "<!-- This Page Intentionally Left Blank -->\n";
	Print::print_html_footer();
	return 0;
};

1;
