/*
 * regedit.c - program to dump and patch windows 9x registrys (regutils package)
 * Copyright (C) 1998 Memorial University of Newfoundland
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <getopt.h>
#include "registry.h"
#include "regformat.h"	/* for *_VALUE defines */
#include "misc.h"

char *progname;
int warnings = 1;
char *tablename="registry";

static int
sortkey(const void *a1, const void *a2)
{
    RegistryKey *k1 = (RegistryKey *) a1, *k2 = (RegistryKey *) a2;
    return strcasecmp(k1->entry->name, k2->entry->name);
}

static int
sortval(const void *a1, const void *a2)
{
    RegistryValue **v1 = (RegistryValue **) a1, **v2 = (RegistryValue **) a2;
    return strcasecmp((*v1)->name, (*v2)->name);
}

/* print out sql safe string ptr. We will malloc a new buffer and return that (it may be longer than the buffer we got) */
char *print_sql_data(char *ptr,int length) {
  int i=0;
  int j=0;

  char *result=malloc(length*2);

  for(i=0;i<length;i++) {
 		switch(*(ptr+i)) {
 		case 0:
		  result[j++]='\\';
		  result[j++]='0';
		  break;
 		case '\'':
 		case '\"':
		  result[j++]='\\';
		  result[j++]=*(ptr+i);
		  break;
 		case '\n':
		  result[j++]='\\';
		  result[j++]='n';
		  break;
 		case '\\':
		  result[j++]='\\';
		  result[j++]='\\';
		  break;
 		default:
		  result[j++]=*(ptr+i);
 		};
 	};
  result[j]=0;
  return (result);
};

void cheap_uni2ascii(char *src, char *dest, int l)
{
                                                                                                                                                             
   for (; l > 0; l -=2) {
      *dest = *src;
      dest++; src +=2;
   }
   *dest = 0;
}

static void
dump_key(RegistryKey key, const char *pathname, FILE *fp, int extendedTypes)
{
    RegistryKey *ch = 0, child;
    RegistryValue **val;
    char childpath[1024];
    int i;
    int nchild, nalloc;
    
    //fprintf(fp, "[%s]\n", pathname);
    if (registry_nvalues(key) != 0) {
	val = (RegistryValue **) xmalloc(sizeof(RegistryValue *) * registry_nvalues(key));
	for (i = 0; i < registry_nvalues(key); i++)
	    val[i] = registry_value(key, i);
	qsort(val, registry_nvalues(key), sizeof(RegistryValue *), sortval);
	for (i = 0; i < registry_nvalues(key); i++) {
	  char *string;
	  char *type;
	  int len;
	  RegistryValue *v = val[i];
	  if (v->name == NULL)
	    continue;
	  //if (v->name[0] == '\0')
	  //	putc('@', fp);
	  //   else
	  //	dump_string(v->name, strlen(v->name), fp);
	  //   putc('=', fp);
	  len = v->datalen;
	  switch (v->type) {
	  case STRING_VALUE:
	    type = strdup("REG_SZ");
	    string = (char *)malloc(len+10);
	    memcpy(string,v->data,len);
	    len++;
	    break;
	  case HEX_VALUE:
	    string = (char *)malloc(len+10);
	    type = strdup("REG_BINARY");
	    memcpy(string,v->data,len);
	    len++;
	    break;
	  case DWORD_VALUE:
	    type = strdup("REG_DWORD");
	    string = (char *)malloc(15);
	    snprintf(string, 14, "0x%08x", *((u_int *) v->data));
	    break;
	  case USTRINGZ_VALUE:
	    type = strdup("REG_SZ");
	    string = (char *)malloc(len+10);
	    cheap_uni2ascii(v->data,string,len);
	    len=len/2;
	    break;
	  case STRINGZ_VALUE:
	    type = strdup("REG_SZ");
	    string = (char *)malloc(len+10);
	    memcpy(string,v->data,len);
	    len++;
	    break;
	  default:
	    type = strdup("Unknown");
	    string = (char *)malloc(len+10);
	    memcpy(string,v->data,len);
	    len++;
	  }

	  {
	    char *temp=strdup(pathname);
	    char *clean_name;
	    char *clean_value;
	    char *clean_path;
	    int i;
	    
	    //Convert all \ in name to /:
	    for(i=0; i<strlen(temp);i++) if(temp[i]=='\\') temp[i]='/';
	    
	    clean_path=print_sql_data(temp,strlen(temp));
	    free(temp);
	    clean_name=print_sql_data(v->name,strlen(v->name));
	    clean_value=print_sql_data(string,len-1);
	    
	    fprintf(fp, " insert into %s set `path`='/%s',`size`='%d',`type`='%s',`key`='%s',`value`='%s' ;\n",tablename,clean_path, v->datalen, type, clean_name,clean_value); 
	    
	    free(clean_value);
	    free(clean_path);
	    free(clean_name);
	  };
	  free(type);
	  free(string);
	  //putc('\n', fp);
	}
	free(val);
    }
    //putc('\n', fp);
    nalloc = 0;
    nchild = 0;
    child = registry_first_subkey(key);
    while (child.entry != NULL) {
	if (nchild == nalloc) {
	    RegistryKey *nch;
	    if (nalloc == 0)
		nalloc = 8;
	    else
		nalloc *= 4;
	    nch = (RegistryKey *) xmalloc(sizeof(RegistryKey) * nalloc);
	    if (nchild != 0) {
		memcpy(nch, ch, sizeof(RegistryKey) * nchild);
		free(ch);
	    }
	    ch = nch;
	}
	ch[nchild++] = child;
	child = registry_next_subkey(child);
    }
    if (nchild != 0) {
	qsort(ch, nchild, sizeof(RegistryKey), sortkey);
	for (i = 0; i < nchild; i++) {
	  sprintf(childpath, "%s\\%s", pathname,
		  registry_key_name(ch[i]));
	  dump_key(ch[i], childpath, fp, extendedTypes);
	}
	free(ch);
    }
}

void
skip_eol(FILE *fp)
{
    int ch;

    while ((ch = getc(fp)) != EOF && ch != '\n')
	;
}

static int
read_string(char *data, int maxlen, int *datalen, FILE *fp)
{
    int len = 0;
    int ch;

    maxlen--;
    for (;;) {
	ch = getc(fp);
	switch (ch) {
	case '"':
	    data[len] = '\0';
	    *datalen = len;
	    return 1;
	case '\\':
	    ch = getc(fp);
	    /* fall through */
	default:
	    if (len < maxlen)
		data[len++] = ch;
	}
    }
}

static int
read_hex(char *data, int maxlen, int *datalenp, u_int *typep, FILE *fp)
{
    int datalen;
    int ch;

    *typep = HEX_VALUE;	/* default */

    if ((ch = getc(fp)) != 'e'
	|| (ch = getc(fp)) != 'x'
	|| ((ch = getc(fp)) != ':' && ch != '('))
    {
	ungetc(ch, fp);
	return 0;
    }

    /* [Patch from Jeff Muizelaar to allow empty hex value.] */
    /* check for explicit type specifier */
    if (ch == '(') {
	if (fscanf(fp, "%x):", typep) != 1)
	    return 0;
    }
    /* check for empty hex entry */
    if ((ch = getc(fp)) == '\n') {
	ungetc(ch, fp);
	datalen = 0;
	return 1;
    }
    ungetc(ch, fp);
    if (fscanf(fp, "%x", &ch) != 1)
	return 0;

    data[0] = ch;
    datalen = 1;
    for (;;) {
	if ((ch = getc(fp)) == ',') {
	    if (fscanf(fp, "%x", &ch) != 1) {
		ch = getc(fp);
		if (ch == '\\') {
		    skip_eol(fp);
		    ungetc(',', fp);
		    continue;
		}
		ungetc(ch, fp);
		break;
	    }
	    if (datalen >= maxlen)
		return 0;
	    data[datalen++] = ch;
	} else {
	    ungetc(ch, fp);
	    *datalenp = datalen;
	    return 1;
	}
    }
    return 0;
}

void
reg_import(Registry *r, FILE *fp)
{
    char name[256], data[8192];
    u_int dword;
    int datalen;
    RegistryKey key;
    int i;
    int ch;

    for (;;) {
	ch = getc(fp);
	if (ch != '[') {
	    if (ch == EOF)
		return;
	    if (ch != '\n')
		skip_eol(fp);
	    continue;
	}
	for (i = 0; i < sizeof data - 1; i++) {
	    ch = getc(fp);
	    if (ch == EOF || ch == ']' || ch == '\n')
		break;
	    data[i] = ch;
	}
	data[i] = '\0';
	if (ch == ']') {
	    if ((ch = getc(fp)) == '-') {
		skip_eol(fp);
		key = registry_key(r, data, 0);
		if (key.entry != NULL)
		    registry_delete_key(key);
		continue;
	    }
	    ungetc(ch, fp);
	} else if (ch == '\n')
	    continue;
	skip_eol(fp);
	key = registry_key(r, data, 1);
	for (; (ch = getc(fp)) == '@' || ch == '"'; skip_eol(fp)) {
	    if (ch == '@')
		name[0] = '\0';
	    else
		read_string(name, sizeof name, &datalen, fp);
	    switch (getc(fp)) {
	    case '=':
		datalen = 0;
		switch (getc(fp)) {
		case '"':
		    if (!read_string(data, sizeof data, &datalen, fp))
			break;
		    registry_set(key, name, data, datalen, STRING_VALUE);
		    continue;
		case 'd':
		    if (fscanf(fp, "word:%x", &dword) != 1)
			break;
		    registry_set(key, name, (char *) &dword, sizeof dword, DWORD_VALUE);
		    continue;
		case 'h':
		    {
			u_int type;

			if (!read_hex(data, sizeof data, &datalen, &type, fp))
			    break;
			registry_set(key, name, data, datalen, type);
			continue;
		    }
		case 'u':
		    {
			char c;
			char tdata[sizeof data / 2 - 1];

			if (fscanf(fp, "stringz:%c", &c) != 1 || c != '"')
			    break;
			if (!read_string(tdata, sizeof tdata, &datalen, fp))
			    break;
			memset(data, 0, datalen * 2 + 2);
			/* `Convert' to unicode */
			for (i = 0; i < datalen; i++)
			    data[i * 2] = tdata[i];
			registry_set(key, name, data, datalen * 2 + 2,
				     USTRINGZ_VALUE);
			continue;
		    }
		case 's':
		    {
			char c;

			if (fscanf(fp, "tringz:%c", &c) != 1 || c != '"')
			    break;
			if (!read_string(data, sizeof data, &datalen, fp)
			    || datalen == sizeof data)
			    break;
			data[datalen++] = '\0';
			registry_set(key, name, data, datalen, STRINGZ_VALUE);
			continue;
		    }
		}
		continue;
	    case '-':
		if ((ch = getc(fp)) != '\n') {
		    ungetc(ch, fp);
		    break;
		}
		/* fall through */
	    case '\n':
		registry_delete_value(key, name);
		ungetc('\n', fp);
		continue;
	    default:
		break;
	    }
	    break;
	}
	ungetc(ch, fp);
    }
}

/* static void */
/* usage(const char *cmd) */
/* { */
/*     fprintf(stderr, */
/* "usage: %s [-vNq] [-f regfile] [-e file [-t topkeyname]] [-i file] [-o output]\n", */
/* 	cmd); */
/*     fprintf(stderr, "\ */
/*     -e file	Write a ascii dump of the registry to file (- for stdout)\n\ */
/*     -f file	Read (binary) registry from file (default is system.dat)\n\ */
/*     -t top	The top level key name (eg, HKEY_LOCAL_MACHINE, HKEY_USER)\n\ */
/*     -i file	Import (ascii) changes from file\n\ */
/*     -o file	Save modified (binary) registry in file (instead of -f file)\n\ */
/*     -v		Verbose mode (debugging)\n\ */
/*     -q		Quite mode - suppress warning messages\n\ */
/*     -N		Don't generated `extended' data types (only string, dword, hex)\n\ */
/* "); */
/*     exit(1); */
/* } */

int xflag;
void license(void)
{
	printf("flag_reg_read - reads a windows registry hive and generates SQL statements\n\
Type flag_reg_read -h for help.\n\
\n\
Copyright (C) 2003, Michael Cohen (scudette@reapoff.no-ip.com)\n\
\n\
This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA\n\
\n\
Heavily based on chntpw. Here is the Copyright for that:\n\
* Copyright (c) 1997-2002 Petter Nordahl-Hagen.\n\
 * Freely distributable in source or binary for noncommercial purposes,\n\
 * but I allow some exceptions to this.\n\
 * Please see the COPYING file for more details on\n\
 * copyrights & credits.\n\
");
};

/* help: output a helpfull message */
void help(void)
{
	printf("flag_reg_dump - Dumps a windows registry hive into SQL insert statements.\n\
Usage: flag_reg_dump [options]\n\
\n\
-l,--license\t\tPrints out the License terms for this product\n\
-h,--help\t\tThis cruft\n\
-f,--file STR\t\tFilename to open (mandatory)\n\
-t,--table STR\t\tTable name to insert entries into (registry)\n\
");
};

char *filename=NULL;

int
main(int argc, char **argv)
{
    extern int optind;
    extern char *optarg;
    //char *filename = "system.dat";
    char *import = NULL;
    char *export = NULL;
    char *output = NULL;
    char *topkeyname = NULL;
    int stdin_used = 0;
    int stdout_used = 0;
    int verbose = 0;
    int version = 0;
    int extendedTypes = 1;
    Registry *r;
    int o;
    int c;

    progname = argv[0];
    if (!progname || !*progname)
	progname = "flag_reg_read_9X";

  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"license", 0, 0, 'l'},
      {"help", 0, 0, 'h'},
      {"file", 1, 0, 'f'},
      {"table",1,0,'t'},
      {0, 0, 0, 0}
    };
    
    c = getopt_long(argc, argv,
		    "lhf:t:",
		    long_options, &option_index);
    if (c == -1)
      break;
    
    switch (c) {
    case 'f':
      filename=optarg;
      break;
    case 't':
      tablename=optarg;
      break;
    case 'h':
      help();
      exit(0);
      break;
    case 'l':
      license();
      exit(0);
      break;
    default:
      printf("Unknown option '%c'", c);
      exit(-1);
    }
  }
  if (optind < argc) {
    printf("non-option ARGV-elements: ");
    while (optind < argc)
      printf("%s ", argv[optind++]);
    printf("\n");
     }
  
  if(!filename) {
    help();
    exit(1);
  };

/*     while ((o = getopt(argc, argv, "f:e:i:No:qt:vVx")) != EOF) { */
/* 	switch (o) { */
/* 	case 'f': */
/* 	    filename = optarg; */
/* 	    break; */
/* 	case 't': */
/* 	    topkeyname = optarg; */
/* 	    break; */
/* 	case 'i': */
/* 	    import = optarg; */
/* 	    break; */
/* 	case 'e': */
/* 	    export = optarg; */
/* 	    break; */
/* 	case 'o': */
/* 	    output = optarg; */
/* 	    break; */
/* 	case 'q': */
/* 	    warnings = 0; */
/* 	    break; */
/* 	case 'v': */
/* 	    verbose = 1; */
/* 	    break; */
/* 	case 'V': */
/* 	    version = 1; */
/* 	    break; */
/* 	case 'N': */
/* 	    extendedTypes = 0; */
/* 	    break; */
/* 	case 'x': */
/* 	    xflag = 1;	*/
/* 	    break; */
/* 	case '?': */
/* 	    usage(progname); */
/* 	} */
/*     } */
/*     if (version) { */
/* 	printf("%s: version %s\n", progname, VERSION); */
/* 	exit(0); */
/*     } */
/*     if (optind != argc) */
/* 	usage(progname); */

  warnings = 0;

    if (!import && !export && !output)
	export = "-";
    if (strcmp(filename, "-") == 0)
	stdin_used++;
    if ((r = registry_open(filename, verbose)) == NULL)
	exit(1);
    if (output)
	registry_rename(r, output);
    if (import) {
	FILE *fp;

	if (strcmp(import, "-") == 0) {
	    if (stdin_used++) {
		fprintf(stderr, "%s: stdin (-) used multiple times\n",
			progname);
		exit(1);
	    }
	    fp = stdin;
	} else if ((fp = fopen(import, "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s - %s\n",
		    progname, import, strerror(errno));
	    exit(1);
	}
	reg_import(r, fp);
	fclose(fp);
    }
    if (export) {
	FILE *fp;
	RegistryKey key;
	char *base;

	if (strcmp(export, "-") == 0) {
	    if (stdout_used++) {
		fprintf(stderr, "%s: stdout (-) used multiple times\n",
		    progname);
		exit(1);
	    }
	    fp = stdout;
	} else if ((fp = fopen(export, "w")) == NULL) {
	    fprintf(stderr, "%s: can't open %s for writing - %s\n",
		progname, export, strerror(errno));
	    exit(1);
	}
	base = strrchr(filename, '/');
	if (base != NULL)
	    base++;
	else
	    base = filename;
	key = registry_key(r, NULL, 0);
#if 0
	fprintf(fp, "REGEDIT4\n\n");
#endif
	if (!topkeyname) {
	    char *p;
	    if (strncasecmp(base, "user.", 5) == 0
		|| ((p = strrchr(base, '/'))
		    && strncasecmp(p + 1, "user.", 5) == 0))
		topkeyname = "HKEY_USERS";
	    else
		topkeyname = "HKEY_LOCAL_MACHINE";
	}
	dump_key(key, topkeyname, fp, extendedTypes);
	fflush(fp);
	/* Bit-wise OR intentional */
	if (ferror(fp) | (fp != stdout && fclose(fp) == EOF)) {
	    fprintf(stderr, "%s: error writing to %s - %s\n",
		progname, export, strerror(errno));
	    exit(1);
	}

    } else
	/* Will do a write if registry was modified */
	if (!registry_close(r))
	    exit(1);

    return 0;
}
