/*
 * regedit.c - program to dump and patch windows 9x registrys (regutils package)
 * Copyright (C) 1998 Memorial University of Newfoundland
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include "registry.h"
#include "regformat.h"	/* for *_VALUE defines */
#include "misc.h"

char *progname;
int warnings = 1;

static void
dump_string(const char *data, int len, FILE *fp)
{
    int i;

    putc('"', fp);
    for (i = 0; i < len; i++) {
	switch (data[i]) {
	case '\0':	/* don't know what win95 regedit does with this */
	    fprintf(fp, "\\0");
	    break;
	case '\\':
	    fprintf(fp, "\\\\");
	    break;
	case '"':
	    fprintf(fp, "\\\"");
	    break;
	default:
	    putc(data[i], fp);
	    break;
	}
    }
    putc('"', fp);
}

/* dump a unicode string */
static void
dump_ustring(const char *data, int len, FILE *fp)
{
    int i;

    putc('"', fp);
    for (i = 0; i < len; i += 2) {
	switch (data[i]) {
	case '\0':	/* don't know what win95 regedit does with this */
	    fprintf(fp, "\\0");
	    break;
	case '\\':
	    fprintf(fp, "\\\\");
	    break;
	case '"':
	    fprintf(fp, "\\\"");
	    break;
	default:
	    putc(data[i], fp);
	    break;
	}
    }
    putc('"', fp);
}

static void
dump_hex(const char *data, int len, FILE *fp, int type)
{
    int i;

    if (type == HEX_VALUE)
	fprintf(fp, "hex:");
    else
	fprintf(fp, "hex(%x):", type);
    for (i = 0; i < len; i++) {
	if (i != 0)
	    putc(',', fp);
	fprintf(fp, "%02x", (unsigned char) data[i]);
    }
}

static int
sortkey(const void *a1, const void *a2)
{
    RegistryKey *k1 = (RegistryKey *) a1, *k2 = (RegistryKey *) a2;
    return strcasecmp(k1->entry->name, k2->entry->name);
}

static int
sortval(const void *a1, const void *a2)
{
    RegistryValue **v1 = (RegistryValue **) a1, **v2 = (RegistryValue **) a2;
    return strcasecmp((*v1)->name, (*v2)->name);
}

static int
odd_bytes_are_zero(const char *str, int len)
{
    int i;

    if (len & 1)
	return 0;
    for (i = 0; i < len; i += 2)
	if (str[i] == '\0' || str[i + 1] != '\0')
	    return 0;
    return 1;
}

static void
dump_key(RegistryKey key, const char *pathname, FILE *fp, int extendedTypes)
{
    RegistryKey *ch = 0, child;
    RegistryValue **val;
    char childpath[1024];
    int i;
    int nchild, nalloc;

    fprintf(fp, "[%s]\n", pathname);
    if (registry_nvalues(key) != 0) {
	val = (RegistryValue **) xmalloc(sizeof(RegistryValue *) * registry_nvalues(key));
	for (i = 0; i < registry_nvalues(key); i++)
	    val[i] = registry_value(key, i);
	qsort(val, registry_nvalues(key), sizeof(RegistryValue *), sortval);
	for (i = 0; i < registry_nvalues(key); i++) {
	    RegistryValue *v = val[i];
	    if (v->name == NULL)
		continue;
	    if (v->name[0] == '\0')
		putc('@', fp);
	    else
		dump_string(v->name, strlen(v->name), fp);
	    putc('=', fp);
	    switch (v->type) {
	    case STRING_VALUE:
		dump_string(v->data, v->datalen, fp);
		break;
	    case HEX_VALUE:
		dump_hex(v->data, v->datalen, fp, v->type);
		break;
	    case DWORD_VALUE:
		fprintf(fp, "dword:%08x", *((u_int *) v->data));
		break;
	    default:
		if (extendedTypes) {
		    int done = 0;
		    switch (v->type) {
		    case USTRINGZ_VALUE:
			done = 1;
			if (v->datalen > 0 && (v->datalen & 1) == 0
			    && v->data[v->datalen - 1] == '\0'
			    && v->data[v->datalen - 2] == '\0'
			    && odd_bytes_are_zero(v->data, v->datalen - 2))
			{
			    fprintf(fp, "ustringz:");
			    dump_ustring(v->data, v->datalen - 2, fp);
			    done = 1;
			}
			break;

		    case STRINGZ_VALUE:
			if (v->datalen > 0
			    && v->data[v->datalen - 1] == '\0'
			    && strlen(v->data) == v->datalen - 1)
			{
			    fprintf(fp, "stringz:");
			    dump_string(v->data, v->datalen - 1, fp);
			    done = 1;
			}
			break;
		    }
		    if (done)
			break;
		}
		dump_hex(v->data, v->datalen, fp, v->type);
		break;
	    }
	    putc('\n', fp);
	}
	free(val);
    }
    putc('\n', fp);
    nalloc = 0;
    nchild = 0;
    child = registry_first_subkey(key);
    while (child.entry != NULL) {
	if (nchild == nalloc) {
	    RegistryKey *nch;
	    if (nalloc == 0)
		nalloc = 8;
	    else
		nalloc *= 4;
	    nch = (RegistryKey *) xmalloc(sizeof(RegistryKey) * nalloc);
	    if (nchild != 0) {
		memcpy(nch, ch, sizeof(RegistryKey) * nchild);
		free(ch);
	    }
	    ch = nch;
	}
	ch[nchild++] = child;
	child = registry_next_subkey(child);
    }
    if (nchild != 0) {
	qsort(ch, nchild, sizeof(RegistryKey), sortkey);
	for (i = 0; i < nchild; i++) {
	    sprintf(childpath, "%s\\%s", pathname,
		registry_key_name(ch[i]));
	    dump_key(ch[i], childpath, fp, extendedTypes);
	}
	free(ch);
    }
}

void
skip_eol(FILE *fp)
{
    int ch;

    while ((ch = getc(fp)) != EOF && ch != '\n')
	;
}

static int
read_string(char *data, int maxlen, int *datalen, FILE *fp)
{
    int len = 0;
    int ch;

    maxlen--;
    for (;;) {
	ch = getc(fp);
	switch (ch) {
	case '"':
	    data[len] = '\0';
	    *datalen = len;
	    return 1;
	case '\\':
	    ch = getc(fp);
	    /* fall through */
	default:
	    if (len < maxlen)
		data[len++] = ch;
	}
    }
}

static int
read_hex(char *data, int maxlen, int *datalenp, u_int *typep, FILE *fp)
{
    int datalen;
    int ch;

    *typep = HEX_VALUE;	/* default */

    if ((ch = getc(fp)) != 'e'
	|| (ch = getc(fp)) != 'x'
	|| ((ch = getc(fp)) != ':' && ch != '('))
    {
	ungetc(ch, fp);
	return 0;
    }

    /* [Patch from Jeff Muizelaar to allow empty hex value.] */
    /* check for explicit type specifier */
    if (ch == '(') {
	if (fscanf(fp, "%x):", typep) != 1)
	    return 0;
    }
    /* check for empty hex entry */
    if ((ch = getc(fp)) == '\n') {
	ungetc(ch, fp);
	datalen = 0;
	return 1;
    }
    ungetc(ch, fp);
    if (fscanf(fp, "%x", &ch) != 1)
	return 0;

    data[0] = ch;
    datalen = 1;
    for (;;) {
	if ((ch = getc(fp)) == ',') {
	    if (fscanf(fp, "%x", &ch) != 1) {
		ch = getc(fp);
		if (ch == '\\') {
		    skip_eol(fp);
		    ungetc(',', fp);
		    continue;
		}
		ungetc(ch, fp);
		break;
	    }
	    if (datalen >= maxlen)
		return 0;
	    data[datalen++] = ch;
	} else {
	    ungetc(ch, fp);
	    *datalenp = datalen;
	    return 1;
	}
    }
    return 0;
}

void
reg_import(Registry *r, FILE *fp)
{
    char name[256], data[8192];
    u_int dword;
    int datalen;
    RegistryKey key;
    int i;
    int ch;

    for (;;) {
	ch = getc(fp);
	if (ch != '[') {
	    if (ch == EOF)
		return;
	    if (ch != '\n')
		skip_eol(fp);
	    continue;
	}
	for (i = 0; i < sizeof data - 1; i++) {
	    ch = getc(fp);
	    if (ch == EOF || ch == ']' || ch == '\n')
		break;
	    data[i] = ch;
	}
	data[i] = '\0';
	if (ch == ']') {
	    if ((ch = getc(fp)) == '-') {
		skip_eol(fp);
		key = registry_key(r, data, 0);
		if (key.entry != NULL)
		    registry_delete_key(key);
		continue;
	    }
	    ungetc(ch, fp);
	} else if (ch == '\n')
	    continue;
	skip_eol(fp);
	key = registry_key(r, data, 1);
	for (; (ch = getc(fp)) == '@' || ch == '"'; skip_eol(fp)) {
	    if (ch == '@')
		name[0] = '\0';
	    else
		read_string(name, sizeof name, &datalen, fp);
	    switch (getc(fp)) {
	    case '=':
		datalen = 0;
		switch (getc(fp)) {
		case '"':
		    if (!read_string(data, sizeof data, &datalen, fp))
			break;
		    registry_set(key, name, data, datalen, STRING_VALUE);
		    continue;
		case 'd':
		    if (fscanf(fp, "word:%x", &dword) != 1)
			break;
		    registry_set(key, name, (char *) &dword, sizeof dword, DWORD_VALUE);
		    continue;
		case 'h':
		    {
			u_int type;

			if (!read_hex(data, sizeof data, &datalen, &type, fp))
			    break;
			registry_set(key, name, data, datalen, type);
			continue;
		    }
		case 'u':
		    {
			char c;
			char tdata[sizeof data / 2 - 1];

			if (fscanf(fp, "stringz:%c", &c) != 1 || c != '"')
			    break;
			if (!read_string(tdata, sizeof tdata, &datalen, fp))
			    break;
			memset(data, 0, datalen * 2 + 2);
			/* `Convert' to unicode */
			for (i = 0; i < datalen; i++)
			    data[i * 2] = tdata[i];
			registry_set(key, name, data, datalen * 2 + 2,
				     USTRINGZ_VALUE);
			continue;
		    }
		case 's':
		    {
			char c;

			if (fscanf(fp, "tringz:%c", &c) != 1 || c != '"')
			    break;
			if (!read_string(data, sizeof data, &datalen, fp)
			    || datalen == sizeof data)
			    break;
			data[datalen++] = '\0';
			registry_set(key, name, data, datalen, STRINGZ_VALUE);
			continue;
		    }
		}
		continue;
	    case '-':
		if ((ch = getc(fp)) != '\n') {
		    ungetc(ch, fp);
		    break;
		}
		/* fall through */
	    case '\n':
		registry_delete_value(key, name);
		ungetc('\n', fp);
		continue;
	    default:
		break;
	    }
	    break;
	}
	ungetc(ch, fp);
    }
}

static void
usage(const char *cmd)
{
    fprintf(stderr,
"usage: %s [-vNq] [-f regfile] [-e file [-t topkeyname]] [-i file] [-o output]\n",
	cmd);
    fprintf(stderr, "\
    -e file	Write a ascii dump of the registry to file (- for stdout)\n\
    -f file	Read (binary) registry from file (default is system.dat)\n\
    -t top	The top level key name (eg, HKEY_LOCAL_MACHINE, HKEY_USER)\n\
    -i file	Import (ascii) changes from file\n\
    -o file	Save modified (binary) registry in file (instead of -f file)\n\
    -v		Verbose mode (debugging)\n\
    -q		Quite mode - suppress warning messages\n\
    -N		Don't generated `extended' data types (only string, dword, hex)\n\
");
    exit(1);
}

int xflag;

int
main(int argc, char **argv)
{
    extern int optind;
    extern char *optarg;
    char *filename = "system.dat";
    char *import = NULL;
    char *export = NULL;
    char *output = NULL;
    char *topkeyname = NULL;
    int stdin_used = 0;
    int stdout_used = 0;
    int verbose = 0;
    int version = 0;
    int extendedTypes = 1;
    Registry *r;
    int o;

    progname = argv[0];
    if (!progname || !*progname)
	progname = "regedit";

    while ((o = getopt(argc, argv, "f:e:i:No:qt:vVx")) != EOF) {
	switch (o) {
	case 'f':
	    filename = optarg;
	    break;
	case 't':
	    topkeyname = optarg;
	    break;
	case 'i':
	    import = optarg;
	    break;
	case 'e':
	    export = optarg;
	    break;
	case 'o':
	    output = optarg;
	    break;
	case 'q':
	    warnings = 0;
	    break;
	case 'v':
	    verbose = 1;
	    break;
	case 'V':
	    version = 1;
	    break;
	case 'N':
	    extendedTypes = 0;
	    break;
	case 'x':
	    xflag = 1;	/* hacking... */
	    break;
	case '?':
	    usage(progname);
	}
    }
    if (version) {
	printf("%s: version %s\n", progname, VERSION);
	exit(0);
    }
    if (optind != argc)
	usage(progname);
    if (!import && !export && !output)
	export = "-";
    if (strcmp(filename, "-") == 0)
	stdin_used++;
    if ((r = registry_open(filename, verbose)) == NULL)
	exit(1);
    if (output)
	registry_rename(r, output);
    if (import) {
	FILE *fp;

	if (strcmp(import, "-") == 0) {
	    if (stdin_used++) {
		fprintf(stderr, "%s: stdin (-) used multiple times\n",
			progname);
		exit(1);
	    }
	    fp = stdin;
	} else if ((fp = fopen(import, "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s - %s\n",
		    progname, import, strerror(errno));
	    exit(1);
	}
	reg_import(r, fp);
	fclose(fp);
    }
    if (export) {
	FILE *fp;
	RegistryKey key;
	char *base;

	if (strcmp(export, "-") == 0) {
	    if (stdout_used++) {
		fprintf(stderr, "%s: stdout (-) used multiple times\n",
		    progname);
		exit(1);
	    }
	    fp = stdout;
	} else if ((fp = fopen(export, "w")) == NULL) {
	    fprintf(stderr, "%s: can't open %s for writing - %s\n",
		progname, export, strerror(errno));
	    exit(1);
	}
	base = strrchr(filename, '/');
	if (base != NULL)
	    base++;
	else
	    base = filename;
	key = registry_key(r, NULL, 0);
#if 0
	fprintf(fp, "REGEDIT4\n\n");
#endif
	if (!topkeyname) {
	    char *p;
	    if (strncasecmp(base, "user.", 5) == 0
		|| ((p = strrchr(base, '/'))
		    && strncasecmp(p + 1, "user.", 5) == 0))
		topkeyname = "HKEY_USERS";
	    else
		topkeyname = "HKEY_LOCAL_MACHINE";
	}
	dump_key(key, topkeyname, fp, extendedTypes);
	fflush(fp);
	/* Bit-wise OR intentional */
	if (ferror(fp) | (fp != stdout && fclose(fp) == EOF)) {
	    fprintf(stderr, "%s: error writing to %s - %s\n",
		progname, export, strerror(errno));
	    exit(1);
	}

    } else
	/* Will do a write if registry was modified */
	if (!registry_close(r))
	    exit(1);

    return 0;
}
