# ******************************************************
# Copyright 2003: Commonwealth of Australia.
#
# Developed by the Computer Network Vulnerability Team,
# Information Security Group.
# Department of Defence.
# ******************************************************
#  Version: FLAG 0.34 (01-11-2003)
# ******************************************************
# 
# * This program is free software; you can redistribute it and/or
# * modify it under the terms of the GNU General Public License
# * as published by the Free Software Foundation; either version 2
# * of the License, or (at your option) any later version.
# *
# * This program is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# * GNU General Public License for more details.
# *
# * You should have received a copy of the GNU General Public License
# * along with this program; if not, write to the Free Software
# * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
##################################################################

package Report;

use DBI;
use CGI qw/-oldstyle_urls -compile :all /;
use GD;
use GD::Text;
use HTML::Entities;
use URI::Escape;
use IPC::Run qw(run start finish pump);
use POSIX qw(setsid getpid);
use strict;
use Digest::MD5;
use POSIX ":sys_wait_h";
use Data::HexDump;
use Carp qw(cluck);

#############################################################
#
# Some package wide variables
#
#############################################################
#co will contain a global reference to the current CGI object. We must populate it in the constructor.
my $co;
my $dbh;
my %CONF;

#Set this to 1 if you want to debug the analysis methods. Debugging messages are found in the debug_analysis file
my $debug_analysis=1;

#The name of the flag analysis database:
my $flag_db; 
my $case;
my $pid; #Global pid holder.
my $flag_head1;
my $flag_head2;
my $flag_header;
my $flag_nav;
my $flag_nav2;
my $flag_footer;
my $flag_footer_plain="</body></html>";

#These pre-filters are bits of code to be run on the log file before each line is loaded to bring the data into a common format. Mostly used by load_log_file(). These functions receive the text to operate on first and return the modified text. If the lines are to be dropped this function must return 0;
my %prefilters=
  ("Month name to numbers" =>sub {
     $_ = shift;
     s/Jan(uary)?/1/; 
     s/Feb(uary)?/2/;
     s/Mar(ch)?/3/;
     s/Apr(il)?/4/;
     s/May/5/;
     s/Jun(e)?/6/;
     s/Jul(y)?/7/;
     s/Aug(ust)?/8/;
     s/Sep(tember)?/9/;
     s/Oct(ober)?/10/;
     s/Nov(ember)?/11/;
     s/Dec(ember)?/12/;
     return $_;
   },
   "None" => sub { return shift },
   "Swap Months and Day" => sub {
     $_ = shift;
     s/^(\d+)\s(\d+)\s/2003:$1:$2:/;
     return $_;
   },
   "DD/MM/YYYY->YYYY/MM/DD"=> sub {
     $_ = shift;
     s/(\d\d)\/([^\/]+)\/(\d\d\d\d)/$3\/$2\/$1/;
     return $_;
   },
   "YYYY-MM-DD HH:MM:SS->YYYY/MM/DD:HH:MM:SS"=> sub {
     $_ = shift;
     s|^(\d\d\d\d)-(\d\d)-(\d\d) (\d\d:\d\d:\d\d)|$1/$2/$3:$4|;
     return $_;
   },
   "Remove Comments (#)"=> sub {
     $_ = shift;
     if (/^#/) {
     	return 0;
     }
     else {
     	return $_;
     }
   },
  );



#These are the recognised types of the parameters, and a short sub to test if they match. Sub should return zero if argument does not match and 1 if its ok.
my %types = (
	     #Checkbox - This represent a group of variables, possibly as an array - each variable may only be alphnumeric
	     "checkbox" => sub {
	       my $var=shift;
	       if ($var=~/[^\s\w\0]/) {
		 return 0;
	       } else {
		 return 1;
	       }
	     },
	     #Web log table - Is the arg a valid web log table?
	     "web_table"=> sub {
	       my $var=shift;
	       return 1 if index($var,'"')>-1;
	       return($dbh->selectrow_array("select * from $case.meta where property=\"web_table\" and value=?",undef , $var));
	     },

	     "firewall_table"=> sub {
	       my $var=shift;
	       return 1 if index($var,'"')>-1;
	       return($dbh->selectrow_array("select * from $case.meta where property=\"firewall\" and value=?",undef , $var));
	     },

	     "flag_db" => sub {
	       my $var=shift;
	       return 0 if index($var,'"')>-1;
	       return($dbh->selectrow_array("select * from $flag_db.meta where property=\"flag_db\" and value=?",undef , $var));
	     },

	     "int" => sub {
	       my $var=shift;

	       return($var=~/^\d+$/);
	     },

	     "string"=>sub {
	       my $var=shift;
	       return $var;
	     },

#This must only have alphanumeric chars and no spaces. Used in table names etc.
	     "alphanum"=>sub {
	       my $var=shift;
	       return ($var=~/^\w+$/);
	     },

#This type is used for filenames, its allowed to have alphanumeric chars, / and ., but not .. sequences. Note that files must be specified relative to the upload directory...
	     "filename"=>sub {
	       my $var=shift;
	       return 0 if(index($var,'..')>-1);
	       #check to see if its an actual file:
	       return 1 if -f $CONF{"UPLOADDIR"}."/".$var;
	       return 0;
	     },

	     "ipaddr"=> sub {
	       my $var=shift;
	       return($var=~/^\d+\.\d+\.\d+\.\d+$/);
	     },

	     #Allow only certain SQL functions in where clauses
	     "whereclause" => sub {
	       my $var=shift;
	       $var=~s/INET_NTOA\(//;
	       return(!($var=~/[( \"\'\`]/));
	     },

	     #This test checks that inputs fields "fieldxxx" are unique
	     "uniq_field"=> sub {
	       my %args=$co->Vars;
	       my %temp=reverse %args;

	       keys %args;
	       while ((my $key,my $value) =each %args) {
		 if ($key=~/^field/ && $temp{$value} ne $key && $value ne "ignore") {
		   throw("<h1> Error:</h1> You have a repeated field $value in this table. You are only allowed to have a single instance of each field.");
		 }
	       }
	       return 1;
	     },
	    );

my %reports;
########################################################
#
# Report Template

########################################################
$reports{"template"}={
	     #the name of this report - will be displayed in the navigation menus
	     "name" => "Template",
	     #This groups the class of report:
	     "family" => "35 TCPDump Analysis",
	     "hidden" => "yes",
	     #A description will be shown in tool tips so it doesnt need to be really long
	     "description" => "Use this report to construct new reports",
	     #This is a hash that specifies the variable name and its type. Types are taken from %types above.
	     "parameters" => {"case"=>"flag_db"},
	       
	     #This will be executed while the analysis is happening to give the user a progress bar:
	     "progress" => sub {
	       return ("Currently calculating the connection table, please wait");
	     },
	     #This will be executed to produce the form. You do not need to put the <form> tags or the submit button here. You need to offer as many input elements here as are required by the parameters section above.
	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result = "<br>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
	       return $result;
	     },
	     
	     #This will be executed to produce the precached material. This function should set up any knowledge base or tables which will be needed for the display part
	     "analyse" => sub {
	       my $reference= shift;
	       
	       my %args =%{$reference} ;

	       return "1";
	     },

	     #This routine will be called in the second phase to display the results:
	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result="<h1>Table</h1>\n";

	       $result.=display_table("select ",[$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=",$co->url(-relative=>'1')."?report=search_ip&case=".$args{"case"}."&src_ip="],\%args,[]);
	       return $result;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       return "1";
	     },
	    };

#########################################################
#
#                TCPdump related reports.
#
#########################################################
$reports{"connection_table"}={
	     "name" => "Connections Table",
	     "family" => "35 TCPDump Analysis",
	     "description" => "Show unique TCP connections",
	     "parameters" => {"case"=>"flag_db"},
	     "progress" => sub {
	       return ("Currently calculating the connection table, please wait");
	     },

	     "form" => sub {
	       my $reference= shift;
	       
	       my %args =%{$reference} ;
	       if(!exists($args{"case"})) {
		 $args{"case"} = "";
	       };

	       my $result = "<br>Select Flag databases to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
	       return $result;
	     },
	     
	     "analyse" => sub {
	       my $reference= shift;
	       
	       my %args =%{$reference} ;

	       #Prepare the connection table:
	       $dbh->do("create temporary table connection_table_temp (con_id int auto_increment, src_ip int unsigned, src_port int, dest_ip int unsigned, dest_port int,  count int,key(con_id), key(src_ip), key(dest_ip),key(src_port),key(dest_port))");

	       $dbh->do("create table if not exists connection_table (con_id int auto_increment, src_ip int unsigned, src_port int, dest_ip int unsigned, dest_port int, count int, key(con_id), key(src_ip), key(dest_ip),key(src_port),key(dest_port))");

	       $dbh->do("insert into connection_table_temp select NULL, ip_src , tcp_srcport , ip_dst , tcp_dstport,count(ip_dst)  from ip,tcp where ip.key_id = tcp.key_id group by ip_src , tcp_srcport , ip_dst , tcp_dstport");

	       $dbh->do("create temporary table connection_table_temp2 select * from connection_table_temp");

	       $dbh->do('insert into connection_table select if(a.con_id<b.con_id,b.con_id,a.con_id) as "con_id",(a.src_ip),a.src_port ,(a.dest_ip),a.dest_port,a.count+ifnull(b.count,0) as "total"  from connection_table_temp2 as a left join connection_table_temp as b on a.src_ip=b.dest_ip and a.dest_ip=b.src_ip and a.src_port=b.dest_port and a.dest_port=b.src_port group by "con_id"');

	       return "analysed! \n";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $result;

	       $result.="<h1>Connection Table</h1>\n";

	       $result.=display_table("select con_id as  \"Connection #\",INET_NTOA(src_ip) as \"Source IP\",src_port as \"Source Port\",INET_NTOA(dest_ip) as \"Dest IP\",dest_port as \"Dest Port\",count as \"# Packets\"  from connection_table ",[$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=",$co->url(-relative=>'1')."?report=search_ip&case=".$args{"case"}."&src_ip="],\%args,[]);
	       return $result;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       $dbh->do("drop table $case.connection_table");
	       return "Cleaned up\n";
	     },
	    };


$reports{'tcptrace'} ={
	      "name"=>"TCP packet trace",
	      "family"=>"35 TCPDump Analysis",
	      "hidden" => "yes",
	      "description"=>"Reassemble the data that forms the tcp stream in the same screen",
	      "parameters"=>{"case" => "flag_db" , "con_id" => "int"},
	      "progress"=>sub{
		my $reference= shift;
		my %args =%{$reference} ;

# (MC)FIXME : I really want to use this one, but since the table is locked due to the insert, we cant select from it yet. (I tried to use insert delayed, but this doesnt work since it inserts from a select so the select must finish before the insert can actually occur).		
#		return("Currently assembling trace for packet. <br> Progress so far:".tcptrace_show_trace($args{"con_id"},10));
		  return("Currently assembling trace for connection ".$args{"con_id"});
	      },
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;

		if(!defined($args{"con_id"})) {
		  $args{"con_id"} = "";
		};

		my $result = "<br>Select TCP connection to operate on <input type=text name=con_id value=".$args{"con_id"}."><br>";

		  return $result;	
	      },

	      "analyse" => sub {
		my $reference=shift;
		my %args=%{$reference} ;
		my $con_id=$args{"con_id"};

		check_prereq({"report"=>"connection_table","case"=>$args{"case"}});
		
		#Create the connection_cache table to store reassembled streams... This is done for speed.
		$dbh->do("create table if not exists connection_cache (con_id int, id int, direction char(1), size int, key(con_id))");

		#Do the actual stream reconstruction and store the result in the connection_cache table.
		my $sth = $dbh->do("insert into connection_cache select $con_id,ip.key_id,\">\",1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len from frame,ip,tcp,data, connection_table where frame.key_id=ip.key_id and tcp.key_id=ip.key_id and ip.key_id = data.key_id and src_ip= ip_src and dest_ip= ip_dst and src_port = tcp_srcport and dest_port = tcp_dstport and con_id = $con_id");
		  
		$dbh->do("insert into connection_cache select $con_id,ip.key_id, \"<\",1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len from frame,ip,tcp,data, connection_table where frame.key_id=ip.key_id and tcp.key_id=ip.key_id and ip.key_id = data.key_id and src_ip= ip_dst and dest_ip= ip_src and src_port = tcp_dstport and dest_port = tcp_srcport and con_id = $con_id");
	      },

	      "reset" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	
		
		$dbh->do("delete from $case.connection_cache where con_id=?",undef,$args{"con_id"});
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		$url =~ s/[\&\;]con_id=[^\&\;]*//;
		my $con_id = $args{"con_id"};
		my $case=$args{"case"};

		my $result="<h1>Data dump for TCP connection</h1>";

		#Print some details about the connection itself. Source ip:port -> dest ip:port
		my @row=$dbh->selectrow_array("select  INET_NTOA(src_ip),src_port,INET_NTOA(dest_ip),dest_port from connection_table where con_id=".$args{"con_id"});

		#count how many bytes were sent/received.
#FIXME: Make this work again.
#		@row=$dbh->selectrow_array("select sum(length(data)) from connection_cache where con_id=$con_id and direction = \"<\"");
#		$result .= "client sent $row[0] bytes,";
#		@row=$dbh->selectrow_array("select sum(length(data)) from connection_cache where con_id=$con_id and direction = \">\"");
#		$result .= "Server sent $row[0] bytes<br>";

		#Give the user a navigation bar
		$result.="<table width=100%><tr><td><a href='$url&con_id=".($con_id==1?$con_id:$con_id-1)."'><abbr title=\"Previous Connection\"><img border=0 src=\"/flag/images/left_arrow.png\"></abbr></a>      <a href='$url&con_id=".($con_id+1)."'><abbr title=\"Next Connection\"><img border=0 src=\"/flag/images/right_arrow.png\"></abbr> </a>              <a href='".$co->url(-relative=>'1')."?report=plot_sequence_number&case=$case&con_id=$con_id'><abbr title=\"Look at Sequence Numbers\"><img border=0 src=\"/flag/images/chart.png\"></abbr></a>    <a href='".$co->url(-relative=>'1')."?report=html_vis&case=$case&con_id=$con_id'><abbr title=\"View as HTML\"><img border=0 src=\"/flag/images/look.png\"></abbr> </a>  <a href='".$co->url(-relative=>'1')."?report=packets_in_connection&case=$case&con_id=$con_id'> <abbr title=\"View packets in the connection\"><img border=0 src=\"/flag/images/complex.png\"></abbr></a></td><td align=right>";
		$result.= "Connection $con_id: <a href=\"".$co->url(-relative=>'1')."?report=search_ip&case=".$args{"case"}."&src_ip=$row[2]\">$row[2]</a>:$row[3] -> <a href=".$co->url(-relative=>'1')."?report=search_ip&case=".$args{"case"}."&src_ip=$row[0]>$row[0]</a>:$row[1]</td></tr></table> ";
		
		return $result.tcptrace_show_trace($con_id,100000,$args{"case"});
	      },
             };

#This funtion wraps and sanitises its input returning a proper HTML string it removes high bit chars as well.
sub wrap_and_sanitise ($$) {
  my $input=shift;
  my $length = shift;
  $input=~s/[\x80-\xFF]/\./g;
  $input =~s/[\x01-\x09\x0e-\x1f]/\./g;
  $input =~s/\x00/\./g;
  $input=~s/\t/   /g;

  #ensure that all long lines have been properly wrapped:
#  $input=~s/([^\n\r]{$CONF{"WRAP"}})/$1 \_<img src=\"\/flag\/images\/next_line.png\"\_>\n/g;
  $input=~s/([^\n\r]{$length})/$1 \_<img src=\"\/flag\/images\/next_line.png\"\_>\n/g;
  $input=~s/\&/\&amp;/g;
  $input=~s/(^|[^\_])\</$1\&lt;/smg;
  $input=~s/(^|[^\_])\>/$1\&gt;/smg;
  $input=~s/_(\<|\>)/$1/g;

  return $input;
};

#Show the trace in a color coded form...
sub tcptrace_show_trace {
    my $con_id=shift;
    my $limit =shift;
    my $case=shift;
    my $result="";
    #Get the data from the connection cache
#    $sth = $dbh->prepare("select id,substring(data.data,size),direction from connection_cache,data where connection_cache.id=data.key_id and length(data.data)>size and con_id=$con_id group by id order by id limit $limit");
    my $sth = $dbh->prepare("select id,substring(data.data,size),direction from connection_cache,data where connection_cache.id=data.key_id and con_id=? group by id order by id limit $limit");
    $sth->execute($con_id);
    
    #Color code the output
    my $state="red";
    
    $result.="<br><font color=red><pre>";
    while(my @row=$sth->fetchrow_array) {
      if (length($row[1])>0) {
	$row[1] = wrap_and_sanitise($row[1],$CONF{"WRAP"});
	if($row[2] eq ">") {
	  if($state eq "blue") {
	    $result.="</font><font color=red>".($row[1]);
	    $state="red";
	  } else {
	    $result.="\n".($row[1]);
	  };
	} else {
	  if($state eq "red") {
	    $result.="</font><font color=blue>".($row[1]);
	    $state="blue";
	  } else {
	    $result.="\n".($row[1]);
	  };
	};
      };
    };

    #Now we make links for URLs that we might find in the code:
    my $url=$co->url(-relative=>'1')."?report=html_vis&case=$case&con_id=$con_id&url=";
    $result=~s/(GET|POST)\s+([^\s\>\<]+)\s/ "\<a href='$url".uri_escape($2)."'\>$1\<\/a\> $2 " /eg;

    return ($result."</pre>");
};

$reports{'html_vis'}={
		      "name"=>"HTML visualisation",
		      "family" => "35 TCPDump Analysis",
		      "hidden" => "yes",
		      "description" => "Replay the result returned from  a web server into the browser",
		      "form"=> $reports{'tcptrace'}{"form"}, #same as report 1
		      "parameters"=> {"case" => "flag_db" , "con_id" => "int"},
		      "progress"=>sub { return "Currently reassembling html_page"},
		      "analyse"=> sub {
			my $reference= shift;
			my %args =%{$reference} ;	
			#Ensure that we ran the tcptrace on this first:
			check_prereq({"report"=>"tcptrace","case"=>$args{"case"},"con_id"=>$args{"con_id"}});
			return 1;
		      }, #same as report 1
		      "reset"=> $reports{'tcptrace'}{"reset"}, #same as report 1

		      "display" => sub {
			my $reference= shift;
			my %args =%{$reference} ;	
			my $result="";

			my $url=$co->self_url;
			$url =~ s/[\&\;]con_id=[^\&\;]*//;
			my $con_id = $args{"con_id"};
			my $content_type;
			
			my $sth = $dbh->prepare("select id,substring(data.data,size),direction from connection_cache,data where connection_cache.id=data.key_id and con_id=$con_id group by id order by id");
			$sth->execute();
			
			while(my @row=$sth->fetchrow_array) {
			  $result.=$row[1];
			};

			#Now if we get a url parameter, we remove everything up to the URL. This should help us handle HTTP/1.1 streams.
			if (exists($args{"url"})) {
			  my $url=decode_entities($args{"url"});
			  #Here we first try to match the URL fully, then if that doesnt work we match it partially. This is because sometimes, we can only supply a fraction of the URL.
			  if (!(($result=~s/^.*?\Q$url\E //ism) || ($result=~s/^.*?\Q$url\E//ism))) {
			    return "Problem: Could not find $url in stream $con_id.$result";
			  };
#			  $result="Concatenated up to URL $url";
			};

			#remove any headers
			$result =~ s/^.*?\r?\n\r?\n//sm;

			if( $result =~ /Content-Type: ([^\r\n]*)/i) {
			  $content_type= $1;
			} else {
			  $content_type="text/html";
			};
			#compressed encodings are dangerous because we cant sanitize those properly.
#			if ($result=~/(Content-Encoding:[^\r\n]*)/i) {
#			  $content_type.="\r\n$1";
#			};

			$result =~ s/^.*?\r?\n\r?\n//sm;

			$result=~s/(GET|HEAD|POST)\s+[^\s]+\s+HTTP\/1\..*$//sm;
			
			$result=sanitize_html($result);

			#Try and find the images in this page from other connections
			$result=~s/(img|type=\"image\")(.*?)nosrc=[\"\']?([^\s\"\']*)[\"\']?/
			  "$1$2 ".find_connection($3,\%args);
			/smieg;

			print $co->header($content_type).$result;
			exit;
		      },
		     };

sub sanitize_html ($) {
  my $result=shift;
			
  #Some sanitisation things:
  $result=~s/<script/<\!-- no_script_allowed/ig;
  $result=~s/<\/script/--><ignore /ig;
  $result=~s/<(object|embed|applet|meta|xml|link)/<no_$1_allowed/ig;
  $result=~s/src=/nosrc=/ig;
  return($result);
};

#This function accepts a URL and returns a fully qualified URL ponting  to the flag reconstructed session for that url.
sub find_connection($$) {
  my $result;
  my $url=shift;
  my $reference= shift;
  my %args =%{$reference} ;	
  
  $url=~s|\.\./||g;
  $url=~s|http://||g;
  $url=~s|//|/|g;

  my $sth = $dbh->prepare("select con_id from hilights where data like \"%$url%\"");
  $sth->execute();
  if (my @row=$sth->fetchrow_array()) {
    return ("src=\"".$co->url(-relative=>'1')."?report=html_vis&case=".$args{"case"}."&con_id=".$row[0]."&url=$url\"");
  } else {
    return("nosrc=$url");
  }
};

$reports{'protocol_seen'} ={
	      "name"=>"Protocols Seen",
	      "family"=>"35 TCPDump Analysis",
	      "description"=>"This report shows all the different protocols seen within the capture with their relative frequency",
	      "parameters"=>{"case" => "flag_db"},
	      "progress"=> sub {return("Calculating protocol distribution, please wait")},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;	
		
		return show_case_form(\%args);	
	      },
			    
	      "analyse" => sub {
		my $reference= shift;
		my %args =%{$reference} ;

		#We cache the results, since it might take a few seconds with very large data sets.
		$dbh->do("create table if not exists protocols_seen select eth_type, count(eth_type ) as count from eth group by eth_type");
		
		return 1;
	      },
	      
	      "reset" => sub {
		$dbh->do("drop table protocols_seen");
		return 1;
	      },
	      
	      "display" => sub {
		  my $reference= shift;
		  my %args =%{$reference} ;	
		  my $result;
		  
		  my $url=$co->self_url;
		  
		  if(exists($args{"graph"})) {
		      
		      if(exists($args{"legend"})) {
			  print add_pie_legend("select string , count from protocols_seen, enum where name=\"eth_type\" and value=eth_type");
			  exit;
		      } elsif(exists($args{"draw"})){
			$result = draw_graph("pie","select string , count from protocols_seen, enum where name=\"eth_type\" and value=eth_type","Different protocols seen");
			  print $result;
			  exit;
		      } else {
			  $result.="<center><h1>Different protocols seen</h1>\n";
			  $result.="<img src=$url&draw=yes><img src=$url&legend=yes></center>";
		      };
		  } else {
		      #Now print the page
		      my $result = "<h1>Different protocols seen</h1><br><a href=$url&graph=yes>Click here to draw graph<a><br>";
		      
		      $result.=display_table("select eth_type as \"Eth Type\" , string as \"Description\" , count as \"Packet #\" from protocols_seen, enum where name=\"eth_type\" and value=eth_type",[$co->url(-relative=>'1')."?report=protocol_distro&case=".$args{"case"}."&protocol="],\%args,[]);
		      
		      return $result;
		  };
	      },
	     };

$reports{'protocol_distro'} ={
	      "name"=>"Examine protocol distribution",
	      "family"=>"35 TCPDump Analysis",
	      "hidden"=>"yes",
	      "description"=>"This report shows all the different protocols transport layer procols seen (e.g. UDP,TCP etc)",
	      "parameters"=>{"case" => "flag_db", "protocol"=>"int"},
	      "progress"=>sub{return("Calculating, please wait")},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"protocol"})) {
		  $args{"protocol"} = "";
		};

		my $result = "<br>Select a protocol to search for <input type=hidden name=case value=".$args{"case"}.">".selector("protocol",$args{"protocol"},"select value,string from enum where name=\"eth_type\"",$dbh);

		  return $result;
	      },

	      "analyse" => sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		#We cache the results, since it might take a few seconds with very large data sets.
		$dbh->do("create table if not exists proto_distro (`proto` int,`ip_proto` int,`count` int)");

		#IP
		if ($args{"protocol"} == 2048) {
		  $dbh->do("insert into proto_distro select 2048,ip_proto , count(ip_proto ) as count from ip group by ip_proto");
		}  elsif ($args{"protocol"} == 2054) {
#		  $dbh->do("insert into ip_proto_breakdown select 1,I.icmp_type,  count(I.icmp_type)  from icmp as I group by I.icmp_type");
		} ;
		
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		my $protocol = $args{"protocol"};
		my $result = "";

		#IP
		if($protocol == 2048) {
		  if(exists($args{"graph"})) {
		    if(exists($args{"legend"})) {
		    print add_pie_legend("select string , count from proto_distro, enum where name=\"ip_proto\" and value=ip_proto  group by ip_proto order  by count desc");
		    exit;
		  }
		    if(exists($args{"draw"})){	   
		      $result = draw_graph("pie","select string , count from proto_distro, enum where name=\"ip_proto\" and value=ip_proto  group by ip_proto order by count desc","IP protocols seen");
		      print $result;
		      exit;
		    } else {
		      $result.="<center><h1>IP protocols seen</h1>\n";
		      $result.="<img src=$url&draw=yes><img src=$url&legend=yes></center>";
		    };
		  } else {
		    #Now print the page
		    $result = "<h1>IP protocols used</h1><br><a href=$url&graph=yes>Click here to draw graph<a><br>";
		    
		    $result.=display_table("select ip_proto as \"IP Protocol\" , string as \"Description\" , count as \"Packet #\" from proto_distro, enum where name=\"ip_proto\" and value=ip_proto group by ip_proto",[$co->url(-relative=>'1')."?report=ip_proto_breakdown&case=".$args{"case"}."&ip_proto="],\%args,[]);
		    };
		}#ARP
		elsif($protocol == 2054 && 0) {
		  #$dbh->do("create temporary table temp1 select eth_type, count(eth_type ) as count  from eth group by eth_type");
 $result.=display_table("select E.eth_src as \"Source\", E.eth_dst as \"Destination\", F.frame_time as \"Time packet transmitted\", F.frame_pkt_len as \"Length of packet\" from eth as E, frame as F where eth_type = 2054 and E.key_id = F.key_id group by eth_dst",$co->url(-relative=>'1')."?report=ip_proto_breakdown&case=".$args{"case"}."&ip_proto=",\%args,[]);
		  
		}
		else {
		  $result = "<h1> Protocol not implemented yet </h1>";
		};
		
		return $result;
	      },
	     };

  $reports{'ip_proto_breakdown'} ={
	      "name"=>"IP protocol break down",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	      "description"=>"This report shows how the IP protocols are broken down",
	      "parameters"=>{"case" => "flag_db", "ip_proto"=>"int"},
	      "progress"=>sub{return ("Calculating, please wait")},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"ip_proto"})) {
		  $args{"ip_proto"} = "";
		};

		my $result = "<br>Select an IP protocol to search for <input type=hidden name=case value=".$args{"case"}.">".selector("ip_proto",$args{"ip_proto"},"select value,string from enum where name=\"ip_proto\"",$dbh);

		  return $result;
	      },

	      "analyse" => sub {
		my $reference= shift;
		my %args =%{$reference} ;

		#We cache the results, since it might take a few seconds with very large data sets.
		$dbh->do("create table if not exists ip_proto_breakdown (`ip_proto` int,`dest_port` int,`Packet Count` int)");

		if ($args{"ip_proto"} == 6) {
		  $dbh->do("insert into ip_proto_breakdown select 6,tcp_dstport , count(tcp_dstport ) as \"Packet Count\" from tcp where tcp_flags =2 group by tcp_dstport");
		} elsif ($args{"ip_proto"} == 17) {
		  $dbh->do("insert into ip_proto_breakdown select 17,U.udp_dstport, count(U.udp_dstport) from udp as U group by U.udp_dstport");
		}  elsif ($args{"ip_proto"} == 1) {
		  $dbh->do("insert into ip_proto_breakdown select 1,I.icmp_type,  count(I.icmp_type)  from icmp as I group by I.icmp_type");
		} ;
		return 1;
	      },

	      "reset" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		$dbh->do("delete from ip_proto_breakdown where ip_proto = ".$args{"ip_proto"});
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		my $ip_proto = $args{"ip_proto"};
		my $result = "";
		
		if($ip_proto == 6) { #TCP
		  if(exists($args{"graph"})) {
		    if(exists($args{"legend"})) {
		      print add_pie_legend("select dest_port as \"Destination Port\", `Packet Count` from ip_proto_breakdown where ip_proto = 6 group by dest_port order by \"Packet Count\" desc");
		      exit;
		    }
		    if(exists($args{"draw"})){
		      $result = draw_graph("pie","select dest_port as \"Destination Port\", `Packet Count` from ip_proto_breakdown where ip_proto = 6 group by dest_port order by \"Packet Count\" desc","TCP ports connected to");
		      print $result;
		      exit;
		    } else {
		      $result.="<center><h1>TCP ports connected to</h1>";
		      $result.="<img src=$url&draw=yes><img src =$url&legend=yes></center>";
		    };
		  } else {
		    #Now print the page
		    $result = "<h1>TCP ports connected to</h1><a href=$url&graph=yes>Click here to draw graph<a><br>";

		    $result.=display_table("select T.dest_port as \"Destination Port\", E.string as \"Description\", `Packet Count` from ip_proto_breakdown as T left join flag.enum as E on T.dest_port = E.value and E.name=\"tcp_proto\" where ip_proto=6",[$co->url(-relative=>'1')."?report=tcp_proto_breakdown&case=".$args{"case"}."&tcp_dstport="],\%args,[]);
		  };
		} elsif($ip_proto == 17) { #UDP
		  $result = "<h1>UDP ports connected to</h1>";
		  
#		  $result.=display_table("select U.udp_dstport as \"Destination Port\", E.string as \"Description\", count(U.udp_dstport) as \"Packet Count\" from udp U left join flag.enum as E on U.udp_dstport = E.value and E.name = \"udp_proto\" group by U.udp_dstport",[$co->url(-relative=>'1')."?report=udp_proto_breakdown&case=".$args{"case"}."&udp_dstport="],\%args);
		  $result.=display_table("select U.dest_port as \"Destination Port\", E.string as \"Description\", `Packet Count` from  ip_proto_breakdown as U left join flag.enum as E on U.dest_port = E.value and E.name = \"udp_proto\" where U.ip_proto=17",[$co->url(-relative=>'1')."?report=udp_proto_breakdown&case=".$args{"case"}."&udp_dstport="],\%args,[]);
		  
		}

		 elsif($ip_proto == 1) { #ICMP
		  $result = "<h1>ICMP Connections</h1>";
		  
		  $result.=display_table("select I.dest_port as \"Echo type (request/reply)\", F.string as \"Description\", `Packet Count` from ip_proto_breakdown as I left join flag.enum as F on I.dest_port = F.value and F.name=\"icmp_type\" where I.ip_proto=1",[$co->url(-relative=>'1')."?report=icmp_proto_breakdown&case=".$args{"case"}."&icmp_type="],\%args,[]);
		  
		}
		
		else {
		  $result = "<h1> Protocol not implemented yet </h1>";
		};

		return $result;
	      },
	     };

  $reports{'tcp_proto_breakdown'} ={
	      "name"=>"TCP protocol break down",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	      "description"=>"Lists TCP connections broken down by destination port",
	      "parameters"=>{"case" => "flag_db", "tcp_dstport"=>"int"},
	      "progress"=>sub{return "Please wait"},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"tcp_dstport"})) {
		  $args{"tcp_dstport"} = "";
		};

		my $result = "<br>Select a TCP destination port <input type=hidden name=case value=".$args{"case"}."><input type=text name=tcp_dstport value=".$args{"tcp_dstport"}.">";

		  return $result;
	      },

	      "analyse" => sub {
		my $reference=shift;
		my %args =%{$reference} ;

		#Check to see if we ran report connection_table which is a prepreq for this one.
		check_prereq({"report"=>"connection_table","case"=>$args{"case"}});
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		
		my $tcp_dstport = $args{"tcp_dstport"};
		my $result = "";

		#Now print the page
		$result = "<h1>Connections to port $tcp_dstport</h1>";
		
		$result.=display_table("select con_id as  \"Connection\",INET_NTOA(src_ip) as \"Source IP\",src_port as \"Source Port\",INET_NTOA(dest_ip) as \"Dest IP\",dest_port as \"Dest Port\",count  from connection_table where dest_port=\"$tcp_dstport\" or src_port = \"$tcp_dstport\" group by con_id",[$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=",$co->url(-relative=>'1')."?report=search_ip&case=".$args{"case"}."&src_ip="],\%args,[]);
		
		return $result;
	      },
	     };

  $reports{'udp_proto_breakdown'} ={
	      "name"=>"UDP protocol break down",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	      "description"=>"This report shows how TCP connections are broken down by destination port",
	      "parameters"=>{"case" => "flag_db", "udp_dstport"=>"int"},
	      "progress"=>sub{},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		my $result = show_case_form(\%args)."<br>Select a UDP destination port<input type=text name=udp_dstport value=".$args{"udp_dstport"}.">";

		  return $result;
	      },

	      "analyse" => sub {
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		
		my $result = "";

		#Now print the page
		$result = "<h1>Packets to UDP port ".$args{"udp_dstport"}."</h1>\n";
		
		$result.=display_table("select udp.key_id as \"Packet ID\",concat(INET_NTOA(ip.ip_src),\" -> \",INET_NTOA(ip.ip_dst)) as \"Communicating IPs\", count(*) as \"Total Packets\" from udp,ip where ip.key_id =udp.key_id and udp.udp_dstport = ".$args{"udp_dstport"}." group by ip.ip_src , ip.ip_dst",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id=",$co->url(-relative=>'1')."?report=udp_packet_stream&case=".$args{"case"}."&connection="],\%args,[]);
		
		return $result;
	      },
	     };

  $reports{'udp_packet_stream'} ={
	      "name"=>"UDP stream lists",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	      "description"=>"Lists all UDP packets exchanged between two hosts",
	      "parameters"=>{"case" => "flag_db", "connection"=>"string"},
	      "progress"=>sub{return("Calculating, please wait")},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		my $result = "<br>Select a TCP destination port <input type=hidden name=case value=".$args{"case"}."><input type=text name=tcp_dstport value=".$args{"tcp_dstport"}.">";

		  return $result;
	      },

	      "analyse" => sub {
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	
		my $url=$co->self_url;
 		my $connection = $args{"connection"};
		my $result;
		my $ip1;
		my $ip2;

		if ( $connection =~ /([^ ]*) -(>|\&gt;) ([^ ]*)/ ) {
		  $ip1 = $1;
		  $ip2 = $3;
		} else {
		  return ("<h1> Error </h1> Could not parse connection $connection");
		};

		#Now print the page
		$result = "<h1>UDP packets between $ip1 and $ip2 ".$args{"udp_dstport"}."</h1>\n";
		
		$result.=display_table("select udp.key_id as \"Packet ID\",INET_NTOA(ip.ip_src) as \"Src IP\", udp_srcport as \"Source Port\" , INET_NTOA(ip.ip_dst) as \"Dest IP\", udp_dstport as \"Dest Port\" from udp,ip where ip.key_id =udp.key_id and (ip_src=INET_ATON(\"$ip1\") and ip_dst=INET_ATON(\"$ip2\") or ip_src=INET_ATON(\"$ip2\") and ip_dst=INET_ATON(\"$ip1\"))",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
		
		return $result;
	      },
	     };


#ICMP stuff
  $reports{'icmp_proto_breakdown'} ={
	      "name"=>"ICMP protocol break down",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	      "description"=>"This report shows how ICMP connections are broken down by their type",
	      "parameters"=>{"case" => "flag_db", "icmp_type"=>"int"},
	      "progress"=>sub{},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"icmp_type"})) {
		  $args{"icmp_type"} = "";
		};

		my $result = "<br>Select a ICMP type <input type=hidden name=case value=".$args{"case"}."><input type=text name=tcp_dstport value=".$args{"tcp_dstport"}.">";

		  return $result;
	      },

	      "analyse" => sub {
		my $reference=shift;
		my %args =%{$reference} ;

		#Check to see if we ran report 0 which is a prepreq for this one.
		check_prereq({"report"=>"connection_table","case"=>$args{"case"}});

		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		
		my $icmp_type = $args{"icmp_type"};
		my $result = "";

		#Now print the page
		$result = "<h1>Protocol type $icmp_type</h1>";
		
		$result.=display_table("select F.key_id,F.frame_time as \"Time\", I.icmp_type, I.icmp_code,F.frame_pkt_len as \"Length of packet\", INET_NTOA(IP.ip_src) as \"Source IP\", INET_NTOA(IP.ip_dst) as \"Destination IP\" from frame as F, icmp as I, ip as IP where I.key_id = F.key_id and I.icmp_type = $icmp_type and I.key_id = IP.key_id",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
		
		return $result;
	      },
	     };

$reports{'packets_in_connection'} ={
	      "name"=>"Packets involved in a connection",
	      "family"=>"35 TCPDump Analysis",
	       "hidden" => "yes",
	       "description"=>"This report will show information about packets that were involved in a particular connection and their details.",
	      "parameters"=>{"case" => "flag_db", "con_id"=>"int"},
	      "progress"=>sub{},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"con_id"})) {
		  $args{"con_id"} = "";
		};

		my $result = "<br>Select a connection ID <input type=hidden name=case value=".$args{"case"}."><input type=text name=con_id value=".$args{"con_id"}.">";

		  return $result;
	      },

	      "analyse" => sub {
		my $reference=shift;
		my %args =%{$reference};
		
		#Check to see if we ran report 0 which is a prepreq for this one.
		check_prereq({"report"=>"tcptrace","case"=>$args{"case"},"con_id"=>$args{"con_id"}});

		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		
		my $con_id = $args{"con_id"};
		my $result = "";


		#Now print the page
		$result = "<h1>Connection Information</h1><br>";
		
		$result.=display_table("select id as \"Packet ID\", INET_NTOA(ip_src) as \"Source IP Address\", tcp_srcport \"TCP source port\", INET_NTOA(ip_dst) as \"Destination IP address\", tcp_dstport \"TCP destination port\", ip_len as \"IP Length\" from connection_cache,ip, tcp where con_id=$con_id and ip.key_id=id and ip.key_id = tcp.key_id group by id",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
		return $result;
	      },
       };

$reports{'plot_sequence_number'} = 
  { "name" => "Plot Sequence Numbers",
    "family" => "35 TCPDump Analysis",
    "hidden"=>"1",
    "description"=>"Plot the sequence numbers for each connection in both the forward and reverse directions against time",
    "parameters"=>{"case" => "flag_db", "con_id"=>"int"},
    "progress"=>sub {return 1},
    "form"=>sub {
      my $reference= shift;
      my %args =%{$reference} ;
      
      if(!defined($args{"key_id"})) {
	$args{"key_id"} = "";
      };
      
      my $result = "<br>Select Flag databases to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
      $result .= "<br>Select a connection to plot: <input type=text name=con_id value=".$args{"con_id"}.">";
      
      return $result;
    },

    "analyse" => sub {
      my $reference=shift;
      my %args=%{$reference} ;
      
      check_prereq({"report"=>"tcptrace","case"=>$args{"case"},"con_id"=>$args{"con_id"}});
    },
    "reset" => sub {return "Resetting Report"},
    "display" => sub {
      my $reference= shift;
      my %args =%{$reference} ;
      my $con_id = $args{"con_id"};
      my $url=$co->self_url;
      my $result;
      if (exists($args{"draw"})) {
	$result=draw_graph("lines","select b.key_id,tcp_seq from connection_cache as a,tcp as b,frame as c where a.id=b.key_id and b.key_id=c.key_id and con_id=\"$con_id\" and direction=\"<\"","");
	print $result;
	exit;
      } elsif (exists($args{"draw2"})) {
	$result=draw_graph("lines","select b.key_id,tcp_seq from connection_cache as a,tcp as b,frame as c where a.id=b.key_id and b.key_id=c.key_id and con_id=\"$con_id\" and direction=\">\"","");
	print $result;
	exit;
      } elsif(exists($args{"graph"}))  {
	$result="<center><h1> Sequence numbers for TCP connection ".$args{"con_id"}."</h1>\n<img src=$url&draw=please width=600 height=400><br><img src=$url&draw2=please width=600 height=400></center>";

	return $result;
      } else {
	$result="<center><h1> Sequence numbers for TCP connection ".$args{"con_id"}."</h1>\n<a href=$url&graph=please>Click here to plot a graph</a>";
	$result.=display_table("select b.key_id as \"Packet ID\",tcp_seq as \"TCP Sequence\",direction as \"Direction\" from connection_cache as a,tcp as b,frame as c where a.id=b.key_id and b.key_id=c.key_id and con_id=\"$con_id\"",[],\%args,[])."</center>";
	return $result;
      }
    },
  };

$reports{'firewall_log_analysis'} = 
  { "name" => "Slice and Dice",
    "family" => "37 FW Logs Analysis",
    "description"=>"Slice and Dice the firewall logs by IPs,Ports, Actions etc.",
    "parameters"=>{"case" => "flag_db","table"=>"firewall_table","where"=>"whereclause"},
    "progress"=>sub {1},
    "form"=>sub {
      my $reference= shift;
      my $src_ip;
      my %args =%{$reference} ;

      my $result="<b>Step 1</b>: Select case to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."<input type=submit value=\"Update\"><p><b>Step 2</b>: Select firewall table to operate on: ".selector("table",$args{"table"},"select value, value from meta where property='firewall' group by value",$dbh);
      $result.="<p><b> (Optional) search limiting parameters</b> (Use % as the wildcard character)<p><b>Step 3</b>:  ".$co->scrolling_list('where',['source_ip','source_port','dest_ip','dest_port','action','type'],$args{"where"},1,0,{'source_ip'=>'Source IP','source_port'=>"Source Port",'dest_ip'=>"Dest IP",'dest_port'=>"Dest Port",'action'=>"Action",'type'=>"Packet Type"})." has the value <input type=text name=group><p><b>Step 4</b>: ";
      return $result;

    },
    "analyse" => sub {
      my $reference= shift;
      my %args =%{$reference} ;
#      $dbh->do("create table firewall_hits select time_stamp, dest_ip_port, floor(unix_timestamp(time_stamp)/60) as minute, count(floor(unix_timestamp(time_stamp)/60)) as count from Firewall group by minute");
      return 1;
    },

    "reset" => sub {1},
    "display" => sub {
      my $reference= shift;
      my %args =%{$reference} ;
      my $url=$co->self_url;
      my $where = "";
      my $minimum;
      my $result;

      if(exists($args{"graph"})) {
	my $sth = $dbh->prepare("Select min(minute) from firewall_hits");
	$sth->execute();
	
	my @row;
	while(@row = $sth->fetchrow_array()) {
	  $minimum = $row[0];
	}
	$sth->finish();
	
	if(exists($args{"draw"})){
	  $result = draw_graph("bar","select (minute - $minimum) as \"Minute Count\", count from firewall_hits group by (minute - $minimum) order by count desc","Type of whatever seen");
	  print $result;
	  exit;
	}
	else {
	  $result.="<center><h1>Distribution of Hit Times</h1>\n";
	  $result.="<img src=$url&draw=yes>";
	   return $result;
	};
      } else 
	{
	  my $result = "<h1> Firewall Logs</h1><table><tr width=100%><td width=30%>";
	  #<br><a href=$url&graph=yes>Click here to draw graph for type of requests<a><br><br>";
	  if (exists($args{"where"}) && length($args{"where"} && length($args{"group"})>1)) {
	    $result.="Limiting search for ".$args{"where"}."=".$args{"group"};
	  };

	  $result.="</td><td width=30%></td><td width=30%>";
	  if(exists($args{"where"}) && exists($args{"group"}) && length($args{"group"})>1) {
	    $where = "where ".$args{"where"}." like \"".$args{"group"}."\"";
	  };

	  #Tell the user how many rows are in this table;
	  my @row=$dbh->selectrow_array("select count(*) from ".$args{"table"}." $where");
	  $result.="<b> Total row count ".$row[0]."</b></td></tr></table><p>";

	  
	  $result.=display_table("select timestamp as \"Time\",source_ip as \"Source IP\", source_port as \"Source Port\",dest_ip as \"Destination IP\" , dest_port as \"Destination Port\", action as \"Action\",type as \"Type\" from ".$args{"table"}." $where",["",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=source_ip&group=",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=source_port&group=",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=dest_ip&group=",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=dest_port&group=",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=action&group=",$co->url(-relative=>'1')."?report=firewall_log_analysis&case=".$args{"case"}."&table=".$args{"table"}."&where=type&group="],\%args,[]);
	  
	  return $result;
      };
    },
  };

$reports{'packet_breakdown'} ={
	      "name"=>"Packet Breakdown",
	      "family"=>"35 TCPDump Analysis",
	      "description"=>"This report will show information about a packet similar in function to the way that Ethereal does.",
	      "parameters"=>{"case" => "flag_db", "key_id"=>"int"},
	      "progress"=>sub{},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		if(!defined($args{"key_id"})) {
		  $args{"key_id"} = "";
		};

		my $result = "<br>Select Flag databases to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
		$result .= "<br>Select a packet to retrieve information about: <input type=text name=key_id value=".$args{"key_id"}.">";

		  return $result;
	      },

	      "analyse" => sub {
	      },

	      "reset" => sub {
		return 1;
	      },

	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;
		
		my $key_id = $args{"key_id"};
		my $result = "";

		#Now print the page
		$result = "<h1>Packet Information - ".$args{"key_id"}."</h1><br>";
		
		my @tables;
		my @tmp;
		my @tmp2;
		my @table_data;
		#This hash represets SQL statements that must be made in order to convert the values stored in the database into human readable quantities. Should this be stored in the database????
		my %translations=
		  (
		   "eth_type"=>'select concat(string," (",value,")") from enum where name="eth_type" and value=? group by value',
		   "ip_dst" => 'select INET_NTOA(?)',
		   "ip_src" => 'select INET_NTOA(?)',
		   "ip_proto"=>'select concat(string," (",value,")") from flag.enum where name="ip_proto" and value=? group by value',
		   "tcp_dstport"=>'select concat(string," (",value,")") from flag.enum where name="tcp_proto" and value=? group by value',
		   "tcp_srcport"=>'select concat(string," (",value,")") from flag.enum where name="tcp_proto" and value=? group by value',
		   "tcp_flags"=>'select concat(string,"\n") from flag.enum where value & ? and name="tcp_flags_txt"',
		   "ip_flags"=>'select concat(string,"\n") from flag.enum where value & ? and name="ip_flags_txt"',
		   "icmp_type"=>'select concat(string," (",value,")") from flag.enum where name="icmp_type" and value=? group by value',
		   "http_request_headers" => 'select REPLACE(http_request_headers,"\\\\r\\\\n","\n") from http where http_request_headers = ?',
		  );		
		# Get the table names
				
		my $query = "select value from meta where property = \"tcpdump_table\" group by value";
		my $sth = $dbh->prepare($query);
		$sth->execute();
				
		while (@tables = $sth->fetchrow_array()){
		  push @tmp2, $tables[0];
		}

		# Now we need to get all the values that a packet can be associated with.
		my $count = 0;
		my $i;
		
		foreach (@tmp2) {
		  my $table_name=$_;
		  $query = "select * from $table_name where key_id = '$key_id'";
		  $sth = $dbh->prepare($query);
		  $sth->execute();
		  my @headers = @{$sth->{NAME}};
		  if (@table_data = $sth->fetchrow_array()) {
		    $result.="<center><b> Table $table_name</b></center>";
		    $result .= "<table width = \"100%\" border = \"4\">";
		    for($i=0; $i<=$#headers; $i++) {
		      if (defined($headers[$i]) && $headers[$i] eq "key_id") {
			next;
		      }
		      $result .= "<th valign=top><strong>".join("<br>",split("_",$headers[$i]))."</strong> </th>";
		    }
		    $result .= "<tr>";
		    for($i=0; $i<=$#table_data; $i++) {
		      if ($headers[$i] eq "key_id") {
			next;
		      }
		      $result.="<td valign=top>\n";
		      if (exists($translations{$headers[$i]})) {
			#We have a translation we need to use:
			my $sth=$dbh->prepare($translations{$headers[$i]});
			if ($sth->execute($table_data[$i])>0) {
			  while(my @row=$sth->fetchrow_array) {
			    $result .=  "<pre>".wrap_and_sanitise($row[0],$CONF{"WRAP"}/2)."</pre>";
			  };
			} else {
			  $result .= "<pre>".wrap_and_sanitise($table_data[$i],$CONF{"WRAP"}/2)."</pre>";
			}
		      } else {
			$result .= "<pre>".(wrap_and_sanitise($table_data[$i],$CONF{"WRAP"}/2) || "")."</pre>";
		      }
		      $result.=" &nbsp;</td>";
		    };
		    $result .= "</tr>";
		    $result .= "</table><br><br>";
		    
		  }
		}

		$result .= "<h2> Hex Dump of the packet </h2>";
		$query = "select data from data where key_id = $key_id";
		$sth = $dbh->prepare($query);
		$sth->execute();
		my @hex= $sth->fetchrow_array();
		#Now we make a hex dump looking thing:
		
       		my $f = new Data::HexDump;
       		$f->data($hex[0]);
       		$result.="\n<pre>".encode_entities($f->dump)."</pre>";

		#Find the connection associated with this packet
		$query = "select tcp_srcport,tcp_dstport from tcp where key_id = '$key_id'";
		$sth = $dbh->prepare($query);
		$sth->execute();
		if (my @ports=$sth->fetchrow_array()) {
		  $query = "select ip_src,ip_dst from ip where key_id = '$key_id'";
		  $sth = $dbh->prepare($query);
		  $sth->execute();
		  if (my @ips=$sth->fetchrow_array()) {
		    $query = "select con_id from connection_table where (src_ip=$ips[0] and dest_ip=$ips[1] and src_port=$ports[0] and dest_port=$ports[1]) or (src_ip=$ips[1] and dest_ip=$ips[0] and src_port=$ports[1] and dest_port=$ports[0])";
		    $sth = $dbh->prepare($query);
		    $sth->execute();
		    if (my @con_id=$sth->fetchrow_array()) {
		      $result.="<br><a href=\"".$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=$con_id[0]\">Click here to see the whole connection</a>";
		    }
		  }
		}
	
		print print_headers(["key_id"],($args{"key_id"}-1),($args{"key_id"}+1),'').$result.$flag_footer;
		exit;
	      },
	  };

$reports{'dns_traffic'}= 
{
 "name"=>"DNS Traffic",
 "family" => "35 TCPDump Analysis",
 "description" => "View DNS traffic",
 "parameters"=>{"case" =>"flag_db"},
 "progress" => sub {
   return ("Currently pondering, please wait");
 },
 "form" => sub {
   		my $reference= shift;
		return show_case_form($reference);
 },
 "analyse" => sub {return 1},
 "display" => sub {
   my $reference= shift;
   my %args =%{$reference} ;
   
   my $result="<h1>DNS table</h1>\n";
   
   $result.=display_table('select dns_data.key_id as "Packet ID",if(isnull(data),"Q","RR") as "Type",dns_data.name as "Name",type as "DNS Type",class as "Class",data as "Data", INET_NTOA(ip_src) as "Source", INET_NTOA(ip_dst) as "Dest" from dns_data,ip where ip.key_id=dns_data.key_id group by `Packet ID`,data',[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
   return $result;
 },
 "reset"=>sub{return},
};

$reports{"trace_info"}={
	     "name" => "Trace Information",
	     "family" => "35 TCPDump Analysis",
	     "description" => "Shows information about important connections",
	     "parameters" => {"case"=>"flag_db"},
	     "progress" => sub {
	       my $reference= shift;
	       my %args =%{$reference};

	       return ("Currently calculating information from TCP data:<p>".get_progress(\%args));
	     },
	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference};

	       my $result = "<br>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
	       return $result;
	     },
	     "analyse" => sub {
	       my $reference= shift;
	       my %args =%{$reference};
	       $dbh->do("CREATE TABLE `hilights` (`key_id` INT NOT NULL,`con_id` INT NOT NULL,`type` VARCHAR(50) NOT NULL,`data` TEXT NOT NULL,INDEX (`key_id`),key(type) )");

	       set_progress("Collecting HTTP requests");
	       #Look for HTTP requests:
	       $dbh->do("insert into hilights select data.key_id, con_id ,\"HTTP:request\",substring(left(data,locate(\"HTTP\",data,1)+7),1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) from data,tcp,frame,ip, connection_table where data.key_id =tcp.key_id and  tcp.key_id=ip.key_id and ip.key_id=frame.key_id and (tcp.tcp_dstport =80 or tcp.tcp_dstport =8080 or  tcp.tcp_dstport =8008) and data like \"%HTTP%\" and ((ip_src = connection_table.src_ip and tcp_srcport=connection_table.src_port and ip_dst=connection_table.dest_ip and tcp_dstport = dest_port) or (ip_src = connection_table.dest_ip and tcp_srcport=connection_table.dest_port and ip_dst=connection_table.src_ip and tcp_dstport = src_port)) group by con_id,data");

	       set_progress("Collecting SMTP RCPT headers (Mail recipients)");
	       #Look for RCPT headers
	       $dbh->do("insert into hilights select data.key_id, con_id ,\"SMTP:RCPT To\",substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) from data,tcp,frame,ip, connection_table where data.key_id =tcp.key_id and  tcp.key_id=ip.key_id and ip.key_id=frame.key_id and tcp.tcp_dstport =25 and substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) like \"RCPT \%\" and ((ip_src = connection_table.src_ip and tcp_srcport=connection_table.src_port and ip_dst=connection_table.dest_ip and tcp_dstport = dest_port) or (ip_src = connection_table.dest_ip and tcp_srcport=connection_table.dest_port and ip_dst=connection_table.src_ip and tcp_dstport = src_port)) group by con_id,data");

	       set_progress("Collecting SMTP Mail senders");
	       #Look for Mail From:
	       $dbh->do("insert into hilights select data.key_id, con_id ,\"SMTP:Mail From\",substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) from data,tcp,frame,ip, connection_table where data.key_id =tcp.key_id and  tcp.key_id=ip.key_id and ip.key_id=frame.key_id and
tcp.tcp_dstport =25 and substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) like \"MAIL From\%\" and ((ip_src = connection_table.src_ip and tcp_srcport=connection_table.src_port and ip_dst=connection_table.dest_ip and tcp_dstport = dest_port) or (ip_src = connection_table.dest_ip and tcp_srcport=connection_table.dest_port and ip_dst=connection_table.src_ip and tcp_dstport = src_port)) group by con_id,data");

	       set_progress("Collecting POP3 usernames and passwords");
	       #POP usernames and passwords:
	       $dbh->do("insert into hilights select data.key_id, con_id ,\"POP:User\",substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) from data,tcp,frame,ip, connection_table where data.key_id =tcp.key_id and  tcp.key_id=ip.key_id and ip.key_id=frame.key_id and tcp.tcp_dstport =110 and substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) like \"USER \%\" and ((ip_src = connection_table.src_ip and tcp_srcport=connection_table.src_port and ip_dst=connection_table.dest_ip and tcp_dstport = dest_port) or (ip_src = connection_table.dest_ip and tcp_srcport=connection_table.dest_port and ip_dst=connection_table.src_ip and tcp_dstport = src_port)) group by con_id,data");

	       $dbh->do("insert into hilights select data.key_id, con_id ,\"POP:Password\",substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) from data,tcp,frame,ip, connection_table where data.key_id =tcp.key_id and  tcp.key_id=ip.key_id and ip.key_id=frame.key_id and tcp.tcp_dstport =110 and substring(data,1+frame_pkt_len-ip_len+ ip_hdr_len+ tcp_hdr_len) like \"PASS \%\" and ((ip_src = connection_table.src_ip and tcp_srcport=connection_table.src_port and ip_dst=connection_table.dest_ip and tcp_dstport = dest_port) or (ip_src = connection_table.dest_ip and tcp_srcport=connection_table.dest_port and ip_dst=connection_table.src_ip and tcp_dstport = src_port)) group by con_id,data");

	       clear_progress();
	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result="<h1>Interesting transactions from TCP streams</h1>\n";
	       my @where=();
	       if (exists($args{"type"}) && $args{"type"}=~/^([^:]*)/) {
		 push @where, " type like '$1\%' ";
		 $result.=" Limiting search to traffic of type '$1'<br>\n";
	       };

	       if (exists($args{"data"})) {
		 push @where, "data like \"%".$args{"data"}."%\" ";
		 $result.="Limiting search to data matching '".$args{"data"}."'<br>\n";
	       };

	       my $where="";
	       if ($#where>=0) {
		 $where="where ".join(' and ',@where);
		 $result.="<a href=".$co->url(-relative=>'1')."?report=trace_info&case=".$args{"case"}.">Click here for an unlimited view</a>";
	       }

	       if (!exists($args{"data"})) {
		 $args{"data"}="";
	       };
	       if (!exists($args{"type"})) {
		 $args{"type"}="";
	       };

	       $result.="<form action=".$co->url(-relative=>'1')."><input type=hidden name=report value=trace_info><input type=hidden name=case value=".$args{"case"}."><input type=hidden name=data value=\"".$args{"data"}."\"> Select type to limit search by: ".selector("type",$args{"type"},"select type,left(type,instr(type,\":\")-1) as protocol from hilights group by protocol")."<input type=submit></form>";

	       $result.="<form action=".$co->url(-relative=>'1')."><input type=hidden name=report value=trace_info><input type=hidden name=case value=".$args{"case"}."><input type=hidden name=type value=\"".$args{"type"}."\"> Search data component for key word: <input type=text name=data><input type=submit></form>";

	       $result.=display_table("select key_id as \"Packet ID\", con_id as \"Connection\",type as \"Transaction Type\",data as \"Data\" from hilights $where",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id=",$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=",$co->url(-relative=>'1')."?report=trace_info&case=".$args{"case"}."&data=".($args{"data"}||'')."&type="],\%args,[]);
	       return $result;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       $dbh->do("drop table hilights");
	       return "1";
	     },
	    };



###############################################################
#
#  Knowledge base
#
###############################################################
#This report draws a network diagram from the knowledge base using dot (from the graphviz package
$reports{'draw_dot'} =
  {
   "name"=>"Draw Network diagram",
   "family" => "40 Knowledge Base",
   "description" => "This report draws a network diagram from the TCP dump",
   "parameters"=>{"case" =>"flag_db","prog"=>"alphanum","top_level"=>"string"},
   "progress"=> sub {return("Creating knowledge base")},
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference};
     #We require the knowledge base to be executed first
     check_prereq({"report"=>"create_knowledge_base","case"=>$args{"case"}});
   },
   "form" => sub {
     my $reference= shift;
     
     my %args =%{$reference} ;
     if(!exists($args{"case"})) {
       $args{"case"} = "";
     };
     
     my $result = "<br>Select Flag databases to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
     $result.="<input type=submit value=Reftesh><br>Root Node type: ".selector("top_level",$args{"top_level"},"select ptype,ptype from knowledge group by ptype",$dbh)."<br>Plotting program: ".$co->scrolling_list('prog',['dot','neato','twopi','text'],$args{"prog"},1);
$result.="<br><h1> Plotting Options</h1>\n<table><tr><td>Show disconnected objects</td><td><input type=checkbox name=show_disconnected value=yes></td></tr>
<tr><td>Show deductions: </td><td>".$co->scrolling_list('deductions',$dbh->selectcol_arrayref('select description from knowledge where link = "transitive" group by  description'),[],5,'true').
     "</td></tr></table>";
     return $result;
   },
   "display" => sub {
     my $reference= shift;
     my %args =%{$reference};
     my $where;
     my $top_level=$args{"top_level"};
     my $result="Drawing network diagram<br>\n";
     my %obj=();
     my $obj_count=0;
     my $graph="strict digraph G{\noverlap=scale;\nconcentrate=true\n";
     my $knowledge="knowledge_node";
     
     #First establish the types of deductions the user wanted to have:
     my @deductions = $co->param("deductions");
     my $condition = "description='".join("' or description='",@deductions)."' ";
     

     #Prototype
     sub find_label($$$$);

     #Function dives into the subtree given by $type,$value for a better label. Labels are given by the hash %labels.
     sub find_label($$$$) {
       my $pname=shift;
       my $labels = shift;
       my $type= shift;
       my $value = shift;
       my @row=();

       #If there are any child nodes - follow them
       my $sth = $dbh->prepare("select name,type,value,ptype,parent from knowledge where pname=? and link='no'");
       $sth->execute($pname);

       #If the label is in the labels hash - update it.
       while (@row=$sth->fetchrow_array) {
	find_label($row[0],$labels,$row[1],$row[2]);
       };

       if (exists( $labels->{$type}  )) {
		 $labels->{$type}.=" ".$value;
       };
     };

     #Find all root nodes from the knowledge table that were also mentioned in transitive links:
     if (!exists($args{"show_disconnected"})) {
       do_query("create temporary table temp select b.name,b.type,b.value from knowledge_node as b, knowledge as a where (b.name = a.name or b.name=a.pname) and link='transitive' and ($condition) group by b.name; CREATE  temporary TABLE `my_knowledge` (`type` varchar (50),`name` VARCHAR(50),`value` VARCHAR(50) NOT NULL)");

       my $sth=$dbh->prepare("select type,name,value from temp");
       $sth->execute();
       while(my @row = $sth->fetchrow_array) {
	 my $dest_type= $row[0];
	 my $dest_name = $row[1];
	 my $dest_value = $row[2];
	 
	 #Follow each node up the tree to find the type we want
	 while ($dest_type ne $top_level && (my @row2=$dbh->selectrow_array("select ptype,pname,parent from knowledge where name='$dest_name' and link='no'"))) {
	   $dest_type=$row2[0];
	   $dest_name=$row2[1];
	   $dest_value=$row2[2];
	 };
	 do_query("insert into my_knowledge set type='$dest_type',value='$dest_value',name='$dest_name'");
       };
       $knowledge="my_knowledge";
     };
     
     #find those nodes matching the $top_level specified.
     my $sth=$dbh->prepare("select name,type,value from $knowledge where type=?");
     $sth->execute($top_level);
     while (my @row=$sth->fetchrow_array) {
       my %labels=("ip_addr"=>" ","eth_addr"=>" ","dns_A"=>' ');
       #Search the tree for a good label:
       find_label($row[0],\%labels,$row[1],$row[2]);
       my $label="";
       foreach my $i (reverse keys %labels) {
	 if ($labels{$i} ne " ") {
       	   $label.=$labels{$i}."\\n";
	 };
       };

       if (!exists($obj{$row[0]})) {
	 $obj{$row[0]}=$obj_count++;
	 $graph.="obj_".$obj{$row[0]}." [shape=box, label=\"$label\", URL=\"".$co->url(-relative=>'1')."?report=display_object&case=".$args{"case"}."&type=".uri_escape($top_level)."&value=".uri_escape($row[2])."\"]\n";
       };
     };

     #Find all the transitive links as connections between objects.
     $sth=$dbh->prepare("select type,name,value,ptype,pname,parent,description from knowledge where link='transitive' and ($condition)");
	
     $sth->execute();
     while (my @row=$sth->fetchrow_array) {
       my $type=$row[0];
       my $name=$row[1];
       my $value=$row[2];
       my $dest_type=$row[3];
       my $dest_name=$row[4];
       my $description=$row[6];
       my @row2;

       #Follow links up the tree until we reach the top_level we are after. for the start of the link...
       while ($type ne $top_level && (@row2=$dbh->selectrow_array("select ptype,pname,parent from knowledge where name='$name' and link='no'"))) {
	 $type=$row2[0];
	 $name=$row2[1];
       };

       #Do same for the destination of the link:
       while ($dest_type ne $top_level && (@row2=$dbh->selectrow_array("select ptype,pname,parent from knowledge where name='$dest_name' and link='no'"))) {
	 $dest_type=$row2[0];
	 $dest_name=$row2[1];
       };
       $graph.="obj_".$obj{$dest_name}." -> obj_".$obj{$name}."\[color=black, label=\"$description\"\];\n";
     };

     $graph.="}";

     #Ensure that we are running the right program here..
     if ($args{"prog"} ne "dot" && $args{"prog"} ne "neato" && $args{"prog"} ne "twopi" && $args{"prog"} ne "text") {
       return $result."<h1> Error </h1> You must specify a valid plotting tool, Currently dot,neato and twopi are supported.";
     };

     if ($args{"prog"} eq "text") {
       return "<pre>$graph</pre>";
     } else {
       my $result="";
       #We use IPC::Run rather than IPC::Open2 for mod_perl compatibility (and its better anyway).
       run([$CONF{"DOTDIR"}.'/'.$args{"prog"},"-Tsvg","-Gstart=rand"],\$graph,\$result);
       print $co->header(-type=>"image/svg+xml").$result;
       exit;
     };
     return($graph);
   },
   "reset"=>sub{return},
  };

$reports{'mac-ip_mapping'}= {
	     "name"=>"Ethernet MAC address analysis",
	     "family" => "40 Knowledge Base",
	     "hidden"=>"yes",
	     "description" => "The report analyses MAC addresses to add to the knowledge base",
	     "parameters"=>{"case" =>"flag_db"},
	     "progress" => sub {
	        return ("Currently pondering, please wait");
	      },
                      "form" => $reports{"connection_table"}{"form"},
	     "analyse" => sub {return},
#Fixme : eventually this sort of thing should be done in the knowledge base stuff and this query should just query it.
                      "display" => sub {
                 	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result="<h1>MAC address - IP Address mapping</h1>\n";

	       $result.=display_table("select eth.key_id as \"Packet\",eth_src as \"MAC\",INET_NTOA(ip_src) as \"IP\" from eth,ip left join icmp on ip.key_id=icmp.key_id where eth.key_id=ip.key_id and isnull(icmp.key_id) group by eth_src,ip_src",[$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
	       return $result;
	      },
	    "reset"=>sub{return},
};

$reports{'create_knowledge_base'} = {
	    "name" => "Create Knowledge base",
	    "description"=>"This report creates a knowledge base of all traffic.",
                     "family" => "40 Knowledge Base",
	    "progress" => sub {
	      my $reference= shift;
	      my %args =%{$reference};

	      return ("Currently building knowledge base: <p> ".get_progress(\%args));
	    },
                     "parameters" => {"case"=>"flag_db"},
	    "form"=> $reports{'connection_table'}{'form'},
	    "analyse" => sub {
	      my $reference= shift;
	      my %args =%{$reference};

	      #The knowledge base is a graph which consists of nodes and edges. Nodes are characterised by their type and name. Each row in the table is an edge between nodes which are infered.
	      set_progress("Creating knowledge base table");
	      #parent is the entity we associate value with. Note that link specifies if the links between the nodes is transitive (i.e. we are linking two distinct object trees at the same level rather than linking nodes within the same tree). - In other words when declaring a property of an object link = NULL, when describing interactions between seperate objects link = "transitive".
# These are the fields used:
	      #type - the type of the node
	      #name - a unique name for the node. (Note that often the best unique name can be made by using a combination of the node value, type and its parent value - The name is a label which is used to access the node uniquely).
	      #value - Usually the label that will be displayed in object trees or network diagrams.
	      #
	      # Note that names are guaranteed to be unique for each node.
	      my $query="CREATE TABLE `knowledge` (`field` VARCHAR(50) NOT NULL,`packet` INT NOT NULL,`type` varchar(50),`name` VARCHAR(50),`value` VARCHAR(50) NOT NULL,`ptype` varchar(50),`pname` VARCHAR(50),`parent` varchar(100), description varchar(100) NOT NULL, link enum('no', 'transitive'), INDEX ( `packet`)) COMMENT = 'Stores knowledge base information';";

	      $query.="insert into knowledge select 'eth_src',eth.key_id,'ip_addr',concat('ip:',INET_NTOA(ip_src)),INET_NTOA(ip_src),'eth_addr',concat('eth:',eth_src),eth_src,'IP Address','no' from eth,ip left join icmp on ip.key_id=icmp.key_id where eth.key_id=ip.key_id and isnull(icmp.key_id) and ip_src>0 group by eth_src,ip_src;";
	      do_query($query);

	      set_progress("Finding accepted TCP connections");

	      #These are the accepted TCP connections (Locating packets with syn/ack)
	      $query="insert into knowledge select 'tcp_srcport',ip.key_id, 'listening_tcp_port',concat('LT_',INET_NTOA(ip_src),':',tcp_srcport),tcp_srcport, 'ip_addr', concat('ip:',INET_NTOA(ip_src)) ,INET_NTOA(ip_src),concat('Listening on port ',tcp_srcport),'no'  from tcp,ip where tcp_flags & 0x12 = 0x12 and tcp.key_id=ip.key_id group by ip_src, tcp_srcport;";

	      #Record when a connection attempt is made:
	      $query.="insert into knowledge select 'ip_dst',ip.key_id, 'ip_addr',concat('ip:',INET_NTOA(ip_dst)),INET_NTOA(ip_dst), 'ip_addr', concat('ip:',(INET_NTOA(ip_src))),INET_NTOA(ip_src) ,concat('Syn:',tcp_dstport),'transitive'  from tcp,ip where tcp_flags = 0x02 and tcp.key_id=ip.key_id group by ip_src, ip_dst,tcp_dstport;";
	      do_query($query);

	      do_query("insert into knowledge select 'con_id',con_id,'ip_addr',concat('ip:',INET_NTOA(src_ip)),concat(INET_NTOA(src_ip),':',src_port),'ip_addr',concat('ip:',INET_NTOA(dest_ip)),concat(INET_NTOA(dest_ip),':',dest_port),'Communication','transitive' from connection_table" );
	      set_progress("Finding Rejected TCP connections");
	      #These are the Rejected TCP connections (Those with rst/ack)
	      $query="insert into knowledge select 'tcp_srcport',ip.key_id, 'closed_tcp_port',concat('CTP_',INET_NTOA(ip_src),':',tcp_srcport),tcp_srcport, 'ip_addr', concat('ip:',INET_NTOA(ip_src)),INET_NTOA(ip_src),concat('Not Listening on port ',tcp_srcport) ,'no' from tcp,ip where tcp_flags & 0x14 = 0x14 and tcp.key_id=ip.key_id group by ip_src, tcp_srcport;";
	      do_query($query);

	      set_progress("Calculating DNS properties for objects");
	      #DNS entries A records firstly:
	      do_query('insert into knowledge select "dns_data.name",dns_data.key_id,"dns_A",concat("A:",name),name,"ip_addr",concat("ip:",data),data,concat("DNS A Record "),"no" from dns_data where type="A" and not isnull(data) group by name,data;
          insert into knowledge select "dns_data.name",dns_data.key_id,concat("dns_",type),concat("DNS",type,":",name),name,"dns_a",concat("A:",data),data,concat("DNS ",type," Record "),"no" from dns_data where (type="NS" or type="CNAME") and not isnull(data) group by name,data;
');

	      set_progress("Performing POP 3 analysis");
	      #POP3 analysis, we note which users sit at each IP:
#	      check_prereq({"report"=>"trace_info","case"=>$args{"case"}});
#	      $query='insert into knowledge select "data.data",hilights.key_id,"POP_user",concat("pu:",data),data,"ip_addr",concat("ip:",INET_NTOA(ip_src)),INET_NTOA(ip_src),"POP User Name","no" from hilights,ip where hilights.key_id=ip.key_id and type="POP:User" group by data';
	      $query="insert into knowledge select 'pop_req_parameter',a.key_id,'pop_user',concat('POPUser:',pop_req_parameter),pop_req_parameter,'ip_addr',concat('ip:',INET_NTOA(ip_src)),INET_NTOA(ip_src),'POP Username','no' from pop as a,ip as b where a.key_id=b.key_id and pop_req_command like '\%user\%' group by pop_req_parameter,ip_src";
	      do_query($query);

	      #Create a knowledge_node table which is simply a summary of all nodes (type,value) pairs in either the child nodes or the parent nodes:
	      do_query("create temporary table temp select type,name,value,packet from knowledge where link='no'; insert into temp select ptype,pname,parent,packet from knowledge where link='no'; create table knowledge_node select * from temp group by name");
	      clear_progress();

	    },
              "display"=> sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result="<h1>Knowledge base</h1>\n";

	       $result.=display_table("select field,type,name,value,ptype,pname,parent,description,packet as \"Packet\" from knowledge",["","","","","","","","",$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id="],\%args,[]);
	       return $result;
              },
              "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       $dbh->do("drop table if exists $case.knowledge");
	       $dbh->do("drop table if exists $case.knowledge_node");
	       return "Cleaned up\n";
	     },
};

#Objects are infered from the knowledge base as we look for them.
$reports{'display_object'} = {
	      "name"=>"Display Object",
	      "family"=>"40 Knowledge Base",
	      "description"=>"This report displays an object from the knowledge base",
	      "parameters"=>{"case" => "flag_db","type"=>"string" , "value"=>"string"},
	      "progress"=>sub{return("Creating knowledge base")},
	      "form"=> sub {
		my $reference= shift;
		my %args =%{$reference};

		my $result=&{$reports{'connection_table'}{'form'}}(\%args);
		my $array_ref=$dbh->selectcol_arrayref('select description from knowledge where link = "transitive" group by  description');

		$result.="<input type=submit value='Update'><br>Select object property type ".selector("type",$args{"type"},"select type,type from knowledge_node group by type",$dbh)."<br>Property value (use % for wildcards) <input type=text size=20 name=value> Show siblings? <input type=checkbox name=show_siblings><br>Show deductions:<br><input type=hidden name=show_none value=1> ".$co->scrolling_list(-name=>'deductions',-values=>$array_ref,-default=>$array_ref,-size=>5,-multiple=>'true');
		
		return $result;
	      },
	      "analyse" => sub {
		my $reference= shift;
		my %args =%{$reference};
		#We require the knowledge base to be executed first
		 check_prereq({"report"=>"create_knowledge_base","case"=>$args{"case"}});
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference};
		sub tree($$$$$);

		#This draws the tree by utilising recursion
		sub tree($$$$$) {
		  my $pname=shift;
		  my $case=shift;
		  my $depth=shift;
		  my $args_ref=shift;
		  my $path_ref=shift;
		  my %args =%{$args_ref} ;
		  #First establish the types of deductions the user wanted to have:
		  my @deductions = $co->param("deductions");
		  if (!$co->param("deductions") && !$co->param("show_none")) {
		    @deductions=@{$dbh->selectcol_arrayref('select description from knowledge where link = "transitive" group by  description')};
		  }
		  my $condition = "description='".join("' or description='",@deductions)."' ";
		  my @clean_deductions = ();
		  foreach my $i (@deductions) {
		    push @clean_deductions, uri_escape($i);
		  }
		  my $deduction = "deductions=".join("&deductions=",@clean_deductions);

		  my $result="";
		  my $sth = $dbh->prepare("select description,type,value,packet,name from knowledge where pname=? and link='no' group by name order by description");
		  $sth->execute($pname);
		  while(my @row=$sth->fetchrow_array) {
		    my $description=$row[0];
		    my $type=$row[1];
		    my $value=$row[2];
		    my $packet=$row[3];
		    my $name = $row[4];
		    
		    if (!exists($path_ref->[$depth]) || $path_ref->[$depth] eq $name ) {
		      my $url = $co->self_url;
		      $result.="<tr><td align=right>(<a href=".$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id=$packet>$packet</a>)</td><td>"."<img src=/flag/images/spacer.png width=20 height=20>" x $depth."<img src=/flag/images/corner.png width=20 height=20><a href=\"".$co->url(-relative=>'1')."?report=display_object&case=".$args{"case"}."&type=$type&value=".uri_escape($value)."&show_siblings=".uri_escape($args{"show_siblings"})."&$deduction\">$description : $value</a></td></tr>\n";

		      $result.=tree($name,$case,$depth+1,\%args,$path_ref);

		      #Now display the transitive links:
		      my $sth = $dbh->prepare("select description,ptype,parent,packet from knowledge where name=? and link='transitive' and ($condition) group by name,pname,description order by description");
		      $sth->execute($name);
		      while (my @row=$sth->fetchrow_array) {
			$result.="<tr><td align=right>($row[3])</td><td>"."<img src=/flag/images/spacer.png width=20 height=20>" x $depth."<img src=/flag/images/corner.png width=20 height=20><a href=\"".$co->url(-relative=>'1')."?report=display_object&case=".$args{"case"}."&type=$row[1]&value=".uri_escape($row[2])."&show_siblings=".uri_escape($args{"show_siblings"})."&$deduction\"><font color=red>".$row[0]." from $row[1] $row[2]</font></a></td></tr>";
		      };

		      #Now display the transitive links:
		      $sth = $dbh->prepare("select description,type,value,packet from knowledge where pname=? and link='transitive' and ($condition) group by name,pname,description order by description");
		      $sth->execute($name);
		      while (my @row=$sth->fetchrow_array) {
			$result.="<tr><td align=right>($row[3])</td><td>"."<img src=/flag/images/spacer.png width=20 height=20>" x $depth."<img src=/flag/images/corner.png width=20 height=20><a href=\"".$co->url(-relative=>'1')."?report=display_object&case=".$args{"case"}."&type=$row[1]&value=".uri_escape($row[2])."&show_siblings=".uri_escape($args{"show_siblings"})."&$deduction&\"><font color=red>".$row[0]." to $row[1] $row[2]</font></a></td></tr>";
		      };

		    };
		  };


		  return $result;
		};
		
		my $pname = $args{"name"};

		my $result="<h1>Examining object</h1><form action=flag method=get><input type=hidden name=case value=".$args{"case"}."><input type=hidden name=report value=".$args{"report"}.">".selector("type",$args{"type"},"select type,type from knowledge_node group by type",$dbh)." is ". "<input type=text size=20 name=value value='".$args{"value"}."'> Show siblings? <input type=checkbox name=show_siblings value=checked ".$args{"show_siblings"}."><input type=submit value=Refresh><table>";
		my $sth;
		my %root_hash=(); #Contains all the root nodes.

		#The path of the tree above where we are is stored here...
		my @path = ();
		my @row;
		my $name;
		my $object_prev='';

		#Search for all nodes satisfying the search criteria: 
		$sth=$dbh->prepare("select type,value,packet,name from knowledge_node where type=? and value like '".$args{"value"}."'");
		$sth->execute($args{"type"});
		
		while (@row=$sth->fetchrow_array) {
		  #If we get a object_start variable, we skip till we get to it
		  if (exists($args{"object_start"}) && $args{"object_start"}) {
		    if ($args{"object_start"} eq $row[3]) {
		      delete $args{"object_start"};
		    } else {
		      next;
		    }
		  }
		  
		  #For each of those nodes, follow the tree upwards till we find the root:
		  my $type = $row[0];
		  my $value = $row[1];
		  my $packet=$row[2];
		  $name = $row[3];
		  @path=();

		  #Record the first object we are displaying. This is used to pass on using our page_next arrow so the next page's page_prev arrow can get us back to here.
		  if (!$object_prev) {$object_prev = $name};
		  
		  push @path, $name;
		  while (my @row2=$dbh->selectrow_array("select ptype,pname,packet,parent from knowledge where name='$name'  and link='no' ")) {
		    $name = $row2[1];
		    $type = $row2[0];
		    $value = $row2[3];
		    $packet=$row2[2];

		    push @path,$name;
		  }

		  #We only show trees that we havent shown before (we may be called on to display the same tree due to more than one matches on the leafs).
		  if (!exists($root_hash{"$name"})) {
		    #If the user wanted to show all the siblings we clear the path
		    if (exists($args{"show_siblings"}) &&$args{"show_siblings"} eq "checked" ) {
		      @path=();
		    };
		    
		    @path=reverse(@path);
		    
		    #Now draw the tree from the root down
		    $result.="<tr><td align=right>(<a href=".$co->url(-relative=>'1')."?report=packet_breakdown&case=".$args{"case"}."&key_id=$packet>$packet</a>)</td><td><a href=\"".$co->url(-relative=>'1')."?report=display_object&case=".$args{"case"}."&type=".uri_escape($type)."&value=".uri_escape($value)."&show_siblings=".uri_escape($args{"show_siblings"})."\">$type: $value</td></tr>\n";
		    $result.=tree($name,$args{"case"},1,\%args,\@path);
		  };
		  
		  #Store the root node:
		  $root_hash{$name}=1;

		  #If the result is too long to fit in one page, we quit from here....
		  my $length = 0;
		  while ($result=~/<tr>/ig) { $length++}
		  if ($length>$CONF{PAGESIZE}) {
		    last;
		  };
		};
		

			 my $array_ref=$dbh->selectcol_arrayref('select description from knowledge where link = "transitive" group by  description');
			 if (exists($args{"deductions"}) || exists($args{"show_none"})) {
			   $result.="</table><p><input type=hidden name=show_none value=1> ".$co->scrolling_list('deductions',$array_ref,[],5,'true')."<input type=submit value='Submit'></form>";
			 } else {
			   $result.="</table><p><input type=hidden name=show_none value=1> ".$co->scrolling_list('deductions',$array_ref,$array_ref,5,'true')."<input type=submit value='Submit'></form>";
			 }

		print print_headers(["object_start","object_prev"],($args{"object_prev"}),"$name&object_prev=$object_prev",'').$result.$flag_footer;
		exit;
#		$result.="<h3>Heuristics </h3>";
		
		#This detects the one nic having multiple IP addresses
#		@row=$dbh->selectrow_array("select count(*) from knowledge where id='$id' and  field='ip_src'");
#		if ( $row[0] > 1) {
#		  $result.="This interfact has ".$row[0]." IP addresses behind it. There are two possibilities here:\n<ul>\n<li>The interface has a number of IP addresses from the same subnet</li><li>This device is actually routing traffic from another subnet</li></ul>";
#		  @row=$dbh->selectrow_array("select abs(max(INET_ATON(value))-min((INET_ATON(value)))),floor(31-log(abs(max(INET_ATON(value))-min((INET_ATON(value)))))) from knowledge where id='$id' and  field='ip_src'");
#		  $result.="There are $row[0] addresses within the range, which would make the network as least a size /$row[1]";
#		  if ($row[1] < 16) {
#		    $result.="<br>It is therefore likely that this object is also a <em>router</em>."
#		  }
#		}

		return $result;
	      },

	      "reset" => sub {
		my $reference= shift;
		my %args =%{$reference};
  
		#We need to reset the knowledge base:
		return reset_report('create_knowledge_base',\%args);
	      }
};


 $reports{'search_ip'}= {
 	     "name"=>"Search IP address traffic",
 	     "family" => "40 Knowledge Base",
 	     "description" => "This report searches for all traffic from a single IP address",
 	     "parameters"=>{"case"=>"flag_db","src_ip"=>"ipaddr"},
 	     "form"=> sub{
 		my $reference= shift;
 		my %args =%{$reference} ;
 
 		if(!defined($args{"src_ip"})) {
 		  $args{"src_ip"} = "";
 		};
 
 		my $result = "<table><tr><td>Select case to operate on:</td><td>";
		$result.= selector("case",$args{"case"}, "select meta.value, meta.value from meta where property='flag_db'",$dbh);
		$result.="</td></tr><tr><td></td></tr><tr><td>IP address traffic to analyse:</td><td>";
		$result.= "<input type=text name=src_ip value=\"".$args{"src_ip"}."\"<br>";
		$result.="</td></tr></table><br>";
 
 
 		return $result;
 	      },
 
     	     "progress" => sub {
 		return 1;
 	      },
 		
 	     "analyse" => sub {
 		# No need to cache the result as fast anyway
 		return 1;
 	      },
 
 	      "reset" => sub {
 		return 1;
 	      },
 
 
  	     "display" => sub {
 		  my $reference= shift;
 		  my %args =%{$reference} ;	
 		  
 		  my $url=$co->self_url;
 
 		      #Now print the page
 		      my $case = $args{"case"};
 		      my $ip = $args{"src_ip"};
 
 		      my $result="<h2>All connections for IP address $ip</h2>\n<br>";
 
 		      $result.= display_table("select con_id as \"Connection\", concat(INET_NTOA(src_ip),\" -> \",INET_NTOA(dest_ip)) as \"Connection details\", count(con_id) as \"Total Con\", sum(count) as \"Packets\" from connection_table where src_ip = INET_ATON('$ip') or dest_ip = INET_ATON('$ip') GROUP BY dest_ip, src_ip", [$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id=",$co->url(-relative=>'1')."?report=search_ip_specific&case=".$args{"case"}."&connection="],\%args,[]);
 	
 		return $result;
 	      },
 
 };
 
 
   $reports{'search_ip_specific'} ={
 	      "name"=>"Search IP traffic 2",
 	      "family"=>"40 Knowledge Base",
 	      "hidden" => "yes",
 	      "description"=>"Locates connection specific to 2 end points",
 	      "parameters"=>{"case" => "flag_db", "connection"=>"string"},
 	      "progress"=>sub{return("Calculating, please wait")},
 	      "form"=>sub {
 		my $reference= shift;
 		my %args =%{$reference} ;
		my $result="<br>Select case to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select web databases to operate on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);
		$result.="<br>Enter connection endpoints like this x.x.x.x -> y.y.y.y <input type=text name=connection size=50><br>"; 
		
		return $result;
 	      },
 
 	      "analyse" => sub {
 		my $reference=shift;
		my %args =%{$reference};

 		#Check to see if we ran report connection_table which is a prepreq for this one.
 		check_prereq({"report"=>"connection_table","case"=>$args{"case"}});
 		return 1;
 	      },
 
 	      "reset" => sub {
 		return 1;
 	      },
 
 	      "display" => sub {
 		my $reference= shift;
 		my %args =%{$reference} ;	
 		my $url=$co->self_url;
 
 		my $db = $args{"case"};
 		my $connection = $args{"connection"};
 		my $result = "";
		my $ip1;
		my $ip2;

		if ( $connection =~ /([^ ]*) -> ([^ ]*)/ ||  $connection =~ /([^ ]*) -\&gt; ([^ ]*)/) {
		  $ip1 = $1;
		  $ip2 = $2;
		} else {
		  return ("<h1> Error </h1> Could not parse connection $connection");
		};

 		#Now print the page
 		$result = "<h2>Traffic between $ip1 and $ip2 </h2>";
 		
 		$result.=display_table("select con_id as  \"Connection\",INET_NTOA(src_ip) as \"Source IP\",src_port as \"Source Port\",INET_NTOA(dest_ip) as \"Dest IP\",dest_port as \"Dest Port\", count  from connection_table where src_ip = INET_ATON('$ip1') AND dest_ip = INET_ATON('$ip2') OR dest_ip = INET_ATON('$ip1') AND src_ip = INET_ATON('$ip2') group by con_id",[$co->url(-relative=>'1')."?report=tcptrace&case=".$args{"case"}."&con_id="],\%args,[]);
 		
 		return $result;
 	      },
 	     };



######################################################
#
#     HTTP log file reports.
#
######################################################
 $reports{'list_web_log'} ={
	      "name"=>"List web log events",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report simply lists the web log requests in a table",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select case to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select web databases to operate on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);
		
		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;

		#Now print the page
		my $result="<h1>Web Server Log in table ".$args{"table"}."</h1>";

		$result.=display_table("select id as \"ID\",timestamp as \"Time\",source_ip as \"Source IP\", action as \"Action\", url as \"URL\",result as \"HTTP result\", size as \"Bytes\" from ".$args{"table"},[],\%args,[]);

		return $result;
	      },
	     };

 $reports{'list_log_file'} ={
	      "name"=>"List log file contents",
	      "family"=>"35 Log Analysis",
	      "description"=>"This report simply lists the log entries in a searchable/groupable table",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $case_db = $args{"case"};
		my $result="<br>Select case to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select log table to operate on: ".selector("table",$args{"table"}," select a.value,a.value from $case_db.meta as a, $flag_db.log_types as b where b.name=a.property",$dbh);
		
		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;

		#Now print the page
		my $result="<h1>Web Server Log in table ".$args{"table"}."</h1>";
		
		# get fields, descriptions, log type etc, since we dont know what they are...
		my @fields;
		my @names;
		my @desc;
		my $type = $dbh->selectrow_array("select property from ".$args{"case"}.".meta where value='".$args{"table"}."'");
		my $sth=$dbh->prepare("Describe ".$args{"table"});
		$sth->execute;
		while( @fields = $sth->fetchrow_array ) {
		  push @names, $fields[0];
		}
		foreach (@names) {
		  push @desc, $dbh->selectrow_array("select description from flag.".$type."_fields where name='$_'");
		}

		# build query string
		my $query = "select ";
		for(my $i=0; $i<@desc; $i++) {
		  $query .= "$names[$i+1] as \"$desc[$i]\",";
		}
		chop $query;
		$query .= " from ".$args{"case"}.".".$args{"table"};

		$result.=display_table($query,[],\%args,[]);

		return $result;
	      },
	     };

 $reports{'popular_webhits'} ={
	      "name"=>"Most popular web links",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows the most accessed URLs",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select case databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);

		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->url(-relative=>'1')."?table=".$args{"table"}."&case=".$args{"case"};

		#Now print the page
		my $result="<h1>Most popular URLs in table ".$args{"table"}."</h1>";

		$result.=display_table("select url, count(url) from ".$args{"table"}." group by url",["$url&report=ip_by_url&url="],\%args,[]);

		return $result;
	      },
	     };

 $reports{'ip_by_url'} ={
	      "name"=>"IP addresses requesting given URLs",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows all the IPs requesting the given URL",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum","url"=>"string"},
	      "progress" => sub {},
	      "hidden"=>1,
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		my $result="<br>Select case databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);
		$result.="<br>URL: <input type=text name=url size=10><br>";
		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->url(-relative=>'1')."?table=".$args{"table"}."&case=".$args{"case"};

		#Now print the page
		my $result="<h1>IP addresses requesting \"".$args{"url"}."\"</h1>";

		$result.=display_table("select source_ip as \"Source IP\", count(source_ip) as \"Total Hits\" from ".$args{"table"}." where url like \"".$args{"url"}."\" group by source_ip" ,["$url&report=requests_from_ip&ip="],\%args,[]);

		return $result;
	      },
	     };

 $reports{'hits_by_ip'} ={
	      "name"=>"Most frequent source IP addresses",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows the most frequent IPs to access this site",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		
		my $result="<br>Select web databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);

		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->url(-relative=>'1')."?table=".$args{"table"}."&case=".$args{"case"};

		#Now print the page
		my $result="<h1>Total hits by IP address in table ".$args{"table"}."</h1>";

		$result.=display_table("select source_ip as \"Source IP\", count(source_ip) as \"Total Hits\" from ".$args{"table"}." group by source_ip" ,["$url&report=requests_from_ip&ip="],\%args,[]);

		return $result;
	      },
	     };

 $reports{'requests_from_ip'} ={
	      "name"=>"Requests from a single IP address",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows those URLs that were accessed by a single IP address",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum","ip"=>"string"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select web databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);
		
		$result.="<br>IP address (Use % as the wildcard) <input type=text name=ip value=\"".$args{"ip"}."\"><br>";
		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;

		#Now print the page
		my $result="<h1>Web Server Log for ".$args{"ip"}." in table ".$args{"table"}."</h1>";

		$result.=display_table("select id as \"ID\", source_ip as \"Source IP\",  timestamp as \"Time\", action as \"Action\", url as \"URL\", result as \"Result\",size as \"Size\"  from ".$args{"table"}." where source_ip like \"".$args{"ip"}."\"","",\%args,[]);

		return $result;
	      },
	     };

 $reports{'common_exploits'} ={
	      "name"=>"Detect web server exploit attempts",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows those URLs match the internal list of attack signatures",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select web databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);

		return $result;
	      },

	      "analyse" => sub {
		# There is no need to cache the result since it is quite quick anyways
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->url(-relative=>'1')."?table=".$args{"table"}."&case=".$args{"case"};

		#Now print the page
		my $result="<h1>Web server exploit attempts in table ".$args{"table"}." by source IP</h1>";

		$result.=display_table("select source_ip,url,description,count(id) as \"Total Signatures\" from ".$args{"table"}.",$flag_db.exploit where url like sig group by source_ip ",["$url&report=requests_from_ip&ip="],\%args,[]);

		return $result;
	      },
	     };

 $reports{'HTTP_bandwith'} ={
	      "name"=>"Bandwith used to send from server",
	      "family"=>"30 WebLog Analysis",
	      "description"=>"This report shows the bandwidth utilised per minute to transmit from the server",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select web databases to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select the table to work on: ".selector("table",$args{"table"},"select value, value from meta where property='web' group by value",$dbh);

		return $result;
	      },

	      "analyse" => sub {
		my $reference= shift;
		
		my %args =%{$reference} ;
		$dbh->do("create table http_bw select timestamp,floor(unix_timestamp(timestamp)/60) as minute,sum(size) as total from ".$args{"table"}." group by minute");

		return 1;
	      },

	      "reset" => sub {
		$dbh->do("drop table http_bw");
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->url(-relative=>'1')."?table=".$args{"table"}."&case=".$args{"case"};

		#Now print the page
		my $result="<h1>Web server exploit attempts in table ".$args{"table"}." by source IP</h1>";

		#find the minimum:
		my $minimum = @{$dbh->selectcol_arrayref("select min(minute) from http_bw where minute>0")}[0];

		  if(!exists($args{"max"})) {
		    $args{"max"}=200;
		  };
		  if(!exists($args{"min"})) {
		    $args{"min"}=100;
		  };

		if(exists($args{"draw"})) {
		  $result=draw_graph("bar","select (minute - $minimum) as \"minute count\",total from http_bw where minute - $minimum >".$args{"min"}." and minute - $minimum<".$args{"max"},"");
		  print $result;
		  exit;
		} else {
		  my $url=$co->self_url;
		  my $orig_url=$url;
		  $url=~s/\&min=[^\&]+//;
		  $url=~s/\&max=[^\&]+//;

		  my $max=$args{"max"};
		  my $min=$args{"min"};
		  my $zoom = int(($max-$min)/3);
		  my $step=int(($max-$min)/1);


		  $result="<center><h1>Bandwith utilised by HTTP table ".$args{"table"}."</h1>\n<img src=$orig_url&draw=please><br><a href=$url&min=".($min-100)."&max=".($max-100).">Left</a>          <a href=$url&min=".($min-$zoom)."&max=".($max+$zoom).">Zoom out</a>                <a href=$url&min=".($min+$zoom)."&max=".($max-$zoom).">Zoom in</a>          <a href=$url&min=".($min+100)."&max=".($max+100).">Right</a></center>";
		  return $result;
	  
		};

	      },
	     };

############################################################
#
#       Cookie log reports
#
############################################################
 $reports{'list_cookie_log'} ={
	      "name"=>"List cookie log",
	      "family"=>"31 Cookie Analysis",
	      "description"=>"This report simply lists the cookie log requests in a table",
	      "parameters"=>{"case" => "flag_db", "table"=>"alphanum"},
	      "progress" => sub {},
	      "form"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result="<br>Select case to operate on: ".selector("case",$args{"case"},"select value,value from $flag_db.meta where property=\"flag_db\"",$dbh)."  <input type=submit value=\"Update\"><p>Select cookie databases to operate on: ".selector("table",$args{"table"},"select value, value from meta where property='cookie' group by value",$dbh);
		
		return $result;
	      },

	      "analyse" => sub {
	    	my $reference= shift;
		my %args =%{$reference} ;

		# Translate all %-encoded entries.
                my %trans;
                my $sth = $dbh->prepare("select content from ".$args{table});
                $sth->execute;
                while (my @row = $sth->fetchrow_array)
                {
                  my $orig = $row[0];
                  $row[0] =~ s/%([A-Fa-f\d]{2})/chr hex $1/eg;
                  $trans{$orig} = $row[0];
                  $orig =~ s/\"/\\\"/g;
                  $trans{$orig} =~ s/\"/\\\"/g;
                }
                $sth->finish;

                foreach my $orig (keys %trans)
                {
                  $dbh->do ("update ".$args{table}." set content=\"".
                                      $trans{$orig}."\" where content=\"$orig\""
                            );
                }

                # Make "c" = cookie get, "s" = cookie set.
                $dbh->do ("update ".$args{table}." set cookie_type = ".
                          "\"Retrieve\" where cookie_type = \"c\"");
                $dbh->do ("update ".$args{table}." set cookie_type = ".
                          "\"Set\" where cookie_type = \"s\"");
		return 1;
	      },

	      "reset" => sub {
		return 1;
	      },
	      "display" => sub {
		my $reference= shift;
		my %args =%{$reference} ;	

		my $url=$co->self_url;

		#Now print the page
		my $result="<h1>Cookie Log in table ".$args{"table"}."</h1>";

		$result.=display_table("select id as \"ID\",timestamp as \"Time\",source_ip as \"Source IP\", source_port as \"Source port\", dest_ip as \"Dest IP\", dest_port as \"Dest Port\", dns as \"Dest hostname\", cookie_type as \"Cookie set/get\", content as \"Cookie content\" from ".$args{"table"},[],\%args,[]);

		return $result;
	      },
	     };

############################################################
#
#       Case management and loading reports
#
############################################################
$reports{'new_case'} = { 
	      "name"=>"Create new case",
	      "family"=>"10 Create new case",
	      "description"=>"Create database for new case to load data into",
	      "parameters"=>{"create_case"=>"alphanum"},
	      "progress"=>sub{},
	      "reset"=>sub{},
	      "form"=> sub {
	    	my $reference= shift;
		my %args =%{$reference} ;
                $args{'create_case'} = '' unless $args{'create_case'};
		my $result = 'Please enter a new case name : <input name="create_case" type="textbox" value= "'.encode_entities($args{"create_case"}).'"> <br><br> ';
		return $result;
	      },

	      "analyse" => sub {
		return 1;
	      },

	      "display"=> sub { 
		my $reference= shift;
		my %args =%{$reference} ;
		my @check;
		my $result;

		my $sth=$dbh->prepare("Select value,property from flag.meta where value='$args{\"create_case\"}'");
		$sth->execute();
		@check = $sth->fetchrow_array;

		if($args{"create_case"} eq $check[0]) {
		    $result = "<br><br>The database <strong> " .$args{"create_case"}. " </strong> already exists. <br><br> Remove the database if necessary via the FLAG homepage. <br>"
		} else {
                   if(exists($args{"create_case"}) && (length($args{"create_case"}) > 1)){
		     my $case = $args{"create_case"};
		     $case =~ s/^\s+//;
		     $sth=$dbh->do("Insert into flag.meta values ('flag_db','$case')");
		     $sth=$dbh->do("Create database if not exists $case");
		     $sth=$dbh->do("Create table if not exists $case.meta(`time` timestamp(14) NOT NULL,property varchar(50), value text)");	       
		     $dbh->do("create table if not exists $case.bookmarks (id int(11) auto_increment, canon text, url text,  description text,  bookmark text ,  PRIMARY KEY  (id),  KEY id (id))");

		     $result = "<br><br>The database for case <strong>".$args{"create_case"}."</strong> has been successfully created.";
		   };
		 };
		return $result;
	      },

    };


$reports{'del_case'} = { 
	      "name"=>"Remove case",
	      "family"=>"70 Remove case",
	      "description"=>"Remove database for specified case",
	      "parameters"=>{"remove_case"=>"flag_db"},
	      "reset"=>sub{},
	      "form"=> sub {
	    	my $reference= shift;
		my %args =%{$reference} ;
		my $result = '<table><tr><td>Please select the database to remove :</td>';
		   $result .= '<td>'.selector("remove_case",$args{"remove_case"},"select meta.value, meta.value from flag.meta where property = 'flag_db'",$dbh). '</td></tr> ';
		   $result .= '<tr><td>Confirm database deletion</td><td><input name="confirm_rm" type="checkbox" value=" "></td></tr></table>';
		return $result;
	      },

	      "analyse" => sub {
 			return 1;
		},

	       "progress" => sub{},

	      "display"=> sub {
 		my $reference= shift;
		my %args =%{$reference} ;
		my $result;

		if(exists($args{"confirm_rm"})) {
		     my $sth=$dbh->do("Delete from flag.meta where value = '$args{\"remove_case\"}'");
		     $sth=$dbh->do("Drop database $args{\"remove_case\"}");
		     $result = "The database <strong> $args{\"remove_case\"} </strong> has been successfully removed.";
		} else {
		     $result = "Please confirm the deletion of database $args{\"remove_case\"}.";
		};
		return $result;	
	      },
};

$reports{'load_log_file'}={
	      "name"=>"Load Log File",
	      "family"=>"20 Load Data",
	      "description"=>"Load Data from log file into Database",
	      "parameters"=>{"table"=>"alphanum", "datafile"=>"filename", "final"=>"uniq_field"},
	      
	      "form"=> \&load_log_form,
	      "reset"=>  sub {
		my $reference= shift;
		my %args =%{$reference} ;

		$dbh->do("drop table ".$args{"table"});
		return 1;
	      },

	      "progress" => sub {
		my $reference= shift;
		my %args =%{$reference} ;

		my $sth= $dbh->prepare("select count(id) from ".$args{"table"});
		$sth->execute;
		my @count=$sth->fetchrow_array;
		return ("So far uploaded $count[0] Rows into table ".$args{"table"});
	      },

	      "display"=> sub {
		my $reference= shift;
		my %args =%{$reference} ;

		my @count = @{$dbh->selectcol_arrayref("select max(id) from ".$args{"table"},undef)};
                if (!$count[0]) { $count[0] = "<strong>WARNING: NO</strong>"; }
		
		return " $count[0] rows of data uploaded to the database <br><br> <a href=".$co->url(-relative=>'1')."?report=load_log_file&case=".$args{"case"}."&table=".$args{"table"}."&reset=yes&final=yes>Click here to reset case </a>\n";
	      },
			   
	      "analyse"=>sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $result;
		my $keys="";
	
		## Creating the database and table then uploading data into table ### 
		my $space = " ";
		my @temp=();
		
		# Get list of fields for table creation
		my $sth = $dbh->prepare("Select name,type,`key` from $flag_db.".$args{"log_type"}."_fields where name not like \"\%ignore\%\"");
		$sth->execute;

		# actual fields
		my %fields;
		foreach (keys %args) {
		  if(/^field\d+$/) {
		    $fields{"$args{$_}"} = "";
		  }
		}
		    
		#Fix up the keys on the right columns. We need to add the keys after the tables have been created or else we take a huge permformance hit during the upload.
		while(my @row = $sth->fetchrow_array) {
		  if(exists($fields{$row[0]})) {
		    push @temp, "$row[0] $row[1]";
		    if ($row[2]) {
		      $keys.="alter table ".$args{"table"}." add key(".$row[0].");";
		    }
		  }
		};
	
		# Create new table and remove meta table entries
		my $case = $args{"case"};
		$sth = $dbh->do("Create table if not exists ".$args{"case"}.".".$args{"table"}." (id int auto_increment,".join (",",@temp).",key(id))");

		# Upload data into table using simple delimiters
		my $ln_count = 5;
		my @fields;
		my $regex;
		my $delmethod = $args{"delmethod"};
		my $inputfile = $CONF{"UPLOADDIR"}.$args{"datafile"};
		
		# Determine which delimiter to use for splitting input then upload into db
		# // Simple delimiter specified
		
		if($delmethod eq "simple") {
		  my @prefilters=$co->param('prefilter');
		  open(FILEHANDLE,"gzip -dcf <$inputfile|") || error("Cannot open file: $inputfile");

		  while(<FILEHANDLE>) {
		    #apply pre-filters
		    foreach my $prefilter (@prefilters) {
		      if (exists($prefilters{$prefilter})) {
			$_= &{$prefilters{$prefilter}}($_);
		      };
		    };
		    # skip where prefilter returns 0
		    if($_ eq 0) {
		      next;
		    }
		    
		    # fix up space delimiter
		    $regex = $args{"delimiter"};
		    if($regex == "") {
		      $regex = " ";
		    }

		    my @temp2 = split ($regex,$_);
		    my @temp=();
		    for(my $i=0; $i<50; $i++) {
		      if(exists($args{"field$i"}) && defined($temp2[$i]) && ($args{"field$i"} ne "ignore")) {
			push @temp, $args{"field$i"}."=\"".$temp2[$i]."\"";
		      }
		    }
		    my $sth=$dbh->prepare("insert into " .$args{"table"}."  set ".join(",",@temp));
		    $sth->execute;
		  }
		}  elsif($args{"delmethod"} eq "preset" || $args{"delmethod"} eq "advanced")  {
		  #// Get regex from db using $preset value
		  if($args{"delmethod"} eq "preset") {
		    $sth=$dbh->prepare("Select regex,name,prefilter from $flag_db.log_preset where name='$args{\"preset\"}'");
		    $sth->execute;
		    my @prereg = $sth->fetchrow_array;
		    if(!defined($prereg[0])) {
		      $sth=$dbh->prepare("Select regex,name,prefilter from $flag_db.log_preset limit 1");
		      $sth->execute; 
		      @prereg = $sth->fetchrow_array;
		    };
		    $regex = $prereg[0];
		  } else {
		    $regex =$args{"delimiter"};
		  };
		  
		  my $linein;
		  my @regtemp;
		  my $j;
		  my @prefilters=$co->param('prefilter');
		  
		  #Big loop for uploading records. FIXME: optimise this loop to achieve faster uploads.
		  open (FILEHANDLE, "gzip -dcf <$inputfile|") or throw("Cannot open file: $inputfile");
		  #In case we kill this process first...
		  local $SIG{TERM} = sub {
		    $dbh->do("insert into meta set property=\"".$args{"log_type"}."\",value=\"".$args{"table"}."\"");
		    $dbh->do("update  meta set property=\"report_executed\"  where property like \"pending%\" and value=\"".canonicalise(\%args)."\""); 
		  };

		  while(<FILEHANDLE>) {
		    foreach my $prefilter (@prefilters) {
		      if (exists($prefilters{$prefilter})) {
			$_= &{$prefilters{$prefilter}}($_);
		      };
		    };
		    # skip where prefilter returns 0
		    if($_ eq 0) {
		      next;
		    }
		    
		    $j=0;
		    @regtemp=();
		    if(!length($_)<=1 && /$regex/o){ 
		      no strict "refs";
		      while(defined($ {$j+1})) {
			if(exists($args{"field$j"}) && $args{"field$j"} ne "ignore" ) {
			  push @regtemp, $args{"field$j"} ."="."\"$ {$j+1}\"";
			};
			$j++;
		      };
		    };
		    if ($#regtemp>0) {
		      $dbh->do("insert into ".$args{"table"}." set ".join(",",@regtemp));
		    };
		  };
		}

		do_query($keys);
		
		$dbh->do("insert into meta set property=\"".$args{"log_type"}."\",value=\"".$args{"table"}."\"");
		return("Log file successfully loaded into " .$args{"table"});
	      },
	     };

$reports{'load_tcp_dump'}={
	     "name"=>"Load TCP dump file",
	     "family"=>"20 Load Data",
	     "description"=>"Enable user to load a tcpdump file into the database",
	     "parameters"=>{"case"=>"flag_db", "datafile"=>"filename"},
	      
	      "form" => sub{
		my $reference= shift;
		my %args =%{$reference} ;

		my $result .= '<table border="0"> <tr><td><strong>Step 1:</strong></td></tr>  <tr> <td>Select Database:</td><td>';
  		$result .= selector("case", $args{"case"}, "select value,value from $flag_db.meta where property=\"flag_db\"");
		$result .= "</td></tr><tr><td>".$args{"newdb"}."</td> </tr> <tr><td></td></tr><tr><td colspan=\"3\"><hr/></td></tr>";
		$result .= "<tr><td><strong>Step 2:</strong></td></tr>";
		$result .= "<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args);
		$result .= "</td><td halign=\"left\"></td></tr><tr><td colspan=\"3\"><hr/></td></tr><tr><td><strong>Step 3:</strong></td><td>Perform basic analysis <input type=checkbox name=preanalyse value=yes checked></td></tr><tr><td colspan=2><hr/></td></tr></table>";
		
		return $result;
		} ,


	      "reset" =>  sub{
		my $reference= shift;
		my %args =%{$reference} ;

		$dbh->do("delete from meta where value like \"report=load_tcp_dump&case=".$args{"case"}."\"")
;
		#Run through all the reports in the database and drop them
		my $sth=$dbh->prepare("select value from meta");
		$sth->execute();
		while(my @row=$sth->fetchrow_array) {
		  if ($row[0] =~ /^report=([^\&]*)/ && $1 ne "load_tcp_dump") {
		    my $query=new CGI($row[0]);
		    my %args = $query->Vars;

		    &{$reports{$1}{"reset"}}(\%args);
		    $dbh->do("delete from meta where value=\"".$row[0]."\"");
		  }
		}
		safe_shell($CONF{"TETHEREAL"}." -G drop | mysql -fu".$CONF{"USERNAME"}." -p".$CONF{"PASSWORD"}." -D".$args{"case"});
		return 1;
	      },
 
	      "progress" => sub {
		my $reference= shift;
		my %args =%{$reference} ;
		my $case=$args{"case"};
		my $sth=$dbh->prepare("select count(*) from frame");
		$sth->execute;
		my @count=$sth->fetchrow_array;
		return (get_progress(\%args)."<p>Uploaded <strong> $count[0] </strong> rows into $case");
	      },

	      "display"=> sub {
		my $result;
		my $reference= shift;
		my %args =%{$reference} ;
		my $case=$args{"case"};
		my $sth=$dbh->prepare("select count(*) from $case.ip");
		$sth->execute;
		my @count=$sth->fetchrow_array;
		if( $count[0] == 0) {
		   $result = "An error was encountered loading data into the database.<br><br>";
		   $result .= "This may be due to permission problems or tcp dump file format.<br><br>";
		   $result .= "Please check user documenation for more details.<br>";
		} else {
		   $result="Data successfully uploaded into database.";
		};
                return $result;
	      },

	      "analyse" => sub{
		my $reference= shift;
		my %args =%{$reference} ;

		my $case = $args{"case"};
		my $datafile = $args{"datafile"};
		set_progress("Loading data into case");
		safe_shell($CONF{"TETHEREAL"}." -G sql | mysql -fD\"$case\" -p".$CONF{"PASSWORD"}." -u".$CONF{"USERNAME"});
		safe_shell($CONF{"TETHEREAL"}." -VxQ -r\"".$CONF{"UPLOADDIR"}.$datafile."\" | mysql -fD\"$case\" -p".$CONF{"PASSWORD"}." -u".$CONF{"USERNAME"});
		# Set up indexing of tables after doing inserts
		set_progress("Adding table indexes");
		do_query("alter table tcp add index(key_id); alter table tcp add index(tcp_srcport); alter table tcp add index(tcp_dstport); alter table ip add index(key_id);  alter table ip add index(ip_dst);  alter table ip add index(ip_src);  alter table frame add index(key_id);  alter table eth add index(key_id); alter table icmp add index(key_id);");

		#Now if the user wanted to do some of the more time consuming analysis we do it here:
		if (exists($args{"preanalyse"})) {
		  set_progress("Now perfoming preset analysis");
		  set_progress("       Building Connection table");
		  check_prereq({"report"=>"connection_table","case"=>$args{"case"}});
		  set_progress("       Extracting interesting strings (Trace Info)");
		  check_prereq({"report"=>"trace_info","case"=>$args{"case"}});
		  set_progress("       Building knowledgebase");
		  check_prereq({"report"=>"create_knowledge_base","case"=>$args{"case"}});
		}
		clear_progress();
	      },	

};

$reports{"load_dd_image"}={
	     "name" => "Load dd partition image",
	     "family" => "20 Load Data",
	     "description" => "Load a new dd image into a case",
	     "parameters" => {"case"=>"flag_db","datafile"=>"filename","mountpoint"=>"string","fstype"=>"checkbox"},
	     "progress" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       
	       return ("Currently calculating the connection table, please wait".get_progress(\%args));
	     },

	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference};

	       my $result = "<br>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
	       $result.="<table><tr><td>dd File:<br>(Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."<input type=submit name=test_fs value='Guess image type'>";
	       if(exists($args{"test_fs"}) && &{$types{"filename"}}($args{"datafile"}) && ((sysopen(FH,$CONF{"UPLOADDIR"}.'/'.$args{"datafile"},0) || ($result.="$!") && 0))) {
		 my $in;
		 my $out="";
		 my $size=10000;
		 sysread FH,$in,$size;
		 run ([$CONF{"FLAG_BIN"}."/file",'-'],\$in,\$out);
		 $out=~s/^[^:]*://;
		 $result.="<br>my guess:".$out;
	       }
	       $result.="<tr><td>Filesystem Type</td><td>".$co->scrolling_list(-name=>'fstype', -values=>['','bsdi','fat','fat12','fat16','fat32','freebsd','linux-ext2','linux-ext3','ntfs','openbsd','solaris'], -default=>[$args{"fstype"}],-size=>'1')."</td></tr>
</td></tr><tr><td>Mount point </td><td><input type=text name=mountpoint size=20></td></tr>
<tr><td>Perform Analysis:</td><td><input type=checkbox name=do_analysis value=yes checked></td></tr></table>";
	       return $result;
	     },

	     "analyse" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $string="datafile=".encode_entities($args{"datafile"})."&mountpoint=".encode_entities($args{"mountpoint"})."&fstype=".encode_entities($args{"fstype"});
	       $dbh->do("insert into meta set property=\"dd_file\",value=?",undef,$string);

	       if ($args{"do_analysis"}) {
		 set_progress("Calculating timelines");
		 check_prereq({"report"=>"timeline","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
		 set_progress("Collating filesystem information");
		 check_prereq({"report"=>"view_filesystem","case"=>$args{"case"},"datafile"=>$args{"datafile"}});	
		 set_progress("Checking file types and calculating Hashes");
		 check_prereq({"report"=>"view_filetypes","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
		 set_progress("Comparing Hashes to DB");
		 check_prereq({"report"=>"md5_comparison","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
	       }
	       clear_progress();
	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $result="<h1> File ".$args{"datafile"}." uploaded to case ".$args{"case"}." </h1>";
	       return $result;
	     },

	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       return "1";
	     },
	    };


$reports{"bookmark"}={
	     "name" => "Bookmark",
	     "family" => "65 Miscelaneous",
	     "hidden" => "yes",
	     "description" => "Create a bookmark of this row",
	     "parameters" => {"case"=>"flag_db","bookmark"=>"string","url"=>"string","description"=>"string"},
	     "progress" => sub {
	       return ("Currently calculating the connection table, please wait");
	     },
	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result = "<table><tr><td>Flag case </td><td bgcolor=".$CONF{"BGCOLOR1"}.">".$args{"case"}."<input type=hidden name='case' value='".$args{"case"}."'></td></tr>\n";
	       my %bm;
	       foreach my $i (split "&",$args{"bookmark"}) {
		 if ($i =~ /^([^=]*)=(.*)$/) {
		   $result.="<tr><td>$1</td><td bgcolor=".$CONF{"BGCOLOR1"}.">$2</td></tr>\n";
		 };
	       }

	       $result.="<tr><td>Enter a Description</td><td><input name=description type=text size=20></td></tr></table> <input type=hidden name=url value='".encode_entities($args{"url"})."'><input type=hidden name=bookmark value='".encode_entities($args{"bookmark"})."'>";
	       return $result;
	     },

	      "analyse" => sub {
	       my $reference= shift;
	       my %args =%{$reference};
	       #Create the bookmark table if it doesnt exists already
	       $dbh->do("create table if not exists bookmarks (id int(11) auto_increment, canon text, url text,  description text,  bookmark text ,  PRIMARY KEY  (id),  KEY id (id))");
	       #Create a CGI object for the referenced URL so we can canonicalise it
	       $args{"url"}=~s/http:\/\/[^\?]*\?//;
	       my $url=new CGI($args{"url"});
	       my %url=$url->Vars;

	       #insert the record in
	       $dbh->do("insert into bookmarks set url=?,canon=?,description=?,bookmark=?",undef,$args{"url"},canonicalise(\%url),$args{"description"},$args{"bookmark"});

	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $result=$co->redirect($args{"url"});
	       print $result;
	       exit(0);
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       $dbh->do("delete from bookmarks where id=?",undef,$args{"id"});
	       print $co->redirect($args{"url"});
	       return "1";
	     },
	    };

$reports{"browse_bookmarks"}={
	     "name" => "View Bookmarks",
	     "family" => "65 Miscelaneous",
#	     "hidden" => "yes",
	     "description" => "View all bookmarks for this case",
	     "parameters" => {"case"=>"flag_db"},
	     "progress" => sub {
	       return ("Currently calculating the connection table, please wait");
	     },
	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result = "<br>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
	       return $result;
	     },

	      "analyse" => sub {
	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $result="<h1>Bookmarks for case ".$args{"case"}."</h1><table><tr>";
	       my $current="";

	       #Find bookmarks in this case, order by reports:
	       my $sth=$dbh->prepare("select url,description,bookmark from bookmarks where canon like \"%&case=".$args{"case"}."%\" order by canon");
	       $sth->execute();
	       
	       while (my @row=$sth->fetchrow_array) {
		 my $parameters = new CGI($row[0]);
		 #break out the bookmark table
		 my @bookmarks = split("&",$row[2]);
		 
		 if ($current ne $parameters->param("report")) {
		   $current = $parameters->param("report");
		   $result.="</tr></table><p>Report <b> ".$reports{$parameters->param("report")}{"name"}."</b><br><table><tr>\n";
		   #First do the headers
		   foreach (@bookmarks) {
		     if (/^([^=]*)=(.*)$/) {
		       $result.="<th>$1</th>";
		     };
		   };
		   $result.="<th>BM Description</th></tr><tr bgcolor=".$CONF{"BOOKMARKCOLOR"}.">";
		 };
		 #Now draw the line
		 foreach (@bookmarks) {
		   if (/^([^=]*)=(.*)$/) {
		     $result.="<td>$2</td>";
		   };
		 };
		 $result.="<td><a href=".$co->url(-relative=>'1')."?".($row[0]).">$row[1]</a></td></tr><tr  bgcolor=".$CONF{"BOOKMARKCOLOR"}.">";
	       };

	       $result.="</tr></table>";
	       return $result;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       return "1";
	     },
	    };


##############################################################
#
# Disk and File forensics
#
##############################################################
$reports{"view_strings"}=
  {
   "name" => "View Strings in Image File",
   "family" => "50 Disk Forensics",
   "description" => "Search an image file and construct a strings database",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
   "progress" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     my @row=$dbh->selectrow_array("select count(offset) from string_$table_name");
     return ("Currently adding strings to database, $table_name please wait. <br>So far added ".$row[0]." strings".get_progress(\%args));
   },
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
     $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td></tr></table>";
     return $result;
   },
   
   "display" => sub {
     my $reference= shift;
     
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     my $result= "<h1>Strings from ".$args{"datafile"}."</h1>";
     $result.=display_table("select offset,if(length(string)>255,concat(left(string,255),\"  ....  \"),string) as \"String\" from string_$table_name",[$co->url(-relative=>'1')."?report=display_sector&case=".$args{"case"}."&datafile=".$args{"datafile"}."&seek="],\%args,[]);
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     #create table if it exists...
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     $dbh->do("create table if not exists string_$table_name (`offset` INT NOT NULL,`string` TEXT NOT NULL,key (offset))");
     my $sth=$dbh->prepare("insert into string_$table_name set offset=?,string=?");

     #FIXME (MC) rewrite this part in C for speed.
     #This part heavily borrowed from : Copyright 1999 Nathan Scott Thompson <quimby at city-net dot com> (CPAN)
     open(IN,$CONF{"UPLOADDIR"}.$args{"datafile"}) or throw("Cant open image ".$CONF{"UPLOADDIR"}.$args{"datafile"});
     my $punctuation = join '\\', split //, q/`~!@#$%^&*()-+={}|[]\:";'<>?,.\//;   #Fix emacs color hilighing "'`
     my $printable = '\w \t\r\n' . $punctuation; #We want to join separate lines into the same data entry so that free text searches make more sense.
     my $chunksize = 1024 * 1024/100; #There seems to be an optimal chunk size when it comes to optimization
     my $offset = 0;
     my $opt_n = 4; #Minimum string length

     set_progress("Started reading file");

     while ( $_ or (read( IN, $_, $chunksize) && update_progress("Currently read up to $offset",$dbh,"Currently")) )
       {
	 $offset += length($1) if s/^([^$printable]+)//o;
	 my $string = '';
	 
	 do {
	   $string .= $1 if s/^([$printable]+)//o;
	 } until ( $_ or ! (read( IN, $_, $chunksize ) && update_progress("Currently read up to $offset",$dbh,"")) );
	 
	 if ( length($string) >= $opt_n )
	   {
	     $sth->execute($offset,$string);
	   }
	 $offset += length($string);
	 #report on the current progress:
       }
     close(IN);
     clear_progress();
   },
   
   #This routine will be called to clean up
   "reset" => sub {
     my $reference= shift;
     my %args =%{$reference} ;

     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     $dbh->do("drop table if exists string_$table_name");
     return "1";
   },
  };

$reports{"display_sector"}=
  {
   "name" => "Display Sector",
   "family" => "50 Disk Forensics",
   #	     "hidden" => "yes",
   "description" => "Displays a sector containing the bytes of interest",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
   
   #This will be executed while the analysis is happening to give the user a progress bar:
   "progress" => sub {
     my $reference= shift;
     my %args =%{$reference} ;

     return ("Currently reading sector information, please wait");
   },
   
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
     $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td></tr></table>";
     return $result;
   },
	
   "analyse" => sub {
     my $reference= shift;
     
     my %args =%{$reference} ;
     
     return "1";
   },

   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $tmp="";

     #Retrieve the fs type for this dd file or fall back on a default.
     my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

     if (!exists($args{"seek"}) || !($args{"seek"}=~/^\d+$/)) {
       throw("Please supply a valid seek arguement ".$args{"seek"}." <br>");
     };

     run([$CONF{"FLAG_BIN"}."/dcat","-s",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype],\$tmp,\$tmp);
     my $block_size = 1024;
     if ($tmp=~/(\d*).*Block Size/i) {
       $block_size=$1;
     }
     my $result;
     $result="<h1>Sector dump for sector ".uc(sprintf("%08lx",int($args{"seek"}/$block_size)))." which starts at byte ".uc(sprintf("%08lx",int($args{"seek"}/$block_size)*$block_size))."</h1>\n";
     if (exists($args{"mode"}) && $args{"mode"} eq "hex") {
       my $data;
       my $url=$co->self_url;
       $url =~ s/([\?\&\;])mode=[^\&\;]*/$1/;
       $result.="<a href=".$url."&mode=data>View data</a>";

       my $f = new Data::HexDump;
       $f->file("<".$CONF{"UPLOADDIR"}.$args{"datafile"}) or throw("Could not open ".$args{"datafile"});
       $f->offset(int($args{"seek"}/$block_size)*$block_size);
       $f->block_size($block_size);
       $result.="\n<pre>".encode_entities($f->dump)."</pre>";
     } else {
       my $url=$co->self_url;
       $url =~ s/([\?\&\;])mode=[^\&\;]*/$1/ig;
       $result.="<a href=".$url."&mode=hex>View hex</a>";
       my $fh= new FileHandle "<".$CONF{"UPLOADDIR"}.$args{"datafile"} or throw("Could not open ".$args{"datafile"});
       my $tmp;
       $fh->seek(int($args{"seek"}/$block_size)*$block_size,0);
       $fh->read($tmp,$block_size);
       $fh->close;
       $result.="\n<pre>".wrap_and_sanitise($tmp,$CONF{"WRAP"})."</pre>";
   } ;

     print print_headers(["seek"],($args{"seek"}-$block_size),($args{"seek"}+$block_size),'').$result.$flag_footer;
     exit;
   },
   
   #This routine will be called to clean up
   "reset" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     return "1";
   },
  };

$reports{"timeline"}=
  {
   "name" => "Browse timeline",
   "family" => "50 Disk Forensics",
   #	     "hidden" => "yes",
   "description" => "Views the Mac times for inodes in a time line",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
   "progress" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     return ("Currently calculating inode information for ".$args{"datafile"}.", please wait<br>".get_progress(\%args));
   },
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."<input type=submit value=\"Update\"></td></tr>";
     $result.=disk_partition_form();
     $result.="</td></tr></table>";
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

     #This table stores the inode information
     $dbh->do("create table if not exists inode_$table_name (`st_ino` int,`st_alloc` char(4),`st_uid` int, `st_gid` int,`st_mtime` int,`st_atime` int,`st_ctime` int,`st_dtime` int,`st_mode` int,`st_nlink`  int,`st_size` int,`st_block0` int,`st_block1` int)");

     #Here we store the file name information
     $dbh->do("create table if not exists file_$table_name (`type` varchar(5),`inode` int,`status` varchar(10), `name` text)");

     $dbh->do("create temporary table file_temp (`type` varchar(5),`inode` int,`status` varchar(10), `name` text)");

     #populate the file table:
     set_progress("Compiling file - inode mappings");
     my $sth = $dbh->prepare("insert into file_temp (`type`,`inode`,`status`,`name`) values (?,?,?,?)");
     my $in="";
     my @ils=($CONF{"FLAG_BIN"}."/fls","-rp",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype);
     my $h=start \@ils, \$in, '>pipe', \*OUT,'2>pipe',\*ERR;
     while ((pump $h) && (my $line=<OUT>)) {
       chomp $line;
       if ($line =~ /^(\S+)\s+(\d+):\s+(\S+)$/) {
	 $sth->execute($1,$2,"-","/".$3);
       } elsif ($line =~ /^(\S+)\s+\*\s+(\d+):\s+(\S+)$/ ) {
	 $sth->execute($1,$2,"deleted","/".$3);
       } elsif ($line =~ /^(\S+)\s+\*\s+(\d+)\(realloc\):\s+(\S+)$/) {
	 $sth->execute($1,$2,"realloc","/".$3);
       }
     };
     finish $h;

     close(OUT);
     close(ERR);

     $dbh->do("insert into file_$table_name select * from file_temp group by file_temp.inode");
     $dbh->do("drop table file_temp");

     set_progress("Computing inode information");
     $in="";
     @ils=($CONF{"FLAG_BIN"}."/ils","-e",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype);
     $h=start \@ils, \$in, '>pipe', \*OUT,'2>pipe',\*ERR;
     while ((pump $h) && (my $line=<OUT>)) {
       chomp $line;
       
       if ($line=~/ERR/) {
	 last;
       };
       
       if ($line=~/^\d/) { #Real data on this line
	 $dbh->do("insert into inode_$table_name (`".join("`,`",@ils)."`) values ('".join("','",split("\\\|",$line))."')") or throw("SQL Error");
       } else { #The column headers are here
	 @ils=split("\\\|",$line);
       } ;
      };
     finish $h;

     close(OUT);
     close(ERR);
     set_progress("Creating indexes on tables");
     #Delete those inodes who have never been used at all:
     do_query("delete from inode_$table_name where st_mtime =0 and st_atime =0 and st_ctime =0 and st_dtime = 0");
     do_query("alter table inode_$table_name add key(st_ino);alter table inode_$table_name add key(st_mtime);alter table inode_$table_name add key(st_atime);alter table inode_$table_name add key(st_ctime);alter table inode_$table_name add key(st_dtime)");
     do_query("alter table file_$table_name add key(inode);alter table file_$table_name add key(status);alter table file_$table_name add key(name)");

     set_progress("creating MAC time table");
     do_query("create temporary table temp select st_ino,status,st_mtime,1 as `m`,0 as `a`,0 as `c`,0 as `d`,name from inode_$table_name left join file_$table_name on st_ino=inode;
 insert into temp select st_ino,status,st_atime,0,1,0,0,name from inode_$table_name left join file_$table_name on st_ino=inode;
insert into temp select st_ino,status,st_ctime,0,0,1,0,name from inode_$table_name left join file_$table_name on st_ino=inode;
insert into temp select st_ino,status,st_dtime,0,0,0,1,name from inode_$table_name left join file_$table_name on st_ino=inode;
 create table if not exists mac_$table_name select st_ino,status,st_mtime,sum(m) as `m`,sum(a) as `a`,sum(c) as `c`,sum(d) as `d`,name from temp where st_mtime>0 group by st_mtime,name order by st_mtime,name;
 alter table  mac_$table_name  add key(st_ino);
 drop table temp;
");

     clear_progress();
     return "1";
   },

   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result="<h1>MAC Times</h1>\n";
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     my $url=$co->self_url;
     $url =~ s/[\&\;]from_time=[^\&\;]*//;

     if (exists($args{"from_time"})) {
       $result.="<b>(<a href=$url>Clear</a>) Only examining data after ".$args{"from_time"}."</b><br>";
       $result.=display_table("select from_unixtime(st_mtime) as `time`,st_ino as `inode`,status, if(m,'m',' ') as `m`,if(a,'a',' ') as `a`,if(c,'c',' ') as `c`, if(d,'d',' ') as `d`,name from mac_$table_name where st_mtime > UNIX_TIMESTAMP('".$args{"from_time"}."')",["$url&from_time=",$co->url(-relative=>'1')."?report=view_inode&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=",'','','','','',$co->url(-relative=>'1')."?report=view_filesystem&case=".$args{"case"}."&datafile=".$args{"datafile"}."&file="],\%args,[]);
     } else {
       $result.=display_table("select from_unixtime(st_mtime) as `time`,st_ino as `inode`,status, if(m,'m',' ') as `m`,if(a,'a',' ') as `a`,if(c,'c',' ') as `c`, if(d,'d',' ') as `d`,name from mac_$table_name",["$url&from_time=",$co->url(-relative=>'1')."?report=view_inode&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=",'','','','','',$co->url(-relative=>'1')."?report=view_filesystem&case=".$args{"case"}."&datafile=".$args{"datafile"}."&file="],\%args,[]);
     }
     
     return $result;
   },

   #This routine will be called to clean up
   "reset" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     $dbh->do("drop table if exists inode_$table_name");
     $dbh->do("drop table if exists file_$table_name");
     $dbh->do("drop table if exists mac_$table_name");
     return "1";
   },
  };

$reports{"view_inode"}=
  {
   "name" => "View Inode",
   "family" => "50 Disk Forensics",
#   "hidden" => "yes",
   "description" => "View information about a specific Inode",
   "parameters" => {"case"=>"flag_db","inode"=>"int","datafile"=>"filename"},
   "progress" => sub {
     return ("Currently calculating the connection table, please wait");
   },

   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
     $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td><tr><td>inode number</td><td><input type=text name=inode></td></tr></table>";
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     check_prereq({"report"=>"timeline","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
     return "1";
   },
   
   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $result="";
     my $in="";
     my $table_name = $args{"datafile"};   
     my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     my $sth=$dbh->prepare("select name from file_$table_name where inode = '".$args{"inode"}."'");
     $sth->execute();

     run([$CONF{"FLAG_BIN"}."/istat",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype,$args{inode}],\$in,\$result);
     
     while(my @name=$sth->fetchrow_array) { $result .= "\nFilename: $name[0]\n"; };
     $result="<h1>Inode Information for Inode ".$args{"inode"}."</h1><a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}.">View file</a>  <a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}."&mode=dump>Save File</a><pre>\n".encode_entities($result)."</pre>";
     return $result;
   },
   
   "reset" => sub {
     return "1";
   },
  };


$reports{"view_file"}=
  {
   "name" => "View file",
   "family" => "50 Disk Forensics",
#   "hidden" => "yes",
   "description" => "View file from a given Inode",
   "parameters" => {"case"=>"flag_db","inode"=>"int","datafile"=>"filename"},
   "progress" => sub {
     return ("Currently retrieving the file for viewing");
   },

   "form" => sub {
     my $reference= shift;
     my %args =%{$reference};
     $args{"inode"}='' if !exists $args{"inode"};
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr><tr><td>";
     $result.=disk_partition_form();
     $result.="</td></tr><tr><td>inode number</td><td><input type=text name=inode value='".$args{"inode"}."'></td></tr></table>";
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     check_prereq({"report"=>"timeline","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
     return "1";
   },
   
   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $result="";
     my $file="";
     my $in="";
     my $table_name = $args{"datafile"};     
     my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     #MC FIXME:: Not scalable to keep file in a scalar.

     my $name=$dbh->selectrow_array("select name from file_$table_name where inode = '".$args{"inode"}."'");
     run([$CONF{"FLAG_BIN"}."/icat",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype,$args{inode}],\$in,\$file);
     $result="<h1>File $name in inode ".$args{"inode"}."</h1>";

     my $url=$co->self_url;
     $url =~ s/[\&\;]mode=[^\&\;]*//;
     if (exists($args{"mode"}) && $args{"mode"} eq "hex") {
       my $f = new Data::HexDump;
       $f->data($file);
       $result.="<a href=$url>Text View</a>  <a href=$url&mode=dump>Save File</a> <a href=$url&mode=view>View file</a><pre>\n".encode_entities($f->dump)."</pre>";
     } elsif(exists($args{"mode"}) && $args{"mode"} eq "dump") {
       if (length($name)<=1) { $name="unknown.bin" };
       $name =~ s/^\/*//;
       print $co->header(-type=>'application/octet-stream',-content_disposition=>"attachment; filename=$name").$file;
       exit;
     } elsif(exists($args{"mode"}) && $args{"mode"} eq "view") {
       my @file=("/usr/bin/file","-bi","-");
       my $filetype="";
       my $header=substr($file,0,1024);
       if($file) {
	 run \@file,\$header,\$filetype;
	 chomp $filetype;
	 print $co->header(-type=>$filetype);
	 print sanitize_html($file);
	 exit;
       } else {
	 return("<h1> Error: </h1> File empty");
       };
     } else {
       $result.="<a href=$url&mode=hex>HexDump</a>  <a href=$url&mode=dump>Save File</a> <a href=$url&mode=view>View file</a><pre>\n".wrap_and_sanitise($file,$CONF{"WRAP"})."</pre>";
   }

     return $result;
   },
   
   "reset" => sub {
     return "1";
   },
  };

#Returns a form indicating links for the partitions currently loaded into the case
sub disk_partition_form {
  my $sth=$dbh->prepare("select value from meta where property=\"dd_file\"");
  my $result='';
  $sth->execute;
  while (my @row=$sth->fetchrow_array) {
    my $hash=new CGI($row[0]);
    my $url=$co->self_url;
    $url=~s/(\&|\?|\;)datafile=[^\&\;]*/$1/;
    $result.="<tr><td><a href=$url&datafile=".$hash->param("datafile").">".$hash->param("datafile")." -> ".$hash->param("mountpoint")."</a></td></tr>\n";
  };
  return $result;
};

#Function returns an attribute (a single encoded parameter) from a property in the meta table. Takes on 2 args, the meta table property, and the attribute. Filter is used to further filter the rows returned.
sub get_meta_attr($$$) {
  my $property=shift;
  my $attr=shift;
  my $filter=shift;

  my $sth=$dbh->prepare("select value from meta where property=\"$property\" and value like \"%$filter%\"");
  my $result='';
  $sth->execute;
  my @row=$sth->fetchrow_array;
  my $hash=new CGI($row[0]);
  return $hash->param($attr);
};

$reports{"view_filesystem"}=
  {
   "name" => "Browse Filesystem",
   "family" => "50 Disk Forensics",
#	     "hidden" => "yes",
   "description" => "View a given file within the filesystem",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename","file"=>"string"},
   "progress" => sub {
     return ("Currently calculating the connection table, please wait");
   },
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     if (!exists($args{"file"}) || !$args{"file"}) {
       $args{"file"}="/";
     };

     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."<input type=submit value=\"Update\"></td></tr><tr><td colspan=2>Mount points (partitions) loaded into this case</td></tr>";
    $result.=disk_partition_form();

     $result.="<tr><td>File name to see (relative to the root of the file system)</td><td><input type=text name=file value='".$args{"file"}."'></td></tr></table>";
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;

     check_prereq({"report"=>"timeline","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
     return "1";
   },

   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result="<h1>Browse File system</h1>\n<table><tr><td valign=top>";
     $args{"count"}=0;
     $args{"found"}=0;

     $result.=print_branch("/",\%args,$args{"file"},'','/');

     # Given a root directory name, this function will produce htmlised output for the subdirectory tree which matches the path given in $path. Note that $path is given relative to $root. $root must end with "/" e.g. /bin/
     sub print_branch {
       my $root=shift;
       my $reference= shift;

       my $path=shift;
       my $preamble=shift;
       my $base=shift;
       my $target=$reference->{"file"};

       #shorten the path by removing the first directory off the end... this will be used in recursion below.
       $path=~s/^\/*([^\/]+)//;

       #This holds the base name of the directory that needs to be expanded.
       my $branch=$1;
       my $result='';
       my $in='';
       my $out='';

       my $table_name = $reference->{"datafile"};
       $table_name=~s/[^a-zA-Z0-9]/_/g;

       my $sth=$dbh->prepare("select type,mid(name,length(\"$root\")+1,if(LOCATE(\"/\",name,length(\"$root\")+1),LOCATE(\"/\",name,length(\"$root\")+1)-length(\"$root\")-1,1000)) as `dirname`,status,inode from file_$table_name where name like \"$root%\" group by `dirname` order by `dirname`");
       $sth->execute() or throw("Error");

       while (my @row=$sth->fetchrow_array) {
	 my $url=$co->self_url;
	 $url=~s/([\?\&\;])file=[^\&\;]*/$1/;
	 my $type=$row[0];
	 my $filename=$row[1];
	 my $status=$row[2];
	 my $inode=$row[3];

	 my $type_img=$type;
	 
	 if ($status ne '-') {
	   $type_img = "<img alt='del' src='/flag/images/no.png' height=16> $type";
	 } elsif ($type eq 'd/d') {
	   $type_img="<img alt='d/d' src='/flag/images/folder.png' height=16>";
	 } elsif ($type eq 'r/r') {
	   $type_img="<img alt='r/r' src='/flag/images/page.png' height=16>";
	 } elsif ($type eq 'l/l') {
	   $type_img="<img alt='l/l' src='/flag/images/link.png' height=16>";
	 };

	   if ($target eq $base.$filename || $target eq $base.$filename."/") {
	     $reference->{"found"}=1;
	     $result.= "$preamble <img src='/flag/images/corner.png' height=16>$type_img <a name='mark'></name><font color=red>$filename</font><br>\n";
	     $reference->{"inode"}=$inode;
	   }  else {
	     $result.= "$preamble <img src='/flag/images/corner.png' height=16>$type_img <a href='$url&file=".uri_escape("$base$filename/")."#mark'>$filename<a><br>\n";
	   };

	   if (!$reference->{"found"}) { $reference->{"count"}++ };
	   
	   if ($filename eq $branch && $type eq "d/d") {
	     $result.=print_branch($base.$filename.'/',$reference,$path,$preamble."&nbsp&nbsp&nbsp&nbsp",$base.$filename.'/');
	   }
       };
       return $result;
     };
   
     my $in='';
     my $out='';
     my $sth;
     my $table_name = $reference->{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

     if (exists($args{"inode"})) {
       run([$CONF{"FLAG_BIN"}."/istat",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype,$args{inode}],\$in,\$out);
       $sth=$dbh->prepare("select name from file_$table_name where inode = '".$args{"inode"}."'");
       $sth->execute();
       while(my @name=$sth->fetchrow_array) { $out .= "\nFilename: $name[0]\n"; };
     } else {
       $args{"inode"}=2;
     } ;


     $result.="</td><td valign=top>".("<img border=0 src=/flag/images/spacer.png height=16 width=16>&nbsp <br>\n" x  $args{"count"} )."<pre>".encode_entities($out)."</pre><br><a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}.">View file</a>  <a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}."&mode=dump>Save File</a></td></tr></table>";


#     $result="<h1>Inode Information for Inode ".$args{"inode"}."</h1><a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}.">View file</a>  <a href=".$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=".$args{"inode"}."&mode=dump>Save File</a><pre>\n".encode_entities($result)."</pre>";


     return $result;
   },
   
   #This routine will be called to clean up
   "reset" => sub {
     return "1";
   },
  };

$reports{"md5_comparison"}=
  {
   "name"=>"MD5 Hash comparison",
   "family"=>"50 Disk Forensics",
   #"hidden" => "yes",
   "description" => "This report will give a table for describing what the type of file this is based on the MD5 hash matches",
   "parameters"=>{"case" => "flag_db", "datafile"=>"filename"},
   "progress"=>sub{return "Please wait, building Hash Database"},
   
   "form"=>sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."<input type=submit value=\"Update\"></td></tr>";
     $result.=disk_partition_form();
     $result.="</td></tr></table>";
     return $result;
   },
   
   "analyse" => sub {
     my $reference=shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     #Check to see if we ran report view_filetypes which is a prepreq for this one.
    check_prereq({"report"=>"view_filetypes","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
     #Create a table for quicker look up, basically ties in all the recognized md5 hashes with the nsrl information into the one table so we do not need to keep searching the nsrl database each time.
     do_query("delete from NSRL_$table_name;alter table NSRL_$table_name drop index `productname`;alter table NSRL_$table_name drop index `inode`;
insert into NSRL_$table_name select a.inode,f.name,a.md5,b.filename as nsrl_filename,c.productname,c.productversion,c.mfgcode from hash_$table_name as a,flag.NSRL_hashes as b,flag.NSRL_products as c, file_$table_name as f, inode_$table_name where a.binary_md5 = b.md5 and b.productcode=c.productcode and a.inode=f.inode and a.inode=st_ino and st_size > '1';
alter table NSRL_$table_name add index(productname);alter table NSRL_$table_name add index(inode)");
     return 1;
   },
   
   "reset" => sub {
     my $reference=shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     $dbh->do("delete from NSRL_$table_name");
     return 1;
   },
   
   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;	
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     my $url=$co->self_url;
     
     my $result = "<h1>Recognized file hashes</h1><h2>image file ".$args{"datafile"}." in case: ".$args{"case"}."</h2>";
     $result .= display_table("select inode as `Inode`,name as `Filename`,nsrl_filename as `Possible Hash match`,productname as `Product`,productversion as `Version`,mfgname as `Maker`,md5 from NSRL_$table_name",[$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=",$co->url(-relative=>'1')."?report=view_filesystem&case=".$args{"case"}."&datafile=".$args{"datafile"}."&file="],\%args,[]);
     return $result;
   },
  };

$reports{"view_filetypes"}={
	     "name" => "View File-Types",
	     "family" => "50 Disk Forensics",
#	     "hidden" => "yes",
	     "description" => "View file types and MD5 hashes assiciated with this inode",
	     "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
	     "progress" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       my @row2=$dbh->selectrow_array("select count(*) from file_$table_name where type like \"r/%\" or type like \"%/r\"");

	       my @row=$dbh->selectrow_array("select count(*) from hash_$table_name");
	       return ("Currently calculating hashes and file types for file ".$args{"datafile"}."<br> So far did $row[0] inodes out of $row2[0] file inodes in file.");
	     },

	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       
	       my $result = "<table><tr><td>Select Flag case to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."<input type=submit value=\"Update\"></td></tr>";
	       $result.=disk_partition_form();
	       $result.="</td></tr></table>";
	       return $result;
	     },
	     
	     "analyse" => sub {
	       my $reference= shift;
	       my %args =%{$reference};
	       my $table_name = $args{"datafile"};
	       my $fstype=get_meta_attr('dd_file','fstype','datafile='.uri_escape($args{'datafile'})) || 'linux-ext2';

	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       check_prereq({"report"=>"timeline","case"=>$args{"case"},"datafile"=>$args{"datafile"}});
	       #Store the Binary MD5 hash into this table.
	       do_query("create table if not exists `hash_$table_name` (`inode` INT NOT NULL,`md5` CHAR(35) NOT NULL, `binary_md5` CHAR(16) BINARY NOT NULL, `filetype` TINYTEXT NOT NULL);
create table if not exists NSRL_$table_name  (inode int(11) NOT NULL default '0', name varchar(200) default NULL,  md5 varchar(35) NOT NULL default '',  nsrl_filename varchar(250) NOT NULL default '',  productname varchar(250) default NULL,  productversion varchar(250) default NULL,  mfgname varchar(200) default NULL);
");

	       my $sth=$dbh->prepare("Select inode from file_$table_name where type like \"r/%\" or type like \"%/r\"");
	       $sth->execute;
	       my @file = ($CONF{"FLAG_BIN"}."/file","-m",$CONF{"FLAG_BIN"}."/magic","-b","-");
	       my @icat=($CONF{"FLAG_BIN"}."/icat",$CONF{"UPLOADDIR"}.$args{"datafile"},'-f',$fstype,0);

	       my $sth2=$dbh->prepare("insert into `hash_$table_name` set inode=?,md5=?,binary_md5=?,filetype=?");

	       # We want to find the files that are associated to each inode, and then find their md5sums.
	       while(my @row=$sth->fetchrow_array) {
		 $icat[2]=$row[0];
		 my $in='';
		 my $ctx = Digest::MD5->new;

		 my $h=start \@icat, \$in ,">pipe",*ICAT or die("Could not start icat");
		 my $header="";
		 my $filetype="";
		 my $enough=0;
		 
		 while(<ICAT>) {
		   if(!$enough && length($header)<1024) {
		     $header.=substr($_,0,1040);
		     if(length(header)>1024) {
		       $enough=1;
		     };
		   };
		   $ctx->add($_);
		 };
		 
		 $h->finish();
		 close ICAT;
		 if($header) {
		   run \@file,\$header,\$filetype;
		 };
		 chomp $filetype;
		 
		 # Now store it in the database.
		 my $hash = $ctx->hexdigest;
		 my $binary_hash = $hash;
		 $binary_hash =~ s/(..)/chr(hex("0x".$1))/ieg;
		 $sth2->execute($row[0],$hash,$binary_hash,$filetype) or warn "Couldn't insert hashes: $!\n";
	       };
	       
	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;
	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       # This bit now has an MD5 file identification part.
	       # if(null,<img = cross>, <img = tick>)
	       # or alternatively for those that know C:  null ? <img = cross> : <img tick>;

	       my $result="<h1>File types and MD5 Hashes</h1><h2>image file ".$args{"datafile"}." in case: ".$args{"case"}."</h2>\n";
#	       $result.=display_table("select a.inode,md5,if(md5,'<img src=\"/flag/images/00016.gif\">','<img src=\"/flag/images/00017.gif\">') as \"MD5 match\",name,c.st_size,filetype from hash_$table_name as a,file_$table_name as b,inode_$table_name as c where a.inode=b.inode and a.inode=c.st_ino",[$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=",'',$co->url(-relative=>'1')."?report=view_filesystem&case=".$args{"case"}."&datafile=".$args{"datafile"}."&file="],\%args);
	       $result.=display_table("select a.inode as `Inode`,a.md5 as `MD5`,d.inode as `
Hash DB`,b.name as `File Name`,c.st_size as `Size`,filetype as `File Type` from hash_$table_name as a,file_$table_name as b,inode_$table_name as c left join NSRL_$table_name as d on a.inode=d.inode where a.inode=b.inode and a.inode=c.st_ino",[$co->url(-relative=>'1')."?report=view_file&case=".$args{"case"}."&datafile=".$args{"datafile"}."&inode=",'','',$co->url(-relative=>'1')."?report=view_filesystem&case=".$args{"case"}."&datafile=".$args{"datafile"}."&file="],\%args,['','', sub {
  my $i=shift;
  my $reference=shift;
  my %args =%{$reference};

  if ($i) {
    return ("<abbr title=\"Recognized Hash Info\"><a href=\"".$co->url(-relative=>'1')."?report=md5_comparison&case=".$args{"case"}."&datafile=".$args{"datafile"}."&where_inode=\'$i\'\"><img src=/flag/images/tick.png border=0></a></abbr>");
  } else {
    return "<img src=/flag/images/question.png border=0>";
  };
}]);
	       return $result;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       $dbh->do("drop table if exists hash_$table_name");
	       $dbh->do("drop table if exists NSRL_$table_name");
	       return "1";
	     },
	    };

$reports{"browse_registry"}=
  {
   "name" => "Browse Registry Hive",
   "family" => "50 Disk Forensics",
   "description" => "Browse a windows registry hive file (found in c:\\winnt\\system32\\config\\)",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
   "progress" => sub {
     return ("Currently loading the registry, please wait");
   },
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
     $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td></tr></table>";
     return $result;
   },
	     
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     $dbh->do("create table if not exists reg_$table_name (`path` CHAR(250), `size` SMALLINT, `type` CHAR(12),`key` VARCHAR(200),`value` text)");
     run([$CONF{"FLAG_BIN"}."/flag_reg_read","-f",$CONF{"UPLOADDIR"}.$args{"datafile"},"-t","reg_$table_name"],'|',["mysql","-f","-u",$CONF{"USERNAME"},"-p".$CONF{"PASSWORD"},$args{"case"}]);
     return "1";
   },

   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     my $result="<h1>Registry Hive in file '".$args{"datafile"}."' case '".$args{"case"}."'</h1>";

     our $url=$co->self_url;
     $url=~s/([\?\&\;])type=[^\&\;]*/$1/g;
     $url=~s/([\?\&\;])file=[^\&\;]*/$1/g;
     $url=~s/([\?\&\;])where_[^=]*=[^\&\;]*/$1/g;

     if (exists($args{"type"}) && $args{"type"} eq "table") {
       $result.="<a href=$url>View in Tree</a><p>\n<table><tr><td NOWRAP valign=top>";
       $result.=display_table("select path,size,type as `Type`,`key` as `Key`, if(length(value)<50,value,concat(left(value,50),' .... ')) as `Value` from reg_$table_name",["$url&file="],\%args,[]);
       return $result;
     } else {
       $result.="<a href=$url&type=table>View in Table</a><p>\n<table><tr><td NOWRAP valign=top>";

       $args{"count"}=0;
       $args{"found"}=0;
       $url=~s/([\?\&\;])limit=[^\&\;]*/$1/g;

       if (!$args{"file"}) {
	 $args{"file"}="/";
       };
       
       $result.=print_reg_branch('/',\%args,$args{"file"},'','/');
       $result.="</td><td NOWRAP valign=top>".("<img border=0 src=/flag/images/spacer.png height=16 width=16>&nbsp <br>\n" x  $args{"count"} );
       $args{"simple"}="yes";
       $args{"file"} =~ s/\/*$//;
       
       $result.=display_table("select `type` as `Type`,`key` as `Key`,if(length(value)<50,value,concat(left(value,50),' .... ')) as `Value` from reg_$table_name where path='".$args{"file"}."'",[],\%args,[]);
       
       $result.="</td></tr></table>";
       
       return $result;
     };

     # Given a root directory name, this function will produce htmlised output for the subdirectory tree which matches the path given in $path. Note that $path is given relative to $root. $root must end with "/" e.g. /bin/
     sub print_reg_branch {
       my $root=shift;
       my $reference= shift;

       my $path=shift;
       my $preamble=shift;
       my $base=shift;
       my $target=$reference->{"file"};

       #shorten the path by removing the first directory off the end... this will be used in recursion below.
       $path=~s/^\/*([^\/]+)//;

       #This holds the base name of the directory that needs to be expanded.
       my $branch=$1;
       my $result='';
       my $in='';
       my $out='';

       my $table_name = $reference->{"datafile"};
       $table_name=~s/[^a-zA-Z0-9]/_/g;

       my $sth=$dbh->prepare("select mid( path,length(\"$root\")+1,if(LOCATE(\"/\",path,length(\"$root\")+1),LOCATE(\"/\",path,length(\"$root\")+1)-length(\"$root\")-1,1000)) as `dirname` from reg_$table_name where path like \"$root%\" group by `dirname` order by `dirname`");
       $sth->execute() or throw("Error");

       my $count=0;
       my @array=();
       my $array_found=0;
       
       my $i=0;
       
       while (my @row=$sth->fetchrow_array) {
	 $array[$i] = $row[0];
	 my $match=$base.$row[0];
	 if ($target =~ /^$match\/?/) {
	   $array_found=$i;
	 };
	 $i++;
       };

#       $result.="My array found is $array_found ".$#array;

       if ($array_found-$CONF{"MAXTREESIZE"}>0) {
	 $result.="$preamble  <img alt=\"Scroll Up\" src=\"/flag/images/up.png\"><br>\n";
       }

       for (my $i=($array_found-$CONF{"MAXTREESIZE"})>0?$array_found-$CONF{"MAXTREESIZE"}:0;$i<=(($array_found+$CONF{"MAXTREESIZE"} < $#array)?$array_found+$CONF{"MAXTREESIZE"}:$#array);$i++) {
	 my $filename=$array[$i];
	 my $type_img="<img alt='d/d' src='/flag/images/folder.png' height=16>";
	 $count++;
	 if ($count>30) {
	   $result.="$preamble Click for more <br>";
	   last;
	 }
	 if ($target eq $base.$filename || $target eq $base.$filename."/") {
	   $reference->{"found"}=1;
	   $result.= "$preamble <img src='/flag/images/corner.png' height=16>$type_img <a name='mark'></name><font color=red>$filename</font><br>\n";
	 }  else {
	   $result.= "$preamble <img src='/flag/images/corner.png' height=16>$type_img <a href='$url&file=".uri_escape("$base$filename/")."#mark'>$filename<a><br>\n";
	 };

	 if (!$reference->{"found"}) { $reference->{"count"}++ };
	 if ($filename eq $branch) {
	   $result.=print_reg_branch($base.$filename.'/',$reference,$path,$preamble."&nbsp&nbsp&nbsp&nbsp",$base.$filename.'/');
	   $count=0;
	 };
       };

       if ($array_found+$CONF{"MAXTREESIZE"}<$#array) {
	 $result.="$preamble  <img alt='Scroll Down' src='/flag/images/down.png'><br>\n";
       }

       return $result;
     };
     

   },
   
   "reset" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     $dbh->do("drop table if exists reg_$table_name");
     return "1";
   },
  };

##############################################################
# Unstructured disk forensics
##############################################################

my @magic=();

#This hash represents all predefined magic sets for extracting files from unstructured data
$magic[0] = {
	    "name" => "Jpeg",
	    "description" => "JPEG images",
	    "start_re" => '/\xff\xd8....(JFIF|Exif)/',
	    "end_re"=>"",
	     #Max size in kb
	    "max_size"=>"200",
	     "extension"=>"jpg",
	     "subdir"=>"jpeg",
	     #This is a subroutine that will be executed on each file in subdir to produce a sensible thumbnail. Takes two arguments, the filename to thumb nail and the basename. The function must return the thumb nail (including any content types needed). Sometimes its more convenient to store this on disk the first time the function is called, entirely up to this function.
	     "thumb"=> sub {
	       my $basename= shift;
	       my $filename = shift;

	       #Check that the thumbs directory is there
	       if (! -d $basename.'/thumbs/') {
		 mkdir $basename.'/thumbs/';
	       }

	       #If the original file is not there - there is nothing we can do.
	       if (!-e $basename.'/'.$filename) {
		 return;
	       };

	       #Have we got the file already stored?
	       if (!-e $basename.'/thumbs/'.$filename) {
		 my $temp="";
		 
		 #If conversion failed for some reason - we return a broken image
		 run(['/usr/bin/convert','-geometry','100x100>',$basename.'/'.$filename,$basename.'/thumbs/'.$filename],\$temp,\$temp);
		 if(!-e $basename.'/thumbs/'.$filename) {
		   my $result="";
		   sysopen(FH,$CONF{"IMAGE_PATH"}.'/broken.png',0) || throw("Unable to open thumbnail image");
		   $result = join("",<FH>);
		   close(FH);
		   return $co->header(-type=>"image/jpeg").$result;
		 };
	       };

	       my $result="";
	       sysopen(FH,$basename.'/thumbs/'.$filename,0) || throw("Unable to open thumbnail image");
	       $result = join("",<FH>);
	       close(FH);
	       return $co->header(-type=>"image/jpeg").$result;
	     },
	     #This method is responsible with viewing the full file. You may want to simply return the file with the appropriate content-type to let the browser handle it - or give some kind of a digest (for example a strings of a word document) of the file. Parameters are the same as thumbs.
	     "view" => sub {
	       my $basename= shift;
	       my $filename = shift;
	       
	       #Check that the directory is there
	       if (! -d $basename) {
		 throw("Error - unable to open directory");
	       }

	       #Have we got the file already stored?
	       if (!-e $basename.'/'.$filename) {
		 throw("File not found");
	       };

	       my $result="";
	       sysopen(FH,$basename.'/'.$filename,0) || throw("Unable to open image");
	       $result = join("",<FH>);
	       close(FH);
	       return $co->header(-type=>"image/jpeg").$result;
	     }
	 };

$magic[1] = {
	    "name" => "PNG",
	    "description" => "Portable Network Graphics images",
	    "start_re" => '/\x89PNG/',
	    "end_re"=>"",
	    "max_size"=>"200",
	     "extension"=>"png",
	     "subdir"=>"png",
	     "thumb" => $magic[0]{"thumb"},
	     "view" => $magic[0]{"view"},
	 };

$magic[2] = {
	     "name"=>"MSOffice",
	     "description" =>"Microsoft Office Files",
	     "start_re" => '/(\376\067\0\043|\320\317\021\340\241\261|\333\245-\0\0\0)/',
	     "end_re" => "",
	      "max_size"=>"200",
	      "extension"=>"doc",
	      "subdir"=>"ms_office",
	     "thumb"=> sub {
	       my $result="";
	       sysopen(FH,$CONF{"IMAGE_PATH"}.'/msoffice.gif',0) || throw("Unable to open msoffice image");
	       $result = join("",<FH>);
	       close(FH);
	       return $co->header(-type=>"image/jpeg").$result;
	     },

	     "view"=> sub {
	       my $result="";
	       my $basename= shift;
	       my $filename = shift;
	       print STDERR "about to run ".'/usr/bin/strings '. $basename.'/'.$filename."\n";
	       run(['/usr/bin/strings',$basename.'/'.$filename],\$result,\$result) ;
	       
	       return $co->header(-type=>"text/html")."<html><body><pre>".wrap_and_sanitise($result,$CONF{"WRAP"})."</pre></body></html";
	     },
	    };

$reports{"extract_files"}={
	     "name" => "Extract files",
	     "family" => "60 Unstructured Forensics",
	     "description" => "Extract files from unstructured data based on common file signatures",
	     "parameters" => {"case"=>"flag_db","datafile"=>"filename","magic"=>"checkbox","/^size_.+/"=>"alphanum"},
	     "progress" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       return ("Currently extracting files from ".$args{"datafile"}.get_progress(\%args));
	     },
	     "form" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $result = "<table><tr><td>Select Flag case to operate on: </td><td>".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
	       $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td></tr></table>";
	       my $column_width = 4; #Number of columns
	       $result.="<table width=100%>";

	       #This is how we can recall the magic CGI array
	       $result.="My array is ".join("<br>",split("\0",$args{"magic"}))."<br>";

	       #Work through the $magic array to display all the magic we have
	       for (my $i=0; $i<=$#magic; $i+=$column_width) {
		 $result.="<tr>";
		 for (my $j=0; $j<=$column_width; $j++) {
		   if (exists($magic[$i+$j])) {
		     $result.="<td><abbr title=\"".$magic[$i+$j]{"description"}. "\"><input name=magic type=checkbox value=\"".$magic[$i+$j]{"name"}."\" checked>".$magic[$i+$j]{"name"}." up to <input type=text size=5 name=\"size_".$magic[$i+$j]{"name"}."\" value=\"".$magic[$i+$j]{"max_size"}."\">kb</td>";
		   } else {
		     $result.="<td></td>";
		   } ;
		 };
		 $result.="</tr>";
	       };


	       $result.="</table>";
	       return $result;
	     },
	     
	     "analyse" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       my $base_dirname=$CONF{"RESULTDIR"}.'/'.$table_name.'_extract';
	       if (! -d $base_dirname) {
		 mkdir $base_dirname || die ("$@");
	       };

	       my $absolute_max_size = 10000000;
	       my @magic_selection = split "\0", $args{"magic"};
	     
	       #Get the user chosen magic and do each one
	       for (my $magic_i=0;$magic_i<=$#magic_selection; $magic_i++) {	
		 foreach my $i (@magic) {
		   my %i = %{$i};

		   #lets do this one
		   if ($i{"name"} eq $magic_selection[$magic_i]) {
		     set_progress("Extracting ".$i{"description"});
		     #Create subdirs. If a subdir is already there - we assume the we did that one before...
		     if (! -d $base_dirname.'/' . $i{"subdir"}) {
		       mkdir $base_dirname.'/' . $i{"subdir"} || die ("$!");
		       
		       my $tmp="";
		       
		       my $max_size = 10000; #Maximum reasonable size (10mb);
		       #Ensure that the maximum size is reasonable
		       if ($args{"size_".$magic_selection[$magic_i]} > $max_size) {
			 $args{"size_".$magic_selection[$magic_i]} = $max_size;
		       };

		       print STDERR ($CONF{"FLAG_BIN"}.'/exgrep','-s',$i{"start_re"},'-m',$args{"size_".$magic_selection[$magic_i]} * 1024,'-x',$i{"extension"},'-f',$CONF{"UPLOADDIR"}.'/'.$args{"datafile"},'-p', $base_dirname.'/'.$i{"subdir"}.'/');
		       run([$CONF{"FLAG_BIN"}.'/exgrep','-s',$i{"start_re"},'-m',$args{"size_".$magic_selection[$magic_i]} * 1024,'-x',$i{"extension"},'-f',$CONF{"UPLOADDIR"}.'/'.$args{"datafile"},'-p', $base_dirname.'/'.$i{"subdir"}.'/'],\$tmp,\$tmp);

		       #Generate the thumbnails in the background to allow the user to browse them immediately...
		       if (fork()) {
			 my $count=0;
			 #Work on each line of output
			 while ($tmp=~/(\d+)/smg){
			   my $file = $1.".".$i{"extension"};
			   my $directory = $base_dirname.'/'.$i{"subdir"}.'/';
			   if (exists($i{"thumb"})) {
			     $count++;
			     &{$i{"thumb"}}($directory,$file);
			   };			
			 };
			 CORE::exit();
		       };
		     };
		   };
		 };

	       };
	       
	       clear_progress();
	       return "1";
	     },

	     "display" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       #This part draws a selection listbox for the category items
	       my @magic_selection = split "\0", $args{"magic"};

	       #Work out the base directory for the extracted files.
	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;

	       my $base_dirname=$CONF{"RESULTDIR"}.'/'.$table_name.'_extract';

	       if (!exists($args{"selected"}) || !in_array($args{"selected"},\@magic_selection)) {
		 $args{"selected"} = $magic_selection[0];
	       }

	       my %selected_magic;
	       #Find the magic record which is currently selected
	       for (my $i=0; $i<=$#magic; $i++) {
		 if ($magic[$i]{"name"} eq $args{"selected"}) {
		   %selected_magic=%{$magic[$i]};
		   last;
		 };
	       };
	       
	       if (!%selected_magic) {
		 throw("Cant find magic type");
	       };
	       
	       #If we need to delete a bunch of files we do so now:
	       if (exists($args{"delete"})) {
		 foreach my $file (split "\0", $args{"delete"}) {
		   $file =~ /([\w\.]+)$/;
		   my $filename=$1;
		   
		   unlink $base_dirname.'/'.$selected_magic{"subdir"}.'/'.$filename;
		   unlink $base_dirname.'/'.$selected_magic{"subdir"}.'/thumbs/'.$filename;
		 }
	       };


	       #We need to produce the thumb nails here...
	       if (exists($args{"thumb"}) && exists($args{"file"})) {
		 $args{"file"} =~ /([\w\.]+)$/;
		 my $filename=$1;
		 
		 #Producing the actual thumbnails is the responsibility of the magic entry
		 print &{$selected_magic{"thumb"}}($base_dirname.'/'.$selected_magic{"subdir"}.'/',$filename);
		 exit;
		 #We need to produce the entire image
	       } elsif (exists($args{"view"}) && exists($args{"file"})) {
		 $args{"file"} =~ /([\w\.]+)$/;
		 my $filename=$1;
		 
		 print &{$selected_magic{"view"}}($base_dirname.'/'.$selected_magic{"subdir"}.'/',$filename);
		 exit;
	       };

	       my $column_width=4;
	       my $row_num=10;

	       #Number of thumbs per page
	       my $number_of_thumbs = $column_width * $row_num;
	       if (!exists($args{"page"})) {
		 $args{"page"}=0;
	       }
	       my $count=$number_of_thumbs*($args{"page"});

	       #Now we open the directory and list it:
	       opendir DH, $base_dirname.'/'.$selected_magic{"subdir"} || throw("Cant open directory");
	       my @files_in_dir = sort(readdir DH);
	       close DH;

	       #Here we draw the HTML form
	       my $result="<h1>Extracted ".$args{"selected"}." files from ".$args{"datafile"}."</h1>\n<form name=frm method=post action=\"".$co->url(-relative=>1)."\">";

	       #Pass the query in hidden parameters
	       foreach my $arg (keys(%args)) {
		 foreach  my $value (split "\0",$args{$arg}) {
		   if ($arg ne 'selected' && $arg ne 'delete' && !($arg=~/^reload_/)) {
		     $result.="<input type=hidden name=\"".$arg."\" value=\"".$value."\">\n";
		   };
		 };
	       };

	       $result.="Select extracted file types: ".$co->scrolling_list(-name=>'selected', -values=>\@magic_selection,-default=>[$args{'selected'}],-size=>'1',-onclick=>"this.form.submit();");

	       $result.="<table width=100%>";
	       my $dir="";
	       do {
		 $result.="<tr width=100%>";
		 for (my $j=0;$j<$column_width;$j++) {
		   #We only want to see the files we are interested in (i.e. with the correct extension
		   do {
		     $dir=$files_in_dir[$count++];
		   } while ($dir && !($dir=~/$selected_magic{"extension"}$/));
		   
		   if ($dir) {
		     my @stat = stat($base_dirname.'/'.$selected_magic{"subdir"}.'/'.$dir);
		     
		     #If the user asked us to reload more (or less) of the file from the image we do that here...
		     if (exists($args{"reload_$dir"}) && int($stat[7]/1024)!=int($args{"reload_$dir"})) {
		       use Fcntl;
		       my $newsize=$args{"reload_$dir"}*1024;
		       #we can only read a reasonable size here...
		       if ($newsize > 10_000_000) {
			 $newsize=10_000_000;
		       };
		       
		       my $offset = 0;		 
		       if ($dir =~ /^(\d+)/) {
			 $offset=$1;
		       };
		       
		       my $table_name = $args{"datafile"};
		       $table_name=~s/[^a-zA-Z0-9]/_/g;
		       
		       my $file="";
		       
		       #open the dd image file:
		       sysopen(FH,$CONF{"UPLOADDIR"}.'/'.$args{"datafile"},0) || throw("Cant open dd file ".$args{"datafile"});
		       sysseek FH,$offset,0;
		       sysread FH,$file,$newsize;
		       close(FH);
		       
		       #Write to the extracted file:
		       sysopen(FH,$base_dirname.'/'.$selected_magic{"subdir"}.'/'.$offset.'.'.$selected_magic{"extension"},O_WRONLY) || throw("Unable to open file for writing ".$base_dirname.'/'.$selected_magic{"subdir"}.'/'.$dir);
		       syswrite(FH,$file);
		       close(FH);
		       
		       #Now regenerate the thumbnail if needed, and reupdate the stat data:
		       unlink $base_dirname.'/'.$selected_magic{"subdir"}.'/thumbs/'.$dir;
		       @stat = stat($base_dirname.'/'.$selected_magic{"subdir"}.'/'.$dir);
		     };

		     my $url=$co->self_url;
		     $url =~ s/[\?\&\;]reload_[^=]*=[^\&\;]*//ig;
		     
		     $result.="<td valign=bottom><table border=3><tr><td><a href=\"$url&view=yes&file=$dir\"><img border=1 src=\"$url&thumb=yes&file=$dir\"></a></td><td valign=bottom>Delete <input type=checkbox name=delete value=\"$dir\"><br><input type=text name=\"reload_$dir\" size=4 value=".int($stat[7]/1024).">kb";

		     $result.="</td></tr></table><br>$dir </td>\n";
		   };
		   
		 };
		 $result.="</tr>";
	       } while($dir && --$row_num);
	       
	       $result.="</table>";
	       $result.="<input type=submit value=update> <label name=label>Select all for delete:</label> <input type=checkbox name=all onclick=\"for(var i=0;i < frm.length;i++){
      		             fldObj = frm.elements[i]; // all form elements
    	                              if (fldObj.type == 'checkbox'){ 
                                                   if (all.checked==true) {
			 fldObj.checked = true;
                                                   } else {
			 fldObj.checked = false;
                                                   }
                                               }};\"></form>";
	       print print_headers(["page","reload_"],($args{"page"}>0?$args{"page"}-1:0),$args{"page"}+1," Page ".edit_parameters_form(\%args,["page","reload_"],"<input size=3 name=page value=\"".$args{"page"}."\">\n")." / ".int($#files_in_dir/ $number_of_thumbs));

	       print $result.$flag_footer;
	       exit;
	     },

	     #This routine will be called to clean up
	     "reset" => sub {
	       my $reference= shift;
	       my %args =%{$reference} ;

	       #When we reset this report, we need to remove all the directories we created:
	       my @magic_selection = split "\0", $args{"magic"};
	       
	       #Work out the base directory for the extracted files.
	       my $table_name = $args{"datafile"};
	       $table_name=~s/[^a-zA-Z0-9]/_/g;
	       
	       my $base_dirname=$CONF{"RESULTDIR"}.'/'.$table_name.'_extract';
	       #Get the user chosen magic and do each one
	       for (my $magic_i=0;$magic_i<=$#magic_selection; $magic_i++) {	
		 foreach my $i (@magic) {
		   my %i = %{$i};
		   
		   #lets do this one
		   if ($i{"name"} eq $magic_selection[$magic_i]) {
		     my $tmp="";

		     run(['/bin/rm','-rf',$base_dirname.'/'.$i{"subdir"}.'/'],\$tmp,\$tmp);
		   };
		 };
	       };

	       return "1";
	     },
	    };

$reports{"windows_history"}=
  {
   "name" => "View Windows Internet History",
   "family" => "60 Unstructured Forensics",
   "description" => "Search an image file and extract URLs visited",
   "parameters" => {"case"=>"flag_db","datafile"=>"filename"},
   "progress" => sub {
     my $reference = shift;
     my %args = %{$reference};
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     my @row=$dbh->selectrow_array("select count(offset) from history_$table_name");
     return ("Extracting URLs from ".$args{"datafile"}.", please wait<br>Found ".$row[0]." URLs".get_progress(\%args));
   },
   "form" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     my $result = "<table><tr><td>Select Flag case to operate on:</td><td> ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh)."</td></tr>";
     $result.="<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args)."</td></tr></table>";
     return $result;
   },
   
   "display" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;

     my $result= "<h1>Internet history from ".$args{"datafile"}."</h1>";
     $result.=display_table("select offset as `Offset`,from_unixtime(timestamp) as `Timestamp`,user as `User`,string as `URL` from history_$table_name",[],\%args,[]);
     return $result;
   },
   
   "analyse" => sub {
     my $reference= shift;
     my %args =%{$reference} ;
     
     #create table if it doesn't exist...
     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     $dbh->do("create table if not exists history_$table_name (`offset` BIGINT NOT NULL,`timestamp` INT NOT NULL,`user` TEXT,`string` TEXT NOT NULL,key (offset))");
     my $sth=$dbh->prepare("insert into history_$table_name set offset=?,timestamp=?,user=?,string=?");

     open(IN,$CONF{"UPLOADDIR"}.$args{"datafile"}) or throw("Can't open image ".$CONF{"UPLOADDIR"}.$args{"datafile"});
     
     set_progress("Extracting URL history from ".$args{"datafile"});
     
     my $blocksize = 4096;
     my $block = "";
     my $bytesread = 0;
     my $offset = 0;
     my $year = 3.1536e+14;                             # Initialise values for date calculations
     my $day = 8.64e+11;				# Year, day and second in Windows format
     my $second = 1.0e+7;
     my $hour = 3.6e+3;					# Hour in Unix format (used after conversion)
     my $timezone = (10*$hour);
     my $progresscheck = 0;

     while (read (IN, my $newblock, $blocksize)) {
	
       $block .= $newblock;
       $bytesread += length $newblock;

       while ($block =~ m/URL\x20.\x00{3}/) {

         my $timestamp = 0;                                     # Initialise timestamp counters
	 
	 $block =~ s/^.*?(?=URL\x20.\x00{3})//s;
	 
	 if (length $block < $blocksize) {
	   read (IN, $newblock, $blocksize);
	   $block .= $newblock;
	   $bytesread += length $newblock;
	 }
	 
	 $offset = $bytesread - length $block;
	 $block =~ s/^.{16}//s;
	 
	 for (my $loop1 = 0; $loop1 < 8; $loop1++) {
	   my $decimal = ord $block;
	   $block =~ s/^.//s;
	   for (my $loop2 = 0; $loop2 < $loop1; $loop2++) {
	     $decimal *= 256;
	   }
	   $timestamp += $decimal;
	 }

	 $timestamp = int(($timestamp-(369*$year)-(89*$day))/$second); # Convert to Unix time
	 $timestamp -= $timezone;		# Convert to local time
	 
	 $block =~ m/^.{1,150}?([\x21-\x7e]+?:\/\/[\x21-\x7e]+)/;
	 
	 if ($1) {
	   my $string = $1;
	   my $user = "";

	   if ($string =~ m/([\x21-\x7e]+?)@[\x21-\x7e]{1,8}:\/\//) {
	   	$user = $1;
	   }
	   $string =~ s/^.*@(?=.+?:\/\/)//;
	   $sth->execute($offset,$timestamp,$user,$string);
	   update_progress("URL found at offset $offset",$dbh,"");
	 }
       }
       $block =~ s/.*(?=.{8}\z)//s;
       $offset = $bytesread - length $block;
       if (($offset-$progresscheck) > 10000000) {
         update_progress("Currently read up to offset $offset",$dbh,"");
	 $progresscheck = $offset;
       }  	
     }
     close(IN);
   },
   
   #This routine will be called to clean up
   "reset" => sub {
     my $reference= shift;
     my %args =%{$reference} ;

     my $table_name = $args{"datafile"};
     $table_name=~s/[^a-zA-Z0-9]/_/g;
     
     $dbh->do("drop table if exists history_$table_name");
     return "1";
   },
  };


##############################################################
# Start of object methods.
##############################################################
#Constructor
sub new($$) {
  my $self = {};
  my $reference=shift;
  my $co_ref = shift;

  #This object will carry the CGI that was passed to it.
  $co=$ {$co_ref};

#  if (defined($co)) { $co->DESTROY };
  #Populate these package wide globals for each new instantiation of this object.
#  $co=new CGI;

  #Load defaults:
  load_defaults();
  $flag_db = $CONF{"FLAGDB"};

  my %args=%{$reference};

  if(!exists($args{"case"})) {
    $args{"case"}="flag";
  };
  
  #Create a database connection...
  $self->{DBH} = connect_to_db($args{"case"});
  
  $dbh=$self->{DBH};

  #Initialise the header whenever the constructor is called:
  $self->{"flag_head2"}= "</td>
    <td width=10><center><img src=\"/flag/images/defence.png\"><br><font size=\"+1\"><a href=\"/flag/html/gpl/\"> &copy;CNVT</a></font></center></td>
	</tr>
      </tbody>
    </table>";
  $flag_head2=$self->{"flag_head2"};
  
  my $url=$co->self_url;
  $url=~s/limit=[^\&\;]*//;
  if(!($url=~/\?/)) {$url.="?"};

  my $home="";

  if(exists($args{"report"})) {
    $home="family=".uri_escape($reports{$args{"report"}}{"family"});
  };

  $self->{"flag_head1"}="<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">
<html>
  <head>
    <title>FLAG - Forensic Log Analysis GUI</title>
  </head>
<body link=blue vlink=blue bgcolor=\"".$CONF{"BGCOLOR"}."\"><table width=\"100%\">
       <tbody>
        <tr>
          <td align=left width=10><img src=\"/flag/images/flag.png\" alt=\"flag_heading\" border=\"0\"></td><td align=center>";
   $flag_head1=$self->{"flag_head1"};

  $self->{"flag_nav"}="<table><tr>
	  <td valign=\"bottom\"><a href=\"flag.pl?$home\"><img src=\"/flag/images/home.png\" border=\"0\"></a></td>";
  $flag_nav=$self->{"flag_nav"};

  my $page_size=$CONF{"PAGESIZE"};
  my $prev_limit = ($args{"limit"} || 0 )-$page_size;
  if ($prev_limit<0) {$prev_limit=0}
 
  $self->{"flag_nav2"}="<td valign=\"bottom\"> <a href=\"$url&limit=$prev_limit\"><img src=\"/flag/images/back.png\"  border=\"0\"></a></td> page ".(int(($args{"limit"}||0)/$page_size)+1)."<td valign=\"bottom\"><a href=\"$url&limit=".(($args{"limit"}||0)+$page_size)."\"><img src=\"/flag/images/forward.png\" border=\"0\"></td> </a><td valign=\"bottom\">
	  <td valign=\"bottom\"><a href=".$co->self_url."&reset=1><img src=\"/flag/images/reset.png\" border=\"0\"></a></td>	  
	  <td valign=\"bottom\"><a href=".$co->self_url."&stop=1><img src=\"/flag/images/stop.png\" border=\"0\"></a></td></tr></table>";
  $flag_nav2=$self->{"flag_nav2"};

  $self->{"flag_header"}="\r\n".$self->{flag_head1}.$self->{flag_nav}.$self->{flag_nav2}.$self->{flag_head2};
  $flag_header=$self->{"flag_header"};

  $self->{"flag_footer"}="<p><center>".$self->{flag_nav}.$self->{flag_nav2}."</center></body></html>";
  $flag_footer=$self->{"flag_footer"};
  $self->{"flag_footer_plain"} = "</body></html>";
  $flag_footer_plain="</body></html>";

  bless($self);
  return($self);
};

#Shows all the reports in a nice table...
sub show($) {
  my $self=shift;
  my $reference=shift;
  my %args=%{$reference} ;

  my $i;
  my $current="";
  my $url=$co->url(-relative=>'1');
  my $result="";

  $result.=$flag_header;
  $result.="<table><tr><td valign=top><h1> Class </h1>";
  #Find all the families by making a cross index in the %families hash. Each member of the hash contains an array of all the names.
  my %families;
  foreach $i (sort(keys %reports)) {
    push @{$families{$reports{$i}{'family'}}}, $i;
  }

  foreach $i (sort(keys %families)) {
    my $name=$i;
    $name=~s/^\d*//g;
    $result.="<a href='flag?family=".uri_escape($i)."'>".$name."</a><br>\n";
  };
  $result.= "</td><td width=10 valign=top>    </td><td valign=top>";

  if (!exists($args{'family'})) {
    $args{'family'}="10 Create new case";
  }

  $result.= "<h1>Reports</h1>";
  foreach $i ( @{$families{$args{'family'}}}) {
    if(!exists($reports{$i}{'hidden'})) {
      $result.= "  <a href='".$co->url(-relative=>'1')."?report=$i'> ".$reports{$i}{'name'}."</a><br>\n";
      $result.="<img src=/flag/images/spacer.png width=40 height=1>".$reports{$i}{'description'}."<br><br>\n";
    };
  };

  $result.="</td></tr></table>";

  return($result.$flag_footer_plain);
};

#This function forms the core of the framework. It checks the URL parameters for the report number that should be executed. The caching status is then checked in the meta table for the case. If the report has been executed before, the display method is called. Otherwise the framework forks and executes the analysis method of the report. If the analysis takes too long, the user will receive a progress indication from the report by calling the progress method repeatadly.
sub process_report($$) {
  my $self=shift;
  my $my_name = shift;
  my $reference = shift;
  my %parameters=%{$reference};
  my $case = $parameters{"case"};

  my $result="";
  my $temp;

  $temp=check_parameters($my_name,\%parameters);
  #Include the consistant header:
  $result.=$self->{"flag_header"};

  #The following fields are essential methods, if any of them is missing, spit the dummy:
  foreach my $word ("progress","display","analyse","name","description","reset","form") {
    if (!exists($reports{$my_name}{$word})) {
      return("$result <h1> Internal Error encountered!!! </h1>\nCould not find method $word in report $my_name. This should not happen<br> It is possible that the report has not beed completed properly. ");
    }
  };

  #If our parameters are bad we will display the form:
  if ($temp) {
    $result.=$temp."<h1>".$reports{$my_name}{"name"}."</h1>\n";
    
    $result.="<form action=".$co->url(-relative=>'1')." method=get name=my_form> <input type=hidden name=report value=\"$parameters{'report'}\">";
    $result.= &{$reports{$my_name}{"form"}}(\%parameters,$self->{DBH});
    $result.= "<input type=submit name=final value=\"Submit\"></form>$flag_footer_plain";
    return $result;

#If we were told to reset the report we do this now...
  } elsif (defined($parameters{"reset"})) {
#    return $result.reset_report($my_name,\%parameters);
    my $result=reset_report($my_name,\%parameters);
    my $url=$co->self_url;

    #Redirect the browser to the canonical version of this report
    $url=$co->url(-relative=>'1')."?".canonicalise(\%parameters);

    #Redirect the browser to this report again, so it will be recalculated.
    return (refresh($result,$url));

    #Here we check to see if another flag instance is already analysing this - if so we kill it and refresh.
  } elsif (defined($parameters{"stop"})) {
    my $pid = $dbh->selectrow_array("select property from $case.meta where property like \"pending%\" and value='".canonicalise(\%parameters)."'");
    $pid=~s/\D//ig;

    my $url=$co->self_url;
    #Remove the stop= from the URL:
    $url=~s/stop=[^\&]*//;

    if (!$pid ) { return(refresh("<h1> Not currently analysing this report.</h1>",$url)) };
    kill 15 => $pid;
    #Now we remove any pending states in case its still there:
    $dbh->do("delete from meta where  property like \"pending%\" and value='".canonicalise(\%parameters)."'");
    $dbh->do("delete from meta where  property like \"progress%\" and value='".canonicalise(\%parameters)."'");

    #Redirect the browser to this report again, so it will be recalculated.
    return ($result."<h1>Just killed report $pid</h1>");

    #Here we check to see if another flag instance is already analysing this - if so we display the progress report.
  } elsif (my $pid = $dbh->selectrow_array("select property from $case.meta where property like \"pending%\" and value='".canonicalise(\%parameters)."'")) {
    $pid=~s/\D//ig;
    #Now we need to check that the pid is still running - we do this by sending a signal to it:
    if (!(kill 0 => $pid)) {
      $dbh->do("delete from $case.meta where property like \"pending\%\" and value=\"".canonicalise(\%parameters)."\"");
      #Sometimes we find that the process went away but didnt clean the meta table in the database, in that case we just try to call the display method which is hopefully ready to be run (This is dangerous since the analysis process might have died and the display method is not ready yet, but seems to work in most cases).
      $result.= &{$reports{$my_name}{"display"}}(\%parameters,$self->{DBH});
      return $result.$self->{flag_footer};
#      return (refresh("<h1>Error</h1> Process $pid appears to have died unexpectedly....",''));
    }

    return refresh(&{$reports{$my_name}{"progress"}}(\%parameters,$self->{DBH}),'');

    #Did we run this report before (i.e. is it already cached) if so we call the display method
} elsif ($dbh->selectrow_array("select * from $case.meta where property=\"report_executed\" and value='".canonicalise(\%parameters)."'")) {
  $result.= &{$reports{$my_name}{"display"}}(\%parameters,$self->{DBH});
  return $result.$self->{flag_footer};
  
  #OK Now we need to do the analysis from scratch
} else {
    #When the child exits we want to display the report and exit - Note that this will override the sleep() in the parent in case the child finishes before the parent is done sleeping.
  local $SIG{CHLD} =sub {
    #We are currently in the signal handler, If we dont ignore SIGCHLD now, and the display routine forks (e.g. IPC::Run), we keep getting called here each time and end up with a very messy recursion bug!
    local $SIG{CHLD} = 'IGNORE';
    my $co=new CGI;
    my $result=&{$reports{$my_name}{"display"}}(\%parameters,$self->{DBH});
    print $co->header.$self->{"flag_header"}.$result.$self->{"flag_footer"};
    
    #Read the child exit status so we dont have zombies around...
    my $kid;
    do {
      $kid = waitpid(-1,WNOHANG);
    } until $kid == -1;
    CORE::exit;
  };

    #Note in the database that we are working on this job...
    if ($case) {
      $dbh->do("insert into $case.meta set property=\"pending \",value=\"".canonicalise(\%parameters)."\"");
    };
    
    $pid=fork();
    
    #Our parent will wait for the child to do the analysis for several seconds. If the child is still going an auto-refresh page is produced.
    if ($pid) { #parent
      $dbh=connect_to_db($case);
      #record the pid in the database so that progress indications work.
     if ($case) {
	$dbh->do("update $case.meta set property=\"pending $pid\" where value=\"".canonicalise(\%parameters)."\" and property != \"report_executed\"");
      }
      #Maximum wait 3 seconds to have the child finish
      sleep(3);
      
      #Output a self refreshed meta page for progress bar every 2 seconds
      return(refresh(&{$reports{$my_name}{"progress"}}(\%parameters),''));
      close(STDOUT);
      CORE::exit;
      #The child will go into the background and begin the analysis.
    } else {#child
      #Children sometimes need to fork as well, in that case we dont care about their children:
      local $SIG{CHLD} = 'IGNORE';
#      local $SIG{CHLD} = sub 
#	print STDERR "Handled SIGCHLD\n";
#	return;
#      };
      
      #Need to create a database handle just for the child to make sure that the parent doesnt terminate the connection on it...
      $self->{DBH} = connect_to_db($case);
      $dbh= $self->{DBH};
      
      #This is required to ensure that mod_perl closes the apache socket
      if(exists $ENV{MOD_PERL}) {
	$parameters{"apache_obj"}->cleanup_for_exec();
      }
     # chdir '/';

      #Close all superfleous filedescriptors - we need to use POSIX's close because we dont have perl descriptors.
      setsid;
      for (my $i=0; $i<=254; $i++) {
	POSIX::close($i);
      }

      #If we need to debug - we throw all debugging information into a file:
      if ($debug_analysis) {
	open(STDERR,">>/tmp/debug_analysis") || die("Cant open debug file");
	print STDERR "\nFlag Debug output ". time."\n-------------------------------------------\n";
      };
      
      local $SIG{TERM} = sub {
	print STDERR "Handled SIGTERM";
	$dbh->do("update  $case.meta set property=\"report_executed\"  where property like \"pending%\" and value=\"".canonicalise(\%parameters)."\""); 
	exit;
      };

      &{$reports{$my_name}{"analyse"}}(\%parameters,$self->{DBH});
      $dbh->do("update  $case.meta set property=\"report_executed\"  where property like \"pending%\" and value=\"".canonicalise(\%parameters)."\"");

      #This is needed because mod_perl overrides this method to Apache::exit and we really do want to quit here becuase we just forked the process. (Hence its useless now)
      CORE::exit(0);
    }
    
#    $result.= &{$reports{$my_name}{"display"}}(\%parameters,$self->{DBH});
#    $result.="</body></html>";
#    return $result;
  };
};

#This subroutine checks that the paramters presented in %paramters are ok to be run in report $report_number. It does this by iterating over the type testing functions in %types to ensure that the paramters conform. Returns -1 or 0 if ok.
sub check_parameters($$) {
  my $report_name=shift;
  my $reference=shift;

  my %parameters=%{$reference};
  my $key;
  my $type;

  #NOTE!!!! You must reset the each iterator here since we are exiting from the loop early sometimes... this really screws up mod_perl....
  keys %{$reports{$report_name}{"parameters"}};

  while(($key,$type) = each (%{$reports{$report_name}{"parameters"}})) {
    my $var=$parameters{$key};
    #If the parameter is a RE we check if it matches. NOTE: If you use a RE to specify a bunch of inputs, you cant count of them being there, this is in contrast to the case where you specify them explicitely (pretty obvious but sometimes I forget that). e.g. specifying "var"=>"alphanumeric" ensures that var must exist, but specifying "/var_.*/"=>"alphanumeric" does not force there to be a var parameter.
    if ($key =~ /^\/(.+)\/$/) {
      my $re=$1;

      keys %parameters;
      #Search through the paramters given for a matching RE.
      while ((my $key, my $var) = each(%parameters)) {
	if ($key=~/$re/) {
	  if(!(&{$types{$type}}($var))){
	    return "<h1> Error</h1>Expecting parameter $key to be of type $type, but instead I got '$var' - Please check this value and try again<p>";
	  };
	};
      };
      next;
    };

    if (!exists($parameters{$key}) || $parameters{$key} eq '') {
      return("<!-- $key not defined -->");
    } elsif(!(&{$types{$type}}($var))){
      return "<h1> Error</h1>Expecting parameter $key to be of type $type, but instead I got '$var' - Please check this value and try again<p>";
    };
  };
  return("");
};

sub DESTROY  {
  my $self = shift;
  $self->{DBH}->DESTROY;
};

##################################################
#
#    Helper functions go here.
#
##################################################

#Draws a selector widget with the query as returned by query
sub selector {
  my $variable = shift;
  my $default = shift;
  my $query = shift;

  my $result="<select name=\"$variable\">\n";
  my $sth;
  my $first;

  $sth=$dbh->prepare($query);
  $sth->execute();

  while(my @row=$sth->fetchrow_array) {
    if(defined($row[0]) && $default && ($row[0] eq $default)) {
      $result.="<option selected value=\"".($row[0])."\">$row[1]</option>\n";
    } else {
      $result.="<option value=\"".($row[0])."\">$row[1]</option>\n";
    };
  };
  $result.="</select>";
};

#Draws a table given a query from the database. link will be put into the anchor href with the first column appended to it. Takes care of order and limits.
sub display_table($$$$) {
  my $result="";
  my $query = shift;
  my $link_ref = shift;
  my $reference = shift;

#this array reference will contain references for functions that must be called in order to translate the values stored in the database prior to displaying them on the screen. It is optional.
  my $translation=shift;

  my %args=();
  if(defined($reference)) {
    %args=%{$reference};
  };

  #process bookmarks
  my $bmh = $dbh->prepare("select id,canon,url,description,bookmark from bookmarks where canon=?");
  $bmh->execute(canonicalise($reference));

  my @bookmarks=();

  while (my @row=$bmh->fetchrow_array) {
      push @bookmarks, \@row;
  };

  my @link;

  if ($link_ref) {
    @link = @{$link_ref};
  };

#Number of table entries per page
  my $page_size=$CONF{"PAGESIZE"};
  my $order="";
  my $old_ordered_value="";
  my $grouped_order=0;
  my $group_clause="";
  my $group_by="";
  my $having_clause="";
  my $group_by_colnum=0;
  my @width;

  if (exists($args{"group_by"}) && !($query=~/group by/i)) {
    my $url=$co->self_url;
    $url=~s/(\&|\?|;)group_by=[^\&\;]*//g;
    $group_clause=" group by `".$args{"group_by"}."`";
    $group_by="&group_by=".uri_escape($args{"group_by"});
    $query=~s/select /select count\(\*\) as "Group count",/i;
    @link=('',@link);
    $result.="(<a href=$url>Clear</a>)<b> Grouping by ".$args{"group_by"}."</b><br>";
  } elsif (exists($args{"order"}) && $args{"order"} eq "Group count") {
    delete($args{"order"});
  } elsif (exists($args{"dorder"}) && $args{"dorder"} eq "Group count") {
    delete($args{"dorder"});
  };

#Initialise order and limit if they are not defined
  if(exists($args{"dorder"})) {
    $order = "order by `".$args{"dorder"}."` desc ";
    $grouped_order=$args{"dorder"};
#    $result.="<b> ordering by ".$args{"dorder"}." in decreasing order</b><br>";
  } elsif(exists($args{"order"})) {
    $order="order by `".$args{"order"}."`";
    $grouped_order=$args{"order"};
#    $result.="<b> ordering by ".$args{"order"}."</b><br>";
  };
  
  $grouped_order=~s/\"//ig;
  if (!exists($args{"limit"})) {
    $args{"limit"}=0;
  }

  #If where clauses were provided, apply them only if the original query did not have where clauses.
#  if (!($query=~/where/i)) {
    foreach my $key (keys(%args)) {
      if ($key =~ /^where_(.*)/) {
	my $where_var=$1;
	my $url=$co->self_url;
	$url=~s/(\&|\?|\;)where_[^=]+=[^\&\;]*/$1/g;

	#If key is in quotes, we match it exactly otherwise if there are no % we add those automatically
	if($args{$key}=~/^\'(.+)\'$/) {
	  $args{$key} = $1;
	} elsif (!($args{$key}=~/\%/)) {
	  $args{$key} = "%".$args{"$key"}."%";
	}

	#We must use having here because we are not allowed to use aliases in where clauses (Why not??? ANSI SQL forbids it) and thats all we have.
	$having_clause = " having `$where_var` like \"".$args{$key}."\" ";
	$result.="(<a href=$url>Clear</a>) <b> Filtering on lines having `$where_var` like \"".$args{$key}."\" </b><br>";
      }
    }
#  };
  
# $result.="My query is $query $group_clause $having_clause ";

  #Run the actual query tacking the order/limit at the end
  my $sth = $dbh->prepare($query." $group_clause $having_clause $order limit ".$args{"limit"}.",$page_size");
  my $rv=$sth->execute();

  #Have we failed this query
  if (!$rv) {
    throw($result."<h1> Error </h1> Got a database error for the Query:<br>\n<pre> $query $group_clause $having_clause $order </pre>");
    #Or maybe nothing was returned?
  } elsif ($rv == 0 && !exists($args{"simple"})) {
    $result.="<h1> No rows returned </h1>\n<!--(Query $query $group_clause $having_clause $order limit ".$args{"limit"}.",$page_size)--> ";
    return($result);
  } elsif ($rv==0) {
    return($result);
  } ;

  #Do the headers of the table as well as the navigation page up/page down links
  my @headers = @{$sth->{NAME}};
  my $url=$co->self_url;
  $url=~s/(\&|\;)limit=[^\&\;]*//;
  $url=~s/(\&|\;)dorder=[^\&\;]*//;
  $url=~s/(\&|\;)order=[^\&\;]*//;

  $result.="<table border=0><tr>";

  for(my $i=0; $i<=$#headers; $i++) {

    #We keep tab on the maximum width of the column to make the text boxes look good
    $width[$i] = length($headers[$i]);

    if($grouped_order && $grouped_order eq $headers[$i]) {
      $grouped_order=$i;
    };
    
    #When doing a group by we must only display the count column and the groupped by column - otherwise it might get confusing for the user.
    if (length($group_clause)>1) {
      if ($headers[$i] ne 'Group count' && $headers[$i] ne $args{"group_by"}) {
	next;
      } else {
	#remember the field we group by
	$group_by_colnum=$i;
      }
    };

    #Simple tables do not have sortable columns.
    if(exists($args{"simple"})) {
      $result.="<th>$headers[$i]</th>\n";
    } elsif(exists($args{"dorder"}) && $args{"dorder"} eq $headers[$i]) {
      $result.="<th bgcolor=\"".$CONF{"HILIGHT"}."\"><abbr title=\"Sort by increasing ".$headers[$i]."\"><a href=$url&order=".uri_escape($headers[$i]).">$headers[$i]</a></abbr></th>\n";
    } elsif(exists($args{"order"}) && $args{"order"} eq $headers[$i]) {
      $result.="<th bgcolor=\"".$CONF{"HILIGHT"}."\"><abbr title=\"Sort by decreasing ".$headers[$i]."\"><a href=$url&dorder=".uri_escape($headers[$i]).">$headers[$i]</a></abbr></th>\n";
    } else {
      $result.="<th><abbr title=\"Sort by decreasing ".$headers[$i]."\"><a href=$url&dorder=".uri_escape($headers[$i]).">$headers[$i]</a></abbr></th>\n";
    };
  };
  #Add a header for bookmarks:
  if (!exists($args{"simple"})) {
    $result.="<th><abbr title=\"Bookmarks\"><img src=\"/flag/images/bookmark.png\"></abbr></th>";
  };
 
  my $count=0;
  my $group_count=0;
  #write out the actual table rows
  while(my @row= $sth->fetchrow_array ) {
    my $bookmarked=undef;

    if(length($grouped_order) && lc($row[$grouped_order]) ne $old_ordered_value) {
      $old_ordered_value=lc($row[$grouped_order]);
      $group_count++;
    };

    #Detect if this line is bookmarked at all...
    for (my $k=0; $k<=$#bookmarks && !defined($bookmarked); $k++) {
      my @bookmark = @{$bookmarks[$k]};
      #parse the bookmark into a hash of column=>values:
      my $bmhash=new CGI($bookmark[4]);
      
      my $flag=1;

      for (my $j=0; $j<=$#row;$j++) {
	$row[$j] =~ s/\s+$//;
	my $param = $bmhash->param($headers[$j]);
	if ($param) {
	  $param =~ s/\s+$//;
	};
	
	if ($row[$j] ne ' ' && $param ne $row[$j]) {
	  $flag=0;
	  #	  $result.= "'".$bmhash->param($headers[$j])."' -> $headers[$j] \n Did not match '". $row[$j]."'<br>\n";
	  last;
	} else {
	  #	  $result.= $bmhash->param($headers[$j])." -> $headers[$j] Did match '". $row[$j]."'<br>\n";
	};
      };

      if ($flag) {$bookmarked=$k};
    }

    if (defined($bookmarked)) {
      $result.='</tr><tr bgcolor='.$CONF{"BOOKMARKCOLOR"}.'  id="el'.$count.'" onmouseover="this.style.backgroundColor =\''.$CONF{"HILIGHT"}.'\'" onmouseout="this.style.backgroundColor =\''.$CONF{"BOOKMARKCOLOR"}.'\'" >';
    } elsif($group_count % 2) {
      $result.='</tr><tr bgcolor='.$CONF{"BGCOLOR1"}.'  id="el'.$count.'" onmouseover="this.style.backgroundColor =\''.$CONF{"HILIGHT"}.'\'" onmouseout="this.style.backgroundColor =\''.$CONF{"BGCOLOR1"}.'\'" >';
    } else {
      $result.='</tr><tr bgcolor='.$CONF{"BGCOLOR"}.'  id="el'.$count.'" onmouseover="this.style.backgroundColor =\''.$CONF{"HILIGHT"}.'\'" onmouseout="this.style.backgroundColor =\''.$CONF{"BGCOLOR"}.'\'" >';
    };

    for (my $i=0;$i<=$#row;$i++) {
      if (defined($row[$i]) && ((defined($row[$i]) && !$width[$i]) || $width[$i] < length($row[$i]))) {
	$width[$i] =  length($row[$i]);
      };

      #Remember the original row item. This will be used to preserve any hyperlinks after the row data is formatted:
      my $orig_row=$row[$i];

      #If we have a translation defined for this column, we call it now, Otherwise the default translation is to escape the entities, and wrap lines. Note that this means that if you dont want to encode the entities you should pass a translation here!!! also if you use a translation, you need to take care of your own line wrapping:
      if(defined($translation) && $translation->[$i]) {
	no strict "refs";
	my $offset=0;

	#When we have a groupped by clause, all the columns move over by 1 spot.
	if($group_by_colnum) {
	  $offset++;
	};
	$row[$i+$offset]=&{$translation->[$i]}($row[$i+$offset],\%args);
      } elsif($row[$i]) {
	#Split very long lines
	$row[$i]=~s/\s*$//;
	$row[$i]=encode_entities($row[$i]);
	$row[$i]=~s/([^\n\r]{$CONF{"WRAP"}})/$1<img border=0 src=\"\/flag\/images\/next_line.png\"><br>/gs;
      } ;

      #When grouping we only show the groupped field and the count for clarity.
      if ($group_by_colnum && $i != 0 && $group_by_colnum != $i) {
	next;
      } elsif ($group_by_colnum && $i) {
	    my $url=$co->self_url;
	    $url=~s/(\&|\?|\;)group_by=[^\&\;]*/$1/;
	    $url=~s/(\&|\;)limit=[^\&\;]*//;
	    $url=~s/(\&|\;)where_[^=]*=[^\&\;]*//;

	    $link[$i]=$url."&where_".uri_escape($headers[$i])."=";
      } ;

      if (!defined($row[$i])) {
	$row[$i]=" ";
      }

      if ($link[$i]) {
	if (!defined($orig_row)) {
	  $orig_row='';
	};

	#Work out where we are going to....
	$link[$i]=~/report=([^\&\;]*)/;
	$result.="<td nowrap align=left><abbr title=\"".$reports{$1}{"name"}." : ".$reports{$1}{"description"}."\"><a href=\"$link[$i]".uri_escape($orig_row)."#mark\">$row[$i]</a></abbr></td>\n";
      } else {
	$result.="<td nowrap align=left>$row[$i]</td>\n";
      }
    }
    
    #If this row has a bookmark we show the description here, otherwise we allow the user to put a bookmark here...
    my $url=$co->self_url;
    $url=~s/(\&|\?|\;)bookmark=[^\&\;]*/$1/;

    my @bm=();
    #prepare the bookmark string:
    for(my $i=0;$i<=$#row;$i++) {
      push @bm, encode_entities("$headers[$i]=".$row[$i]);
    };
    
    my $bm = join('&',@bm);
    if (defined($bookmarked) && !exists($args{"simple"})) {
      my @bookmark = @{$bookmarks[$bookmarked]};
#      $result.="<td><a href=".$co->self_url."&id=".$bookmark[0]."&reset=1>$bookmark[3]</td>";
      $result.="<td nowrap><abbr title=\"Clear bookmark\"><a href=".$co->url(-relative=>'1')."?report=bookmark&case=".$args{"case"}."&bookmark=".uri_escape($bm)."&url=".uri_escape($url)."&id=".$bookmark[0]."&reset=1&description=$bookmark[3]>$bookmark[3]</a></abbr></td>";
    } elsif (!exists($args{"simple"})) {
      $result.="<td nowrap><abbr title=\"Bookmark this entry\"><a href=".$co->url(-relative=>'1')."?report=bookmark&case=".$args{"case"}."&bookmark=".uri_escape($bm)."&url=".uri_escape($url)."><img border=0 src=/flag/images/bookmark.png></a></abbr></td>";
    };

  };
  $result.="</tr><tr>\n";
   #If we get a simple in the args we dont do this fancy stuff.
  if (!($query=~/group by/i) && !exists($args{"simple"})) {
    $result.= "</tr><tr><td nowrap colspan=20 align=center>Click here to group by Column </td></tr><tr>";

    $url=$co->self_url;
    $url=~s/(\&|\?|\;)group_by=[^\&\;]*/$1/;
    $url=~s/(\&|\;)limit=[^\&\;]*//;
    $url=~s/(\?|\&|\;)where_[^=]*=[^\&\;]*/$1/;

    #Group by clauses:
    foreach my $header (@headers) {
      if ($header eq 'Group count') {
	$result.="<td nowrap> </td>";
      } else {
	$result.="<td nowrap align=center><a href=$url&group_by=".uri_escape($header).">$header</a></td>";
      }
    };
  };

  if (!$group_by_colnum && !exists($args{"simple"})) {
    #Provide users with ways to filter by where clauses.
    $query=~s/([\'\"])/\\$1/g;
    $result.= "\n<tr><td nowrap colspan=20><table width=\"100%\"><tr><td nowrap><a onclick=\"alert('Query used is ".encode_entities("$query $group_clause $having_clause $order limit ".$args{"limit"}.",$page_size")."')\"><img src=/flag/images/Bookmarks-prelight.png></a></td><td nowrap colspan=20 align=center>Enter a term to filter on field (% is wildcard)</td></tr></table> </td></tr><tr>";
    for (my $i=0; $i<=$#headers; $i++) {
      $result.="<td nowrap><form method=get action=\"".$co->url(-relative=>1)."\"><input size=\"".($width[$i]<$CONF{"WRAP"}?$width[$i]:$CONF{"WRAP"})."\" name=\"where_$headers[$i]\">";
      foreach my $arg (keys(%args)) {
	if (!($arg =~/^where_/ || $arg=~/^limit/)) {
	  $result.="<input type=hidden name=\"".$arg."\" value=\"".$args{$arg}."\">";
	};
      }
      #    $result.="<br><input type=submit value='submit'></form></td>\n";
      $result.="</form></td>\n";
    };
    $result.="<tr>";
    foreach my $header (@headers) {
      if ($header eq 'Group count') {
	$result.="<td nowrap> </td>";
      } else {
	$result.="<td nowrap align=center>$header</td>";
      }
    };
  };

  $result.="</tr></table>";
  return $result;
};

#Function checks that the analysis part of named function was run, or else it gets run now.
sub check_prereq(%) {
  my $reference= shift;
  my %args =%{$reference} ;

  if(!($dbh->selectrow_array("select * from $case.meta where property=\"report_executed\" and value='".canonicalise(\%args)."'"))) {
    &{$reports{$args{'report'}}{"analyse"}}(\%args);
    $dbh->do("insert into $case.meta set property=\"report_executed\",value=\"".canonicalise(\%args)."\"");
  };
};


#********************************************************************
# Load log form : Draws the form for loading the logs in. 
#Expects a hash reference with all the arguments in it
# *******************************************************************
sub load_log_form {
  my $reference= shift;
  my %args=%{$reference};
  my $result="";
  
  # setup some defaults for vars
  foreach $result ("case","newdb","table","datafile","delmethod","preset","log_type") {
    if(!exists($args{$result})) {
      $args{$result} = "";
    };
  };
  
  #If no preset is given but delmethod is 
  if ($args{"delmethod"} eq "preset") {
    %args=%{load_preset($reference)};
  };

  # Step 1 - Allow user to specify database and name the new table to be created.
  $result .= '<table border="0"> <tr><td><strong>Step 1:</strong></td></tr>  <tr> <td>Select Database:</td>     <td>';
  $result .= selector("case", $args{"case"}, "select value,value from $flag_db.meta where property=\"flag_db\"");
  $result .= "</td></tr><tr> <td>Enter Name for table:</td> <td><input name=\"table\" type=\"textbox\" value=\"".$args{"table"} ."\" /></td><td halign=\"left\"></td></tr> <tr><td colspan=\"3\"><hr /></td></tr>";
 
  
  # Step 2: Enter filename to load
  $result .= "<tr><td><strong>Step 2:</strong></td></tr>";
  $result .= "<tr><td>Enter name of file to load:<br> (Files are taken from <br>".$CONF{"UPLOADDIR"}.")</td><td>".select_file('datafile',\%args);
#  $result .= scrolling_list(-name=>'datafile', -values=>\@filelist, -default=>[$args{"datafile"}],-size=>7, -onChange=>"this.form.submit();", -multiple=>"false");
  $result .= "</td><td halign=\"left\"> <input type=\"submit\" value=\"Update\"/></td></tr><tr><td colspan=\"3\"><hr/></td></tr>";

# Step 4 - Choose Log Class
  $result .= '<tr><td><strong>Step 3:</strong></td></tr> <tr> <td>Choose Log Class:</td> <td>';
  $result .= selector("log_type",$args{"log_type"},"select name,description from $flag_db.log_types");
  $result .= '</td> <td>  <input type="submit" value="Update" /> </td> </tr> <tr><td colspan="3"><hr /></td></tr>';

  #Step 3 - Specify or select a delimiter
  $result.= "<tr><td><strong>Step 4:</strong></td></tr> <tr> <td>Field Separator Type:</td> <td>".$co->radio_group(-name=>"delmethod", -values=>['simple','advanced','preset'], -default=>$args{"delmethod"},-onclick=>"this.form.submit();")."</td></tr><tr>";

  # Simple (char) Field Delimiter
  if($args{"delmethod"} eq "simple") {
    #Ensure that the value of delimiter at present is actually within the list of delimiters provided in the database
    $args{"delimiter"} = @{$dbh->selectcol_arrayref("select delimiter from $flag_db.simple_delimiters where delimiter=?",undef,$args{"delimiter"})}[0];
    
    # If we dont have a valid delimeter we just use space.
    if(!defined($args{"delimiter"}) || $args{"delimiter"} eq "") {
      $args{"delimiter"} = " ";
    }
    
    $result .= '<td>Simple Field Separator:</td><td>';
    $result .= selector("delimiter",$args{"delimiter"}, "select delimiter,description from $flag_db.simple_delimiters");
    $result .= '</td>';
    
  }  # Advanced (regex) Field Delimiter
  elsif($args{"delmethod"} eq "advanced") {
    $result .= '<td>Enter Regular Expression:</td> <td><input name="delimiter" type="textbox" size=50 value="';
    #$result .= uri_escape($args{"delimiter"});
    $result .= $args{"delimiter"} unless $args{"delimiter"} =~ /[\"\']/;
    $result .= '" /></td>';
  }
  # Preset Log File Type
  elsif($args{"delmethod"} eq "preset") {
  
    $result .= '<td>Select Preset File Filter:</td><td>';
    $result .= selector("preset",$args{"preset"}, "select name,name from $flag_db.log_preset where log_type='".$args{"log_type"}."'");
    $result .= '</td>';

    
  } else {
    # Confirm a field separator type has been selected
    $result .=  "<tr><td><strong> Please select a separator type </strong></td></tr></table>";
    return $result;
  };

  $result .= '<td><input type="submit" name=override value="Update" /></td></tr><tr><td colspan=\"3\"><hr /></td></tr><tr><td colspan=3>First few lines from the file</td></tr>';
  
  
  # Step 5 - Load first 3 lines from text file
  if ($args{"table"} eq "") {
    $result .= "<strong> Please enter a name for the database table. </strong><br><br>";
    return $result;
  } elsif ($args{"datafile"} eq " ") {
    $result .= "<strong>Please enter the name of a log file to load.</strong><br><br>" ;
    return $result;
  };

  my @lines=();
  my $inputfile = $CONF{"UPLOADDIR"}.$args{"datafile"};
  open (FILEHANDLE, "gzip -dcf <$inputfile|") or error("Cannot open file: $inputfile");
  while (defined(my $info = <FILEHANDLE>) && $#lines < 2) {
    push @lines, $info;
  };
  close(FILEHANDLE);

  #print the unparsed lines:
  $result.='Raw text from file: <table border="0" bgcolor="lightgrey"><tr><td>'.join('</td></tr><tr><td>',@lines)."</td></tr></table>\n";

  $result.='<table><tr><td colspan=3><br><b>Step 5: </b></td></tr><tr><td valign=center>Prefilter to use on the data:</td><td width=50></td><td> '.$co->scrolling_list(-name=>'prefilter',-values=>[sort(keys(%prefilters))],-size=>4,-multiple=>'true')."</td><td><input type=submit value='Update'></td></tr></table>";

  my @prefilters=$co->param('prefilter');
  @lines = ();
  open (FILEHANDLE, "gzip -dcf <$inputfile|") or error("Cannot open file: $inputfile");
  while(defined(my $line = <FILEHANDLE>) && $#lines < 2) {
    foreach my $prefilter (@prefilters) {
      if (exists($prefilters{$prefilter})) {
        $line = &{$prefilters{$prefilter}}($line);
      }
    }
    # only add if prefilter doesnt return 0
    if($line ne 0) {
      push(@lines, $line);
    }
  };
  close(FILEHANDLE);

  #print the parsed lines:
  $result.='<br>Processed text from file (After applying pre-filters): <table border="0" bgcolor="lightgrey"><tr><td>'.join('</td></tr><tr><td>',@lines)."</td></tr></table><tr><td colspan=\"3\"><hr /></td></tr>";

  my @line;
  if($args{"delimiter"} eq "") {
    $args{"delimiter"}=" ";
  };
  
  $result.='<strong>Step 6:</strong><br>';
  $result.="Assign name to each field:<br><table border=\"1\" bgcolor=\"lightgrey\">\n";
  
  if($args{"delmethod"} eq "simple") {
    foreach (@lines) {
      @line = split /$args{"delimiter"}/, $_;
      $result .= "<tr><td>".join("</td><td>",@line)."</td></tr><tr>";
    }
  } else {
    my $unmatched = 0;
    foreach (@lines) {
      /$args{"delimiter"}/ || do {$unmatched++; next;};
      my $j=1;
      @line=();
      no strict "refs";
      while(defined($ {$j})) {
	push @line, "${$j++}";
      };
      $result .= "<tr><td>".join("</td><td>",@line)."</td></tr><tr>";
    };
    $result .= "<strong>Lines not matched to regular expression</strong>" if $unmatched;
  }
  
  # print field headings select box as taken from database
  for(my $j=0; $j<$#line+1; $j++) {
    $result.="<td>".selector("field$j",$args{"field$j"},"select name,description from $flag_db.".$args{"log_type"}."_fields")."</td>";	
  };

  $result.="</td></tr></table><p>";

  return $result;
};

# load preset
sub load_preset {
  my $reference= shift;
  my %args =%{$reference} ;
  
  my $sth;
  
  if (exists($args{"preset"}) && length($args{"preset"})>1) {
    $sth=$dbh->prepare("select regex,log_type,field_map,prefilter from $flag_db.log_preset where name='".$args{"preset"}."'");
  } else {
    #If we dont have a preset specified we select the first one there of the specified type
    $sth=$dbh->prepare("select regex,log_type,field_map,prefilter from $flag_db.log_preset where  log_type='".$args{"log_type"}."' limit 1");
  }
  $sth->execute();
  
  my @row=$sth->fetchrow_array;
  
  $args{"delimiter"} = $row[0];
  $args{"log_type"} = $row[1];
  if (!exists($args{'prefilter'}) || exists($args{'override'})) {
    $co->param('prefilter',split(',',$row[3]));
  }
  my @field_map=split ",",$row[2];
  
  for(my $i=0; $i<= $#field_map;$i++) {
    $args{"field$i"}=$field_map[$i];
  };

  return \%args;
}

sub draw_graph($$$) {
  my $type=shift;
  my $query=shift;
  my $title=shift;
  my @x;
  my @y;
  my $my_graph;
  my $x_coord = 400 ;
  my $y_coord = 300;

  #Create a temporary table with the results so we can scale the graph properly.
  my $sth = $dbh->prepare("create temporary table temp_graph $query");
  $sth->execute();

  $sth=$dbh->prepare("select * from temp_graph");
  $sth->execute();
  my @headers=@{$sth->{NAME}};

  my @row=$dbh->selectrow_array("select max($headers[0]),min($headers[0]),max($headers[1]),min($headers[1]) from temp_graph");

  if($type eq "pie") {
    use GD::Graph::pie;
    $my_graph = new GD::Graph::pie($x_coord ,$y_coord );
    $my_graph->set(
		   #        title                   => $title,
		   label                   => '',
		   axislabelclr    => 'black',
		   suppress_angle => '10',
		   '3d' => '1',
		   transparent     => 0,
		  );
    $my_graph->set(dclrs =>  [qw(red blue green purple orange brown)]);
    $my_graph->set_title_font("ariel.ttf",18);
    $my_graph->set_label_font('20thcent.ttf', 12);
    $my_graph->set_value_font('cetus.ttf', 10);

  } elsif($type eq "bar") {
    use GD::Graph::bars;
    $my_graph = new GD::Graph::bars(400,300);

    $my_graph->set(
		   y_tick_number       => "10",
		   x_tick_number =>"10",
		   x_label_skip=>1,
		   x_min_value=>$row[1],
		   x_max_value=>$row[0],
		   y_max_value=>$row[2],
		   y_min_value=>$row[3],
		   box_axis=>1,
		   axis_space=>10,
		   overwrite           => 1,
		   transparent         => 0,
		  );
    
    
  } elsif($type eq "lines") {
    use GD::Graph::lines;
    $my_graph = new GD::Graph::lines(600,400);

    $my_graph->set(
		   y_tick_number       => "10",
		   x_tick_number =>"10",
		   x_label_skip=>1,
		   x_min_value=>$row[1],
		   x_max_value=>$row[0],
		   y_max_value=>$row[2],
		   y_min_value=>$row[3],
		   box_axis=>1,
		   axis_space=>10,
		   overwrite           => 1,
		   transparent         => 0,
		  );
    
    
  } else{return "error"};
  
  while(my @r= $sth->fetchrow_array ) {
    push (@x,$r[0]);
    push (@y,$r[1]);
  };

  my @data=(\@x,\@y);

  $my_graph->plot(\@data);
  my $result= header(-type => "image/png", -expires => "-1d").$my_graph->gd->png();
};

sub add_pie_legend($) {
  my $q = new CGI;

  my $query = shift; # Get the querry that we will use to get the legend elements
  my @row;        # Put the result of the querry in here
  my @x;          # Then, once we've got that array of @rows, we'll just plonk 
                  # each string into the @x array.

  my $sth = $dbh->prepare($query);
  $sth->execute();

  while(my @row= $sth->fetchrow_array ) {
    push(@x,$row[0]);
  };

  use constant X_HORIZONTAL => 0;
  use constant Y_VERTICAL => 20;

  my $im = new GD::Image(300,300);
  my $count = 0;
  my $x = 50;
  my $y = 50;

  my $white = $im->colorAllocate(255,255,255);
  my $black = $im->colorAllocate(0,0,0);
  my $red = $im->colorAllocate(255,0,0);
  my $blue = $im->colorAllocate(0,0,255);
  my $green = $im->colorAllocate(50,200,0);
  my $purple = $im->colorAllocate(200,0,255);
  my $orange = $im->colorAllocate(255,200,0);
  my $brown = $im->colorAllocate(154,102,051);


  $im->transparent($white);
  $im->interlaced('true');

  my @colour;
  $colour[0] = $red;
  $colour[1] = $blue;
  $colour[2] = $green;
  $colour[3] = $purple;
  $colour[4] = $orange;
  $colour[5] = $brown;


  my $buffer_size = @colour;
  $count = 0;

  foreach (@x){
    $im->string(gdLargeFont, $x += X_HORIZONTAL, $y += Y_VERTICAL, $_, $black);
    $im->filledRectangle(10,$y,25,$y+10, $colour[$count]);
    $count++;
    if (($count % $buffer_size) == 0) {
      $count = 0;
    }
  }

  print $q->header(-type => "image/png", -expires => "-1d");
  binmode STDOUT;
  print $im->png;
  exit;
};

#Return a new handler to the database...
sub connect_to_db($) {
  my $case=shift;
  my $dbh=DBI->connect("DBI:mysql:database=$case",$CONF{"USERNAME"},$CONF{"PASSWORD"},{ RaiseError => 0, AutoCommit => 1 });
  $dbh->{"InactiveDestroy"}=1;
  
  if (!$dbh) {
    throw("<h1> Error - Unable to open database $case.</h1> An error was encountered - Did you remember to create the database first?");
  }

#Change this to 1 to increase debugging
  $dbh->trace($debug_analysis);

  return $dbh;
};

sub show_case_form($) {
    my $reference= shift;
    my %args =%{$reference};

    my $result ="<br>Select database to operate on: ".selector("case",$args{"case"},"select $flag_db.meta.value, $flag_db.meta.value from $flag_db.meta where property='flag_db'",$dbh);
    return $result;
};

sub safe_shell($){
	#We must ignore the sigchild whilest we are executing the shell to make sure that the reaper subroutine does not catch this...
	my $TEMP = $SIG{CHLD};

	local $SIG{CHLD} = 'IGNORE';
	my $shell = shift;

	my $result=system("$shell");
	local $SIG{CHLD} = $TEMP; 
	return $result;
};

#This function returns the canonical form of the current requst. The canonical form is defined as the sorted urlified key=value pairs of the parameters defined in the $reports{name}{"parameters"} hash. This is used to uniquely identify the request in order to manage the caching.
sub canonicalise($) {
  my $reference= shift;
  if(!$reference) {
    return("");
  };
  my %args =%{$reference} ;

  #Spit the dummy if there is no report
  my $report=$args{"report"} || return ("Error: cant find %args");
  my $result="report=$report";
  foreach my $i (sort(keys(%{$reports{$report}{"parameters"}}))) {
    #If $i represents a RE, we use it to grab all the matching args:
    if ($i=~/^\/(.+)\//) {
      my $re=$1;
      foreach my $key (sort(keys(%args))) {
	if ($key =~ /$re/) {
	  $result.="&".encode_entities("$key=".$args{$key});
	};
      };
      next;
    };

    #We may need to account for arrays here
    foreach  my $value (split "\0", $args{$i}) {
      $result.="&".encode_entities("$i=".$value);
    };
  };
  return $result;
};

sub reset_report($$) {
  my $my_name = shift;
  my $reference = shift;
  my %parameters=%{$reference};
  
  my $result= "Resetting report $my_name\n";
  $dbh->do("delete from meta where property=\"report_executed\" and value=\"".canonicalise(\%parameters)."\"");
  $result.= &{$reports{$my_name}{"reset"}}(\%parameters);
  return $result;
};

#Loads defaults from configure.pl Defaults will be loaded into the %CONF hash.
sub load_defaults() {
  open(FILEHANDLE,"<configure.pl") or throw("<h1> Error - No configuration File can be opened, please create configure.pl in this directory</h1>");
  
  while (<FILEHANDLE>) {
    if (/^([^\#=]*)=(.*$)/) {
      $CONF{uc($1)}=$2;
    };
  };
    close(FILEHANDLE);

    #Now check that we have a set of mandatory entries:
    foreach my $i ("FLAG_BIN","DOTDIR","PASSWORD","USERNAME","UPLOADDIR") {
      if (!$CONF{$i}) {
	throw("Error, mandatory configuration $i is missing from configure.pl");
      }
    }
    
    #Some built in defaults:
    $CONF{"BGCOLOR"}="#FFFFFF" unless exists($CONF{"BGCOLOR"});
    $CONF{"BGCOLOR1"}="#e0e0e0" unless exists($CONF{"BGCOLOR1"});
    $CONF{"BOOKMARKCOLOR"}="#ffe5bf" unless exists($CONF{"BOOKMARKCOLOR"});
    $CONF{"HILIGHT"}="#CCFFCC" unless exists($CONF{"HILIGHT"});
    $CONF{"FLAGDB"}="flag" unless exists($CONF{"FLAGDB"});
    #Number of rows in each table
    $CONF{"PAGESIZE"}="50" unless exists($CONF{"PAGESIZE"});
    #Number of columns to wrap at.
    $CONF{"WRAP"}="80" unless exists($CONF{"WRAP"});
    #Path to flag images relative to the flag_server
    $CONF{"IMAGE_PATH"}="images" unless exists($CONF{"IMAGE_PATH"});
    #This directory must be trusted (i.e. do not allow untrusted users write access here)...
    $CONF{"TMPDIR"}="/tmp/" unless exists($CONF{"TMPDIR"});
    $CONF{"RESULTDIR"}="results" unless exists($CONF{"RESULTDIR"});
    $CONF{"TETHEREAL"}=$CONF{"FLAG_BIN"}.'/tethereal' unless exists($CONF{"TETHEREAL"});
    $CONF{"MAXTREESIZE"}=15 unless exists($CONF{"MAXTREESIZE"});
};

  #Throws a fatal error
  sub throw($) {
    my $arg=shift;
    if($dbh) {
    print $co->header.$flag_header.$arg."FLAG Database error: <br><pre>".$dbh->errstr."</pre>";
  } else {
    print $co->header.$flag_header.$arg."FLAG general error. Is mysql running?";
  } ;
    print $flag_footer_plain;
    exit;
  };

#returns a refresh header derived from the flag header variable
sub refresh($$) {
  my $arg=shift;
  my $url=shift;

  if (!$url || length($url)<1) {
    $url=$co->self_url;
  }
  my $my_header =  $flag_header;
  my $replacement="<body";
  substr($my_header, index($my_header,$replacement),length($replacement)) ="<body bgcolor=".$CONF{"BGCOLOR"}."><META HTTP-EQUIV=Refresh Content=\"2; URL=".$url."\"";
  return $my_header.$arg.$flag_footer_plain;
}


#This function is used to customise the navigation headers. takes: var is the navigation variable that should be scrapped from the URL. previous is a url to send the back button to, next is a url to send the next button to.
sub print_headers($$$) {
   my $reference=shift;
   my @vars=@{$reference};
   my $previous=shift;
   my $next=shift;
   my $page_count = shift;

   my $result=$co->header.$flag_head1.$flag_nav;
   
   my $url=$co->self_url;
   #remove the variable from the URL
   foreach my $var (@vars) {
     $url=~s/$var.*?=[^\&\;]*//ig;
   };

   $url =~ s/\&+/\&/g;

   $flag_nav2="$page_count <td valign=\"bottom\"> <a href=\"$url&$vars[0]=$previous\"><img src=\"/flag/images/back.png\"  border=\"0\"></a></td><td valign=\"bottom\"><a href=\"$url&$vars[0]=$next\"><img src=\"/flag/images/forward.png\" border=\"0\"></td>	  <td valign=\"bottom\"><a href=".$co->self_url."&reset=1><img src=\"/flag/images/reset.png\" border=\"0\"></a></td>	  
	  <td valign=\"bottom\"><a href=".$co->self_url."&stop=1><img src=\"/flag/images/stop.png\" border=\"0\"></a></td></tr></table>";

   $result .=  $flag_nav2.$flag_head2;
   $flag_footer="<p><center>".$flag_nav.$flag_nav2."</center></body></html>";
   return $result;
 };

#Sets the meta table entries to advertise the current progress of the analysis.
sub set_progress($) {
  my $message=shift;
  my $pid=POSIX::getpid();
  
#  if (($dbh->do("update meta set value=\"$message\" where property=\"progress $pid\"")) eq '0E0') {
    $dbh->do("insert into meta set value=\"$message\",property=\"progress $pid\"");
#  }
};
		 
sub update_progress($$$) {
  my $message=shift;
  my $dbh=shift;
  my $like = shift;
  my $pid=POSIX::getpid();
  
  $dbh->do("update meta set value=\"$message\" where property=\"progress $pid\" and value like \"\%$like\%\"");
  return 1
};

#Clears all progress indications from the meta table;
sub clear_progress() {
  my $pid=POSIX::getpid();

  $dbh->do("delete from meta where property=\"progress $pid\"");
}

#Retrieves the current progress message
sub get_progress($) {
  my $reference=shift;
  my %args=%{$reference};

  #We need to find the pid of the required process:
  my $pid = $dbh->selectrow_array("select property from meta where property like \"pending%\" and value='".canonicalise(\%args)."'");
  $pid=~s/\D//ig;

  #my @message= $dbh->selectrow_array("select value from meta where property=\"progress $pid\"");
#  return $message[0];
  #Set the simple attribute on the args so the display_table draws a simple table.
  $args{"simple"} = "yes";
  return display_table("select `time` as `Timestamp`,value as \"Progress Indication\" from meta where property=\"progress $pid\" order by `Timestamp`",[],\%args,[]);
};

#Maintains a log with timing of queries.
sub do_query($){
  my $query=shift;
  my $time;
  my $pid=POSIX::getpid();

  foreach my $i (split (";",$query)) {
    $time=time;
    if($i) {
      $dbh->prepare($i)->execute() || cluck "Execute failed";
      $dbh->do("insert into meta set property=\"log $pid ".(time-$time)."\",value=?",undef,$i);
    };
  }
}

#This returns a HTML rendered selection box of the files in the UPLOAD directory. Accepts the name of the argument to return.
sub select_file($$) {
  my $result="";
  my $datafile= shift;
  my $reference=shift; 
  my %args=%{$reference};

  opendir(DIRECTORY, $CONF{"UPLOADDIR"}) or throw("Can't open upload directory, ".$CONF{"UPLOADDIR"});
  my @filelist = ();
  foreach my $i (sort(readdir( DIRECTORY ))) {
    if ($i ne '.' && $i ne '..') {
      push @filelist, $i;
    }
  };
  closedir DIRECTORY;
  

  $result .= scrolling_list(-name=>$datafile, -values=>\@filelist, -size=>7, -multiple=>"false",-default=>[$args{$datafile}]);
  return $result;
}

#A translation function to hex encode a binary md5 value
sub md5_hex {
  my $str=shift;
  my $result='';
  for(my $i=0;$i<length($str);$i++) {
    $result.=sprintf("%02x",ord(substr($str,$i,1)));
  };
  return $result;
}

#searches for needle in array haystack
sub in_array() {
  my $needle=shift;
  my $haystack=shift;

  foreach my $i (@{$haystack}) {
    if ($needle eq $i) {
      return 1;
    }
  }
  return 0;
};

#This subroutine prints a form with all variables in %args as hidden parameters. The text within $form is inserted into the form. Parameters names in @exclude are then excluded from the form.
sub edit_parameters_form () {
  my $reference = shift;
  my %args=%{$reference};

  $reference = shift;
  my @exclude = @{$reference};
  my $text=shift;

  my $result="";
  $result.="<form method=get action=\"".$co->url(-relative=>1)."\">$text";
      foreach my $arg (keys(%args)) {
	my $found=0;
	foreach my $var (@exclude) {
	  if ($arg=~/$var/i) {
	    $found =1;
	    last;
	  }
	};
	
	if (!$found) {
	  $result.="<input type=hidden name=\"".$arg."\" value=\"".$args{$arg}."\">\n";
	};
      }
  #    $result.="<br><input type=submit value='submit'></form></td>\n";
  $result.="</form>\n";
  
  return $result;
};

return 1;
