/*
** The Sleuth Kit
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2003 Brian Carrier.  All rights reserved 
**
** TASK
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
**
** renamed to have consistant name (was unrm)
**
** 
** Copyright (c) 1997,1998,1999, International Business Machines
** Corporation and others. All Rights Reserved.
**
*/

/* TCT */
/*++
 * NAME
 *	dls 1
 * SUMMARY
 *	disk data recovery
 * SYNOPSIS
 * .ad
 * .fi
 *	\fBdls\fR [\fB-bevV\fR] [\fB-f \fIfstype\fR]
 *		\fIdevice\fR [\fIstart-stop\fR ...]
 * DESCRIPTION
 *	\fBdls\fR opens the named \fIdevice\fR and copies data blocks.
 *	By default, \fBdls\fR copies unallocated data blocks only.
 *
 * 	Arguments:
 * .IP \fB-b\fR
 *	With file systems that have logical blocks that consist of fragments,
 *	don't insert null-byte padding to preserve logical block alignment
 *	in the output.
 *	This option is a no-op with the LINUX ext2fs file system, where
 *	logical blocks and fragments have the same size.
 * .IP \fB-e\fR
 *	Copy every block. The output should be similar to dd(1).
 * .IP "\fB-f\fI fstype\fR"
 *	Specifies the file system type. The default file system type
 *	is system dependent. With most UNIX systems the default type
 *	is \fBufs\fR (Berkeley fast file system). With Linux the default
 *	type is \fBext2fs\fR (second extended file system).
 * .IP \fB-v\fR
 *	Turn on verbose mode, output to stderr.
 * .IP \fB-V\fR
 *	Turn on verbose mode, output to stdout.
 * .IP \fIdevice\fR
 *	Disk special file, or regular file containing a disk image.
 *	On UNIX systems, raw mode disk access may give better performance
 *	than block mode disk access.  LINUX disk device drivers support
 *	only block mode disk access.
 * .IP "\fIstart-stop\fR ..."
 *	Examine the specified block number or number range. Either the
 *	\fIstart\fR, the \fIstop\fR, or the \fI-stop\fR may be omitted.
 *	If \fB-b\fR is not specified, the start block must be
 *	aligned to a logical block boundary (e.g. a multiple of 8 in
 *	the case of an FFS file system). With the LINUX ext2fs file system,
 *	the start block number must be >= 1.
 * BUGS
 *	\fBdls\fR should support more file system types. Right now, support
 *	is limited to \fBext2fs\fR when built on Linux, and \fBufs\fR when
 *	built on Solaris and BSD systems.
 * LICENSE
 *	This software is distributed under the IBM Public License.
 * AUTHOR(S)
 *	Wietse Venema
 *	IBM T.J. Watson Research
 *	P.O. Box 704
 *	Yorktown Heights, NY 10598, USA
 --*/

#include "fs_tools.h"
#include "error.h"
#include "split_at.h"

FILE   *logfp;

/* atoblock - convert string to block number */

DADDR_T atoblock(const char *str)
{
    char   *cp;
    DADDR_T addr;

    if (*str == 0)
	return (0);
    addr = STRTOUL(str, &cp, 0);
    if (*cp || cp == str)
	error("bad block number: %s", str);
    return (addr);
}

/* usage - explain and terminate */

static void usage()
{
    printf("usage: %s [-belvV] [-f fstype] device [block... ]\n", progname);
	printf("\t-b: no block padding\n");
	printf("\t-e: every block\n");
	printf("\t-l: print details in time machine list format\n");
	printf("\t-s: print slack space of FAT and NTFS images (other flags are ignored\n");
	printf("\t-v: verbose to stderr\n");
	printf("\t-V: print version\n");
    printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
	fs_print_types();
	exit(1);
}

static void
print_list_head(FS_INFO *fs, char *image)
{
    char    hostnamebuf[BUFSIZ];
    unsigned long now;
	char unit[32];
  
    if (gethostname(hostnamebuf, sizeof(hostnamebuf) - 1) < 0)
        error("gethostname: %m");
    hostnamebuf[sizeof(hostnamebuf) - 1] = 0;
    now = time((time_t *) 0);

    switch (fs->ftype & FSMASK) {
      case EXT2FS_TYPE:
      case FFS_TYPE:
		strncpy(unit, "fragment", 32);
        break;
      case FATFS_TYPE:
		strncpy(unit, "sector", 32);
        break;
      case NTFS_TYPE:
		strncpy(unit, "cluster", 32);
        break;
      default:
        printf("Unsupported File System\n");
        exit(1);
    }
	
    /*
     * Identify table type and table origin.
     */  
    printf("class|host|image|start_time|unit\n");
    printf("dls|%s|%s|%lu|%s\n", hostnamebuf, image, now, unit);

	printf("addr|alloc\n");
	return;
}

static u_int8_t
print_list(FS_INFO *fs, DADDR_T addr, char *buf, int flags, char *ptr)
{
	printf("%lu|%s\n", (ULONG)addr, (flags & FS_FLAG_ALLOC)?"a":"f");
	return WALK_CONT;
}


/* print_block - write data block to stdout */

static u_int8_t
print_block(FS_INFO *fs, DADDR_T addr, char *buf, int flags, char *ptr)
{
    if (verbose)
		fprintf(logfp, "write block %lu\n", (ULONG) addr);
    if (fwrite(buf, fs->block_size, 1, stdout) != 1)
		error("write stdout: %m");
	
	return WALK_CONT;
}



/* SLACK SPACE */
static OFF_T flen;

static u_int8_t
slack_file_act (FS_INFO *fs, DADDR_T addr, char *buf, int size,
  int flags, char *ptr)
{
	if (flen <= 0) 
		fwrite(buf, size, 1, stdout);	
	else
		flen -= size;

	return WALK_CONT;
}

static u_int8_t
slack_inode_act(FS_INFO *fs, INUM_T inum, FS_INODE *fs_inode, int flags,
  char *ptr)
{

	if (verbose)
		fprintf (logfp,
		  "slack_inode_act: Processing meta data: %lu\n",
		  (ULONG)inum);

	flen = fs_inode->size;
	if ((fs->ftype & FSMASK) == FATFS_TYPE) {
		fs->file_walk(fs, fs_inode, 0, 0, 
		  (FS_FLAG_CONT | FS_FLAG_ALLOC | FS_FLAG_SLACK),
		  slack_file_act, ptr);
	}
	else {
		FS_DATA *fs_data = fs_inode->attr;

		/* For NTFS we go through each non-resident attribute */
		while ((fs_data) && (fs_data->flags & FS_DATA_INUSE)) {

			if (fs_data->flags & FS_DATA_NONRES) {
				fs->file_walk(fs, fs_inode, fs_data->type, fs_data->id, 
				  (FS_FLAG_CONT | FS_FLAG_ALLOC | FS_FLAG_SLACK),
				  slack_file_act, ptr);
			}

			fs_data = fs_data->next;
		}
	}

	return WALK_CONT;
}






/* main - open file system, list block info */

int     main(int argc, char **argv)
{
    FS_INFO *fs;
    char   *start;
    char   *last;
    DADDR_T bstart;
    DADDR_T blast;
    int     ch;
    int     flags = FS_FLAG_UNALLOC | FS_FLAG_ALIGN | FS_FLAG_META; 
    char   *fstype = DEF_FSTYPE;
	char	list = 0, slack = 0;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "bef:lsvV")) > 0) {
	switch (ch) {
	default:
	    usage();
	case 'b':
	    flags &= ~FS_FLAG_ALIGN;
	    break;
	case 'e':
	    flags |= FS_FLAG_ALLOC;
	    break;
	case 'f':
	    fstype = optarg;
	    break;
	case 'l':
		list = 1;
	    break;
	case 's':
		slack = 1;
		break;
	case 'v':
	    verbose++;
	    logfp = stderr;
	    break;
	case 'V':
		print_version();
		exit(0);
	}
    }

    if (optind >= argc)
	usage();

    /*
     * Open the file system.
     */
    fs = fs_open(argv[optind++], fstype);


	if (slack) {
		if (((fs->ftype & FSMASK) != NTFS_TYPE) &&
		  ((fs->ftype & FSMASK) != FATFS_TYPE)) {
			printf ("Slack option only allowed with FAT and NTFS file systems\n");
			exit (1);
		}

		if ((list) || (optind < argc)) {
			fprintf(stderr, "Other options igroned with the slack space flag, try again\n");
			exit (1);
		}
	
		fs->inode_walk(fs, fs->first_inum, fs->last_inum, 
		  (FS_FLAG_ALLOC | FS_FLAG_USED | FS_FLAG_LINK | FS_FLAG_NOABORT), 
		  slack_inode_act, (char *)0);
	
		exit(0);
	}

	if (list)
		print_list_head(fs, argv[optind-1]);

    /*
     * Output the named data blocks, subject to the specified restrictions.
     */
    if (optind < argc) {
	while ((start = argv[optind]) != 0) {
	    last = split_at(start, '-');
	    bstart = (*start ? atoblock(start) : fs->start_block);
		if (bstart < fs->start_block)
			bstart = fs->start_block;

	    blast = (!last ? bstart : *last ? atoblock(last) : fs->last_block);
		if (blast > fs->last_block)
			blast = fs->last_block;

		if (list)
	    fs->block_walk(fs, bstart, blast, flags, print_list, (char *) fs);
		else
	    fs->block_walk(fs, bstart, blast, flags, print_block, (char *) fs);
	    optind++;
	}
    }

    /*
     * Output all blocks, subject to the specified restrictions.
     */
    else {
		if (list)
	    	fs->block_walk(fs, fs->start_block, fs->last_block, 
			   flags, print_list, (char *)fs);
		else
			fs->block_walk(fs, fs->start_block, fs->last_block,
		       flags, print_block, (char *) fs);
    }
    fs->close(fs);
    exit(0);
}
