/*++
* NAME
*	fs_io 3
* SUMMARY
*	file system I/O routines
* SYNOPSIS
*	#include "fstools.h"
*
*	void	fs_read_block(fs, buf, len, addr, comment)
*	FS_INFO *fs;
*	FS_BUF	*buf;
*	int	len;
*	DADDR_T	addr;
*	const char *comment;
*
*	void	fs_read_random(fs, buf, len, offs, comment)
*	FS_INFO *fs;
*	char	*buf;
*	int	len;
*	OFF_T	offs;
*	const char *comment;
* DESCRIPTION
*	fs_read_block() reads a block of data from the named file system.
*
*	fs_read_random() reads from an arbitrary position.
*
*	Arguments:
* .IP fs
*	File system handle.
* .IP buf
*	Result buffer pointer.
* .IP len
*	Amount of bytes to be read.
* .IP addr
*	Block number.
* .IP fd
*	Open device file.
* .IP offs
*	Byte offset.
* .IP comment
*	Text logged with verbose logging.
* LICENSE
*	This software is distributed under the IBM Public License.
* AUTHOR(S)
*	Wietse Venema
*	IBM T.J. Watson Research
*	P.O. Box 704
*	Yorktown Heights, NY 10598, USA
--*/

#include "fs_tools.h"
#include "error.h"
#include "fs_types.h"

/* fs_read_block - read a block from a file system */

void    fs_read_block(FS_INFO *fs, FS_BUF *buf, int len, DADDR_T addr,
		              const char *comment)
{
    char   *myname = "fs_read_block";
    OFF_T   offs;

    /*
     * Sanity checks.
     */

	if (len % fs->dev_bsize)
		error("%s: Block Read Request with length (%d) not a multiple of %d (%s)", 
		  myname, len, fs->dev_bsize, comment);

    if (len > buf->size)
		error("%s: Buffer length is too short for read (%d > %d) (%s)", 
		  myname, len, buf->size, comment);

	if (addr > fs->last_block)
		error("%s: Read Error (address is too large for image %lu) (%s)", 
		  myname, (ULONG)addr, comment);

    /*
     * Bummer: with the 32-bit file model, Solaris 7 read() won't go past 2GB
     * seek offsets, so we must use pread() instead of read(). When printing
     * disk offsets, convert to double in case they have huge file offset
     * types and the number does not fit in an unsigned long.
     */
    buf->addr = addr;
    offs = (OFF_T) addr *fs->block_size;

    if (verbose)
		fprintf(logfp, "%s: read block %lu offs %llu len %d (%s)\n",
		  myname, (ULONG) addr, (ULLONG) offs, len, comment);

#ifdef USE_PREAD
    if (pread(fs->fd, buf->data, len, offs) != len)
		error("read block error (%u@%.0f): %m (%s)", len, (double) offs,
		  comment);
#else
    if (fs->seek_pos != offs) {
		if (LSEEK(fs->fd, offs, SEEK_SET) != offs)
			error("read block lseek error (%llu): %m (%s)", 
			  (ULLONG) offs, comment);
	}
    if (read(fs->fd, buf->data, len) != len)
		error("read block read error (%d@%llu): %m (%s)", 
		  len, (ULLONG) offs, comment);
#endif
    fs->seek_pos = offs + len;
    buf->used = len;
}

/* fs_read_random - random-access read */

void    fs_read_random(FS_INFO *fs, char *buf, int len, OFF_T offs,
		               const char *comment)
{
    char   *myname = "fs_read_random";
    int     count;

    if (verbose)
	fprintf(logfp, "%s: read offs %.0f len %d (%s)\n",
		myname, (double) offs, len, comment);

    if (LSEEK(fs->fd, offs, SEEK_SET) != offs)
		error("read random seek error: offset %llu: %m (%s)", 
		  (ULLONG) offs, comment);

    if ((count = read(fs->fd, buf, len)) != len)
		error("read random read error (%d@%llu): %m (%s)", 
		  len, (ULLONG) offs, comment);

    fs->seek_pos = offs + len;
}
