/*
 * The Sleuth Kit
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003 Brian Carrier.  All rights reserved
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 */

#ifndef _MM_TOOLS_H
#define _MM_TOOLS_H


 /*
  * External interface.
  */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <string.h>

#include "mm_types.h"

 /*
  * Verbose logging.
  */
extern FILE *logfp;

 /*
  * Solaris 2.x. Build for large files when dealing with filesystems > 2GB.
  * With the 32-bit file model, needs pread() to access filesystems > 2GB.
  */
#if defined(SUNOS5)
#define SUPPORTED
#include <sys/sysmacros.h>
#define STRTOUL		strtoul

#define u_int8_t	uint8_t
#define u_int16_t	uint16_t
#define u_int32_t	uint32_t
#define u_int64_t	uint64_t

#endif

 /*
  * SunOS 4.x cannot handle filesystems > 2GB.
  */
#if defined(SUNOS4)
#define SUPPORTED
#define STRTOUL		strtol

extern char *optarg;
extern int optind;

#define u_int8_t	uint8_t
#define u_int16_t	uint16_t
#define u_int32_t	uint32_t
#define u_int64_t	uint64_t

#endif

 /*
  * FreeBSD can handle filesystems > 2GB.
  */
#if defined(FREEBSD2) || defined(FREEBSD3) || defined(FREEBSD4) || defined(FREEBSD5)
#include <sys/uio.h>
#define SUPPORTED
#define STRTOUL		strtoul
#endif /* FREEBSD */

 /*
  * BSD/OS can handle filesystems > 2GB.
  */
#if defined(BSDI2) || defined(BSDI3) || defined(BSDI4)
#define SUPPORTED
#define STRTOUL		strtoul
#endif /* BSDI */

 /*
  * OpenBSD looks like BSD/OS 3.x.
  */
#if defined(OPENBSD2) || defined(OPENBSD3)
#define SUPPORTED
#define STRTOUL		strtoul
#endif

#if defined(DARWIN)
#define SUPPORTED
#define STRTOUL		strtoul
#endif /* DARWIN */


 /*
  * Linux 2.
  */
#if defined(LINUX2) 
#define SUPPORTED
#include <linux/types.h>
#define STRTOUL		strtoul
#endif /* LINUX */

#if defined(CYGWIN)
#define SUPPORTED
#define STRTOUL		strtoul
#endif /* CYGWIN */




 /*
  * Catch-all.
  */
#ifndef SUPPORTED
#error "This operating system is not supported"
#endif


typedef unsigned long ULONG;

/* Flags for the return value of walk actions */
#define WALK_CONT	0x0
#define WALK_STOP	0x1

/* Structures */
typedef struct MM_INFO MM_INFO;
typedef struct MM_PART MM_PART;

/* walk action functions */
typedef u_int8_t (*MM_PART_WALK_FN) (MM_INFO *, int, MM_PART *, int, char *);

/* Walk flags */
// #define MM_FLAG_UNUSED	(1<<0)		/* Fill in the unused space */


/***************************************************************
 * MM_INFO: Allocated when a disk is opened
 */
struct MM_INFO {
	int     fd;				/* open raw device */
	daddr_t	sect_offset;		/* sector offset that partition was
					   started from */
	u_int8_t  mmtype;		/* type of media management */
	char 	*str_type;
	u_int8_t  flags;		/* flags for image, see below */
	MM_PART *part_list;		/* linked list of partitions */

	u_int16_t	first_part;	/* number of first partition */
	u_int16_t	last_part;	/* number of last partition */

	/* media management type specific function pointers */
	void    (*part_walk) (MM_INFO *, u_int16_t, u_int16_t, int, 
	  MM_PART_WALK_FN, char *);
	void    (*open) (char *, unsigned char);
	void    (*close) (MM_INFO *);
};

/* flag for MM_INFO flags */
#define MM_LIT_ENDIAN	0x01
#define MM_BIG_ENDIAN	0x02




/***************************************************************
 * Generic structures  for partitions / slices
 */

struct MM_PART {
    MM_PART *prev;
    MM_PART *next;

    daddr_t start;
    daddr_t len;
    char *desc;
	int8_t table_num;
	int8_t slot_num;

};

extern void mm_part_unused(MM_INFO *);
extern void mm_part_print(MM_INFO *);
extern MM_PART *mm_part_add(MM_INFO *, daddr_t, daddr_t, char *, 
  int8_t, int8_t);
extern void mm_part_free(MM_INFO *);



/**************************************************************8
 * Generic routines.
 */
extern MM_INFO *mm_open(const char *, const char *, daddr_t);
//extern void mm_read_block(FS_INFO *, FS_BUF *, int, daddr_t, const char *);
extern void mm_print_types();
extern void print_version();

 /*
  * Support for DOS Partitions
  */
extern MM_INFO *dos_open(const char *, unsigned char ftype, daddr_t);

extern MM_INFO *mac_open(const char *, unsigned char ftype, daddr_t);

extern MM_INFO *bsd_open(const char *, unsigned char ftype, daddr_t);

extern MM_INFO *sun_open(const char *, unsigned char ftype, daddr_t);
/* 
 * Endian Ordering 
 */
extern u_int8_t guessu16(MM_INFO *, u_int8_t *, u_int16_t);
extern u_int8_t guessu32(MM_INFO *, u_int8_t *, u_int32_t);

#define getu16(mm, x)   \
    ((mm->flags & MM_LIT_ENDIAN) ? \
      (((u_int8_t *)x)[0] + (((u_int8_t *)x)[1] << 8)) :    \
      (((u_int8_t *)x)[1] + (((u_int8_t *)x)[0] << 8)) )

/* 32-bit values */
#define getu32(mm, x)   \
    ( (mm->flags & MM_LIT_ENDIAN)  ?    \
     ((((u_int8_t *)x)[0] <<  0) + \
      (((u_int8_t *)x)[1] <<  8) + \
      (((u_int8_t *)x)[2] << 16) + \
      (((u_int8_t *)x)[3] << 24) ) \
    :   \
     ((((u_int8_t *)x)[3] <<  0) + \
      (((u_int8_t *)x)[2] <<  8) + \
      (((u_int8_t *)x)[1] << 16) + \
      (((u_int8_t *)x)[0] << 24) ) )

#endif
