/*
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003 Brian Carrier.  All rights reserved
 *
 * DOS Partition Tables
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"
#include "mymalloc.h"
#include "error.h"

#include "dos.h"


/* Check the extended partition flags */
#define dos_is_ext(x)	\
	((((x) == 0x05) || ((x) == 0x0F) || ((x) == 0x85)) ? 1 : 0)

/* 
 * dos_get_desc
 *
 * Return a buffer with a string description of the partition type
 * 
 * From: http://www.win.tue.nl/~aeb/partitions/partition_types-1.html 
 */
static char *
dos_get_desc(u_int8_t ptype)
{
#define DESC_LEN 64
	char *str = mymalloc(DESC_LEN);

	switch (ptype) {
		case 0x00:
			snprintf(str, DESC_LEN, "Empty (0x00)");
			break;
		case 0x01:
			snprintf(str, DESC_LEN, "DOS FAT12 (0x01)");
			break;
		case 0x04:
		case 0x06:
			snprintf(str, DESC_LEN, "DOS FAT16 (0x%.2X)", ptype);
			break;
		case 0x05:
			snprintf(str, DESC_LEN, "DOS Extended (0x05)");
			break;
		case 0x07:
			snprintf(str, DESC_LEN, "NTFS (0x07)");
			break;
		case 0x08:
			snprintf(str, DESC_LEN, "AIX Boot (0x08)");
			break;
		case 0x09:
			snprintf(str, DESC_LEN, "AIX Data (0x09)");
			break;
		case 0x0b:
		case 0x0c:
			snprintf(str, DESC_LEN, "Win95 FAT32 (0x%.2X)", ptype);
			break;
		case 0x0e:
			snprintf(str, DESC_LEN, "Win95 FAT16 (0x0E)");
			break;
		case 0x0F:
			snprintf(str, DESC_LEN, "Win95 Extended (0x0F)");
			break;
		case 0x11:
			snprintf(str, DESC_LEN, "DOS FAT12 Hidden (0x11)");
			break;
		case 0x12:
			snprintf(str, DESC_LEN, "Hibernation (0x12)");
			break;
		case 0x14:
		case 0x16:
			snprintf(str, DESC_LEN, "DOS FAT16 Hidden (0x%.2X)", 
					ptype);
			break;
		case 0x1b:
		case 0x1c:
			snprintf(str, DESC_LEN, "Win95 FAT32 Hidden (0x%.2X)", 
					ptype);
			break;
		case 0x1e:
			snprintf(str, DESC_LEN, "Win95 FAT16 Hidden (0x1E)");
			break;
		case 0x42:
			snprintf(str, DESC_LEN, "Win LVM / Secure FS (0x42)");
			break;
		case 0x44:
			snprintf(str, DESC_LEN, "GoBack (0x44)");
			break;
		case 0x50:
		case 0x51:
		case 0x53:
		case 0x54:
			snprintf(str, DESC_LEN, "OnTrack Disk Manager (0x%.2X)",
					ptype);
			break;
		case 0x55:
			snprintf(str, DESC_LEN, "EZ-Drive (0x55)");
			break;
			return "EZ-Drive";
		case 0x57:
			snprintf(str, DESC_LEN, "DrivePro (0x57)");
			break;
		case 0x61:
			snprintf(str, DESC_LEN, "SpeedStor (0x61)");
			break;
		case 0x63:
			snprintf(str, DESC_LEN, "UNIX System V (0x63)");
			break;
		case 0x64:
		case 0x65:
		case 0x66:
		case 0x67:
		case 0x68:
		case 0x69:
			snprintf(str, DESC_LEN, "Novell Netware (0x%.2X)",
					ptype);
			break;
		case 0x82:
			snprintf(str, DESC_LEN, "Linux Swap / Solaris x86 (0x82)");
			break;
		case 0x83:
			snprintf(str, DESC_LEN, "Linux (0x83)");
			break;
		case 0x84:
			snprintf(str, DESC_LEN, "Hibernation (0x84)");
			break;
		case 0x85:
			snprintf(str, DESC_LEN, "Linux Extended (0x85)");
			break;
		case 0x86:
			snprintf(str, DESC_LEN, "NTFS Volume Set (0x86)");
			break;
		case 0x8e:
			snprintf(str, DESC_LEN, "Linux Logical Volume Manager (0x8e)");
			break;
		case 0x93:
			snprintf(str, DESC_LEN, "Linux Hidden (0x93)");
			break;
		case 0x9f:
			snprintf(str, DESC_LEN, "BSD/OS (0x9F)");
			break;
		case 0xa0:
		case 0xa1:
			snprintf(str, DESC_LEN, "Hibernation (0x%.2X)",
					ptype);
			break;
		case 0xa5:
			snprintf(str, DESC_LEN, "FreeBSD (0xA5)");
			break;
		case 0xa6:
			snprintf(str, DESC_LEN, "OpenBSD (0xA6)");
			break;
		case 0xa7:
			snprintf(str, DESC_LEN, "NextSTEP (0xA7)");
			break;
		case 0xa8:
			snprintf(str, DESC_LEN, "Mac OS X (0xA8)");
			break;
		case 0xa9:
			snprintf(str, DESC_LEN, "NetBSD (0xA9)");
			break;
		case 0xab:
			snprintf(str, DESC_LEN, "Mac OS X (0xAB)");
			break;
		case 0xb7:
			snprintf(str, DESC_LEN, "BSDI (0xB7)");
			break;
		case 0xb8:
			snprintf(str, DESC_LEN, "BSDI Swap (0xB8)");
			break;
		case 0xbe:
			snprintf(str, DESC_LEN, "Solaris 8 Boot (0xBE)");
			break;
		case 0xc2:
			snprintf(str, DESC_LEN, "Hidden Linux (0xC2)");
			break;
		case 0xc3:
			snprintf(str, DESC_LEN, "Hidden Linux Swap (0xC3)");
			break;
		case 0xc6:
		case 0xc7:
			snprintf(str, DESC_LEN, "Corrupted Windows NT Volume / Stripe Set (0x%.2X)",
					ptype);
			break;
		case 0xee:
			snprintf(str, DESC_LEN, "GPT Safety Partition (0xEE)");
			break;
		case 0xfb:
			snprintf(str, DESC_LEN, "VMWare File System (0xFB)");
			break;
		case 0xfc:
			snprintf(str, DESC_LEN, "VMWare Swap (0xFC)");
			break;
			return "VMware Swap";	
		case 0xfd:
			snprintf(str, DESC_LEN, "Linux RAID (0xFD)");
			break;
		default: 
			snprintf(str, DESC_LEN, "Unknown Type (0x%.2X)",
					ptype);
			break;
	}

	return str;
}

/* 
 * Load an extended partition table into the structure in MM_INFO.
 *
 * sect_cur: The sector where the extended table is located
 * sect_ext_base: The sector of the primary extended table (this does
 *   not change for recursive calls)
 * table: a counter that identifies the table depth 
 *   (increases by 1 for each recursive call)
 * 
 * For the primary extended table, sect_cur == sect_ext_base 
 *
 */
static void 
dos_load_ext_table(MM_INFO *mm, daddr_t sect_cur, daddr_t sect_ext_base, 
  int table)
{
	dos_sect sect;
	int i;
	off_t ret;
	off_t addr = (off_t)sect_cur * 512;
	char *table_str;

	if (verbose)
		fprintf (logfp,
		  "dos_load_ext: Table Sector: %lu, Primary Base Sector: %lu\n",
		  (ULONG)sect_cur, (ULONG)sect_ext_base);

	/* seek to the table location */
	if  (addr != (ret = lseek(mm->fd, addr, SEEK_SET))) {
		error ("Error seeking to extended table sector: %lu  Ret: %lu", 
				 (ULONG)addr, (ULONG)ret);
	}

	/* Read the table */
	if (sizeof(sect) != read (mm->fd, (char *)&sect, sizeof(sect))) {
		error ("Error reading extended table sector %lu\n", 
		  (ULONG)sect_cur);
	}

	/* Sanity Check */
	if (getu16(mm, sect.magic) != DOS_MAGIC) {
		error ("Invalid extended partition table in sector %lu\n", 
				(ULONG)sect_cur);
	}

	/* Add an entry of 1 length for the table  to the internal structure */
	table_str = mymalloc(32);
	snprintf(table_str, 32, "Extended Table (#%d)", table);
	mm_part_add(mm, sect_cur, 1, table_str, table, -1);

	/* Cycle through the four partitions in the table 
	 *
	 * When another extended partition is found, it is processed
	 * inside of the loop
	 */
	for (i = 0; i < 4; i++) {
		dos_part *part = &sect.ptable[i];
	
		/* Get the starting sector and size, we currently
		 * ignore CHS */
		u_int32_t part_start = getu32(mm, part->start_sec);
		u_int32_t part_size = getu32(mm, part->size_sec);

		if (verbose)
			fprintf(logfp,
			  "load_ext: %d:%d    Start: %lu   Size: %lu  Type: %d\n",
			  table, i, (ULONG)part_start, (ULONG)part_size, part->ptype);

		if (part_size == 0)
			continue;

		/* partitions are addressed differently 
		 * in extended partitions */
		if (dos_is_ext(part->ptype))  {

			/* part start is added to the start of the
			 * first extended partition (the primary
			 * extended partition) */
			mm_part_add(mm, sect_ext_base + part_start, 
			  part_size, dos_get_desc(part->ptype), 
			  table, i);


			/* Process the extended partition */
			dos_load_ext_table(mm, sect_ext_base + part_start, 
			  sect_ext_base, table + 1);
		}

		else {
			/* part_start is added to the start of the 
			 * current partition for the actual
			 * starting location */
			mm_part_add(mm, sect_cur + part_start, 
			  part_size, dos_get_desc(part->ptype), 
			  table, i);
		}
	}

	return;
}


/* 
 * Load the primary partition table (MBR) into the internal
 * data structures in MM_INFO
 *
 * This will automatically call load_ext_table for extended
 * partitions
 *
 * sect_cur is the addres of the table to load
 */
static void 
dos_load_prim_table(MM_INFO *mm, daddr_t sect_cur)
{
	dos_sect sect;
	int i;
	off_t ret;
	off_t addr = (off_t)sect_cur * 512;
	char *table_str;

	if (verbose)
		fprintf (logfp,
		  "dos_load_prim: Table Sector: %lu\n",
		  (ULONG)sect_cur);

	/* seek to the table */
	if  (addr != (ret = lseek(mm->fd, addr, SEEK_SET))) {
		error ("Error seeking to primary table sector: %lu  Ret:%lu", 
				 (ULONG)addr, (ULONG)ret);
	}

	/* Read the sector */
	if (sizeof(sect) != read (mm->fd, (char *)&sect, sizeof(sect))) {
		error ("Error reading primary table sector %lu\n", 
		  (ULONG)sect_cur);
	}

	/* Sanity Check */
	if (guessu16(mm, sect.magic, DOS_MAGIC)) {
		error ("File is not a DOS partition (invalid primary magic) (Sector: %lu)\n", 
				(ULONG)sect_cur);
	}

	/* Add an entry of 1 sector for the table  to the internal structure */
	table_str = mymalloc(32);
	snprintf(table_str, 32, "Primary Table (#0)");
	mm_part_add(mm, sect_cur, 1, table_str, -1, -1);

	/* Cycle through the partition table */
	for (i = 0; i < 4; i++) {
		dos_part *part = &sect.ptable[i];

		/* We currently ignore CHS */
		u_int32_t part_start = getu32(mm, part->start_sec);
		u_int32_t part_size = getu32(mm, part->size_sec);

		if (verbose)
			fprintf(logfp,
			  "load_pri:0:%d    Start: %lu   Size: %lu  Type: %d\n",
			  i, (ULONG)part_start, (ULONG)part_size, part->ptype);

		if (part_size == 0)
			continue;

		/* Add the partition to the internal structure */
		mm_part_add(mm, part_start, part_size, 
		  dos_get_desc(part->ptype), 0, i);

		/* If it is an extended partition, process it now */
		if (dos_is_ext(part->ptype))  {
			dos_load_ext_table(mm, part_start, part_start, 1);
		}
	}
	return;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
dos_part_walk(MM_INFO *mm, u_int16_t start, u_int16_t last, int flags, 
  MM_PART_WALK_FN action, char *ptr)
{
	MM_PART *part;
	int cnt = 0;

	if (start < mm->first_part || start > mm->last_part)
		error ("dos_part_walk: Invalid starting partition: %d", start);
	if (last < mm->first_part || last > mm->last_part)
		error ("dos_part_walk: Invalid ending partition: %d", last);

	/* Cycle through the partitions */
	part = mm->part_list; 
	while ((part != NULL) && (cnt <= last)) {

		/* Can we do the action yet? */
		if (cnt >= start)
			action(mm, cnt, part, 0, ptr);

		part = part->next;
		cnt++;
	}

	return;
}


void 
dos_close (MM_INFO *mm)
{
	mm_part_free(mm);
	close(mm->fd);
	free(mm);
}


/* 
 * Given the path to the file, open it and load the internal
 * partition table structure
 *
 * offset is the sector offset to the begiinning of the structure
 * (which maynot be the beginning of a partition)
 */
MM_INFO *
dos_open(const char *path, unsigned char type, daddr_t sect_offset)
{
	char *myname = "dos_open";

	MM_INFO *mm = (MM_INFO *)mymalloc(sizeof(*mm));

	if (type != MM_DOS)
		error ("%s: Invalid media type for DOS tools", myname);

	/* Open the image */
	if ((mm->fd = open(path, O_RDONLY)) < 0)
		error("%s: open %s: %m", myname, path);

	/* If an offset was given, then use that */
    if (sect_offset) 
        mm->sect_offset = sect_offset;
    else
        mm->sect_offset = DOS_PART_OFFSET;

	/* inititialize settings */
	mm->mmtype = type;
	mm->str_type = "DOS Partition Table";
	mm->part_list = NULL;
	mm->flags = 0;

	/* Assign functions */
	mm->part_walk = dos_part_walk;
	mm->close = dos_close;

	/* Load the partitions into the sorted list */
	dos_load_prim_table(mm, mm->sect_offset);

	/* fill in the sorted list with the 'unknown' values */
	mm_part_unused(mm);

	return mm;
}
