/*
 * The Sleuth Kit
 *
 * $Date: 2005/06/13 19:27:25 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 *
 * mac: Mac partition structures
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"
#include "mac.h"


/* 
 * Process the partition table at the sector address 
 * 
 * It is loaded into the internal sorted list 
 */
static uint8_t
mac_load_table(MM_INFO * mm, uint8_t test)
{
    mac_part part;
    char *table_str;
    uint32_t idx, max_part;

    if (verbose)
	fprintf(logfp, "mac_load_table: Sector: %" PRIuDADDR "\n",
		mm->sect_offset);

    /* The table can be variable length, so we loop on it 
     * The idx variable shows which round it is
     * Each structure is 512-bytes each
     */

    max_part = 1;		/* set it to 1 and it will be set in the first loop */
    for (idx = 0; idx < max_part; idx++) {
	uint32_t part_start;
	uint32_t part_size;
	char *str;


	/* Read the entry */
	if (mm_read_block_nobuf
	    (mm, (char *) &part, sizeof(part),
	     (DADDR_T) mm->sect_offset + idx) != sizeof(part)) {
	    if (test) {
		fprintf(stderr,
			"Error reading partition entry %" PRIuDADDR "\n",
			mm->sect_offset + idx);
		return 1;
	    }
	    else {
		error("Error reading partition entry %" PRIuDADDR "\n",
		      mm->sect_offset + idx);
	    }

	}


	/* Sanity Check */
	if (idx == 0) {
	    /* Set the endian ordering the first time around */
	    if (guessu16(mm, part.magic, MAC_MAGIC)) {
		if (test)
		    return 1;
		else
		    error("Invalid Mac partition table (Sector: %"
			  PRIuDADDR ") %" PRIx16 "\n",
			  (mm->sect_offset + idx), getu16(mm, part.magic));
	    }

	    /* Get the number of partitions */
	    max_part = getu32(mm, part.pmap_size);
	}
	else if (getu16(mm, part.magic) != MAC_MAGIC) {
	    if (test)
		return 1;
	    else
		error("Invalid Mac partition table entry (Sector: %"
		      PRIuDADDR ") %" PRIx16 "\n", (mm->sect_offset + idx),
		      getu16(mm, part.magic));
	}


	part_start = getu32(mm, part.start_sec);
	part_size = getu32(mm, part.size_sec);

	if (verbose)
	    fprintf(logfp,
		    "mac_load: %d  Starting Sector: %" PRIu32 "  Size: %"
		    PRIu32 " Type: %s\n", idx, part_start, part_size,
		    part.type);

	if (part_size != 0) {

	    str = mymalloc(sizeof(part.name));
	    strncpy(str, (char *)part.type, sizeof(part.name));

	    mm_part_add(mm, (DADDR_T) part_start, (DADDR_T) part_size,
			MM_TYPE_VOL, str, -1, idx);
	}
    }

    /* Add an entry for the table length */
    table_str = mymalloc(16);
    snprintf(table_str, 16, "Table");
    mm_part_add(mm, mm->sect_offset, max_part, MM_TYPE_DESC, table_str, -1,
		-1);

    return 0;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
mac_part_walk(MM_INFO * mm, PNUM_T start, PNUM_T last, int flags,
	      MM_PART_WALK_FN action, char *ptr)
{
    MM_PART *part;
    unsigned int cnt = 0;

    if (start < mm->first_part || start > mm->last_part)
	error("Invalid starting partition: %" PRIuPNUM "", start);

    if (last < mm->first_part || last > mm->last_part)
	error("Invalid ending partition: %" PRIuPNUM "", last);

    part = mm->part_list;
    while ((part != NULL) && (cnt <= last)) {

	if (cnt >= start)
	    action(mm, cnt, part, 0, ptr);

	part = part->next;
	cnt++;
    }

    return;
}


void
mac_close(MM_INFO * mm)
{
    mm_part_free(mm);
    free(mm);
}

MM_INFO *
mac_open(IMG_INFO * img_info, DADDR_T offset, uint8_t test)
{
    MM_INFO *mm = (MM_INFO *) mymalloc(sizeof(*mm));

    mm->img_info = img_info;
    mm->mmtype = MM_MAC;
    mm->str_type = "MAC Partition Map";

    /* If an offset was given, then use that too */
    mm->sect_offset = offset + MAC_PART_OFFSET;

    /* inititialize settings */
    mm->part_list = NULL;
    mm->first_part = mm->last_part = 0;
    mm->endian = 0;
    mm->dev_bsize = 512;
    mm->block_size = 512;

    /* Assign functions */
    mm->part_walk = mac_part_walk;
    mm->close = mac_close;

    /* Load the partitions into the sorted list */
    if (mac_load_table(mm, test)) {
	mac_close(mm);
	return NULL;
    }

    /* fill in the sorted list with the 'unknown' values */
    mm_part_unused(mm);

    return mm;
}
