/*
 * libewf file reading
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <stdlib.h>

#include "ewf_crc.h"
#include "ewf_file_header.h"

#include "libewf_definitions.h"
#include "libewf_endian.h"
#include "libewf_notify.h"
#include "libewf_md5.h"
#include "libewf_file.h"
#include "libewf_read.h"
#include "libewf_section.h"
#include "libewf_segment_table.h"

/* Reads and processes sections of a segment file
 * Returns the last section read, or NULL on error
 */
EWF_SECTION *libewf_read_sections_from_segment( LIBEWF_HANDLE *handle, uint32_t segment )
{
	EWF_SECTION *section     = NULL;
	char *filename           = NULL;
	int file_descriptor      = 0;
	uint64_t next_offset     = 0;

	/* The first offset is directly after the file header (13 byte)
	 */
	uint64_t previous_offset = EWF_FILE_HEADER_SIZE;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: invalid handle.\n" );

		return( NULL );
	}
	if( libewf_segment_table_values_is_set( handle->segment_table, segment ) == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: missing a segment file for segment %" PRIu32 ".\n", segment );

		return( NULL );
	}
	file_descriptor = libewf_segment_table_get_file_descriptor( handle->segment_table, segment );

	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: invalid file descriptor.\n" );

		return( NULL );
	}
	while( 1 )
	{
		section = libewf_section_read( handle, file_descriptor, handle->segment_table->section_list[ segment ], previous_offset );

		if( section == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: unable to read section.\n" );

			return( NULL );
		}
		next_offset = libewf_endian_convert_64bit( section->next );

		/* Check if the section alignment is correct.
		 * The done and next section point back at themselves,
		 * they should be the last section withing the segment file
		 */
		if( previous_offset < next_offset )
		{
			/* Seek the next section, it should be within the segment file
			 * this is required to skip the actual data within the segment file
			 */
			if( libewf_common_lseek( file_descriptor, next_offset, SEEK_SET ) != next_offset )
			{
				filename = libewf_segment_table_get_filename( handle->segment_table, segment );

				LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: next section not found segment file: %s.\n", filename );

				ewf_section_free( section );

				return( NULL );
			}
			previous_offset = next_offset;
		}
		else if( ewf_section_is_type_next( section ) || ewf_section_is_type_done( section ) )
		{
			break;
		}
		else
		{
			LIBEWF_WARNING_PRINT( "libewf_sections_read_segment: section skip for section type: %s not allowed.\n", section->type );

			ewf_section_free( section );

			return( NULL );
		}
		ewf_section_free( section );
	}
	return( section );
}

/* Reads a certain chunk within the sectors section according to the offset table
 * Returns the amount of bytes read, or -1 on error
 */
int64_t libewf_read_chunk( LIBEWF_HANDLE *handle, uint32_t chunk, void *buffer, uint64_t buffer_size )
{
	uint64_t size       = 0;
	uint64_t offset     = 0;
	int32_t count       = 0;
	int file_descriptor = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: invalid handle.\n" );

		return( -1 );
	}
	if( handle->index_build == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: index was not build.\n" );

		return( -1 );
	}
	if( chunk >= handle->offset_table->amount )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: chunk: %" PRIu32 " not in offset table.\n", chunk );

		return( -1 );
	}
	file_descriptor = handle->offset_table->file_descriptor[ chunk ];
	size            = handle->offset_table->size[ chunk ];
	offset          = handle->offset_table->offset[ chunk ];

	LIBEWF_VERBOSE_PRINT( "libewf_read_chunk: read file descriptor: %d, for offset: %" PRIu64 ", for size: %" PRIu64 ".\n", file_descriptor, offset, size );

	if( size == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: invalid chunk offset data - size of chunk is zero.\n" );

		return( -1 );
	}
	else if( size > buffer_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: size of chunk larger than specified buffer size.\n" );

		return( -1 );
	}
	count = ewf_sectors_chunk_read( buffer, file_descriptor, offset, size );

	if( count < size )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_chunk: cannot read chunk: %" PRIu32 " from file.\n", chunk );

		return( -1 );
	}
	return( count );
}

/* Reads a random offset within the data within the EWF file
 * Returns the amount of bytes read, or -1 on error
 */
int64_t libewf_read_random( LIBEWF_HANDLE *handle, void *buffer, uint64_t size, uint64_t offset )
{
	LIBEWF_CHUNK_CACHE *chunk_cache = NULL;
	EWF_CRC *calculated_crc         = NULL;
	EWF_CRC stored_crc              = 0;
	void *data_set                  = NULL;
	uint64_t media_size             = 0;
	uint64_t buffer_offset          = 0;
	uint64_t available              = 0;
	int64_t chunk_read_count        = 0;
	int64_t count_read              = 0;
	uint32_t chunk                  = 0;
	uint32_t chunk_data_size        = 0;
	uint32_t raw_data_size          = 0;
	int8_t result                   = 0;
	uint8_t percentage              = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_random: invalid handle.\n" );

		return( -1 );
	}
	if( handle->index_build == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_random: index was not build.\n" );

		return( -1 );
	}
	if( handle->chunk_cache == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_random: invalid chunk cache.\n" );

		return( -1 );
	}
	media_size = libewf_handle_media_size( handle );

	if( offset >= media_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_random: offset beyond media size.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_read_random: reading from offset: %" PRIu64 " size: %" PRIu64 ".\n", offset, size );

	/* The chunk we are after
	 */
	chunk = (uint32_t) ( offset / handle->chunk_size );

	if( chunk >= handle->offset_table->amount )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_random: attempting to read past the end of the file.\n");

		return( -1 );
	}
	/* Offset within the decompressed buffer
	 */
	buffer_offset = offset % handle->chunk_size;

	/* Reallocate the chunk cache if the chunk size is not the default chunk size
	 * this prevents multiple reallocation of the chunk cache
	 */
	chunk_data_size = handle->chunk_size + EWF_CRC_SIZE;

	if( chunk_data_size > handle->chunk_cache->allocated_size )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_read_random: reallocating chunk data size: %" PRIu32 ".\n", chunk_data_size );

		chunk_cache = libewf_chunk_cache_realloc( handle->chunk_cache, chunk_data_size );

		if( chunk_cache == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_read_random: unable to reallocate chunk cache.\n");

			return( -1 );
		}
		handle->chunk_cache = chunk_cache;
	}
	while( size > 0 )
	{
		/* If we no longer have any more blocks the end of file was reached
		 */
		if( chunk >= handle->offset_table->amount )
		{
			break;
		}
		/* Work out if this is a cache miss
		 */
		if( handle->chunk_cache->identifier != chunk )
		{
			/* Determine the size of the chunk including the CRC
			 */
			chunk_data_size = handle->offset_table->size[ chunk ];

			if( chunk_data_size > handle->chunk_cache->allocated_size )
			{
				LIBEWF_VERBOSE_PRINT( "libewf_read_random: reallocating chunk data size: %" PRIu32 ".\n", chunk_data_size );

				chunk_cache = libewf_chunk_cache_realloc( handle->chunk_cache, chunk_data_size );

				if( chunk_cache == NULL )
				{
					LIBEWF_WARNING_PRINT( "libewf_read_random: unable to reallocate chunk cache.\n");

					return( -1 );
				}
				handle->chunk_cache = chunk_cache;
			}
			/* Prevent data contamination, wipe the cache buffers clean
			 */
			chunk_cache = libewf_chunk_cache_wipe( handle->chunk_cache );

			if( chunk_cache == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_read_random: unable to wipe chunk cache.\n");

				return( -1 );
			}
			handle->chunk_cache = chunk_cache;

			/* Read the chunk data
			 */
			chunk_read_count = libewf_read_chunk( handle, chunk, handle->chunk_cache->read, handle->chunk_cache->allocated_size );

			if( chunk_read_count == -1 )
			{
				LIBEWF_WARNING_PRINT( "libewf_read_random: unable to read chunk.\n");

				return( -1 );
			}
			/* The size of the data within the chunk is the amount of bytes read
			 * minus the 4 bytes for the CRC
			 */
			chunk_data_size = (uint32_t) (chunk_read_count - EWF_CRC_SIZE);

			percentage = ( ( handle->offset_table->last > 0 ) ? ( ( chunk * 100 ) / handle->offset_table->last ) : 1 );

			/* Determine if the chunk is compressed
			 */
			if( handle->offset_table->compressed[ chunk ] == 1 )
			{
				raw_data_size = handle->chunk_size;

				/* Try to decompress the chunk
				 */
				result = ewf_sectors_chunk_uncompress( handle->chunk_cache->data, &raw_data_size, handle->chunk_cache->read, chunk_read_count );

				LIBEWF_VERBOSE_PRINT( "libewf_read_random: chunk %" PRIu32 " of %" PRIu64 " (%" PRIu8 "%%) is COMPRESSED.\n", ( chunk + 1 ), handle->offset_table->amount, percentage );

				if( result != 1 )
				{
					LIBEWF_WARNING_PRINT( "libewf_read_random: unable to uncompress chunk.\n" );

					data_set = (void *) libewf_handle_set_crc_error_chunk( handle, chunk );

					if( data_set == NULL )
					{
						LIBEWF_WARNING_PRINT( "libewf_read_random: unable to set CRC error chunk.\n" );

						return( -1 );
					}
					if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
					{
						return( -1 );
					}
					/* The chunk cache will remain empty on error (only 0 values)
					 */
				}
				handle->compression_used    = 1;
				handle->chunk_cache->amount = raw_data_size;
			}
			else if( handle->offset_table->compressed[ chunk ] == 0 )
			{
				LIBEWF_VERBOSE_PRINT( "libewf_read_random: chunk %" PRIu32 " of %" PRIu64 " (%" PRIu8 "%%) is UNCOMPRESSED.\n", ( chunk + 1 ), handle->offset_table->amount, percentage );

				calculated_crc = ewf_crc_calculate( handle->chunk_cache->read, ( chunk_read_count - EWF_CRC_SIZE ), 1 );

				if( calculated_crc == NULL )
				{
					LIBEWF_WARNING_PRINT( "libewf_read_random: unable to calculate CRC.\n" );

					return( -1 );
				}
				stored_crc = libewf_endian_convert_32bit( &handle->chunk_cache->read[ chunk_read_count - EWF_CRC_SIZE ] );

				LIBEWF_VERBOSE_PRINT( "libewf_read_random: CRC for chunk: %" PRIu32 " (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", ( chunk + 1 ), stored_crc, *calculated_crc );

				if( stored_crc != *calculated_crc )
				{
					LIBEWF_WARNING_PRINT( "libewf_read_random: CRC does not match for chunk: %" PRIu32 " (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", chunk, stored_crc, *calculated_crc );

					data_set = (void *) libewf_handle_set_crc_error_chunk( handle, chunk );

					if( data_set == NULL )
					{
						LIBEWF_WARNING_PRINT( "libewf_read_random: unable to set CRC error chunk.\n" );

						ewf_crc_free( calculated_crc );

						return( -1 );
					}
					if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
					{
						ewf_crc_free( calculated_crc );

						return( -1 );
					}
					/* The chunk cache will remain empty on error (only 0 values)
					 */
				}
				else
				{
					data_set = libewf_common_memcpy( (void *) handle->chunk_cache->data, (void *) handle->chunk_cache->read, chunk_data_size );

					if( data_set == NULL )
					{
						LIBEWF_WARNING_PRINT( "libewf_read_random: unable to set chunk cache.\n" );

						ewf_crc_free( calculated_crc );

						return( -1 );
					}
				}
				handle->chunk_cache->amount = chunk_data_size;

				ewf_crc_free( calculated_crc );
			}
			else
			{
				LIBEWF_WARNING_PRINT( "libewf_read_random: unsupported compressed chunk value.\n" );

				return( -1 );
			}
			handle->chunk_cache->identifier = chunk;
		}
		/* The available amount of data within the cached chunk
		 */
		available = handle->chunk_cache->amount - buffer_offset;

		/* Correct the available amount of bytes is larger
		 * than the requested amount of bytes
		 */
		if( available > size )
		{
			available = size;
		}
		/* Copy the relevant data to buffer
		 */
		data_set = libewf_common_memcpy( ( buffer + count_read ), (void *) &handle->chunk_cache->data[ buffer_offset ], available );

		if( data_set == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_read_random: unable to set buffer.\n" );

			return( -1 );
		}
		size          -= available;
		count_read    += available;
		buffer_offset  = 0;

		chunk++;
	}
	if( handle->swap_byte_pairs == 1 )
	{
		libewf_endian_swap_byte_pairs( buffer, count_read );
	}
	return( count_read );
}

/* Reads the data to a file descriptor
 * Returns a -1 on error, the amount of bytes read on success
 */
int64_t libewf_read_to_file_descriptor( LIBEWF_HANDLE *handle, int output_file_descriptor, uint64_t read_size, uint64_t read_offset, void (*callback)( uint64_t bytes_read, uint64_t bytes_total ) )
{
	LIBEWF_MD5_CTX md5;

	EWF_MD5HASH *calculated_md5hash          = NULL;
	LIBEWF_STRING *calculated_md5hash_string = NULL;
	LIBEWF_STRING *stored_md5hash_string     = NULL;
	uint8_t *data                            = NULL;
	int64_t read_count                       = 0;
	int64_t total_read_count                 = 0;
	int64_t write_count                      = 0;
	uint64_t media_size                      = 0;
	uint64_t size                            = 0;
	uint8_t read_all                         = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: invalid handle.\n" );

		return( -1 );
	}
	if( handle->index_build == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: index was not build.\n" );

		return( -1 );
	}
	media_size = libewf_handle_media_size( handle );

	if( ( read_size == 0 ) || ( read_size > media_size ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: invalid size.\n" );

		return( -1 );
	}
	if( read_offset >= media_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: invalid offset.\n" );

		return( -1 );
	}
	if( ( read_size + read_offset ) > media_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to export beyond size of media.\n" );

		return( -1 );
	}
	read_all = ( ( read_size == media_size ) && ( read_offset == 0 ) );
	data     = (uint8_t *) libewf_common_alloc( handle->chunk_size * sizeof( uint8_t ) );

	if( data == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to allocate data.\n" );

		return( -1 );
	}
	LIBEWF_MD5_INIT( &md5 );

	while( total_read_count < read_size )
	{
		size = read_size - total_read_count;

		if( size > handle->chunk_size )
		{
			size = handle->chunk_size;
		}
		read_count = libewf_read_random( handle, data, size, read_offset );

		if( read_count <= -1 )
		{
			LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: error reading data.\n" );

			libewf_common_free( data );

			return( -1 );
		}
		read_offset += size;

		LIBEWF_MD5_UPDATE( &md5, data, read_count );

		write_count = libewf_common_write( output_file_descriptor, data, read_count );

		if( write_count < read_count )
		{
			LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: error writing data.\n" );

			libewf_common_free( data );

			return( -1 );
		}
		total_read_count += read_count;

		if( callback != NULL )
		{
			callback( total_read_count, read_size );
		}
  	}
	libewf_common_free( data ) ;

	calculated_md5hash = ewf_md5hash_alloc();

	if( calculated_md5hash == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to create MD5 hash.\n" );

		return( -1 );
	}
  	LIBEWF_MD5_FINAL( calculated_md5hash, &md5 );

	calculated_md5hash_string = ewf_md5hash_to_string( calculated_md5hash );

	if( calculated_md5hash_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to create MD5 hash string.\n" );

		ewf_md5hash_free( calculated_md5hash );

		return( -1 );
	}
	/* Only verify MD5 if all data was read
	 */
	if( read_all )
	{
		/* If MD5 hash is NULL no hash section was found in the file
		 */
		if( handle->md5hash != NULL )
		{
			stored_md5hash_string = ewf_md5hash_to_string( handle->md5hash );

			if( stored_md5hash_string == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: unable to read stored MD5 hash.\n" );

				ewf_md5hash_free( calculated_md5hash );
				libewf_string_free( calculated_md5hash_string );

				return( -1 );
			}
			LIBEWF_VERBOSE_PRINT( "libewf_read_to_file_descriptor: MD5 hash stored: %s, calculated: %s.\n", (char *) stored_md5hash_string, (char *) calculated_md5hash_string );

			if( libewf_common_memcmp( calculated_md5hash, handle->md5hash, 16 ) != 0 )
			{
				LIBEWF_WARNING_PRINT( "libewf_read_to_file_descriptor: MD5 hash does not match.\n" );

				ewf_md5hash_free( calculated_md5hash );
				libewf_string_free( calculated_md5hash_string );
				libewf_string_free( stored_md5hash_string );

				return( -1 );
			}
			libewf_string_free( stored_md5hash_string );
		}
		else
		{
			LIBEWF_VERBOSE_PRINT( "libewf_read_to_file_descriptor: MD5 hash stored: NONE, calculated: %s.\n", (char *) calculated_md5hash_string );
		}
	}
	else
	{
		LIBEWF_VERBOSE_PRINT( "libewf_read_to_file_descriptor: MD5 hash calculated: %s.\n", (char *) calculated_md5hash_string );
	}
	return( total_read_count );
}

