/*
 * img_cat
 * The Sleuth Kit 
 *
 * $Date: 2007/12/19 23:11:56 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2006 Brian Carrier, Basis Technology.  All rights reserved 
 *
 * This software is distributed under the Common Public License 1.0
 *
 */
#include "tsk/libtsk.h"
#include <errno.h>
#include <string.h>

#ifdef TSK_WIN32
#include <fcntl.h>
#endif

static TSK_TCHAR *progname;

static void
usage()
{
    TFPRINTF(stderr, _TSK_T("usage: %s [-vV] [-i imgtype] image\n"),
        progname);
    tsk_fprintf(stderr,
        "\t-i imgtype: The format of the image file (use 'i list' for supported types)\n");
    tsk_fprintf(stderr, "\t-v: verbose output to stderr\n");
    tsk_fprintf(stderr, "\t-V: Print version\n");

    exit(1);
}


int
MAIN(int argc, TSK_TCHAR ** argv)
{
    TSK_IMG_INFO *img;
    TSK_TCHAR *imgtype = NULL;
    int ch;
    TSK_OFF_T done;
    ssize_t cnt;

    progname = argv[0];

    while ((ch = getopt(argc, argv, _TSK_T("i:vV"))) > 0) {
        switch (ch) {
        case _TSK_T('?'):
        default:
            TFPRINTF(stderr, _TSK_T("Invalid argument: %s\n"),
                argv[optind]);
            usage();

        case _TSK_T('i'):
            imgtype = optarg;
            if (TSTRCMP(imgtype, _TSK_T("list")) == 0) {
                tsk_img_print_types(stderr);
                exit(1);
            }
            break;

        case _TSK_T('v'):
            tsk_verbose++;
            break;

        case _TSK_T('V'):
            tsk_print_version(stdout);
            exit(0);
        }
    }

    /* We need at least one more argument */
    if (optind >= argc) {
        tsk_fprintf(stderr, "Missing image name\n");
        usage();
    }

    if ((img =
            tsk_img_open(imgtype, argc - optind,
                (const TSK_TCHAR **) &argv[optind])) == NULL) {
        tsk_error_print(stderr);
        exit(1);
    }

#ifdef TSK_WIN32
    if (-1 == _setmode(_fileno(stdout), _O_BINARY)) {
        tsk_errno = TSK_ERR_FS_WRITE;
        snprintf(tsk_errstr, TSK_ERRSTR_L,
            "dls_lib: error setting stdout to binary: %s",
            strerror(errno));
        return 1;
    }
#endif

    for (done = 0; done < img->size; done += cnt) {
        char buf[16 * 1024];
        size_t len;

        if (done + sizeof(buf) > img->size) {
            len = (size_t)(img->size - done);
        }
        else {
            len = sizeof(buf);
        }

        cnt = img->read_random(img, 0, buf, len, done);
        if (cnt != len) {
            if (cnt >= 0) {
                tsk_fprintf(stderr,
                    "img_cat: Error reading image file at offset: %"
                    PRIuOFF ", len: %" PRIuOFF ", return: %" PRIuOFF "\n",
                    done, len, cnt);
            }
            else {
                tsk_error_print(stderr);
            }
            img->close(img);
            exit(1);
        }

        if (fwrite(buf, cnt, 1, stdout) != 1) {
            tsk_errno = TSK_ERR_IMG_WRITE;
            snprintf(tsk_errstr, TSK_ERRSTR_L,
                "img_cat: Error writing to stdout:  %s", strerror(errno));
            tsk_error_print(stderr);
            img->close(img);
            exit(1);
        }
    }

    img->close(img);
    exit(0);
}
