/*
** fls
** The @stake Sleuth Kit (TASK)
**
** Given an image and directory inode, display the file names and 
** directories that exist (both active and deleted)
**
** Brian Carrier [carrier@atstake.com]
** Copyright (c) 2002 @stake Inc.  All rights reserved
**
**
** TCTUTILs
** Brian Carrier [carrier@cerias.purdue.edu]
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
** 
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"

#include "ntfs.h"


void usage(char *myProg) {
	printf("usage: %s [-adDFlpurV] [-f fstype] [-m dir/] [-z ZONE] image [inode]\n", 
	  myProg);
    printf("\tIf [inode] is not given, 2 is used\n");
	printf("\t-a: Display \".\" and \"..\" entries\n");
	printf("\t-d: Display deleted entries only\n");
	printf("\t-D: Display directory entries only\n");
	printf("\t-F: Display file entries only (NOTE: This was -f in TCTUTILs)\n");
	printf("\t-l: Display long version (like ls -l)\n");
	printf("\t-m: Display output in mactime input format with\n");
	printf("\t      dir/ as the actual mount point of the image\n");
	printf("\t-p: Display full path for each file\n");
	printf("\t-u: Display undeleted entries only\n");
	printf("\t-r: Recurse on directory entries\n");
	printf("\t-V: Print version\n");
	printf("\t-z: Time zone of original machine (i.e. EST or GMT) (only useful with -l)\n");
    printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
	fs_print_types();

	exit(1);
}

FILE *logfp;

/*directory prefix for printing mactime output */
static char *macpre = NULL;	

static int localFlags;

/* Local Flags */
#define LCL_DOT		0x001
#define LCL_LONG	0x002
#define LCL_FILE	0x004
#define LCL_DIR		0x008
#define LCL_FULL	0x010
#define LCL_MAC		0x020



static void
printit (FS_INFO *fs, FS_DENT *fs_dent, int flags, char *ext_inode, 
  char *ext_name)
{
	int i;

	if (!(localFlags & LCL_FULL)) {
		for (i=0; i<fs_dent->pathdepth;i++) 
			fprintf(stdout, "+");

		if (fs_dent->pathdepth)
			fprintf(stdout, " ");
	}


	if (localFlags & LCL_MAC) {
		fs_dent_print_mac(stdout, fs_dent, flags, fs, macpre);
	}

	else if (localFlags & LCL_LONG) {

		if (LCL_FULL & localFlags) 
			fs_dent_print_long(stdout, fs_dent, flags, fs, ext_inode, ext_name);
		else {
			char *tmpptr = fs_dent->path;
			fs_dent->path = NULL;
			fs_dent_print_long(stdout, fs_dent, flags, fs, ext_inode, ext_name);
			fs_dent->path = tmpptr;
		}

	}
	else {
		if (LCL_FULL & localFlags) 
			fs_dent_print(stdout, fs_dent, flags, fs, ext_inode, ext_name);
		else {
			char *tmpptr = fs_dent->path;
			fs_dent->path = NULL;
			fs_dent_print(stdout, fs_dent, flags, fs, ext_inode, ext_name);
			fs_dent->path = tmpptr;
		}
		printf("\n");
	}
}


static u_int8_t
print_dent (FS_INFO *fs, FS_DENT *fs_dent, int flags, char *ptr) 
{
    if ((ISDOT (fs_dent->name) ) && ((localFlags & LCL_DOT) == 0))
		return WALK_CONT;

	/* only print dirs if LCL_DIR is set and only print everything
	** else if LCL_FILE is set (or we aren't sure what it is)
	*/
	if ( ((localFlags & LCL_DIR) && 
		 ((fs_dent->fsi) && 
		 ((fs_dent->fsi->mode & FS_INODE_FMT) == FS_INODE_DIR))) || 
		((localFlags & LCL_FILE) && 
		 (((fs_dent->fsi) && 
		 ((fs_dent->fsi->mode & FS_INODE_FMT) != FS_INODE_DIR)) ||
		 (!fs_dent->fsi)))) {


		/* Make a special case for NTFS so we can identify all of the
		 * alternate data streams!
		 */
		if (((fs->ftype & FSMASK) == NTFS_TYPE) && (fs_dent->fsi)) {

			FS_DATA *fs_data = fs_dent->fsi->attr;

			while ((fs_data) && (fs_data->flags & FS_DATA_INUSE)) {
				if (fs_data->type == NTFS_ATYPE_DATA) {
					char ext_i[32];
					char ext_n[256];
					snprintf(ext_i, 32, "-%d-%d", fs_data->type, fs_data->id);

					/* The default data attribute does not get the :XX */
					if (strcmp(fs_data->name, "$Data") == 0)
						ext_n[0] = '\0';
					else
						snprintf(ext_n, 256, ":%s", fs_data->name);
					printit(fs, fs_dent, flags, ext_i, ext_n);
				}
				else if (fs_data->type == NTFS_ATYPE_IDXROOT) {
					printit(fs, fs_dent, flags, "", "");
				}

				fs_data = fs_data->next;
			}
		}
		else {
			printit(fs, fs_dent, flags, "", "");
		}
	}
	return WALK_CONT;
}


int 
main(int argc, char **argv) 
{
	char *fstype = DEF_FSTYPE;
	int inode;
	int flags = FS_FLAG_ALLOC | FS_FLAG_UNALLOC;
	char ch;
	FS_INFO 	*fs;
	extern int optind;
	progname = argv[0];

	localFlags =  LCL_DIR | LCL_FILE;

	while ((ch = getopt(argc, argv, "adDf:Fm:luprVz:")) > 0) {
		switch (ch) {
		case '?':
		default: 
			usage(argv[0]);
		case 'a':
			localFlags |= LCL_DOT;
			break;
		case 'd':
			flags &= ~FS_FLAG_ALLOC;
			break;
		case 'D':
			localFlags &= ~LCL_FILE;
			localFlags |= LCL_DIR;
			break;
        case 'f':
            fstype = optarg;
            break;
		case 'F':
			localFlags &= ~LCL_DIR;
			localFlags |= LCL_FILE;
			break;
		case 'l':
			localFlags |= LCL_LONG;
			break;
		case 'm':
			localFlags |= LCL_MAC;
			macpre = optarg;
			break;
		case 'p':
			localFlags |= LCL_FULL;
			break;
		case 'u':
			flags &= ~FS_FLAG_UNALLOC;
			break;
		case 'r':
			flags |= FS_FLAG_RECURSE;
			break;
		case 'V':
			print_version();
			exit(0);
		case 'z':
            {
            char envstr[32];
            snprintf(envstr, 32, "TZ=%s", optarg);
            if (0 != putenv(envstr)) {
                    error ("error setting environment");
            }

            /* we should be checking this somehow */
            tzset();
            }
            break;

		}
	}

	/* only the image and optional inode are left */
	if ((optind == argc) || ((optind+2) < argc))
		usage(argv[0]);


	/* Set the full flag to print the full path name if recursion is
	** set and we are only displaying files or deleted files
	*/
	if ((flags & FS_FLAG_RECURSE) && (
	  ((flags & FS_FLAG_UNALLOC) && (!(flags & FS_FLAG_ALLOC))) ||
	  ((localFlags & LCL_FILE) && (!(localFlags & LCL_DIR))) )) {

		localFlags |= LCL_FULL;
	}

	/* set flag to save full path for mactimes style printing */
	if (localFlags & LCL_MAC) {
		localFlags |= LCL_FULL;
	}

	/* we need to append a / to the end of the directory if
	 * one does not already exist
	 */
	if (macpre) {
		int len = strlen (macpre);
		if (macpre[len - 1] != '/') {
			char *tmp = macpre;
			macpre = (char *)malloc(len + 2);
			strncpy (macpre, tmp, len + 1);
			strncat (macpre, "/", len + 2);
		}
	}

	/* open image */
	fs = fs_open(argv[optind++], fstype);

	/* was the inode given? */
	if ((optind+1) != argc)
		inode = fs->root_inum;
	else 
		inode = atoi(argv[optind]);

	/* begin walk */
	fs->dent_walk(fs, inode, flags, print_dent, (char *)0); 

	fs->close(fs);

	exit (0);
}

