/*
** istat
** The @stake Sleuth Kit (TASK)
**
** Display all inode info about a given inode.  This is a more verbose 
** version of 'ils -a'.  The body of this program was built around
** it (ils.c).
**
** Brian Carrier [carrier@atstake.com]
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
**
**
** TCTUILs
** Brian Carrier [carrier@cerias.purdue.edu]
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"
#include "ffs.h"
#include "ext2fs.h"

#include <time.h>

FILE   *logfp;

extern char *tzname[2];


/* When > -1 this is the number of blocks to print, used for -b arg */
static int numblock = -1; 

/* indirect block accounting */
#define INDIR_SIZ 	64
static DADDR_T indirl[INDIR_SIZ];
static	unsigned char	indir_idx;

/* atoinum - convert string to inode number */
INUM_T  
atoinum(const char *str)
{
    char   *cp, *dash;
    INUM_T  inum;

    if (*str == 0)
		return (0);

	/* if we are given the inode in the inode-type-id form, then ignore
	 * the other stuff w/out giving an error 
	 *
	 * This will make scripting easier
	 */
	if ((dash = strchr(str, '-')) != NULL) {
		*dash = '\0';
	}
    inum = STRTOUL(str, &cp, 0);
    if (*cp || cp == str)
		error("bad inode number: %s", str);
    return (inum);
}

/* usage - explain and terminate */

static void usage() {
    printf("usage: %s [-b num] [-f fstype] [-z zone] [-vV] image inum\n", progname);
	printf("\t-v: verbose output to stderr\n");
	printf("\t-V: print version\n");
	printf("\t-b num: force the display of NUM address of block pointers\n");
	printf("\t-z zone: time zone of original machine (i.e. EST or GMT)\n");
    printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
	fs_print_types();
	exit(1);
}

/* global value so that there are always 8 blocks wide */
static int printidx = 0;
#define EDGE "  "
#define WIDTH	8


static u_int8_t
print_addr_act (FS_INFO *fs, DADDR_T addr, char *buf,
  int size, int flags, char *ptr)
{
	if (flags & FS_FLAG_CONT) {
		int i, s;
		/* cycle through the fragments if they exist */
		for (i = 0, s = size; s > 0; s-= fs->block_size, i++) {
			if (printidx == 0)
				printf ("%s", EDGE);

			printf("%lu ", (unsigned long) addr + i);
			if (++printidx == WIDTH) {
				printf("\n");
				printidx = 0;
			}
		}
	}
	else if (flags & FS_FLAG_META) {
		if (indir_idx < INDIR_SIZ) 
			indirl[indir_idx++] = addr;
	}
	return WALK_CONT;
}

static void
print_inode_ntfs(FS_INFO *fs, FS_INODE *inode)
{
	FS_DATA *data;
	int	flags = (FS_FLAG_CONT | FS_FLAG_AONLY | FS_FLAG_SLACK |
	  FS_FLAG_ALLOC | FS_FLAG_UNALLOC);

	data = inode->attr;
	
	while ((data) && (data->flags & FS_DATA_INUSE)) {
	
		printf("Type: %i-%i    Name: %s   %sResident   size: %d\n", 
		  data->type, data->id, data->name,
		  (data->flags & FS_DATA_NONRES)?"Non-":"", (int)data->size);

		if (data->flags & FS_DATA_NONRES) {
			printidx = 0;
			fs->file_walk(fs, inode, data->type, data->id, flags, 
			   print_addr_act, "");
			if (printidx != 0) 
				printf("\n");
		}

		data = data->next;
	}

	return;
}



/* print_inode - list generic inode */
static void
print_inode(FS_INFO *fs, INUM_T inum, FS_INODE *fs_inode)
{
	struct tm *tmTime;
	time_t timetmp; 
	char ls[12];
	int flags = (FS_FLAG_AONLY | FS_FLAG_CONT | FS_FLAG_META |
		  FS_FLAG_SLACK | FS_FLAG_ALLOC | FS_FLAG_UNALLOC );

	printf("inode: %lu\n", (ULONG) inum);
	printf("%sAllocated\n", (fs_inode->flags&FS_FLAG_ALLOC)?"":"Not ");

	if ((fs->ftype & FSMASK) == FFS_TYPE) {
		FFS_INFO *ffs = (FFS_INFO *) fs;
		printf("Group: %lu\n", (ULONG)ffs->cg_num);
	}
	else if ((fs->ftype & FSMASK) == EXT2FS_TYPE) {
		EXT2FS_INFO *ext2fs = (EXT2FS_INFO *) fs;
		printf("Group: %lu\n", (ULONG)ext2fs->grpnum);
	}

	if (fs_inode->link) 
		printf("symbolic link to: %s\n", fs_inode->link);

	printf("uid / gid: %d / %d\n", (int)fs_inode->uid, (int)fs_inode->gid);

	/* The permissions are on the last 3 nibbles as such:
	** There are 4 groups of 3 bits.  The gsb is for Setuid, the next
	** grp is own, next grp is grp, and last grp for other
	**
	** for setuid the bits are: user,grp,text
	** for rwx the bits are rwx ..
	**
	*/
	make_ls(fs_inode->mode, ls, 12);
	printf("mode: %s\n", ls);

	printf("size: %lu\n", (ULONG) fs_inode->size);
	printf("num of links: %lu\n", (ULONG) fs_inode->nlink);

	timetmp = fs_inode->mtime;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);

	if ((fs->ftype & FSMASK) == FATFS_TYPE) 
		printf("Written:\t");
	else
		printf("Modified:\t");

	printf("%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[(tmTime->tm_isdst==0)?0:1]);

	timetmp = fs_inode->atime;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);


	/* FAT only gives the day of last access */
	printf("Accessed:\t%.2d.%.2d.%.4d",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900);

	if ((fs->ftype & FSMASK) != FATFS_TYPE) 
		printf (" %.2d:%.2d:%.2d\t(%s)\n",
		  tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		  tzname[(tmTime->tm_isdst==0)?0:1]);
	else
		printf(" 00:00:00\t(GMT)\n");

	timetmp = fs_inode->ctime;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);

	
	if ((fs->ftype & FSMASK) == FATFS_TYPE) 
		printf("Created:\t");
	else
		printf("Changed:\t");

	printf("%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[(tmTime->tm_isdst==0)?0:1]);

	if (fs->flags & FS_HAVE_DTIME) {
		timetmp = fs_inode->dtime;
		if (timetmp < 0)
			timetmp = 0;
		tmTime = localtime(&timetmp);
		printf("Deleted:\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s)\n",
			tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
			tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
			tzname[(tmTime->tm_isdst==0)?0:1]);
	}

	/* A bad hack to force a specified number of blocks */
	if (numblock > -1)
		fs_inode->size = numblock * fs->file_bsize;

	/* print the name if it exists: NTFS and FAT only */
	if (fs_inode->name) {
		FS_NAME *fs_name = fs_inode->name;
		printf("Name: ");
		while (fs_name) {
			printf("%s", fs_name->name);
			fs_name = fs_name->next;
			if (fs_name)
				printf(", ");
			else
				printf("\n");
		}
	}


	/* print the attributes list for NTFS */
	if ((fs->ftype & FSMASK) == NTFS_TYPE) {
		printf("\nAttributes: \n");
		print_inode_ntfs(fs, fs_inode);

		return;
	}
	else if ((fs->ftype & FSMASK) == FATFS_TYPE) 
		printf ("Sectors:\n");
	else
		printf ("Direct Blocks:\n");

	indir_idx = 0;

	fs->file_walk(fs, fs_inode, 0, 0, flags, print_addr_act, "");

	if (printidx != 0)
		printf("\n");


	/* print indirect blocks */
	if (indir_idx > 0) {
		int i;
		printf ("Indirect Blocks:\n%s", EDGE);

		printidx = 0;

		for (i = 0; i < indir_idx; i++) {
			if (printidx == 0)
				printf("%s", EDGE);
			
			printf("%lu ", (unsigned long) indirl[i]);
			if (++printidx == WIDTH) {
				printf("\n");
				printidx = 0;
			}
		}
		if (printidx != 0)
			printf("\n");
	}
	return;
}


int
main(int argc, char **argv) 
{
	INUM_T	inum;
    int     ch;
    char   *fstype = DEF_FSTYPE;
	FS_INODE *fs_inode;
	FS_INFO		*fs;
    progname = argv[0];

    while ((ch = getopt(argc, argv, "b:f:vVz:")) > 0) {
	switch (ch) {
	default:
	    usage();
    case 'b':
		numblock = atoi(optarg);
		if (numblock < 0) {
			printf("invalid argument: must be positive: %d\n", numblock);
			usage();
		}
		break;
	case 'f':
	    fstype = optarg;
	    break;
	case 'v':
	    verbose++;
	    logfp = stderr;
	    break;
	case 'V':
	    verbose++;
	    logfp = stdout;
	    break;
	case 'z':
		{
		char envstr[32];
		snprintf(envstr, 32, "TZ=%s", optarg);
		if (0 != putenv(envstr)) {
				error ("error setting environment");
		}

		tzset();
		}
		break;

	}
    }

    if ((optind+2) != argc)
		usage();

	/*
     * Open the file system.
     */
    fs = fs_open(argv[optind], fstype);
	inum = atoinum(argv[++optind]);

	fs_inode = fs->inode_lookup(fs, inum);
	if (!fs_inode)
		error ("error looking up inode");

	print_inode(fs, inum, fs_inode);

    fs->close(fs);
    exit(0);
}
