/*
** bcat
** TCTUTILs Package
**
** Given an image , block number, and size, display the contents
** of the block to stdout.
** 
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.3 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
** 
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR 
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"
#include <ctype.h>

/* local flags */
#define HEX 0x1
#define ASCII 0x2
#define HTML 0x4
#define STAT 0x8

#define SWAP_TYPE "swap"
#define PAGESZ	4096

void 
usage(char *myProg)  
{
	printf("usage: %s [-ahw] [-f fstype] image block_num bytes\n", myProg);
	printf("\t-a displays in all ASCII \n");
	printf("\t-f use fstype of fs (only swap currently supported)\n");
	printf("\t-h displays in hexdump-like fashion\n");
	printf("\t-s display basic block stats such as block size, fragments, etc.\n");
	printf("\t-w displays in web-like (html) fashion\n");
	printf("\t\tblock_num is used as page_num with 4096 byte pages\n");
	exit(1);
}

void 
stats (FS_INFO *fs) 
{
	if (!fs)
		return;
	printf("%d: Block Size\n", fs->block_size);
	printf("%d: File Block Size\n", fs->file_bsize);
	printf("%d: Indir Addr Block Size\n", fs->addr_bsize);
	printf("%d: Num Frags\n", fs->block_frags);
}

FILE *logfp;

int 
main(int argc, char **argv) 
{
	FS_INFO *fs = NULL;
	DADDR_T block;
	char   *fstype = DEF_FSTYPE;
	int size;
	FS_BUF *buf;
	char format = 0;
	char ch;
	extern int optind;
	FILE *hSwap = NULL;

	while ((ch = getopt(argc, argv, "ahwf:s")) > 0) {
		switch (ch) {
		case 'a':
			format |= ASCII;
			break;
		case 'h':
			format |= HEX;
			break;
		case 'w':
			format |= HTML;
			break;
		case 'f':
			fstype = optarg;
			break;
		case 's':
			format |= STAT;
			break;
		case '?':
		default:
			usage(argv[0]);
		}
	}

	if (format & STAT) {
		if (optind + 1 != argc)
			usage(argv[0]);
		if (strcmp(SWAP_TYPE, fstype) == 0) {
			printf("swap type not supported for stat option\n");
			usage(argv[0]);
		}
	}
	else if (optind + 3 != argc) 
		usage(argv[0]);

	if ((format & ASCII) && (format & HEX)) {
		printf("Ascii and Hex flags can not be used together\n");
		usage(argv[0]);
	}

	/* open appropriate file */
	if (strcmp(SWAP_TYPE, fstype) == 0) {
		if (NULL == (hSwap = fopen(argv[optind++], "r")))
			error("error opening: %s\n", argv[optind-1]);
	}
	else {
		fs = fs_open(argv[optind++], fstype);
	}

	if (format & STAT) {
		stats(fs);
		fs->close(fs);
		return 0;
	}

	block = atoi(argv[optind++]);
	size = atoi(argv[optind++]);

	if (format & HTML) {
		printf("<HTML>\n");
		printf("<HEAD>\n");
		printf("<TITLE>%s   Block: %i   Size: %i bytes</TITLE>\n",
			argv[optind-3], block, size);
		printf("</HEAD>\n");
		printf("<BODY>\n"); 

	}

	if (strcmp(SWAP_TYPE, fstype) != 0) {
#if defined (HAVE_UFS_FFS)
		if (size < 0 || size > MAXBSIZE) 
#elif defined (HAVE_EXT2FS)
		if (size < 0 || size > EXT2_MAX_BLOCK_SIZE) 
#endif
		{
			error("Invalid size: %i\n", size);
		} 
		else if (size % DEV_BSIZE) {
			error ("Size is not multiple of %i\n", DEV_BSIZE);
		}
	}

	buf = fs_buf_alloc(size);

	/* Read the data */
	if (strcmp(SWAP_TYPE, fstype) == 0) {
		if(fseek(hSwap, PAGESZ*block, SEEK_SET)) 
			error ("error seeking to: %d\n", PAGESZ*block);

		if (size != fread(buf->data, 1, size, hSwap)) 
			error ("error reading file\n");
	}
	else {
		fs_read_block(fs, buf, size, block, "");
	}

	/* do a hexdump like printout */
	if (format & HEX) {
		unsigned int idx1, idx2;

		if (format & HTML) 
			printf("<TABLE BORDER=0>\n");

		for (idx1 = 0; idx1 < size; idx1+=16) {
			if (format & HTML) 
				printf("<TR><TD>%i</TD>", idx1);
			else
				printf("%i\t", idx1);
			

			for (idx2 = 0; idx2 < 16; idx2++) {
				if ((format & HTML) && (0 == (idx2%4)) )
					printf("<TD>");

				printf("%.2x", buf->data[idx2+idx1] & 0xff);

				if (3 == (idx2 % 4)) {
					if (format & HTML) 
						printf("</TD>");
					else
						printf(" ");
				}
			}

			printf("\t");
			for (idx2 = 0; idx2 < 16; idx2++) {
				if ((format & HTML) && (0 == (idx2%4)) )
					printf("<TD>");

				if ((isascii(buf->data[idx2+idx1])) && 
				  (!iscntrl(buf->data[idx2+idx1])))
					printf("%c", buf->data[idx2+idx1]);
				else
					printf(".");

				if (3 == (idx2 % 4)) {
					if (format & HTML) 
						printf("</TD>");
					else
						printf(" ");
				}
			}

			if (format & HTML) 
				printf("</TR>");

			printf("\n");
		}


		if (format & HTML) 
			printf("</TABLE>\n");
		else
			printf("\n");

	} /* end of if hexdump */

	/* print in all ASCII */
	else if (format & ASCII) {
		int iIdx; 
		for (iIdx = 0; iIdx < size; iIdx++) {

			if ((isprint(buf->data[iIdx])) || (buf->data[iIdx] == '\t')) {
				printf("%c", buf->data[iIdx]);
			}
			else if ((buf->data[iIdx] == '\n') || (buf->data[iIdx] == '\r')) {
				if (format & HTML) 
					printf("<BR>");
				printf("%c", buf->data[iIdx]);
			}
			else
				printf(".");
		}
		if (format & HTML) 
			printf("<BR>");

		printf("\n");	
	}

	/* print raw */
	else  {
		if (fwrite(buf->data, size, 1, stdout) != 1)
			error("write: %m");

		if (format & HTML) 
			printf("<br>\n");
		else
			printf("\n");
	} 

	fs_buf_free(buf);

	if (strcmp(SWAP_TYPE, fstype) == 0) 
		fclose(hSwap);
	else 
		fs->close(fs);

	if (format & HTML) 
		printf("</BODY>\n</html>\n");

	return 0;
}

