/*
** ext2_dent
** TCTUTILs package
**
** Manipulate directory entries on an EXT2FS
**
** This file requires The Coroners Toolkit (TCT) 
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.4 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.  
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "fs_tools.h"
#include "fs_dent.h"
#include "mymalloc.h"
#include "error.h"

#ifdef DIR_SUPPORT
#ifdef HAVE_EXT2FS


typedef struct ext2_dir_entry_2 dir_entry;

#define MAX_NAM_DIR	EXT2_NAME_LEN

void 
ext2_dent_alloc(FS_DENT *fs_dent) {
	fs_dent->name = (char *)mymalloc(MAX_NAM_DIR+1);
	if (!fs_dent->name)
		fprintf(stderr, "ext2_dent_alloc: error allocating name space\n");

	fs_dent->maxnamlen = MAX_NAM_DIR;
	fs_dent->fs_type = FS_DENT_EXT2;
}

static void 
ext2_dent_copy(dir_entry *ext2_dent, FS_DENT *fs_dent, FS_INFO *fs) 
{
	fs_dent->inode = ext2_dent->inode;
	fs_dent->nextLink = ext2_dent->rec_len;

	fs_dent->reclen = EXT2_DIR_REC_LEN(ext2_dent->name_len);

	fs_dent->namlen = ext2_dent->name_len;

	if (fs_dent->fs_type != FS_DENT_EXT2) 
		fs_dent_realloc(fs_dent, FS_DENT_EXT2);

	/* ext2 does not null terminate */
	if (ext2_dent->name_len > fs_dent->maxnamlen) 
		error("ext2_dent_copy: Name Space too Small");

	/* Copy and Null Terminate */
	strncpy(fs_dent->name, ext2_dent->name, ext2_dent->name_len);
	fs_dent->name[ext2_dent->name_len] = '\0';	

	if ((fs != NULL) && (fs_dent->inode)) {
		/* Get inode */
		fs_dent->fsi = fs->inode_lookup(fs, fs_dent->inode);

		/* MAKE THIS MORE PORTABLE!! */
		fs_dent->ent_type = (fs_dent->fsi->mode & 0xf000) >> 12;

	} 
	else {
		fs_dent->fsi = NULL;
		fs_dent->ent_type = FS_DENT_UNDEF;
	}
}


/* 
**
** Read contents of directory block
**
** if entry is active call actionAct, if it is deleted then call actionDel
** iTotalLen is the total length of blocks to read left
*/
static int 
ext2_dent_parse_block(FS_INFO *fs, FS_BUF *buf, int iTotalLen,
  int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, char *ptr) 
{

	int idellen = 0;
	int idx, reclen, size;
	int inode;
	dir_entry *dirPtr;
	FS_DENT *fs_dent;

	//size = MIN(iTotalLen, buf->size);
	size = buf->size;

	fs_dent = fs_dent_alloc(FS_DENT_EXT2);

	/* update each time by the actual length instead of the
	** recorded length so we can view the deleted entries 
	*/
	for (idx = 0; idx < size; idx += reclen)  {

		int namelen;
		dirPtr = (dir_entry *) &buf->data[idx];

		reclen = EXT2_DIR_REC_LEN(dirPtr->name_len);
		inode = dirPtr->inode;
		namelen = dirPtr->name_len;

		/* 
		** Check if we may have a valid direct entry.  If we don't
		** then increment to the next word and try again.  
		**
		** although inodes can really only be >= 2, we have to check
		** for >= 0 to find deleted ones
		**
		*/
		if ((inode > fs->last_inum) || 
		   (inode < 0) || 
		   (namelen > MAX_NAM_DIR) || 
		   (namelen <= 0) || 
		   (dirPtr->rec_len < EXT2_DIR_REC_LEN(1)) ||
		   (dirPtr->rec_len % 4) ||
		   (dirPtr->rec_len < namelen + 8) ) {

			reclen = 4;
			if (idellen > 0)
				idellen -= 4;
			continue;
		}

		/* If the inode value is 0 and it is the first entry, then 
		** it has been deleted so print it out.
		** Otherwise inodes of 0 should not be in the middle of
		** the block with FFS.  They will exist for deleted entries
		** under EXT2FS though, so make a special conditional 
		** using name length for it
		*/
		if ((inode > 1) || ((inode == 0) && 
		  ((idx == 0) || (namelen > 0)) ) )  {
			ext2_dent_copy(dirPtr, fs_dent, fs);

			/* Do we have a deleted entry? */
			if ((idellen > 0) || (inode == 0)) {
				actionDel(fs_dent, flags, ptr);
				idellen -= reclen;
			}
			/* We have a non-deleted entry */
			else {
				actionAct(fs_dent, flags, ptr);
			}

			/* If the actual length is shorter then the 
			** recorded length, then the next entry(ies) have been 
			** deleted.  Set idellen to the length of data that 
			** has been deleted
			**
			** Because we aren't guaranteed with Ext2FS that the next
			** entry begins right after this one, we will check to
			** see if the difference is less than a possible entry
			** before we waste time searching it
			*/
			if ((dirPtr->rec_len - reclen >= EXT2_DIR_REC_LEN(1)) 
			  && (idellen <= 0))
				idellen = dirPtr->rec_len - reclen;

		} /* end of if for inode == 0 etc */
		else {
			/* We are here because the inode was 0 and namelen was 0
			** increment to next word and try again
			*/
			reclen = 4;
		}

	} /* end for */
	return size;

} /* end ext2_dent_parse_block() */


/* 
** given the pointer (_addr_) to an array of _num_ direct addresses,
** read the block into buf[0] and pass it to parse_block
**
** return the size of the file still left to read
*/
static int
ext2_dent_walk_direct (FS_INFO *fs, FS_BUF *buf[], DADDR_T *addr, int num,
  int size, int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, 
  char *ptr) 
{
	unsigned int i;

	for (i = 0; i < num && size > 0; i++) {
		/* read block full block so that we can look at all possible 
		** entries*/
		fs_read_block(fs, buf[0], buf[0]->size, 
		  addr[i], "");

		size -= ext2_dent_parse_block(fs, buf[0], size, flags, 
		  actionAct, actionDel, "");

	} /* end for direct */

	return size;
}

/*
** given the address of an array of level _level_  indirect addresses,
** read the array and redirect it one more level or call __direct if 
** level 1
**
** return the size of file still left
*/
static int
ext2_dent_walk_indirect (FS_INFO *fs, FS_BUF *buf[], DADDR_T *addr, int level,
  int size, int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, 
  char *ptr) 
{ 
    DADDR_T *iaddr;
    int i,num;

	if (!addr)
		return 0;

    fs_read_block(fs, buf[level], buf[level]->size, (DADDR_T)addr, "");

    iaddr = (DADDR_T *)buf[level]->data;
	num = buf[level]->size/sizeof(*iaddr);

    if (level == 1) {
		size = ext2_dent_walk_direct (fs, buf, iaddr, num, size, flags,
		  actionAct, actionDel, ptr);
    }
    else {
        for (i = 0; size > 0 && i < num; i++) {
			size = ext2_dent_walk_indirect (fs, buf, iaddr, level-1, size, flags,
			  actionAct, actionDel, ptr);
        }
    }

	return size;
}

/* 
** The main function to do directory entry walking
**
** actionAct is called on active directory entries and actionDel is called
** on deleted entries
**
** this calls ext2_dent_walk_direct and _indirect
*/
void 
ext2_dent_walk(FS_INFO *fs, INUM_T inode, int flags, 
  FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, char *ptr) 
{
	FS_BUF **buf;
	int size, i;
	FS_INODE *fs_inode;

	if (inode < 2)
		error("invalid inode value: %i\n", inode);

	fs_inode = fs->inode_lookup(fs, inode);
	size = fs_inode->size;

	/* Allocate a buffer for each level of indirection */
	buf = (FS_BUF **) mymalloc(sizeof(*buf) * (fs_inode->indir_count+1));

	buf[0] = fs_buf_alloc(fs->block_size);
	for (i = 1; i < fs_inode->indir_count; i++)
		buf[i] = fs_buf_alloc(fs->addr_bsize);

	/* 
	** Do the Direct Blocks First 
	*/
	size = ext2_dent_walk_direct (fs, buf, fs_inode->direct_addr, 
	  fs_inode->direct_count, size, flags, actionAct, actionDel, ptr);

	/* Do we need indirects too? */
    if (size > 0) {
		for (i = 1; size > 0 && i <= fs_inode->indir_count; i++) {

			size = ext2_dent_walk_indirect (fs, buf, 
			  (DADDR_T *)fs_inode->indir_addr[i-1], i, size, flags,
			  actionAct, actionDel, ptr);
		}

	} /* end of indirect */


	/* Free up memory */
	for (i = 0; i < fs_inode->indir_count; i++)
		fs_buf_free(buf[i]);
	free (buf);
}

#endif /* HAVE_EXT2 */
#endif /* DIR_SUPPORT */
