/*
** ffs_dent
** TCTUTILs package
**
** Manipulate direct entries on an FFS fs with fs_dent
**
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.4 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"

#ifdef DIR_SUPPORT
#ifdef HAVE_UFS_FFS

#if defined(OPENBSD2)
#include <ufs/ufs/dir.h>
#endif


#include "fs_dent.h"
#include "mymalloc.h"
#include "error.h"

typedef struct direct dir_entry;

void 
ffs_dent_alloc(FS_DENT *fs_dent) 
{
	fs_dent->name = (char *)mymalloc(MAXNAMLEN);
	fs_dent->maxnamlen = MAXNAMLEN;
	fs_dent->fs_type = FS_DENT_FFS;
}

/* 
** copy OS specific directory inode to generic FS_DENT
*/
static void 
ffs_dent_copy(struct direct *ffs_dent, FS_DENT *fs_dent, FS_INFO *fs) 
{
	fs_dent->inode = ffs_dent->d_ino;
	fs_dent->nextLink = ffs_dent->d_reclen;

#if defined(OPENBSD2)
    fs_dent->reclen = DIRSIZ(0, ffs_dent);
#elif defined(SUNOS5)
	fs_dent->reclen = DIRSIZ(ffs_dent);
#endif

	if ((fs != NULL)  && (fs_dent->inode)) {
		/* Get inode */
		fs_dent->fsi = fs->inode_lookup(fs, fs_dent->inode);

		switch (fs_dent->fsi->mode & IFMT) {
		  case IFIFO:
			fs_dent->ent_type = FS_DENT_FIFO;
			break;
		  case IFCHR:
			fs_dent->ent_type = FS_DENT_CHR;
			break;
		  case IFDIR:
			fs_dent->ent_type = FS_DENT_DIR;
			break;
		  case IFBLK:
			fs_dent->ent_type = FS_DENT_BLK;
			break;
		  case IFREG:
			fs_dent->ent_type = FS_DENT_REG;
			break;
		  case IFLNK:
			fs_dent->ent_type = FS_DENT_LNK;
			break;
		  case IFSOCK:
			fs_dent->ent_type = FS_DENT_SOCK;
			break;
#if defined (OPENBSD2)
		  case IFWHT:
			fs_dent->ent_type = FS_DENT_WHT;
			break;
#endif
#if defined (SUNOS5)
		  case IFSHAD:
			fs_dent->ent_type = FS_DENT_SHAD;
			break;
#endif
		  default:
			fs_dent->ent_type = FS_DENT_UNDEF;
		}
	}
	else {
		fs_dent->ent_type = FS_DENT_UNDEF;
		fs_dent->fsi = NULL;
	}

	fs_dent->namlen = ffs_dent->d_namlen;

	if (fs_dent->fs_type != FS_DENT_FFS) 
		fs_dent_realloc(fs_dent, FS_DENT_FFS);

	/* ffs null terminates so we can strncpy */
	strncpy(fs_dent->name, ffs_dent->d_name, fs_dent->maxnamlen);
}


/* Scan all direct entries and if the entry is deleted call actionDel
** if the entry is active then call actionAct
**
** iTotalLen is the total size of entries to read
**
** return how much was read this time
*/
static int 
ffs_dent_parse_block (FS_INFO *fs, FS_BUF *buf, int iTotalLen,
  int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, char *ptr) 
{
	unsigned int idx;
	int inode, dellen = 0, reclen = 0, size;

	dir_entry *dirPtr;
	FS_DENT *fs_dent;

	size = buf->size;

	fs_dent = fs_dent_alloc(FS_DENT_FFS);

	/* update each time by the actual length instead of the
	** recorded length so we can view the deleted entries 
	*/
	for (idx = 0; idx < size; idx += reclen) {
		int namelen;
		dirPtr = (dir_entry *)&buf->data[idx];

#if defined(OPENBSD2)
		reclen = DIRSIZ(0, dirPtr);
#elif defined(SUNOS5)
		reclen = DIRSIZ(dirPtr);
#endif

		inode = dirPtr->d_ino;
		namelen = dirPtr->d_namlen;

		/* Perform a couple sanity checks 
		** OpenBSD never zeros the inode number, but solaris
		** does.  These checks will hopefully catch all non
		** entries */
		if ((inode > fs->last_inum) ||
		   (inode < 0) ||
		   (namelen > MAXNAMLEN) || 
		   (namelen <= 0) || 
		   (dirPtr->d_reclen <= 0) ||
		   (dirPtr->d_reclen % 4) ) {
		
			reclen = 4;
			if (dellen > 0)
				dellen -= 4;
			continue;
		}

		ffs_dent_copy(dirPtr, fs_dent, fs);

		/* Do we have a deleted entry? */
		if ((dellen > 0) || (inode == 0)) {
			actionDel (fs_dent, flags, ptr);
			dellen -= reclen;
		} 
		else {
			actionAct (fs_dent, flags, ptr);
		}

		/* If we have some slack, the set dellen */
		if ((dirPtr->d_reclen != reclen) && (dellen <= 0))
			dellen = dirPtr->d_reclen - reclen;

	} /* end for size */

	return size;

} /* end ffs_dent_parse_block */



/*
** given the pointer (_addr_) to an array of _num_ direct addresses,
** read the block into buf[0] and pass it to parse_block
**
** return the size of the file still left to read
*/
static int
ffs_dent_walk_direct (FS_INFO *fs, FS_BUF *buf[], DADDR_T *addr, int num,
  int size, int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel,
  char *ptr)
{
    unsigned int i, nchnk, cidx;

    /* Directory entries are written in chunks of DIRBLKSIZ
    ** determine how many chunks of this size we have to read to
    ** get a full block
    **
    ** Entries do not cross over the DIRBLKSIZ boundary
    */
    nchnk = (fs->block_frags * fs->block_size) / DIRBLKSIZ;

    for (i = 0; i < num && size > 0; i++) {
        for (cidx = 0; cidx < nchnk && size > 0; cidx++) {

            /* read full directory chunk so that all entries can
			** be analyzed
			*/
            fs_read_random(fs, buf[0]->data, buf[0]->size,
              (addr[i]*fs->block_size + cidx*DIRBLKSIZ), ptr);

            size -= ffs_dent_parse_block(fs, buf[0], size, flags,
              actionAct, actionDel, "");

        } /* end for fragment */
    } /* end for direct */

    return size;
}

/*
** given the address of an array of level _level_  indirect addresses,
** read the array and redirect it one more level or call __direct if
** level 1
**
** return the size of file still left
*/
static int
ffs_dent_walk_indirect (FS_INFO *fs, FS_BUF *buf[], DADDR_T *addr, int level,
  int size, int flags, FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel,
  char *ptr)
{
    DADDR_T *iaddr;
    int i,num;

    if (!addr)
        return 0;

    fs_read_block(fs, buf[level], buf[level]->size, (DADDR_T)addr, "");

    iaddr = (DADDR_T *)buf[level]->data;
    num = buf[level]->size/sizeof(*iaddr);

    if (level == 1) {
        size = ffs_dent_walk_direct (fs, buf, iaddr, num, size, flags,
          actionAct, actionDel, ptr);
    }
    else {
        for (i = 0; size > 0 && i < num; i++) {
            size = ffs_dent_walk_indirect (fs, buf, iaddr, level-1, size, flags,
              actionAct, actionDel, ptr);
        }
    }

    return size;
}

/* Process _inode_ as a directory inode and process the data blocks
** as file entries.  Call actionDel on deleted entries and actionAct
** on active entries
**
*/
void 
ffs_dent_walk(FS_INFO *fs, INUM_T inode, int flags, 
  FS_DENT_WALK_FN actionAct, FS_DENT_WALK_FN actionDel, char *ptr) 
{
	FS_BUF **buf;
	int  size, i;
	FS_INODE *fs_inode;

	if (inode < 2)
		error("invalid inode value: %i\n", inode);

	fs_inode = fs->inode_lookup(fs, inode);
	size = fs_inode->size;

    /* Allocate a buffer for each level of indirection */
    buf = (FS_BUF **) mymalloc(sizeof(*buf) * (fs_inode->indir_count+1));

    buf[0] = fs_buf_alloc(DIRBLKSIZ);
    for (i = 1; i < fs_inode->indir_count; i++)
        buf[i] = fs_buf_alloc(fs->addr_bsize);

	/* 
	** Do the Direct Blocks First 
	*/
	size = ffs_dent_walk_direct (fs, buf, fs_inode->direct_addr,
      fs_inode->direct_count, size, flags, actionAct, actionDel, ptr);

    /* Do we need indirects too? */
    if (size > 0) {
        for (i = 1; size > 0 && i <= fs_inode->indir_count; i++) {

            size = ffs_dent_walk_indirect (fs, buf,
              (DADDR_T *)fs_inode->indir_addr[i-1], i, size, flags,
              actionAct, actionDel, ptr);
        }

    } /* end of indirect */

    /* Free up memory */
    for (i = 0; i < fs_inode->indir_count; i++)
        fs_buf_free(buf[i]);
    free (buf);
}

#endif /* HAVE_UFS_FFS */
#endif /* DIR_SUPPORT */
