/*
** istat
** TCTUTILs package
**
**
** Display all inode info about a given inode.  This is a more verbose 
** version of 'ils -a' in TCT.  The body of this program was built around
** it (ils.c).
**
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.3 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"
#include "mymalloc.h"

FILE   *logfp;
static FS_INFO *fs;

/* atoinum - convert string to inode number */
INUM_T  
atoinum(const char *str)
{
    char   *cp;
    INUM_T  inum;

    if (*str == 0)
	return (0);
    inum = STRTOUL(str, &cp, 0);
    if (*cp || cp == str)
	error("bad inode number: %s", str);
    return (inum);
}

/* usage - explain and terminate */

static void usage() {
    error("usage: %s [-f fstype] [-z hour] [-vV] image inum", progname);
}

/* global value so that there are always 8 blocks wide */
static int printidx = 0;

/* difference in hours between current timezone and that of image
** for example if image is from Central and we are examining on
** Eastern then the diff would be -1
*/
static signed int zonediff = 0;

/* Print the block numbers blocks in the direct inode format */
static int 
print_direct(DADDR_T *addr, int num, int size) 
{
	int i;
	for (i=0; i<num && size > 0; i++) {
		printf("%lu ", (unsigned long) addr[i]);
		if (++printidx == 8) {
	   		printf("\n");
			printidx = 0;
		}
		size -= fs->file_bsize;
	}
	return (size<0?0:size);
}

/* cycle through the indirect inodes and call _direct when that
** level is reached.  
** the level argument is 0 for the first level of indirection and
** one larger for each higher level
*/
static int 
print_indirect(FS_BUF *buf[], DADDR_T addr, int level, int size) 
{
	DADDR_T *iaddr;
	int i;

	if (!addr)
		return 0;

	fs_read_block(fs, buf[level], buf[level]->size, addr, "");

	iaddr = (DADDR_T *)buf[level]->data;
	if (level == 0) {
		size = print_direct(iaddr,buf[level]->size/sizeof(*iaddr), size);
	}
	else {
		for (i=0; size > 0 && i < buf[level]->size/sizeof(*iaddr); i++) {
			size = print_indirect(buf, iaddr[i], level-1, size);
		}
	}

	return size;
}

/* print_inode - list generic inode */
static void 
print_inode(INUM_T inum, FS_INODE *fs_inode, int flags, char *ptr) 
{
	struct tm *tmTime;
	int  size, level;
	FS_BUF **buf;
	short m, s;
	char x;
	time_t timetmp; 
	signed int zonesec;

	printf("inode: %lu\n", (ULONG) inum);
	printf("uid / gid: %d / %d\n", (int)fs_inode->uid, (int)fs_inode->gid);

	/* The permissions are on the last 3 nibbles as such:
	** There are 4 groups of 3 bits.  The gsb is for Setuid, the next
	** grp is own, next grp is grp, and last grp for other
	**
	** for setuid the bits are: user,grp,text
	** for rwx the bits are rwx ..
	**
	*/
	printf("mode: ");
	s = (fs_inode->mode & 0xe00) >> 9;

	m = (fs_inode->mode & 0x1c0) >> 6;
	if (s&4) 
		x = m&1 ? 's':'S';
	else
		x = m&1 ? 'x':'-';
	printf("%c%c%c", m&4?'r':'-', m&2?'w':'-', x);

	m = (fs_inode->mode & 0x038) >> 3;
	if (s&2) 
		x = m&1 ? 's':'S';
	else
		x = m&1 ? 'x':'-';
	printf("%c%c%c", m&4?'r':'-', m&2?'w':'-', x);

	m = fs_inode->mode & 0x007;
	if (s&1) 
		x = m&1 ? 't':'T';
	else
		x = m&1 ? 'x':'-';
	printf("%c%c%c\n", m&4?'r':'-', m&2?'w':'-', x);

	printf("size: %lu\n", (ULONG) fs_inode->size);
	printf("num of links: %lu\n", (ULONG) fs_inode->nlink);

	zonesec = zonediff * 3600;

	timetmp = fs_inode->mtime + zonesec;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);

	printf("Modified:\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s%s%d)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[0], (zonediff<0)?"":"+", zonediff);

	timetmp = fs_inode->atime + zonesec;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);

	printf("Accessed:\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s%s%d)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[0], (zonediff<0)?"":"+", zonediff);

	timetmp = fs_inode->ctime + zonesec;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);

	printf("Created:\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s%s%d)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[0], (zonediff<0)?"":"+", zonediff);

#ifdef HAVE_DTIME
	timetmp = fs_inode->dtime + zonesec;
	if (timetmp < 0)
		timetmp = 0;
	tmTime = localtime(&timetmp);
	printf("Deleted:\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d\t(%s%s%d)\n",
		tmTime->tm_mon+1, tmTime->tm_mday, tmTime->tm_year+1900,
		tmTime->tm_hour, tmTime->tm_min, tmTime->tm_sec,
		tzname[0], (zonediff<0)?"":"+", zonediff);
#endif

	size = fs_inode->size;

	printf ("Blocks:\n");
	size = print_direct(fs_inode->direct_addr, fs_inode->direct_count, size);

	if (size <= 0) {
		printf("\n");
		return;
	}

	/* allocate */
	buf = (FS_BUF **) mymalloc(sizeof(*buf) * fs_inode->indir_count);
	for (level=0; level < fs_inode->indir_count; level++)
		buf[level] = fs_buf_alloc(fs->addr_bsize);

	for (level=0; size > 0 && level < fs_inode->indir_count; level++) {
		size = print_indirect(buf, fs_inode->indir_addr[level], 
			level, size); 
	}

	printf("\n");

	/* Cleanup */
	for (level=0; level < fs_inode->indir_count; level++)
		fs_buf_free(buf[level]);
}


int
main(int argc, char **argv) 
{
	INUM_T	inode;
    int     ch;
    char   *fstype = DEF_FSTYPE;
    progname = argv[0];

    while ((ch = getopt(argc, argv, "f:vVz:")) > 0) {
	switch (ch) {
	default:
	    usage();
	case 'f':
	    fstype = optarg;
	    break;
	case 'v':
	    verbose++;
	    logfp = stderr;
	    break;
	case 'V':
	    verbose++;
	    logfp = stdout;
	    break;
	case 'z':
		zonediff = atoi(optarg);
		break;
	}
    }

    if ((optind+2) != argc)
		usage();

    /*
     * Open the file system.
     */
    fs = fs_open(argv[optind], fstype);
	inode = atoinum(argv[++optind]);
	fs->inode_walk(fs, inode, inode, ~0, print_inode, NULL);

    fs->close(fs);
    exit(0);
}
