/*
** find_file
** TCTUTILs Package
**
** Find the file that uses the specified inode (including deleted files)
** 
** This file requires The Coroners Toolkit (TCT) 
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.4 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/
#include "fs_tools.h"
#include "error.h"
#include "fs_dent.h"

void 
usage(char *myProg) 
{
	printf("usage: %s [-adu] image inode\n", myProg);
	printf("\t-a: Find all occurrences\n");
	printf("\t-d: Find deleted entries ONLY\n");
	printf("\t-u: Find undeleted entries ONLY\n");
	exit(1);
}

FILE *logfp;

static FS_INFO *fs;

static char cDirType;

/* local flags */
static unsigned char localflags;
#define FIND_ALL 0x1
#define SKIP_UND 0x2
#define SKIP_DEL 0x4

static unsigned int inode = 0;
static unsigned char found = 0;


static unsigned int depth = 0;  /* how deep in the directory tree are we */
#define MAX_DEPTH   32
static char didx[MAX_DEPTH];  /* index in dirs string to where '/' is for
                        ** given depth */
#define DIR_STRSZ   256
static char dirs[DIR_STRSZ];    /* The current directory name string */

static void 
find_del_file (FS_DENT *fs_dent, int flags, char *ptr) 
{

	/* We found it! */
	if (fs_dent->inode == inode) {
		found = 1;
		if (!(localflags & SKIP_DEL)) {
			printf("* %s%s\n", dirs, fs_dent->name);

			if (!(localflags & FIND_ALL)) {
				fs->close(fs);
				exit(0);
			}
		}
	}
	return;
}

static void 
find_act_file (FS_DENT *fs_dent, int flags, char *ptr) 
{
	/* We found it! */
	if (fs_dent->inode == inode) {
		found = 1;
		if (!(localflags & SKIP_UND)) {
			printf("%s%s\n", dirs, fs_dent->name);

			if (!(localflags & FIND_ALL)) {
				fs->close(fs);
				exit(0);
			}
		}
	}

	if ((!ISDOT (fs_dent->name)) && (fs_dent->ent_type & FS_DENT_DIR)) {
		/* go down a level */
		if (depth < MAX_DEPTH) {
			didx[depth] = strlen(dirs);
			strncpy(&dirs[(int)didx[depth]], fs_dent->name,
			DIR_STRSZ - (int)didx[depth]);
			strncat(dirs, "/", DIR_STRSZ);
		}
		depth++;

		fs_dent_walk(fs, fs_dent->inode, flags, find_act_file, 
		   find_del_file, (char *)0, cDirType);

		depth--;
		if (depth < MAX_DEPTH)
			dirs[(int) didx[depth]] = '\0';

	}

}


int 
main(int argc, char **argv)
{
	char *fstype = DEF_FSTYPE;
	int flags = FS_FLAG_LINK | FS_FLAG_UNLINK;
	char ch;
	extern int optind;

	while ((ch = getopt(argc, argv, "adfu")) > 0) {
		switch (ch) {
		case 'a':
			localflags |= FIND_ALL;
			break;
		case 'd':
			localflags |= SKIP_UND;
			break;
		case 'u':
			localflags |= SKIP_DEL;
			break;
		case '?':
		default: 
			usage(argv[0]);
		}
	}

	if ((localflags & SKIP_DEL) && (localflags & SKIP_UND))  {
		printf("-d and -u must not be used together\n");
		usage(argv[0]);
	}
	else if (optind+2 != argc)
		usage(argv[0]);


#if defined (HAVE_UFS_FFS)
	cDirType = FS_DENT_FFS;
#endif
#if defined (HAVE_EXT2FS)
	cDirType = FS_DENT_EXT2;
#endif

	inode = atoi(argv[optind+1]);

	/* Since we start the walk on inode 2, then this will not show
	** up in the above functions, so do it now
	*/
	if (inode == 2) {
		if (!(localflags & SKIP_UND) ) {
			printf("/\n");

			if (!(localflags & FIND_ALL)) 
				return 0;
		}
	}

	/* open image */
	fs = fs_open(argv[optind], fstype);

	found = 0;
	strncpy(dirs, "/", DIR_STRSZ);

	fs_dent_walk(fs, 2, flags, find_act_file, find_del_file, 
	  (char *)0, cDirType);

	fs->close(fs);

	if (!found)
		printf("inode not currently used\n");

	return 0;
}

