/*
** find_inode
** TCTUTILs Package
**
** Given an image  and block number, identify which inode it is used by
** If the entry can not be found in an inode, then determine which
** inode it most likely is from 
** 
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.3 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"
#include "mymalloc.h"

static DADDR_T block;
static DADDR_T closeblock = 0;
static INUM_T closeinode = 0;
static INUM_T curinode;

static FS_INFO *fs;
static FS_INODE *fs_inode;

FILE *logfp;


/*
** Go through direct blocks to find the one we want
**
** return size left 
** exit when block has been found
**
** Look for exact block number in the inodes.  
** It will not be there if it is not used or if it is in a larger block.
** So, save the inode number that has a block closest to our block with
** the following properties:
** 	-It  must be less than the one we want
**	-It must be no further away than the possible number of fragments
**	-It can not cross fragment boundaries
*/
static int 
find_direct(DADDR_T *addr, int num, int size) 
{
	int i;
	for (i=0; i<num && size > 0; i++) {
		if (addr[i] == block) {
			printf("%i\n", (int)curinode); 
			fs->close(fs);
			exit (0);
		}
		/* Does this block meet the requirements from above? */
		else if ((fs_inode->direct_addr[i] < block) && 
 		  (fs_inode->direct_addr[i] > closeblock) &&
		  (block - fs_inode->direct_addr[i] < fs->block_frags) &&
		  ((fs_inode->direct_addr[i] / fs->block_frags) == 
		  (block / fs->block_frags ))) {
			closeblock = fs_inode->direct_addr[i];
			closeinode = curinode;
		}
		size -= fs->file_bsize;
	}
	return (size < 0 ? 0 : size);
}

/*
** Go through indirect entries and call _direct
**
** level is 0 on the first level of indirection
*/
static int 
find_indirect(FS_BUF *buf[], DADDR_T addr, int level, int size) 
{
    DADDR_T *iaddr;
	int i;

	if (!addr)
		return 0;

	fs_read_block(fs, buf[level], buf[level]->size, addr, "");

	iaddr = (DADDR_T *)buf[level]->data;
	if (level == 0) {
		size = find_direct(iaddr,buf[level]->size/sizeof(*iaddr), size);
	}
	else {
		for (i=0; size > 0 && i < buf[level]->size/sizeof(*iaddr); i++) {
			size = find_indirect(buf, iaddr[i], level-1, size);
		}
	}
	return size;
}


/*
** find_block
**
** Main function that calls the direct and indirect calls
*/
static void 
find_block(INUM_T inum, FS_INODE *fsi, int flags, char *ptr) 
{
	FS_BUF  **buf;
	int size, level;

	curinode = inum;
	fs_inode = fsi;

	size = find_direct(fsi->direct_addr, fsi->direct_count, fsi->size);
	if (size <= 0)
		return;

	/* allocate */
	buf = (FS_BUF **) mymalloc(sizeof(*buf) * fsi->indir_count);
	for (level=0; level < fsi->indir_count; level++)
		buf[level] = fs_buf_alloc(fs->addr_bsize);

	for (level=0; size > 0 && level < fsi->indir_count; level++) {
		size = find_indirect(buf, fsi->indir_addr[level],
		  level, size);
	}

	/* Cleanup */
	for (level=0; level < fsi->indir_count; level++)
		fs_buf_free(buf[level]);
}

int
main(int argc, char **argv) 
{
	char   *fstype = DEF_FSTYPE;
	int flags = 0xffffffff;
	
	if (argc != 3) {
		printf("usage: %s image block_num\n", argv[0]);
		exit(1);
	}

	block = atoi(argv[2]);

	fs = fs_open(argv[1], fstype);

	fs->inode_walk(fs, fs->root_inum, fs->last_inum,
		flags, find_block, (char *) 0);

	if (closeinode == 0) {
		printf("No inode or potential fragment base was found\n");
	} 
	else {
		printf("Block %i could be part of file block %i for inode %i\n", 
		  block, closeblock, (int) closeinode);
	}
	fs->close(fs);

	return 0;
}

