/*
** fls
** TCTUTILs package
**
** Given an image and directory inode, display the file names and 
** directories that exist (both active and deleted)
** 
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.4 $
**
**	Platforms: This code has been tested on OpenBSD 2.8, Linux, and 
**     Solaris 2.7.  
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
** 
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"
#include "fs_dent.h"


void usage(char *myProg) {
	printf("usage: %s [-adDflur] [-g dir] [-z hour] image inode\n", 
	  myProg);
	printf("\t-a: Display \".\" and \"..\" entries\n");
	printf("\t-d: Display deleted entries only\n");
	printf("\t-D: Display directory entries only\n");
	printf("\t-f: Display file entries only\n");
	printf("\t-l: Display long version (like ls -l)\n");
	printf("\t-m: Display deleted files in mactime like fashion with\n");
	printf("\t      prefix as the actual mount point of the image\n");
	printf("\t-p: Display full path for each file\n");
	printf("\t-u: Display undeleted entries only\n");
	printf("\t-r: Recurse on directory entries\n");
	printf("\t-z: Number of hours different from current time zone\n");
	exit(1);
}

FILE *logfp;

static FS_INFO *fs;

#define DIR_SEP "+"

static signed int zonediff = 0;  /* number of hours difference */
static char *macpre;	/*directory prefix for printing mactime output */

static int localFlags;
static char cDirType;

/* Local Flags */
#define LCL_DOT		0x001
#define LCL_REC		0x002
#define LCL_UNLINK	0x004
#define LCL_LINK	0x008
#define LCL_LONG	0x010
#define LCL_FILE	0x020
#define LCL_DIR		0x040
#define LCL_FULL	0x080
#define LCL_MAC	0x100



/* Store directory names when we go recursive */
static unsigned int depth = 0;	/* how deep in the directory tree are we */
#define MAX_DEPTH	32
static char didx[MAX_DEPTH];  /* index in dirs string to where '/' is for 
                        ** given depth */
#define DIR_STRSZ	256
static char dirs[DIR_STRSZ];	/* The current directory name string */


/* Print a deleted entry from fs_dent_walk */
static void 
printDelDir (FS_DENT *fs_dent, int flags, char *ptr) 
{
	int i;

	/* leave if we aren't printing deleted */
	if ((localFlags & LCL_UNLINK) == 0) 
		return;


	/* only print dirs if LCL_DIR is set and only print everything
	** else if LCL_FILE is set
	*/
	if ( ((localFlags & LCL_DIR) && 
		 ((fs_dent->ent_type & FS_DENT_MASK) == FS_DENT_DIR) ) ||
		((localFlags & LCL_FILE) && 
		 ((fs_dent->ent_type & FS_DENT_MASK) != FS_DENT_DIR))) {

		if (!(localFlags & LCL_FULL)) {
			for (i=0; i<depth;i++) 
				fprintf(stdout, DIR_SEP);

			if (depth)
				fprintf(stdout, " ");
		}

		/* print a * if it is deleted */
		if (!(localFlags & LCL_MAC)) {
			if ((fs_dent->ent_type & FS_DENT_MASK) < FS_DENT_MAX_STR)
				fprintf(stdout, "%s ", fs_dent_str[fs_dent->ent_type]);

			fprintf(stdout, "* ");
		}


		if (localFlags & LCL_MAC) {
			fs_dent_print_mac(stdout, fs_dent, fs, dirs, zonediff, 1);
		}
		else if (localFlags & LCL_LONG) {
			if (LCL_FULL & localFlags) 
				fs_dent_print(stdout, fs_dent, fs, dirs, zonediff);
			else
				fs_dent_print(stdout, fs_dent, fs, NULL, zonediff);
		}
		else {
			if (LCL_FULL & localFlags) 
				fprintf(stdout,"%i:\t%s%s\n",fs_dent->inode,dirs,fs_dent->name);
			else
				fprintf(stdout, "%i:\t%s\n", fs_dent->inode, fs_dent->name);
		}
	}
}

/* print active entry from fs_dent_walk */
static void 
printActDir (FS_DENT *fs_dent, int flags, char *ptr)  
{
	int i;

	if ((ISDOT (fs_dent->name) ) && ((localFlags & LCL_DOT) == 0))
		return;

	/* Are we supposed to print active entries? */
	if ((localFlags & LCL_LINK))  {
		if ( ((localFlags & LCL_DIR) && 
			 ((fs_dent->ent_type & FS_DENT_MASK) == FS_DENT_DIR) ) ||
			((localFlags & LCL_FILE) && 
			 ((fs_dent->ent_type & FS_DENT_MASK) != FS_DENT_DIR))) {

			if (!(localFlags & LCL_FULL)) {
				for (i=0; i<depth;i++) { 
					fprintf(stdout, DIR_SEP);
				}

				if (depth)
					fprintf(stdout, " ");
			}
			
			if ((fs_dent->ent_type & FS_DENT_MASK) < FS_DENT_MAX_STR)
				fprintf(stdout, "%s ", fs_dent_str[fs_dent->ent_type]);

			if (localFlags & LCL_LONG) {
				if (LCL_FULL & localFlags) 
					fs_dent_print(stdout, fs_dent, fs, dirs, zonediff);
				else
					fs_dent_print(stdout, fs_dent, fs, NULL, zonediff);
			}
			else {
				if (LCL_FULL & localFlags) 
					fprintf(stdout,"%i:\t%s%s\n", fs_dent->inode, dirs,
					  fs_dent->name);
				else
					fprintf(stdout, "%i:\t%s\n", fs_dent->inode, fs_dent->name);
			}
		}

	}

	/* Are we going to recurse? */
	if ((localFlags & LCL_REC) && (!ISDOT(fs_dent->name)) && 
	  (fs_dent->ent_type & FS_DENT_DIR))  {

		/* If we are print full path, then append the new directory */
		if (LCL_FULL & localFlags)   {
			if (depth < MAX_DEPTH) {
				didx[depth] = strlen(dirs);
				strncpy(&dirs[(int)didx[depth]], fs_dent->name, 
				  DIR_STRSZ - (int)didx[depth]);
				strncat(dirs, "/", DIR_STRSZ);
			}
		}
		depth++;

		fs_dent_walk(fs, fs_dent->inode, flags, printActDir, 
		   printDelDir, (char *)0, cDirType);
		depth--;

		/* remove the last directory name from the list */
		if (LCL_FULL & localFlags)  {
			if (depth < MAX_DEPTH) 
				dirs[(int) didx[depth]] = '\0';
		}
	}
}


int 
main(int argc, char **argv) 
{
	char *fstype = DEF_FSTYPE;
	int inode;
	int flags = FS_FLAG_LINK | FS_FLAG_UNLINK;
	char ch;
	extern int optind;

	localFlags = LCL_LINK | LCL_UNLINK | LCL_DIR | LCL_FILE;

	while ((ch = getopt(argc, argv, "adDfm:luprz:")) > 0) {
		switch (ch) {
		case '?':
		default: 
			usage(argv[0]);
		case 'a':
			localFlags |= LCL_DOT;
			break;
		case 'd':
			localFlags &= ~LCL_LINK;
			break;
		case 'D':
			localFlags &= ~LCL_FILE;
			break;
		case 'f':
			localFlags &= ~LCL_DIR;
			break;
		case 'l':
			localFlags |= LCL_LONG;
			break;
		case 'm':
			localFlags |= LCL_MAC;
			macpre = optarg;
			break;
		case 'p':
			localFlags |= LCL_FULL;
			break;
		case 'u':
			localFlags &= ~LCL_UNLINK;
			break;
		case 'r':
			localFlags |= LCL_REC;
			break;
		case 'z':
			zonediff = atoi(optarg);
			break;
		}
	}

	if ((optind+2) != argc)
		usage(argv[0]);

#if defined (HAVE_UFS_FFS)
	cDirType = FS_DENT_FFS;
#endif
#if defined (HAVE_EXT2FS)
	cDirType = FS_DENT_EXT2;
#endif


	/* Set the full flag to print the full path name if recursion is
	** set and we are only displaying files or deleted files
	*/
	if ((localFlags & LCL_REC) && (
	  ((localFlags & LCL_UNLINK) && (!(localFlags & LCL_LINK))) ||
	  ((localFlags & LCL_FILE) && (!(localFlags & LCL_DIR))) )) {

		localFlags |= LCL_FULL;
	}

	/* set appropriate flags for mactimes style printing */
	if (localFlags & LCL_MAC) {
		localFlags &= ~LCL_LINK;
		localFlags |= (LCL_FULL | LCL_UNLINK | LCL_REC | LCL_FILE 
		  | LCL_DIR);

		strncpy(dirs, macpre, DIR_STRSZ);
		if (dirs[strlen(dirs)-1] != '/') 
			strncat(dirs, "/", DIR_STRSZ - strlen(dirs));
	}

	/* Put this here to catch command line option too */
	else if (localFlags & LCL_FULL)
		strncpy(dirs, "/", DIR_STRSZ);
	
	/* open image */
	fs = fs_open(argv[optind++], fstype);
	inode = atoi(argv[optind]);

	/* begin walk */
	fs_dent_walk(fs, inode, flags, printActDir, printDelDir, 
	  (char *)0, cDirType);

	fs->close(fs);

	return 0;
}

