/*
** blockcalc
** TCTUTILs package
**
** Calculates the corresponding block number between 'unrm' and 'dd' images
** when given an 'unrm' block number, it determines the block number it
** had in a 'dd' image.  When given a 'dd' image, it determines the
** value it would have in a 'unrm' image (if the block is unallocated)
**
** THIS SHOULD BE MADE FASTER
**
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 1.4 $
**
**  Platforms: This code has been tested on OpenBSD 2.8, Linux, and
**     Solaris 2.7.
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/
#include "fs_tools.h"
#include "error.h"
#include "split_at.h"

FILE   *logfp;

static FS_INFO *fs;
static int count;
static int uncnt = 0;

char *progname;

#define CALC_DD		0x1
#define CALC_UNRM	0x2

static void 
usage(char *prog)
{
	printf("usage: %s [-du] block dd_image\n", prog);
	printf("Slowly calculates the opposite block number\n");
	printf("\tOne of the following must be given:\n");
	printf("\t-d: The given block number is the 'dd' block num\n");
	printf("\t-u: The given block number is the 'unrm' block num\n");
	exit(1);
}


/* function used when -d is given
**
** keeps a count of unallocated blocks seen thus far
**
** If the specified block is allocated, an error is given, else the
** count of unalloc blocks is given 
**
** This is called for all blocks (alloc and unalloc)
*/
static void 
count_dd(DADDR_T addr, char *buf, int flags, char *ptr)
{
	if (flags & FS_FLAG_UNALLOC) 
		uncnt++;

	if (--count == 0) {
		if (flags & FS_FLAG_UNALLOC) 
			printf("%d\n", uncnt);
		else
			printf("ERROR: block is allocated, it will not be in an unrm image\n");

		fs->close(fs);
		exit (0);
	}
}

/*
** count how many unalloc blocks there are.
**
** This is called for unalloc blocks only
*/
static void 
count_unrm(DADDR_T addr, char *buf, int flags, char *ptr)
{
	if (--count == 0) {
		printf("%d\n", addr);
		fs->close(fs);
		exit (0);
	}
}

int 
main(int argc, char **argv)
{
    int     flags;
    char   *fstype = DEF_FSTYPE;
	char 	ch, type = 0;

	count = -1;

    while ((ch = getopt(argc, argv, "d:u:")) > 0) {
        switch (ch) {
        case '?':
        default:
            usage(argv[0]);

		case 'd':
			type |= CALC_DD;
			flags = FS_FLAG_ALLOC | FS_FLAG_UNALLOC | FS_FLAG_ALIGN;
			count = atoi(optarg);
			break;

		case 'u':
			type |= CALC_UNRM;
			flags = FS_FLAG_UNALLOC | FS_FLAG_ALIGN;
			count = atoi(optarg);
			break; 
		}
	}

	if (((optind+1) != argc) || (!type) || (count < 0))
		usage(argv[0]);

	if ((type & CALC_DD) && (type & CALC_UNRM)) {
		printf("Only one block type can be given\n");
		usage(argv[0]);
	}
	progname = argv[0];

    fs = fs_open(argv[optind++], fstype);
	if (!fs) {
		error("error opening: %s\n", argv[optind-1]);
		return 1;
	}


	if (type == CALC_UNRM)
		fs->block_walk(fs, fs->start_block, fs->last_block,
		  flags, count_unrm, (char *) fs);
	else if (type == CALC_DD)
		fs->block_walk(fs, fs->start_block, fs->last_block,
		  flags, count_dd, (char *) fs);

    fs->close(fs);
	printf("Block too large\n");
    exit(0);
}
