/*
** fs_dent
**
** Display and manipulate directory entries 
** This file contains generic functions that call the appropriate function
** depending on the file system type
**
** This file requires The Coroners Toolkit (TCT)
**    www.fish.com/tct/
**
** Brian Carrier [carrier@cerias.purdue.edu]
**
**  $Revision: 0.7 $
**
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/
#include <time.h>
#include "fs_tools.h"
#include "fs_dent.h"
#include "mymalloc.h"
#include "error.h"

#ifdef DIR_SUPPORT

#if defined(HAVE_UFS_FFS)
#include "ffs_dent.h"
#endif

#if defined(HAVE_EXT2FS)
#include "ext2_dent.h"
#endif

/* ascii version of file types based on types given in fs_dent.h */
char fs_dent_str[FS_DENT_MAX_STR][2] = {"?","f","c","","d","","b","","r","",
  "l","","s","h","w"};


/* walk the directory inodes */
void 
fs_dent_walk(FS_INFO *fs, INUM_T iNode, int flags, FS_DENT_WALK_FN actionAct,
   FS_DENT_WALK_FN actionDel, char *ptr, char fstype) 
{

#if defined(HAVE_UFS_FFS)
	if (FS_DENT_FFS == fstype) {
		ffs_dent_walk(fs, iNode, flags, actionAct, actionDel, ptr);
		return;
	}
#endif

#if defined(HAVE_EXT2FS)
	if (FS_DENT_EXT2 == fstype) {
		ext2_dent_walk(fs, iNode, flags, actionAct, actionDel, ptr);
		return;
	}
#endif

	error("unknown file system type: %s", fstype);

}

/* Allocate a fs_dent structure */
FS_DENT *
fs_dent_alloc(char fstype)  
{
	FS_DENT *fs_dent;

	fs_dent = (FS_DENT *)mymalloc(sizeof(*fs_dent));
	if (!fs_dent){
		error("error allocating memory");
		return NULL;
	}

	fs_dent->ent_type = FS_DENT_UNDEF;

#if defined(HAVE_UFS_FFS)
	if (fstype == FS_DENT_FFS) {
		ffs_dent_alloc(fs_dent);
		return fs_dent;
	}
#endif

#if defined(HAVE_EXT2FS)
	if (fstype == FS_DENT_EXT2) {
		ext2_dent_alloc(fs_dent);
		return fs_dent;
	}
#endif
	error("fs_dent_alloc: Unknown Dir Type: %i\n", fstype);
	return NULL;
}

void fs_dent_realloc (FS_DENT *fs_dent, char fstype) {
	if (fstype == fs_dent->fs_type)
		return;

	free (fs_dent->name);

#if defined(HAVE_UFS_FFS)
	if (fstype == FS_DENT_FFS) {
		ffs_dent_alloc(fs_dent);
		return;
	}
#endif

#if defined(HAVE_EXT2FS)
	if (fstype == FS_DENT_EXT2) {
		ext2_dent_alloc(fs_dent);
		return;
	}
#endif

	error("fs_dent_realloc: Unknown Dir Type: %i\n", fstype);
}

void fs_dent_free(FS_DENT *fs_dent) {
	free (fs_dent->name);
	free (fs_dent);
}


/* Print contents of  fs_dent entry format like ls -l
**
** All elements are tab delimited 
**
** If dir is NULL, then skip else use it has the full directory name
**  It needs to end with "/"
**
** zonedif is how many hours different the captured data is from the
** local time zone.  For example, if an image from Central Time (chicago)
** is being analyzed in eastern time (boston) the value would be -1
** 
*/
void fs_dent_print(FILE *hFile, FS_DENT *fs_dent, FS_INFO *fs, char *dir,
  signed int zonedif) {
	signed int zonesec;
	time_t tmptime;
	FS_INODE *fs_inode = fs_dent->fsi;

	if ((fs == NULL) || (fs_inode == NULL))  {
		fprintf(hFile, "%i:\t", fs_dent->inode);
		if (dir != NULL) 
			fprintf(hFile, "%s", dir);
		fprintf(hFile, "%s", fs_dent->name);
		fprintf(hFile, "\t00.00.0000 00:00:00\t00.00.0000 00:00:00");
		fprintf(hFile, "\t00.00.0000 00:00:00\t0\t0\t0\n");
	}
	else {
		struct tm *tmTime;

		fprintf(hFile, "%i:\t", fs_dent->inode);
		if (dir != NULL) 	
			fprintf(hFile, "%s", dir);
		fprintf(hFile, "%s", fs_dent->name);

		zonesec = zonedif*3600;

		tmptime = fs_inode->mtime + zonesec;
		if (tmptime < 0)
			tmptime = 0;
		tmTime = localtime(&tmptime);

		fprintf(hFile, "\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d",
		  (int) tmTime->tm_mon+1, (int) tmTime->tm_mday, 
		  (int) tmTime->tm_year+1900, (int) tmTime->tm_hour,
		  (int) tmTime->tm_min, (int) tmTime->tm_sec);

		tmptime = fs_inode->atime + zonesec;
		if (tmptime < 0)
			tmptime = 0;
		tmTime = localtime(&tmptime);

		fprintf(hFile, "\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d",
		  (int) tmTime->tm_mon+1, (int) tmTime->tm_mday, 
		  (int) tmTime->tm_year+1900, (int) tmTime->tm_hour,
		  (int) tmTime->tm_min, (int) tmTime->tm_sec);

		tmptime = fs_inode->ctime + zonesec;
		if (tmptime < 0)
			tmptime = 0;
		tmTime = localtime(&tmptime);

		fprintf(hFile, "\t%.2d.%.2d.%.4d %.2d:%.2d:%.2d",
		  (int) tmTime->tm_mon+1, (int) tmTime->tm_mday, 
		  (int) tmTime->tm_year+1900, (int) tmTime->tm_hour,
		  (int) tmTime->tm_min, (int) tmTime->tm_sec);

		fprintf(hFile, "\t%lu\t%lu\t%lu\n", (ulong)fs_inode->size, 
		  (ulong)fs_inode->gid, (ulong)fs_inode->uid);
	}
}


/*
** Print output in the format that mactime reads.
** This allows the deleted files to be inserted to get a better
** picture of what happened
**
** Prepend dir when printing full file name
**  dir needs to end with "/" 
**
** zonedif is how many hours different the captured data is from the
** local time zone.  For example, if an image from Central Time (chicago)
** is being analyzed in eastern time (boston) the value would be -1
**
** set del to 1 if it is deleted, or 0 else
**
** fs is not required (only used for block size).  
*/
void fs_dent_print_mac(FILE *hFile, FS_DENT *fs_dent, FS_INFO *fs, char *dir,
  signed int zonedif, unsigned char del) {

	signed int zonesec = zonedif*3600;
	FS_INODE *fs_inode;
	char  x;
	mode_t s, m;
	int type;

	if ((!hFile) || (!fs_dent) || (!dir)) 
		return;

	fs_inode = fs_dent->fsi;

	/* md5 */
	fprintf(hFile, "000|");

	/* file name */
	fprintf(hFile, "%s%s <%lu>%s|", dir, fs_dent->name, 
	  (ULONG)fs_dent->inode,(del?" (deleted)":""));


	/* device, inode, and mode val */
	fprintf(hFile, "0|%lu|%lu|", (ULONG)fs_dent->inode, 
	  (fs_inode)?(ULONG)fs_inode->mode:0);


	/* mode as string */
	type = fs_dent->ent_type & FS_DENT_MASK;
	if (type == FS_DENT_CHR)
		fprintf(hFile, "c");
	else if (type == FS_DENT_DIR)
		fprintf(hFile, "d");
	else if (type == FS_DENT_BLK)
		fprintf(hFile, "b");
	else if (type == FS_DENT_LNK)
		fprintf(hFile, "l");
	else if (type == FS_DENT_SOCK)
		fprintf(hFile, "s");
	else
		fprintf(hFile, "-");

	if (fs_inode) {
		/* get "set" bits */
		s = (fs_inode->mode & 0xe00) >> 9;

		/* do user */
		m = (fs_inode->mode & 0x1c0) >> 6;
		if (s&4)
			x = m&1 ? 's':'S';
		else
			x = m&1 ? 'x':'-';
		fprintf(hFile,"%c%c%c", m&4?'r':'-', m&2?'w':'-', x);

		/* do group */
		m = (fs_inode->mode & 0x038) >> 3;
		if (s&2)
			x = m&1 ? 's':'S';
		else
			x = m&1 ? 'x':'-';
		fprintf(hFile, "%c%c%c", m&4?'r':'-', m&2?'w':'-', x);

		/* do other */
		m = fs_inode->mode & 0x007;
		if (s&1)
			x = m&1 ? 't':'T';
		else
		x = m&1 ? 'x':'-';
		fprintf(hFile, "%c%c%c|", m&4?'r':'-', m&2?'w':'-', x);

		/* num link, uid, gid, rdev */
		fprintf(hFile, "%d|%d|%d|0|", (int)fs_inode->nlink, 
		  (int)fs_inode->uid, (int)fs_inode->gid);

		/* size, atime, mtime, ctime */
		fprintf(hFile, "%lu|%lu|%lu|%lu|", (ULONG)fs_inode->size,
		  (ULONG)(fs_inode->atime>zonesec ? fs_inode->atime+zonesec : 0), 
		  (ULONG)(fs_inode->mtime>zonesec ? fs_inode->mtime+zonesec : 0), 
		  (ULONG)(fs_inode->ctime>zonesec ? fs_inode->ctime+zonesec : 0));

	}
	else {
		fprintf(hFile, "---------|0|0|0|0|0|0|0|0|");
	}


	/* block size and num of blocks */
	fprintf(hFile, "%lu|0\n", 
	  ((fs)?(ULONG)fs->file_bsize:0), 0);
}

#endif /* DIR_SUPPORT */
