/* checkset.h - interface for checkset type, a set of rules about what
 *              runtime checks to perform
 *
 * this is a packed type instead of a struct.  a byte array seems most
 * convenient, since I want speed, compactness in memory, and portability.
 *
 * hopefully the macros here will make it stable.
 *
 * the data looks like this:
 *
 * ----------------------------------------------------------------------
 * |  n switches | boolean flags | switch 0 | switch 1 | ... | switch n |
 * ----------------------------------------------------------------------
 * */
#ifndef	ELC_CHECKSET_H
#define	ELC_CHECKSET_H

#define	CHECKSET_INIT(c, n) do {		\
    (c)[0]	 = n;				\
    (c)[1]	 = 0;				\
} while (0)

#define	CHECKSET_SIZEOF(n)	((n)+2) /* number of bytes needed for a
					 * checkset with n switches */

#define	CHECKSET_SWITCHSTART(c)	((c)+2)	/* eval to pointer to switches */

/* ------------ macros for switches that inherit */
#define	CHECKSET_LEN(c)	((c)[0])	/* get the number of switches in c */

/* lookup checkset from c at zero-based index i */
#define	CHECKSET_GET(c,i)	((c)[(i)+2])

/* set checkset at index i to value v */
#define	CHECKSET_SET(c,i,v)	 do {		\
    (c)[(i)+2]	 = v;				\
} while (0)

/* ----------- booleans that don't inherit from directory to its subdirs */
#define	CHECKSET_IGNORE		(0x1)
#define	CHECKSET_NOCHILD	(0x2)
#define CHECKSET_SETBOOLEAN(c, b, v) do {	\
    if (v)					\
      (c)[1]	 |= b;				\
    else					\
      (c)[1]	 &= ~b;				\
} while (0)
#define	CHECKSET_SETIGNORE(c,v) CHECKSET_SETBOOLEAN(c,CHECKSET_IGNORE,v)
#define	CHECKSET_SETNOCHILD(c,v) CHECKSET_SETBOOLEAN(c,CHECKSET_NOCHILD,v)

#define	CHECKSET_GETIGNORE(c)	( (c)[1] & CHECKSET_IGNORE)
#define	CHECKSET_GETNOCHILD(c)	( (c)[1] & CHECKSET_NOCHILD)

typedef char	 *checkset;	/* a checkset is a pointer to a sequence
				 * of char of length checkset[0] + 2 */

#endif
