#include	<config.h>
#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#include	<fcntl.h>
#include	<dirent.h>
#include	<utime.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/time.h>
#include	<errno.h>
#include	<openssl/sha.h>
#include	"cdb.h"
#include	"cdb_make.h"
#include	"cdb_get.h"
#include	"cdb_put.h"
#include	"elcerror.h"
#include	"hashtbl/hashtbl.h"
#include	"hashtbl/xstrdup.h"
#include	"xstradd.h"
#include	"options.h"
#include	"rules.h"
#include	"checkset.h"
#include	"elcwft.h"
#include	"eachfile.h"
#include	"utilities.h"
#include	"elcerror_p.h"
#include	"rules_p.h"
#include	"utilities_p.h"
#include	"eachfile_p.h"
#ifdef		ELC_FIND_LEAKS
#include	"leakfind.h"
#endif
#define		ALGO_NAME	"SHA-1"

typedef struct dbinfo {	/* file information for the database */
  struct stat	stat;
  unsigned char	sum[SHA_DIGEST_LENGTH];
} dbinfo;

typedef struct fileinfo {	/* database file information */
  dbinfo	dbinf;
  unsigned	did_sum: 1;	/* flag shows whether sum member
				 * contains a valid checksum */
} fileinfo;

#if 1
inline static void do_checksum(dbinfo *dbinf, const char *fname)
{
    SHA_CTX	context;
    char	buf[BUFSIZ];
    int		n;
    int		fd	 = open(fname, O_RDONLY);
    struct utimbuf	utb;	/* for resetting access time */

    if (fd == -1)
      die(__FUNCTION__, "Error: opening file (%s): %s",
	  fname, strerror(errno));
    SHA1_Init(&context);
    while ( (n = read(fd, buf, BUFSIZ)) )
      SHA1_Update(&context, buf, n);
    close(fd);
    SHA1_Final(dbinf->sum, &context);

    /* reset access time */
    utb.actime	 = dbinf->stat.st_atime;
    utb.modtime	 = dbinf->stat.st_mtime;
    if (utime(fname, &utb) == -1)
      warn(PROGNAME, __FUNCTION__,
	  "Warning: resetting access time for file (%s): %s",
	  fname, strerror(errno));
}
#else
inline static void do_checksum(dbinfo *dbinf, const char *fname)
{
    printf("debug (%s): %s\n", __FUNCTION__, fname);
    memset(dbinf->sum, 0, SHA_DIGEST_LENGTH);
}
#endif

static void show_diffs(const char *path, unsigned long diffs)
{
    printf("changed: %s: ", path);
    if (diffs & RULE_INODE)
      putchar('i');
    if (diffs & RULE_PERMS)
      putchar('p');
    if (diffs & RULE_NLINK)
      putchar('l');
    if (diffs & RULE_UID)
      putchar('u');
    if (diffs & RULE_GID)
      putchar('g');
    if (diffs & RULE_SIZE)
      putchar('z');
    if (diffs & RULE_ATIME)
      putchar('a');
    if (diffs & RULE_MTIME)
      putchar('m');
#if 0
    if (diffs & RULE_CTIME)
      putchar('c');
#endif
    putchar('\n');
}

static void report_stat_differences(const char *path, unsigned long flags,
				    fileinfo *currinf, dbinfo *old)
{
    struct stat		*sa	 = &old->stat;
    struct stat		*sb	 = &currinf->dbinf.stat;
    unsigned long	diffs	 = 0;
    
    if ((flags & RULE_INODE)
	&& (sa->st_ino != sa->st_ino))
      diffs	 |= RULE_INODE;
    if ((flags & RULE_PERMS)
	&& (sa->st_mode != sb->st_mode))
      diffs	 |= RULE_PERMS;
    if ((flags & RULE_NLINK)
	&& (sa->st_nlink != sa->st_nlink))
      diffs	 |= RULE_NLINK;
    if ((flags & RULE_UID)
	&& (sa->st_uid != sb->st_uid))
      diffs	 |= RULE_UID;
    if ((flags & RULE_GID)
	&& (sa->st_gid != sb->st_gid))
      diffs	 |= RULE_GID;
    if ((flags & RULE_SIZE)
	&& (sa->st_size != sb->st_size))
      diffs	 |= RULE_SIZE;
    if ((flags & RULE_ATIME)
	&& (sa->st_atime != sb->st_atime))
      diffs	 |= RULE_ATIME;
    if ((flags & RULE_MTIME)
	&& (sa->st_mtime != sb->st_mtime))
      diffs	 |= RULE_MTIME;
#if 0
    if ((flags & RULE_CTIME)
	&& (sa->st_ctime != sb->st_ctime))
      diffs	 |= RULE_CTIME;
#endif

    if (diffs)
      show_diffs(path, diffs);
}

static void report_differences(options *opts, fileinfo *currinf,
				      const char *path)
{
    dbinfo		old;
    struct cdb		*knowndb	 = &opts->knowndb;
    size_t		knownsiz	 = cdb_datalen(knowndb);
    unsigned long	flags		 = rules_for_path(opts, path);
    
    if (knownsiz != sizeof(old)
	&& knownsiz != sizeof(old.stat))
      die(__FUNCTION__, "Error: bad db entry for file (%s)", path);
    if (cdb_get(knowndb, &old) == -1)
      die(__FUNCTION__, "Error: cdb_get entry for file (%s)", path);

    if (S_ISREG(currinf->dbinf.stat.st_mode)) {	/* if it's a regular file */
      if (knownsiz != sizeof(dbinfo)) /* maybe known has no checksum */
	printf("%s changed to regular file\n", path);
      else if (flags & RULE_SUM) {
	if (! currinf->did_sum) {
	  do_checksum(&currinf->dbinf, path);
	  currinf->did_sum	 = 1;
	}
	if (memcmp(currinf->dbinf.sum, old.sum, sizeof(old.sum))) {
	  printf("changed: %s: " ALGO_NAME ": ", path);
	  hexprint(stdout, currinf->dbinf.sum, sizeof(currinf->dbinf.sum));
	  fputs(" => ", stdout);
	  hexprint(stdout, old.sum, sizeof(old.sum));
	  putchar('\n');
	}
      }
    }
    report_stat_differences(path, flags, currinf, &old);
}

inline static void do_check(options *opts,
			    char *path, size_t pathlen, fileinfo *inf)
{
    struct cdb	*db	 = &opts->knowndb;
    int		err;

    if ( (err = cdb_find(db, path, pathlen)) == -1)
      die(__FUNCTION__,
	  "Error: looking up file (%s) in known database (%s): %s",
	  path, opts->knowndbname, strerror(errno));
    else if (!err)
      printf("new: %s\n", path); /* this file wasn't in known db */
    else
      report_differences(opts, inf, path);
}

inline static void do_update(options *opts,
			     char *path, size_t pathlen, fileinfo *inf)
{
    struct cdb_make	*db	 = &opts->currdb;
    dbinfo		*dbinf	 = &inf->dbinf;
    size_t		datasiz;
    
    if (S_ISREG(dbinf->stat.st_mode)) {
      do_checksum(dbinf, path);
      inf->did_sum	 = 1;
      datasiz		 = sizeof(*dbinf);
    } else {
      datasiz		 = sizeof(dbinf->stat);
    }
    if (cdb_put(db, path, pathlen, dbinf, datasiz) == -1)
      DIE("adding record to current-state db");
}

wft_ret_t process_file(const char *path, const struct stat *sb, void *data)
{
    options	*opts		 = (options *) data;
    fileinfo	inf;		/* file info for the DB */
    checkset	cset;
    size_t	plen		 = strlen(path);
    char	*pathcopy	 = xstrdup(path);
    char	*p		 = pathcopy;

    inf.did_sum	 = 0;
    
    if (p[0] == '/' && p[1] == '/') {
      ++p;		/* pass the first of double initial slashes */
      --plen;
    }
    
    if (plen > 2
	&& p[plen - 2] == '/'
	&& p[plen - 1] == '.') {
      /* ignore trailing "/." */
      p[plen - 2]	 = '\0';
      plen		 -= 2;
    } else if (plen == 2
	       && p[0] == '/'
	       && p[1] == '.') {
      /* it's "/." */
      p[1]	 = '\0';
      --plen;
    }

#if 0
    printf("debug (%s): path (%s)\n", __FUNCTION__, p); /* debug */
    usleep(5);			/* debug */
#endif

    cset	 = hashtbl_lookup(opts->ruleset, p, plen);

    /* see whether checkset includes boolean to ignore the file */
    if (cset && (CHECKSET_GETIGNORE(cset))) { 
      printf("debug: ignoring file (%s)\n", p);
      free(pathcopy);
      return WFT_PRUNE;
    }

    memcpy(&inf.dbinf.stat, sb, sizeof(inf.dbinf.stat));

    if (opts->do_update)
      do_update(opts, p, plen, &inf);

    if (opts->do_check)
      do_check(opts, p, plen, &inf);

    free(pathcopy);

    /* see whether the nochildren boolean is true in this checkset */
    if (cset && (CHECKSET_GETNOCHILD(cset)))
      return WFT_PRUNE;

    return WFT_PROCEED;
}
