/*
integrit - file integrity verification system
Copyright (C) 2000 Ed Cashin

You can redistribute this program and/or modify it under the terms of
the Artistic License as published by the Open Source Initiative,
currently at the following URL:

    http://www.opensource.org/licenses/artistic-license.html

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

*/
#include	<config.h>
#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#include	<stdlib.h>
#include	<errno.h>
#include	<fcntl.h>
#include	<sys/stat.h>
#include	<openssl/md5.h>
#include	"cdb.h"
#include	"cdb_make.h"
#include	"hashtbl/hashtbl.h"
#include	"elcwft.h"
#include	"checkset.h"
#include	"options.h"
#include	"rules.h"
#include	"eachfile.h"
#include	"missing.h"
#include	"utilities.h"
#include	"elcerror.h"
#include	"elcerror_p.h"
#include	"utilities_p.h"
#include	"options_p.h"
#include	"elcwft_p.h"
#include	"eachfile_p.h"
#include	"missing_p.h"
#ifdef		ELC_FIND_LEAKS
#include	"leakfind.h"
#endif

static void open_known_cdb(options *opts)
{
    int		fd	 = open(opts->knowndbname,
				O_RDONLY | O_NDELAY);

    if (fd == -1)
      die(__FUNCTION__, "Error: opening known-state database (%s): %s",
	  opts->knowndbname, strerror(errno));
    cdb_init(&opts->knowndb, fd);
}

static void open_current_cdb(options *opts)
{
    int			fd;

    fd	 = open(opts->currdbname,
		O_TRUNC | O_CREAT | O_WRONLY | O_NDELAY, 0640);
    if (fd == -1)
      DIE("opening current-state database");
    memset(&opts->currdb, 0, sizeof(opts->currdb));
    if (cdb_make_start(&opts->currdb, fd) == -1)
      DIE("start cdb_make");
}

static void close_current_cdb(options *opts)
{
    if (cdb_make_finish(&opts->currdb) == -1)
      DIE("finishing current-state database");
    close(opts->currdb.fd);
}    

static void close_known_cdb(options *opts)
{
    close(opts->knowndb.fd);
    cdb_free(&opts->knowndb);
}

static void get_currdb_checksum(options *opts, char sumbuf[MD5_DIGEST_LENGTH])
{
    MD5_CTX	context;
    char	buf[BUFSIZ];
    char	*fname	 = opts->currdbname;
    int		n;
    int		fd	 = open(fname, O_RDONLY);

    if (fd == -1)
      die(__FUNCTION__, "Error: opening file (%s): %s",
	  fname, strerror(errno));
    MD5_Init(&context);
    while ( (n = read(fd, buf, BUFSIZ)) )
      MD5_Update(&context, buf, n);
    close(fd);
    MD5_Final(sumbuf, &context);
}

static void show_currdb_checksum(options *opts, char *sumbuf, size_t n)
{
    char	*fname	 = opts->currdbname;

    fputs(PROGNAME ": current-state db md5sum -------------- \n"
	  PROGNAME ": ", stdout);
    hexprint(stdout, sumbuf, n);
    fputs("  ", stdout);
    puts(fname);
}

int main(int argc, char *argv[])
{
    options	opts;
    char	newdb_checksum[MD5_DIGEST_LENGTH];

    options_init(&opts);
    options_set(&opts, argc, argv);
    options_announce(&opts);

    /* todo: use shared file locking with fcntl */
    if (opts.do_check)
      open_known_cdb(&opts);
    if (opts.do_update)
      open_current_cdb(&opts);
    
    if (opts.do_check || opts.do_update)
      if (walk_file_tree(opts.root, process_file, &opts) == -1)
	DIE("walk_file_tree");
    if (opts.do_update) {
      close_current_cdb(&opts);
      get_currdb_checksum(&opts, newdb_checksum);
    }

    if (opts.do_check) {
      if (opts.do_update)
	check_for_missing(&opts);	/* only do this after new current cdb
					   is closed (i.e. cdb_make is done) */
      else if (opts.verbose > 0)
	puts(PROGNAME ": not doing update, so no check for missing files");
      
      close_known_cdb(&opts);
    }
    if (opts.do_update)
      show_currdb_checksum(&opts, newdb_checksum, sizeof(newdb_checksum));

    options_destroy(&opts);
#ifdef		ELC_FIND_LEAKS
    GC_gcollect();		/* find the leaks before exiting */
#endif

    return 0;
}
