/*
hashtbl
Copyright (C) 2000 Ed Cashin

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include	<../config.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<syslog.h>
#include	<errno.h>
#include	"errmacs.h"
#include	"xstrdup.h"
#include	"hashtbl.h"
#ifdef	ELC_FIND_LEAKS
#include	"../leakfind.h"	/* checking for memory leaks */
#endif
/* #define	DEBUG */

#ifndef	HASHTBL_GROW_DELTA
#define	HASHTBL_GROW_DELTA	2	/* on grow, increase by this factor */
#endif

#ifndef	HASHTBL_SHRINK_DELTA
#define	HASHTBL_SHRINK_DELTA	HASHTBL_GROW_DELTA	/* on shrink,
							 * divide capacity
							 * by this factor */
#endif

#ifndef	HASHTBL_FULLISH
#define	HASHTBL_FULLISH	0.60	/* ((entries / capacity) > this) means
				 * it's time to grow */
#endif

#ifndef	HASHTBL_EMPTYISH
#define	HASHTBL_EMPTYISH	0.05	/* ((entries / capacity) < this) means
					 * it's time to shrink */
#endif

#ifndef	HASHTBL_BIG_CAPACITY
#define	HASHTBL_BIG_CAPACITY	240101	/* if capacity bigger than this,
					 * we may occasionally shrink. */
#endif

#ifndef	HASHTBL_DEFAULT_CAPACITY
#define	HASHTBL_DEFAULT_CAPACITY	109
#endif

void hashtbl_resize(hashtbl_t *h, size_t capacity);

/* this code for primes is modified a bit from glib's gprimes.c */
static uint32_t hashtbl_closest_prime(uint32_t num)
{
    static const uint32_t	primelist[] = {
      11,      19,      37,      73,      109,      163,      251,
      367,     557,     823,     1237,    1861,     2777,     4177,
      6247,    9371,    14057,   21089,   31627,    47431,    71143,
      106721,  160073,  240101,  360163,  540217,   810343,   1215497,
      1823231, 2734867, 4102283, 6153409, 9230113,  13845163,
    };
    const static size_t		N = sizeof(primelist) / sizeof(primelist[0]);
    int32_t			i;

    for (i = 0; i < N; i++)
      if (primelist[i] > num)
	return primelist[i];

    return primelist[N - 1];
}


/* These two functions are modified only slightly from the public domain
 * code in Dan Bernstein's cdb package.
 * */
#define	CDB_HASHSTART	5381

inline static uint32_t cdb_hashadd(uint32_t h, uint8_t c)
{
    h += (h << 5);
    return h ^ c;
}

inline static uint32_t cdb_hash(const char *key, size_t len)
{
    uint32_t	h;

    h	 = CDB_HASHSTART;
    while (len) {
      h	 = cdb_hashadd(h, *key++);
      --len;
    }
    return h;
}

void hashtbl_init(hashtbl_t *h, size_t capacity)
{
    int		i;
    hashnode_t	**tbl;

    if (! capacity)
      capacity	 = HASHTBL_DEFAULT_CAPACITY;

    h->capacity	 = capacity;
    h->entries	 = 0;

    if (! (tbl = malloc(sizeof(hashnode_t *) * capacity)) )
      fatal_error("malloc table");
    for (i = 0; i < capacity; ++i)
      tbl[i]	 = NULL;
    h->tbl	 = tbl;
}

void hashtbl_destroy(hashtbl_t *h)
{
    free(h->tbl);
}

void hashtbl_free(hashtbl_t *h)
{
    hashnode_t	**tbl	 = h->tbl;
    hashnode_t	*n;
    int		i;
    int		end	 = h->capacity;
    int		left	 = h->entries;

    for (i = 0; (i < end) && left; ++i)
      if ( (n = tbl[i]) ) {
	free(n->key);
	free(n->data);
	free(n);
	--left;
      }
}

inline static hashnode_t *hashtbl_find(hashtbl_t *h,
				       const char *key, size_t keylen)
{
    hashnode_t	*np;
    hashnode_t	**tbl	 = h->tbl;
    
    for (np = tbl[cdb_hash(key, keylen) % h->capacity]; np; np = np->next)
      if (! strcmp(key, np->key))
	return np;

    return NULL;
}

void *hashtbl_lookup(hashtbl_t *h, const char *key, size_t keylen)
{
    hashnode_t	*np;

    if (! (np = hashtbl_find(h, key, keylen)) )
      return NULL;
    return np->data;
}

void hashtbl_grow(hashtbl_t *h)
{
    hashtbl_resize(h, hashtbl_closest_prime(h->capacity
					    * HASHTBL_GROW_DELTA));
}

void hashtbl_shrink(hashtbl_t *h)
{
    hashtbl_resize(h, hashtbl_closest_prime(h->capacity
					    / HASHTBL_SHRINK_DELTA));
}

void hashtbl_resize(hashtbl_t *h, size_t capacity)
{
    size_t	oldcapacity	 = h->capacity;
    hashnode_t	**oldtbl	 = h->tbl;
    hashnode_t	**newtbl;
    hashnode_t	*np, *oldnext;
    int		i;
    uint32_t	hashval;

#ifdef	DEBUG
    fprintf(stderr,
	    "debug: resizing table: %d --> %d ... ", oldcapacity, capacity);
#endif
    if (! (newtbl = malloc(sizeof(hashnode_t *) * capacity)) ) {
      pwarn("could not resize hash table");
      return;
    }
    for (i = 0; i < capacity; ++i)
      newtbl[i]	 = NULL;
    h->tbl	 = newtbl;
    h->capacity	 = capacity;

    for (i = 0; i < oldcapacity; ++i) {
      for (np = oldtbl[i]; np; np = oldnext) {
	oldnext		 = np->next;
	hashval		 = cdb_hash(np->key, np->keylen) % capacity;
	np->next	 = newtbl[hashval];
	newtbl[hashval]	 = np;
      }
    }
    free(oldtbl);
#ifdef	DEBUG
    fputs("done.\n", stderr);
#endif
}

/* hashtbl doesn't do memory management of the stored data.
 * When an entry is replaced by hashtbl_store, it returns the pointer
 * to data that was replaced.
 *
 * The caller can use the returned pointer to free the associated
 * memory appropriately.
 */
void *hashtbl_store(hashtbl_t *h, const char *key, size_t keylen, void *data)
{
    void	*replaced	 = NULL;
    hashnode_t	**tbl		 = h->tbl;
    hashnode_t	*np;
    uint32_t	hashval		 = cdb_hash(key, keylen) % h->capacity;

    for (np = tbl[hashval]; np; np = np->next) {
      if (! strcmp(key, np->key)) { /* replace entry with identical key */
	replaced	 = np->data;
	np->data	 = data;
	break;
      }
    }
    if (! replaced) {
      if (! (np = malloc(sizeof(hashnode_t))) )
	fatal_error("malloc hash table node");
      if (! (np->key = xstrdup(key)))
	fatal_error("xstrdup hash key");
      np->next		 = tbl[hashval];
      tbl[hashval]	 = np;
      np->data		 = data;
      np->keylen	 = keylen;
      ++h->entries;

      if (((float) h->entries / h->capacity) > HASHTBL_FULLISH)
	hashtbl_grow(h);
    }

    return replaced;
}

inline void hashtbl_node_destroy(hashnode_t *np)
{
    /* hashtbl allocates memory for its nodes' keys */
    free(np->key);
}

/* hashtbl doesn't do memory management of the stored data.
 * When an entry is deleted by hashtbl_remove, it returns the pointer
 * to data that was removed.
 *
 * The caller can use the returned pointer to free the associated
 * memory appropriately.
 */
void *hashtbl_remove(hashtbl_t *h, const char *key, size_t keylen)
{
    void	*rmdata	 = NULL;
    hashnode_t	**tbl		 = h->tbl;
    hashnode_t	*np, *rmnode;
    uint32_t	hashval		 = cdb_hash(key, keylen) % h->capacity;

    if (! (np = tbl[hashval]) )
      return NULL;

    /* test the entry in the table first; then go through the list
     * if necessary */
    if (! strcmp(np->key, key)) {
      rmdata		 = np->data;
      tbl[hashval]	 = np->next;
      hashtbl_node_destroy(np);
      free(np);
      --h->entries;
    } else {
      for (; (rmnode = np->next); np = np->next) {
	if (! strcmp(key, rmnode->key)) {
	  rmdata	 = rmnode->data;
	  np->next	 = rmnode->next;
	  hashtbl_node_destroy(rmnode);
	  free(rmnode);
	  --h->entries;
	  break;
	}
      }
    }

    if ((h->capacity > HASHTBL_BIG_CAPACITY)
	&& (((float) h->entries / h->capacity) < HASHTBL_EMPTYISH))
      hashtbl_shrink(h);

   return rmdata;
}
