/*
integrit - file integrity verification system
Copyright (C) 2000 Ed Cashin

You can redistribute this program and/or modify it under the terms of
the Artistic License as published by the Open Source Initiative,
currently at the following URL:

    http://www.opensource.org/licenses/artistic-license.html

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

*/
#include	<config.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<fcntl.h>
#include	<dirent.h>
#include	<utime.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/time.h>
#include	<errno.h>
#include	<openssl/sha.h>
#include	"cdb.h"
#include	"cdb_make.h"
#include	"cdb_get.h"
#include	"cdb_put.h"
#include	"elcerror.h"
#include	"hashtbl/hashtbl.h"
#include	"hashtbl/xstrdup.h"
#include	"xstradd.h"
#include	"options.h"
#include	"rules.h"
#include	"checkset.h"
#include	"elcwft.h"
#include	"eachfile.h"
#include	"utilities.h"
#include	"xml.h"
#include	"elcerror_p.h"
#include	"rules_p.h"
#include	"utilities_p.h"
#include	"eachfile_p.h"
#include	"xml_p.h"
#ifdef		ELC_FIND_LEAKS
#include	"leakfind.h"
#endif
#define		ALGO_NAME	"SHA-1"

typedef struct dbinfo {	/* file information for the database */
  struct stat	stat;
  unsigned char	sum[SHA_DIGEST_LENGTH];
} dbinfo;

typedef struct fileinfo {	/* database file information */
  dbinfo	dbinf;
  char		*path;		/* null-terminated */
  size_t	pathlen;	/* save on strlen calls,
				 * since we know it already */
  unsigned long	ruleset;
  unsigned	found_ruleset: 1; /* flag indicates whether ruleset
				   * member has been initialized */
  unsigned	did_sum: 1;	/* flag shows whether sum member
				 * contains a valid checksum */
} fileinfo;

static void fileinfo_init(fileinfo *inf, char *path, size_t pathlen)
{
    inf->ruleset	 = 0;
    inf->found_ruleset	 = 0;
    inf->did_sum	 = 0;
    inf->path		 = path;
    inf->pathlen	 = pathlen;
}

static unsigned long fileinfo_ruleset(options *opts, fileinfo *inf)
{
    if (! inf->found_ruleset) {
      inf->ruleset		 = rules_for_path(opts, inf->path);
      inf->found_ruleset	 = 1;
    }

    return inf->ruleset;
}

inline static void do_checksum(options *opts, fileinfo *inf)
{
    SHA_CTX	context;
    char	buf[BUFSIZ];
    int		n;
    int		fd	 = open(inf->path, O_RDONLY);
    dbinfo	*dbinf	 = &inf->dbinf;
    struct utimbuf	utb;	/* for resetting access time */

    if (fd == -1)
      die(__FUNCTION__, "Error: opening file (%s): %s",
	  inf->path, strerror(errno));
    SHA1_Init(&context);
    while ( (n = read(fd, buf, BUFSIZ)) )
      SHA1_Update(&context, buf, n);
    close(fd);
    SHA1_Final(dbinf->sum, &context);

    /* reset access time on request */
    if (fileinfo_ruleset(opts, inf) & RULE_RESET_ATIME) {
#ifdef	DEBUG
      fprintf(stderr, "debug (%s): resetting atime for file (%s)\n",
	      __FUNCTION__, inf->path);
#endif
      utb.actime	 = dbinf->stat.st_atime;
      utb.modtime	 = dbinf->stat.st_mtime;
      if (utime(inf->path, &utb) == -1)
	warn(__FUNCTION__,
	     "Warning: resetting access time for file (%s): %s",
	     inf->path, strerror(errno));
    }
}

static void show_diff_xml_long(FILE *out, char T, const char *type,
			       long old, long new)
{
    xml_start_print(out, type);
    XML_START_PRINT(out, "old");
    fprintf(out, "%ld", old);
    XML_END_PRINT(out, "old");
    XML_START_PRINT(out, "new");
    fprintf(out, "%ld", new);
    XML_END_PRINT(out, "new");
    xml_end_print(out, type);
}

static void show_diff_lines_long(FILE *out, char T, const char *type,
				 long old, long new)
{
    fprintf(out, "%c(%ld:%ld) ", T, old, new);
}

static void show_diff_lines_time(FILE *out, char T, const char *type,
				 time_t old, time_t new)
{
    const int	buf_max	 = 16;	/* 20001212-010101 + null */
    char	buf[buf_max];

    putc(T, out);
    putc('(', out);

    if (! strftime(buf, buf_max, "%Y%m%d-%H%M%S", localtime(&old)) )
      DIE("strftime");
    fputs(buf, out);

    putc(':', out);

    if (! strftime(buf, buf_max, "%Y%m%d-%H%M%S", localtime(&new)) )
      DIE("strftime");
    fputs(buf, out);

    fputs(") ", out);
}

static void show_diff_xml_octal(FILE *out, char T, const char *type,
				unsigned long old, unsigned long new)
{
    xml_start_print(out, type);
    XML_START_PRINT(out, "old");
    fprintf(out, "%lo", old);
    XML_END_PRINT(out, "old");
    XML_START_PRINT(out, "new");
    fprintf(out, "%lo", new);
    XML_END_PRINT(out, "new");
    xml_end_print(out, type);
}

static void show_diff_lines_octal(FILE *out, char T, const char *type,
				  unsigned long old, unsigned long new)
{
    fprintf(out, "%c(%lo:%lo) ", T, old, new);
}

static void show_diffs(options *opts,
		       const char *path, unsigned long diffs,
		       const struct stat *sa, const struct stat *sb)
{
    void (*show_long)(FILE *out, char T, const char *type, long old, long new);
    void (*show_time)(FILE *out, char T, const char *type,
		      time_t old, time_t new);
    void (*show_oct)(FILE *out, char T, const char *type,
		     unsigned long old, unsigned long new);

    /* for showing the last twelve bits (four octal digits) */
    const	unsigned perm_mask	 = 07777;

    if (opts->output == OUTPUT_XML) {
      show_long	 = show_diff_xml_long;
      show_time	 = show_diff_xml_long;
      show_oct	 = show_diff_xml_octal;
      XML_CHANGE_START_PRINT(stdout, "stat", path);
    } else {
      show_long	 = show_diff_lines_long;
      show_time	 = show_diff_lines_time;
      show_oct	 = show_diff_lines_octal;
      fprintf(stdout, "changed: %s   ", path);
    }

    if (diffs & RULE_INODE)
      show_long(stdout, 'i', "inode", sa->st_ino, sb->st_ino);
    if (diffs & RULE_PERMS)
      show_oct(stdout, 'p', "permissions",
	       sa->st_mode & perm_mask, sb->st_mode & perm_mask);
    if (diffs & RULE_NLINK)
      show_long(stdout, 'l', "nlinks", sa->st_nlink, sb->st_nlink);
    if (diffs & RULE_UID)
      show_long(stdout, 'u', "uid", sa->st_uid, sb->st_uid);
    if (diffs & RULE_GID)
      show_long(stdout, 'g', "gid", sa->st_gid, sb->st_gid);
    if (diffs & RULE_SIZE)
      show_long(stdout, 'z', "size", sa->st_size, sb->st_size);
    if (diffs & RULE_ATIME)
      show_time(stdout, 'a', "access_time", sa->st_atime, sb->st_atime);
    if (diffs & RULE_MTIME)
      show_time(stdout, 'm', "modification_time", sa->st_mtime, sb->st_mtime);
    if (diffs & RULE_CTIME)
      show_time(stdout, 'c', "change_time", sa->st_ctime, sb->st_ctime);

    if (opts->output == OUTPUT_XML)
      XML_END_PRINT(stdout, "change");

    putc('\n', stdout);
}

static void report_stat_differences(options *opts,
				    fileinfo *currinf, dbinfo *old)
{
    struct stat		*sa	 = &old->stat;
    struct stat		*sb	 = &currinf->dbinf.stat;
    const char *path		 = currinf->path;
    unsigned long	flags	 = fileinfo_ruleset(opts, currinf);
    unsigned long	diffs	 = 0;
    
    if ((flags & RULE_INODE)
	&& (sa->st_ino != sb->st_ino))
      diffs	 |= RULE_INODE;
    if ((flags & RULE_PERMS)
	&& (sa->st_mode != sb->st_mode))
      diffs	 |= RULE_PERMS;
    if ((flags & RULE_NLINK)
	&& (sa->st_nlink != sb->st_nlink))
      diffs	 |= RULE_NLINK;
    if ((flags & RULE_UID)
	&& (sa->st_uid != sb->st_uid))
      diffs	 |= RULE_UID;
    if ((flags & RULE_GID)
	&& (sa->st_gid != sb->st_gid))
      diffs	 |= RULE_GID;
    if ((flags & RULE_SIZE)
	&& (sa->st_size != sb->st_size))
      diffs	 |= RULE_SIZE;
    if ((flags & RULE_ATIME)
	&& (sa->st_atime != sb->st_atime))
      diffs	 |= RULE_ATIME;
    if ((flags & RULE_MTIME)
	&& (sa->st_mtime != sb->st_mtime))
      diffs	 |= RULE_MTIME;
    if ((flags & RULE_CTIME) && (! (flags & RULE_RESET_ATIME))
	&& (sa->st_ctime != sb->st_ctime))
      diffs	 |= RULE_CTIME;

    if (diffs)
      show_diffs(opts, path, diffs, sa, sb);
}

/* report that a file has changed from a non-regular file
 * to a regular file */
static void report_2regfile(options *opts, const char *path)
{
    switch (opts->output) {
      case OUTPUT_XML:
	XML_CHANGE_START_PRINT(stdout, "filetype", path);
	XML_ELEMENT_PRINT(stdout, "old", "not regular file");
	XML_ELEMENT_PRINT(stdout, "new", "regular file");
	XML_END_PRINT(stdout, "change");
	putc('\n', stdout);
	break;
      case OUTPUT_LINES:
	printf("changed: %s became regular file\n", path);
	break;
      default:
	abort();		/* shouldn't happen */
    }
}

/* report that the checksum has changed. */
static void report_sumchange(options *opts, const char *path,
			     const unsigned char *old, size_t oldsiz,
			     const unsigned char *new, size_t newsiz)
{
    switch (opts->output) {
      case OUTPUT_LINES:
	printf("changed: %s   s(", path);
	hexprint(stdout, old, oldsiz);
	putc(':', stdout);
	hexprint(stdout, new, newsiz);
	fputs(")\n", stdout);
	break;
      case OUTPUT_XML:
	XML_CHANGE_START_PRINT(stdout, ALGO_NAME, path);
	XML_START_PRINT(stdout, "old");
	hexprint(stdout, old, oldsiz);	
	XML_END_PRINT(stdout, "old");
	XML_START_PRINT(stdout, "new");
	hexprint(stdout, new, newsiz);	
	XML_END_PRINT(stdout, "new");
	XML_END_PRINT(stdout, "change");
	putc('\n', stdout);
	break;
      default:
	abort();		/* shouldn't happen */
    }
}

static void report_differences(options *opts, fileinfo *currinf)
{
    dbinfo		old;
    struct cdb		*knowndb	 = &opts->knowndb;
    size_t		knownsiz	 = cdb_datalen(knowndb);
    char		*path		 = currinf->path;
    unsigned long	flags		 = fileinfo_ruleset(opts, currinf);
    
    if (knownsiz != sizeof(old)
	&& knownsiz != sizeof(old.stat))
      die(__FUNCTION__, "Error: bad db entry for file (%s)", path);
    if (cdb_get(knowndb, &old) == -1)
      die(__FUNCTION__, "Error: cdb_get entry for file (%s)", path);

    if (S_ISREG(currinf->dbinf.stat.st_mode)) {	/* if it's a regular file */
      if (knownsiz != sizeof(dbinfo)) /* maybe known has no checksum */
	report_2regfile(opts, path);
      else if (flags & RULE_SUM) {
	if (! currinf->did_sum) {
	  do_checksum(opts, currinf);
	  currinf->did_sum	 = 1;
	}
	if (memcmp(currinf->dbinf.sum, old.sum, sizeof(old.sum)))
	  report_sumchange(opts, path, old.sum, sizeof(old.sum),
			   currinf->dbinf.sum, sizeof(currinf->dbinf.sum));
      }
    }
    report_stat_differences(opts, currinf, &old);
}

static void report_newfile(options *opts, fileinfo *inf)
{
    char	*path	 = inf->path;

    switch (opts->output) {
      case OUTPUT_XML:
	XML_CHANGE_START_PRINT(stdout, "newfile", path);
	XML_END_PRINT(stdout, "change");
	putc('\n', stdout);
	break;
      case OUTPUT_LINES:
	printf("new: %s\n", path); /* this file wasn't in known db */
	break;
      default:
	abort();		/* this shouldn't happen */
	break;
    }
}    

inline static void do_check(options *opts, fileinfo *inf)
{
    struct cdb	*db	 = &opts->knowndb;
    int		err;

    if ( (err = cdb_find(db, inf->path, inf->pathlen)) == -1)
      die(__FUNCTION__,
	  "Error: looking up file (%s) in known database (%s): %s",
	  inf->path, opts->knowndbname, strerror(errno));
    else if (!err)
      report_newfile(opts, inf);
    else
      report_differences(opts, inf);
}

inline static void do_update(options *opts, fileinfo *inf)
{
    struct cdb_make	*db	 = &opts->currdb;
    dbinfo		*dbinf	 = &inf->dbinf;
    size_t		datasiz;
    
    if (S_ISREG(dbinf->stat.st_mode)) {
      /* BUGGISH: don't test inf->did_sum, since this function's called
       *          before do_check (very minorly buggish :)
       */
      do_checksum(opts, inf);
      inf->did_sum	 = 1;
      datasiz		 = sizeof(*dbinf);
    } else {
      datasiz		 = sizeof(dbinf->stat);
    }
    if (cdb_put(db, inf->path, inf->pathlen, dbinf, datasiz) == -1)
      DIE("adding record to current-state db");
}

wft_ret_t process_file(const char *path, const struct stat *sb, void *data)
{
    options	*opts		 = (options *) data;
    fileinfo	inf;		/* initialize before use */
    checkset	cset;
    size_t	plen		 = strlen(path);
    char	*pathcopy	 = xstrdup(path);
    char	*p		 = pathcopy;
#if defined(DEBUG) && 0
    static int	counter;
#endif

    if (p[0] == '/' && p[1] == '/') {
      ++p;		/* pass the first of double initial slashes */
      --plen;
    }
    
    if (plen > 2
	&& p[plen - 2] == '/'
	&& p[plen - 1] == '.') {
      /* ignore trailing "/." */
      p[plen - 2]	 = '\0';
      plen		 -= 2;
    } else if (plen == 2
	       && p[0] == '/'
	       && p[1] == '.') {
      /* it's "/." */
      p[1]	 = '\0';
      --plen;
    }

#if defined(DEBUG) && 0
    fprintf(stderr, "debug (%s): count (%d) path (%s)\n",
	    __FUNCTION__, ++counter, p); /* debug */
    usleep(5);			/* debug */
#endif

    /* initialize inf now that we've settled on a file path */
    fileinfo_init(&inf, p, plen);

    cset	 = hashtbl_lookup(opts->ruleset, p, plen);

    /* see whether checkset includes boolean to ignore the file */
    if (cset && (CHECKSET_GETIGNORE(cset))) { 
#ifdef	DEBUG
      fprintf(stderr, "debug: ignoring file (%s)\n", p);
#endif
      free(pathcopy);
      return WFT_PRUNE;
    }

    memcpy(&inf.dbinf.stat, sb, sizeof(inf.dbinf.stat));

    if (opts->do_update)
      do_update(opts, &inf);

    if (opts->do_check)
      do_check(opts, &inf);

    free(pathcopy);

    /* see whether the nochildren boolean is true in this checkset */
    if (cset && (CHECKSET_GETNOCHILD(cset))) {
#ifdef	DEBUG
      fprintf(stderr, "debug: no children for file (%s)\n", path);
#endif
      return WFT_PRUNE;
    }

    return WFT_PROCEED;
}
