/*
integrit - file integrity verification system
Copyright (C) 2000 Ed Cashin

You can redistribute this program and/or modify it under the terms of
the Artistic License as published by the Open Source Initiative,
currently at the following URL:

    http://www.opensource.org/licenses/artistic-license.html

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

*/
#include	<config.h>
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<ctype.h>
#include	<errno.h>
#include	"cdb.h"
#include	"cdb_make.h"
#include	"hashtbl/hashtbl.h"
#include	"hashtbl/xstrdup.h"
#include	"checkset.h"
#include	"options.h"
#include	"rules.h"
#include	"xml.h"
#include	"elcerror.h"
#include	"elcerror_p.h"
#include	"checkset_p.h"
#include	"options_p.h"
#include	"xml_p.h"
#ifdef		ELC_FIND_LEAKS
#include	"leakfind.h"
#endif

#define	EATSPACE(buf)	do {				\
    while (*(buf)					\
	   && (*(buf) != '\n')				\
	   && (isspace(*(buf))))	/* skip whitespace */	\
      ++(buf);						\
} while (0)    

inline static void chomp(char *chp)
{
    /* remove trailing newlines from a one-line string */
    for( ; *chp; ++chp)
      if(*chp == '\n')
	*chp = '\0';
}

char *options_output_str(options *o)
{
    switch (o->output) {
      case OUTPUT_LINES:
	return "human-readable";
	break;
      case OUTPUT_XML:
	return "xml";
	break;
      default:
	DIE("unknown value for output member in options");
	break;
    };
    return "teddy bear froo froo gum"; /* not reached */
}

static void options_announce_lines(options *o, FILE *out)
{
    fprintf(out, PROGNAME ": ---- integrit, version %s -----------------\n",
	    INTEGRIT_VERSION);
    fprintf(out, PROGNAME ": %27s : %s\n", "output", options_output_str(o));
    fprintf(out, PROGNAME ": %27s : %s\n", "conf file", o->conffile);
    fprintf(out, PROGNAME ": %27s : %s\n", "known db", o->knowndbname);
    fprintf(out, PROGNAME ": %27s : %s\n", "current db", o->currdbname);
    fprintf(out, PROGNAME ": %27s : %s\n", "root", o->root);
    fprintf(out, PROGNAME ": %27s : %s\n", "do check",
	    o->do_check ? "yes" : "no");
    fprintf(out, PROGNAME ": %27s : %s\n", "do update",
	    o->do_update ? "yes" : "no");
}

static void options_announce_xml(options *o, FILE *out)
{
    XML_START_PRINT(out, "options");
    XML_ELEMENT_PRINT(out, "output", options_output_str(o));
    XML_ELEMENT_PRINT(out, "conffile", o->conffile);
    XML_ELEMENT_PRINT(out, "knowndb", o->knowndbname);
    XML_ELEMENT_PRINT(out, "currentdb", o->currdbname);
    XML_ELEMENT_PRINT(out, "root", o->root);
    XML_ELEMENT_PRINT(out, "check", o->do_check ? "yes" : "no");
    XML_ELEMENT_PRINT(out, "update", o->do_update ? "yes" : "no");
    XML_END_PRINT(out, "options");
    putc('\n', out);
}

void options_announce(FILE *out, options *o)
{
    switch (o->output) {
      case OUTPUT_LINES:
	options_announce_lines(o, out);
	break;
      case OUTPUT_XML:
	options_announce_xml(o, out);
	break;
      default:
	DIE("unknown value for output member in options");
	break;
    }
}

void options_init(options *o)
{
    o->conffile		 = NULL;
    o->knowndbname	 = NULL;
    o->currdbname	 = NULL;
    o->root		 = NULL;
    if (! (o->ruleset = malloc(sizeof(hashtbl_t))) )
      DIE("malloc hashtbl");
    hashtbl_init(o->ruleset, 20);
    o->verbose		 = 1;
    o->do_check		 = 0;
    o->do_update	 = 0;
    o->default_flags	 = RULE_SUM | RULE_INODE | RULE_PERMS
      | RULE_NLINK | RULE_UID | RULE_GID | RULE_MTIME | RULE_CTIME;
    o->output		 = OUTPUT_LINES; /* human-readable output */
}

void options_destroy(options *o)
{
    free(o->knowndbname);
    free(o->currdbname);
    free(o->root);
    hashtbl_free(o->ruleset);
    hashtbl_destroy(o->ruleset);
    free(o->ruleset);
}

/* call EATSPACE before calling this */
inline static int blank_or_comment(const char *buf)
{
    switch (*buf) {
      case '\0':		/* empty string counts as blank */
	return 1;
	break;
      case '\n':		/* blank line */
	return 1;
	break;
      case '#':			/* comment */
	return 1;
	break;
      default:
	return 0;
	break;
    }
}

inline static void die_noprop(const char *prop, const char *func)
{
    fprintf(stderr, PROGNAME " (%s) Error: no value for property: %s\n",
	    func, prop);
    exit(EXIT_FAILURE);
}

inline static void do_assignment(options *o, char *buf, char *eq)
{
    char	*property;
    char	*val	 = eq + 1;

    *eq	 = '\0';
    if ( (property = strstr(buf, "known")) ) {
      EATSPACE(val);
      if (! *val)
	die_noprop(property, __FUNCTION__);
      chomp(val);
      o->knowndbname	 = xstrdup(val);
    } else if ( (property = strstr(buf, "current")) ) {
      EATSPACE(val);
      if (! *val)
	die_noprop(property, __FUNCTION__);
      chomp(val);
      o->currdbname	 = xstrdup(val);
    } else if ( (property = strstr(buf, "root")) ) {
      EATSPACE(val);
      if (! *val)
	die_noprop(property, __FUNCTION__);
      chomp(val);
      o->root	 = xstrdup(val);
    } else {
      die(__FUNCTION__, "Error: unknown property: %s\n", property);
    }
}

static void options_add_checkset(options *o, char *namebuf,
				 size_t namebuf_chars, checkset cset)
{
    hashtbl_t	*h	 = o->ruleset;
    checkset	old;

    if ( (old = hashtbl_lookup(h, namebuf, namebuf_chars)) ) {
      fprintf(stderr, "Warning: overwriting old checkset (");
      checkset_show(stderr, old);
      fprintf(stderr, ") for file (%s)\n", namebuf);
      free(old);
    }
    hashtbl_store(h, namebuf, namebuf_chars, cset);
}

inline static void do_rule(options *o, char *buf)
{
    checkset	cset;
    char	namebuf[BUFSIZ];
    int		namebuf_chars	 = 0;
    int		n_switches;
    char	ignore		 = 0;
    char	nochildren	 = 0;

    EATSPACE(buf);
    switch (*buf) {
      case '!':
	ignore		 = 1;
	++buf;
	break;
      case '=':
	nochildren	 = 1;
	++buf;
	break;
      default:
	break;	
    }

    EATSPACE(buf);
    for ( ; *buf && (*buf != '\n'); ++buf) {
      if (namebuf_chars == (BUFSIZ - 3)) /* space for two new chars,
					  * plus null char */
	die(__FUNCTION__, "Error: filename too long in config");

      if (*buf == '\\') {
	if (isspace(*(buf + 1)))
	  namebuf[namebuf_chars++]	 = *(++buf);
	else
	  namebuf[namebuf_chars++]	 = '\\';
      } else if (isspace(*buf)) {
	break;			/* end of the name */
      } else {
	namebuf[namebuf_chars++]	 = *buf;
      }
    }
    namebuf[namebuf_chars]	 = '\0';

    EATSPACE(buf);
    n_switches	 = strlen(buf);
    /* trim trailing whitespace */
    while (n_switches && isspace(buf[n_switches - 1])) {
      buf[n_switches - 1]	 = '\0';
      --n_switches;
    }
    if (strspn(buf, "SsIiPpLlUuGgZzAaMmCcRr") != n_switches)
      die(__FUNCTION__,
	  "Error: unrecognized check switch in conf file rule for %s",
	  namebuf);

    if (! (cset = malloc(CHECKSET_SIZEOF(n_switches))) ) 
      DIE("malloc checkset");
    checkset_init(cset, n_switches);
    CHECKSET_SETIGNORE(cset, ignore);
    CHECKSET_SETNOCHILD(cset, nochildren);
    strcpy(CHECKSET_SWITCHSTART(cset), buf);
    
    chomp(namebuf);
#ifdef	DEBUG
    fprintf(stderr, "debug: path (%s) checkset (", namebuf);
    checkset_show(stderr, cset);	/* debug */
    fputs(")\n", stderr);
#endif
    
    options_add_checkset(o, namebuf, namebuf_chars, cset);
}

static void usage(void)
{
    fputs("usage:\n"
	  "\n"
	  "      integrit -C conffile [-x] [-u] [-c]\n"
	  "      integrit -V\n"
	  "      integrit -h\n"
	  "\n"
	  "options:\n"
	  "\n"
	  "      -C	specify configuration file\n"
	  "      -x	use XML output instead of abbreviated output\n"
	  "      -u	do update: create current state database\n"
	  "      -c	do check: verify current state against known db\n"
	  "      -V	show integrit version info and exit\n"
	  "      -h	show this help      \n\n", stderr);
}

void parse_args(options *o, int argc, char *argv[])
{
    int		c;
    opterr	 = 0;

    while ( (c = getopt(argc, argv, "hVvqC:cux")) != -1) {
      switch (c) {
	case 'h':
	  usage();
	  exit(EXIT_SUCCESS);
	  break;
	case 'V':
	  puts(PROGNAME " version " INTEGRIT_VERSION);
	  exit(EXIT_SUCCESS);
	  break;
	case 'v':
	  ++o->verbose;
	  break;
	case 'q':
	  --(o->verbose);
	  break;
	case 'C':
	  o->conffile	 = optarg;
	  break;
	case 'c':
	  o->do_check	 = 1;
	  break;
	case 'u':
	  o->do_update	 = 1;
	  break;
	case 'x':
	  o->output	 = OUTPUT_XML;
	  break;
	case '?':
	  if (isprint(optopt))
	    warn(__FUNCTION__, "Error: unknown option `-%c'.\n", optopt);
	  else
	    warn(__FUNCTION__,
		"Error: unknown option character `\\x%x'.\n", optopt);
	  usage();
	  exit(EXIT_FAILURE);
	  break;
	default:
	  abort();		/* this shouldn't happen */
	  break;
      }	/* end switch */
    } /* end while getopt */

    if (! o->conffile)
      die(__FUNCTION__, "Error: no conffile on command line");
}

void options_set(options *o, int argc, char *argv[])
{
    char	buf[BUFSIZ];
    char	*cp;
    char	*equalsign;
    FILE	*conf;

    parse_args(o, argc, argv);
    if (! (conf = fopen(o->conffile, "r")) )
      DIE("opening conf file");

    while (fgets(buf, BUFSIZ, conf)) {
      cp	 = buf;
      EATSPACE(cp);
      if (blank_or_comment(cp))
	continue;
      else if (*cp == '=')
	do_rule(o, cp);
      else if ( (equalsign = strchr(cp, '=')) )
	do_assignment(o, cp, equalsign);
      else
	do_rule(o, cp);
    }
    if (! o->knowndbname)
      die(__FUNCTION__, "Error: known database unspecified\n");
    if (! o->currdbname)
      die(__FUNCTION__, "Error: current database unspecified\n");
    if (! o->root)
      die(__FUNCTION__, "Error: root search directory unspecified\n");
}
