/*
integrit - file integrity verification system
Copyright (C) 2000 Ed Cashin

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include	<config.h>
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<limits.h>
#include	<errno.h>
#include	"cdb.h"
#include	"cdb_make.h"
#include	"cdb_seq.h"
#include	"hashtbl/hashtbl.h"
#include	"options.h"
#include	"elcerror.h"
#include	"missing.h"
#include	"xml.h"
#include	"missing_p.h"
#include	"elcerror_p.h"
#include	"xml_p.h"
#ifdef		ELC_FIND_LEAKS
#include	"leakfind.h"
#endif

static void print_missing_xml(FILE *out, options *opts,
			      const char *key, size_t klen)
{
    XML_START_PRINT(out, "missing");

    for (; klen--; ++key)
      xml_putc(out, *key);

    XML_END_PRINT(out, "missing");
    putc('\n', out);
}

static void print_missing_abbrev(FILE *out, options *opts,
			      const char *key, size_t klen)
{
    fputs("missing: ", stdout);
    while (klen--)
      putchar(*key++);
    putchar('\n');
}

static void do_currdb_check(options *opts, struct cdb *curr_cdb,
			    char *key, size_t klen)
{
    int		ret;

    if ( (ret = cdb_find(curr_cdb, key, klen)) == -1)
      die(__FUNCTION__,
	  "Error: looking up file (%s) in current-state database (%s): %s",
	  key, opts->currdbname, strerror(errno));
    else if (! ret) {
      if (opts->output == OUTPUT_XML)
	print_missing_xml(stdout, opts, key, klen);
      else
	print_missing_abbrev(stdout, opts, key, klen);
    }
}

/* based on open_known_cdb in main.c */
void open_current_state_cdb(const char *dbname, struct cdb *curr_cdb)
{
    int		fd	 = open(dbname, O_RDONLY | O_NDELAY);

    if (fd == -1)
      die(__FUNCTION__, "Error: opening current-state database (%s): %s",
	  dbname, strerror(errno));
    cdb_init(curr_cdb, fd);
}

void check_for_missing(options *opts)
{
    cdb_seq	knownseq;
    struct cdb	curr_cdb;
    int		err;
    unsigned	ksiz	 = 1024; /* beginning key space allotment */
    char	*key	 = malloc(ksiz);
    size_t	klen, vlen;

    if (! key)
      DIE("malloc key");

    open_current_state_cdb(opts->currdbname, &curr_cdb);

    if ((opts->verbose > 0)
	&& (opts->output != OUTPUT_XML))
      puts(PROGNAME ": checking for missing files --------------");

    knownseq.fd	 = opts->knowndb.fd;
    if (cdb_seq_start(&knownseq) == -1)
      DIE("cdb_seq_start");
    while (! cdb_seq_eod(&knownseq)) { /* while not end of data */
      if ( (err = cdb_seq_sizes(&knownseq, &klen, &vlen)) == -1)
	DIE("cdb_seq_sizes");
      else if (err == 1)	/* no more in db sequence */
	break;
      while (klen > ksiz) {
	if (ksiz > (UINT_MAX / 2))
	  DIE("key size too big");
	ksiz	 *= 2;
	if (! (key = realloc(key, ksiz)) )
	  DIE("realloc key");
      }
      if (cdb_seq_getkey(&knownseq, key, klen, vlen) == -1)
	DIE("cdb_seq_getkey");
      do_currdb_check(opts, &curr_cdb, key, klen);
    }

    free(key);
}
