/*
 * This file is part of Distributed John,
 * Copyright (C) 2001-2003 by Luis Parravicini
 * http://ktulu.com.ar
 */

#define _XOPEN_SOURCE


#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/select.h>
#include <netdb.h>

#include "defs.h"
#include "word.h"
#include "netutils.h"
#include "ipc.h"
#include "intl.h"

extern int  inet_aton(const char *cp, struct in_addr * addr);
#ifndef __FreeBSD__
extern void srandom(unsigned int seed);
#endif
extern long int random(void);

/* #define MAX_RENEWAL	3 */
#define MAX_WORK	50
long	UNIT_TTL = 600;	/* time to live of a packet (in seconds) */
/* amount of words per packet */
long	MAX_WPU = 10000;/* 15:00 in a P133, 6:30 in a K6-2 500 0:45 in a PIII 600, 0:15 in a Xeon 2.2GHz */
long	SAVE_DELAY = 300;/* every SAVE_DELAY seconds the state is saved. */ 
#define setSerial(X) time(&X);

/* packet's possible states */
#define W_FREE		0
#define W_WORKING	1
#define W_REASSIGN	2
/*
 * in how many possible states can be a work packet (all the above states
 * (W_FREE, W_WORKING, W_REASSIGN)).
 */
#define MAX_STATES	3

struct trabajo {
  time_t	pedido;
  long int	id;
/*  time_t	renovado;
  char		renovadas; */
  struct sockaddr_in client;
  struct word palabra1;
  struct word palabra2;
  int		cant;
  char		estado;
  int		sig;	
};

struct data {
	struct trabajo packets[MAX_WORK];
	int cant[MAX_STATES]; /* cantidad de elementos usados en cada estado */
	int idx[MAX_STATES]; /* primer elemento dentro del arreglo segun
				el estado (W_FREE,W_WORKING,W_REASSIGN) */
	struct word	palabra;	/* palabra actual */
	struct word	ultpalabra; /* ultima palabra a probar */
	struct word	primpalabra;/* esta palabra no se prueba, sino la siguiente */
};

#define FLAG_ELAPSED	1
#define FLAG_HUMAN	2
#define FLAG_NUM	4


int		flags;
struct data	*d;
int		fin = 0;
char		now[20];
struct in_addr	bindto;
long int	last_id;
int		shmid[2];
int		verbose;
struct sockaddr_in sa;		/* Addres of the client	*/
int server;			/* Server socket	*/


#ifndef MIN
#define MIN(a,b)	(((a)<(b)) ? (a) : (b))
#endif


/* returns a string with the current time. */
char* ahora()
{
	time_t t = time(NULL);
	char *aux = ctime(&t);
	/* esto vale o es muy chancho?? */
	aux[strlen(aux)-6] = 0;
	return aux;
/*  strftime(now, sizeof(now), "%b %d %T",  */
}

void pferror(FILE *f, char *msg)
{
	fprintf(stderr, msg);
	fclose(f);
}

/* generate new packet id */
long int next_id()
{
	srandom(time(NULL));
	last_id += (double)random()*100/RAND_MAX;
	return last_id;
}

/* saves the state to a file. */
int savestate()
{
	FILE *out;
	char s[MAX_WORD], s1[MAX_WORD], s2[MAX_WORD];
	int i;

	if ((out=fopen("djohn.restore", "w")) == NULL) {
		fprintf(stderr, _("can't open djohn.restore for writing\n"));
		return -1;
	}

	fprintf(out, "%s\n%03d\n", inet_ntoa(bindto), charsetl);
	for (i = 0; i < charsetl; i++) {
		fprintf(out, "%02x", charset[i]);
		if (i != charsetl-1) fprintf(out, " ");
	}
	fprintf(out, "\n%s\n%s\n%s\n", wordtostr(&d->primpalabra, &s[0]),
		wordtostr(&d->ultpalabra, &s1[0]), wordtostr(&d->palabra,
		&s2[0]));
	fprintf(out, "%ld\n", UNIT_TTL);
	for (i = 0; i < MAX_WORK; i++)
		if (d->packets[i].estado == W_WORKING ||
		d->packets[i].estado == W_REASSIGN)
			fprintf(out, "%s\n%s\n", wordtostr(
				&d->packets[i].palabra1, &s[0]),
				wordtostr(&d->packets[i].palabra2, &s1[0]));
	fclose(out);
	return 0;
}

int send_to_client(int socket, char *buf, int buf_size, struct sockaddr_in *sa)
{
	int written;

	if ((written=sendto(socket, buf, buf_size, 0, (struct sockaddr *)sa,
	sizeof(*sa))) == -1) {
		perror(_("send_to_client"));
		return -1;
	}
	if (written != buf_size) {
		fprintf(stderr, _("send_to_client: couldn't send all the "\
			"data to the client.\n"));
		return -1;
	}

	return 0;
}


/*
 * reads a line from the file, if line == -1 does not print a message if
 * an error occurred.
 */
int freadline(char *buf, int buf_size, FILE *in, int line)
{
	if (fgets(buf, buf_size, in) == NULL || buf[strlen(buf)-1] != '\n') {
		if (line != -1) {
			char s[1024];	/* this SHOULD be enough space */
			sprintf(s, _("error reading line %u.\n"), line);
			pferror(in, s);
		}
		return -1;
	}
	if (strlen(buf)) buf[strlen(buf)-1] = 0;
	return 0;
}

/* restores state saved in a file. */
int restorestate()
{
	FILE *in;
	char s1[770], s2[770];
	int i, j, pos, line = 0;

	if ((in=fopen("djohn.restore", "r")) == NULL) {
		fprintf(stderr, _("restorestate: can't open djohn.restore\n"));
		return -1;
  	}

	/* get the ip to bind the socket */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if (!inet_aton(s1, &bindto)) {
		pferror(in, _("restorestate: error converting server address\n"));
		return -1;
	}

	/* get the charset length */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if (strlen(s1) < 3 || sscanf(s1, "%d", &charsetl) == 0) {
		pferror(in, _("restorestate: error converting charsetl\n"));
		return -1;
	}

	/* get the charset */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	charset = (unsigned char *)malloc(sizeof(unsigned char)*charsetl);
	i = 0;
	pos = 0;
	do {
		if (sscanf(&s1[i], "%x", &j) == 0 || j > 256) {
			pferror(in, _("restorestate: error converting "\
				"charset\n"));
			return -1;
		}
		charset[pos++] = j;
		i += 3;
	} while (i < strlen(s1));
	if (pos != charsetl) {
		pferror(in, _("restorestate: too few bytes read\n"));
		return -1;
        }

	/* get the first word */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if ((i=strtoword(s1, &d->primpalabra))) {
		show_error(i, _("first word:"));
		fclose(in);
		return -1;
	}

	/* get the last word */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if ((i=strtoword(s1, &d->ultpalabra))) {
    		show_error(i, _("last word:"));
		fclose(in);
		return -1;
	}

	/* get the current word */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if (strlen(s1) >= sizeof(d->palabra)) {
		pferror(in, _("restore: error reading current word\n"));
		return -1;
	}
	if ((i=strtoword(s1, &d->palabra))) {
    		show_error(i, _("current word:"));
		fclose(in);
		return -1;
	}

	/* get the ttl */
	if (freadline(s1, sizeof(s1), in, line++))
		return -1;
	if (!sscanf(s1, "%ld", &UNIT_TTL)) {
		pferror(in, _("restorestate: error converting unit_ttl\n"));
		return -1;
	}
	if (UNIT_TTL < 0) {
		pferror(in, _("restorestate: unit_ttl is < 0\n"));
		return -1;
	}

	i = 0;
	memset(d->cant, 0, sizeof(d->cant));
	d->idx[W_REASSIGN] = d->idx[W_WORKING] = -1;
	for (i = 0; i < MAX_WORK; i++) {
		d->packets[i].estado = W_FREE;
		d->packets[i].sig = (i == MAX_WORK-1 ? -1 : i+1);
	}
	i = 0;
	do {
		if (freadline(s1, sizeof(s1), in, -1))
			break;
		line++;
		if (freadline(s2, sizeof(s2), in, line++))
			return -1;
		if (strlen(s1) == 0 || strlen(s2) == 0) {
			pferror(in, _("restore: there's something wrong "\
				"inside djohn.restore\n"));
			return -1;
		}
		d->packets[i].estado = W_REASSIGN;
		d->packets[i].sig = d->idx[W_REASSIGN];
		d->idx[W_REASSIGN] = i;
		strtoword(s1, &d->packets[i].palabra1);
		strtoword(s2, &d->packets[i++].palabra2);
		d->cant[W_REASSIGN]++;
	} while (1);
	d->cant[W_FREE] = MAX_WORK - d->cant[W_REASSIGN];
	d->idx[W_FREE] = (d->cant[W_FREE] == 0 ? -1 : i);
	fclose(in);
	return 0;
}

void salida2(void)
{
	shmctl(shmid[0], IPC_RMID, NULL);
	if (shmid[1] != -1)
		shmctl(shmid[1], IPC_RMID, NULL);
}

void salida(int signum)
{
	savestate();
	exit(0);
}

/*
 * tells to a client to sleep until a there may be work to do. This is done
 * by making an estimate of when the first assigned work packet will timeout.
 */
int sleep_client(int s, struct sockaddr_in client)
{
	int i;
	long int t, t0;
	struct sleep sl;

	if (d->idx[W_WORKING] == -1) t = UNDEF_SLEEP;
	else {
		t0 = t = time(NULL);
		for (i=d->idx[W_WORKING]; i != -1; i=d->packets[i].sig)
			if (t > d->packets[i].pedido)
				t = d->packets[i].pedido;
		if ((t=UNIT_TTL - (t0 - t)) < 0) t = 1;
	}
	sl.action = NOW_SLEEP;
	sl.secs = t;
	if (send_to_client(s, (char *)&sl, sizeof(sl), &client))
		return -1;

	return 0;
}

struct word *nextword(struct word *palabra, int cant)
{
	int i, j;

	while (cant && !fin) {
		/* based in generate() from LanMan in john.ini */
		i = j = palabra->length;
		i--;
		while (++palabra->data[i] == charsetl)
			if (i)
				palabra->data[i--] = 0;
			else
				if (j < MAX_WORD-1) {
					palabra->data[0] = 0;
					palabra->data[palabra->length++] = 0;
					break;
				}
		if (!wordcmp(*palabra, d->ultpalabra)) break;
		cant--;
	}
	if (!wordcmp(*palabra, d->ultpalabra))
		fin = 1;
	else
		if (fin) memcpy(palabra, &d->ultpalabra, sizeof(struct word));

	return palabra;
}

/* sends a work packet to a client */
int sendwork(int socket, int addr, int pos)
{
	struct sockaddr_in sa;
	int escrito;
	struct firstword fw;
	char s[MAX_WORD], s1[MAX_WORD];

	fw.action = FIRST_WORD;
	fw.unit_id = d->packets[pos].id;
	memcpy(&fw.word1, &d->packets[pos].palabra1, sizeof(struct word));
	memcpy(&fw.word2, &d->packets[pos].palabra2, sizeof(struct word));
	memset((char *)&sa, 0, sizeof(sa));

	if (verbose)
		printf("%ld\t%s\t%s", d->packets[pos].id,
			wordtostr(&d->packets[pos].palabra1, &s[0]),
			wordtostr(&d->packets[pos].palabra2, &s1[0]));

	sa.sin_family = AF_INET;
	sa.sin_addr.s_addr = d->packets[pos].client.sin_addr.s_addr;
	sa.sin_port = d->packets[pos].client.sin_port;
	if (send_to_client(socket, (char *)&fw, sizeof(fw), &sa))
		return -1;

	return escrito;
}

/* creates a new work packet */
int creatework(struct sockaddr_in client, int pos)
{
	int oldpos = d->idx[pos], i;

	d->packets[d->idx[pos]].estado = W_WORKING;
	d->packets[d->idx[pos]].client = client;
	d->packets[d->idx[pos]].pedido = time(NULL);
	d->packets[d->idx[pos]].id = next_id();
/*  d->packets[d->idx[pos]].renovado = d->packets[d->idx[pos]].pedido;
  d->packets[d->idx[pos]].renovadas = MAX_RENEWAL; */
	d->packets[d->idx[pos]].cant = MAX_WPU;
	if (pos == W_FREE) {
		memcpy(&d->packets[d->idx[pos]].palabra1, &d->palabra,
			sizeof(struct word));
		memcpy(&d->packets[d->idx[pos]].palabra2, nextword(&d->palabra,
			d->packets[d->idx[pos]].cant-1), sizeof(struct word));
		nextword(&d->palabra, 1);
	}

	i = d->packets[d->idx[pos]].sig;
	d->packets[d->idx[pos]].sig = d->idx[W_WORKING];
	d->idx[W_WORKING] = d->idx[pos];
	d->idx[pos] = i;

	return oldpos;
}

/* mark the units which have timedout to be reassigned */
void expireunits()
{
	time_t now = time(NULL);
	int i, j;

	i = d->idx[W_WORKING];
	j = -1;
	while (i != -1) {
		if (d->packets[i].estado == W_WORKING && now - d->packets[i].pedido > UNIT_TTL) {
			if (verbose)
				printf("%s %s:%d\tEXPIRED\t%ld\n", ahora(),
				inet_ntoa(d->packets[i].client.sin_addr),
				ntohs(d->packets[i].client.sin_port),
				d->packets[i].id);
			d->cant[W_REASSIGN]++;
			d->cant[W_WORKING]--;
			if (d->idx[W_WORKING] == i)
				d->idx[W_WORKING] = d->packets[i].sig;
			if (j != -1)
				d->packets[j].sig = d->packets[i].sig;
			d->packets[i].estado = W_REASSIGN;
			d->packets[i].sig = d->idx[W_REASSIGN];
			d->idx[W_REASSIGN] = i;
		}
		j = i;
		i = d->packets[i].sig;
	}
}

#ifdef DEBUGGING
void dumpvars()
{
	int i;
	char s[MAX_WORD];

	printf("word=[%s]\tUNIT_TTL=[%ld]\n", wordtostr(&d->palabra, &s[0]),
		UNIT_TTL);
	printf("cant[FREE]=%d\tcant[WORKING]=%d\tcant[REASSIGN]=%d\n",
		d->cant[W_FREE], d->cant[W_WORKING], d->cant[W_REASSIGN]);
	printf("idx[FREE]=%d\tidx[WORKING]=%d\tidx[REASSIGN]=%d\n",
		d->idx[W_FREE], d->idx[W_WORKING], d->idx[W_REASSIGN]);
	printf("WORKING=[");

	i = d->idx[W_WORKING];
	while (i != -1) {
		printf("%d", i);
		i = d->packets[i].sig;
		if (i != -1) printf(",");
	}
	printf("]\n");

	printf("FREE=[");
	i = d->idx[W_FREE];
	while (i != -1) {
		printf("%d", i);
		i = d->packets[i].sig;
		if (i != -1) printf(",");
	}
	printf("]\n");

	printf("REASSIGN=[");
	i = d->idx[W_REASSIGN];
	while (i != -1) {
		printf("%d", i);
		i = d->packets[i].sig;
		if (i != -1) printf(",");
	}
	printf("]\n");
}
#endif

/* sends the current charset to the client */
int sendcharset(int socket, struct sockaddr_in client)
{
	struct sockaddr_in sa;
	int escrito;
	struct charsetdata gcs;

	gcs.action = CHARSET_DATA;
	gcs.length = charsetl;
	memcpy(gcs.charset, charset, charsetl);
	memset((char *)&sa, 0, sizeof(sa));
	sa.sin_family = AF_INET;
	sa.sin_addr.s_addr = client.sin_addr.s_addr;
	sa.sin_port = client.sin_port;
	if (send_to_client(socket, (char *)&gcs, charsetl+2, &sa))
		return -1;

	return escrito;
}

/* saves the passwords containes in the struct to a file. */
int savepasswd(struct finishwork *fw)
{
	FILE *out;
	char *aux;
	int cant;

	if ((out=fopen("djohn.pot", "a")) == NULL) {
		fprintf(stderr, _("can't append djohn.pot\n"));
		return -1;
	}
	aux = (char *)fw;
	aux += sizeof(struct finishwork);
	cant = fw->cracked;
	while (cant--) {
		fputs(aux, out);
		fputs("\n", out);
		aux += strlen(aux)+1;
	}
	fclose(out);

	return -1;
}

void opt_init_djohnd(int argc, char **argv)
{
	int i, ip, restore;
	long ttl, words;
	char *charset, *fw, *lw;
#ifdef DEBUGGING
	char s[MAX_WORD], s1[MAX_WORD];
#endif

	charset = fw = lw = NULL;
	restore = 0;
	d->primpalabra.length = d->ultpalabra.length = charsetl = ip = 0;
	while ((i=getopt(argc, argv, "vw:b:c:f:l:t:rh")) != -1)
		switch (i) {
			case 'b':
				if (!inet_aton(optarg, &bindto)) {
					fprintf(stderr, _("invalid ip\n"));
					exit(1);
				}
				ip = 1;
				break;
			case 'c':
				charset = optarg;
				break;
			case 'f':
				fw = optarg;
				break;
			case 'l':
				lw = optarg;
				break;
			case 'r':
				restore = 1;
				break;
			case 'v':
				verbose++;
				break;
			case 't':
				if (sscanf(optarg, "%ld", &ttl)) {
					if (ttl < 0) {
						fprintf(stderr, _("the packet's ttl must be > 0\n"));
						exit(1);
					}
					UNIT_TTL = ttl;
				} else {
					fprintf(stderr, _("the packet's ttl must be a long\n"));
					exit(1);
				}
				break;
			case 'w':
				if (sscanf(optarg, "%ld", &words)) {
					if (words < 0) {
						fprintf(stderr, _("the words per packet must be > 0\n"));
						exit(1);
					}
					MAX_WPU = words;
				} else {
					fprintf(stderr, _("the words per packet must be a long\n"));
					exit(1);
				}
				break;
			case 'h':
			printf("Distributed John\n");
			printf("Copyright (C) 2001-2003 Luis Parravicini\n\n");
			printf(_("usage: djohnd [options]\n"));
			printf(_("-b ip : binds the server socket to 'ip'\n"));
			printf(_("-c charset : sets the charset\n"));
			printf(_("-f first_word : sets the first word to be tried\n"));
			printf(_("-l last_word : sets the last word to be tried\n"));
			printf(_("-t ttl : sets the packet's ttl (in seconds) (default: 600)\n"));
			printf(_("-w n : sets the amount of words per packet (default: 10000)\n"));
			printf(_("-r : restore previous session\n"));
			printf(_("-v : be verbose\n"));
			printf(_("-h : this help message\n"));
			exit(0);
			case ':': exit(1);
		}

	/* INADDR_ANY */
	if (!ip) inet_aton("0.0.0.0", &bindto);

	if (restore && restorestate()) {
		fprintf(stderr, _("using command line arguments\n"));
		restore = 0;
	}
	if (!restore) {
		if (charset == NULL || mkcharset(charset) < 2) {
			fprintf(stderr, _("the charset must have at least 2 characters\n"));
			exit(1);
		}
		if ((i=strtoword(fw, &d->primpalabra))) {
			show_error(i, _("first word:"));
			exit(1);
		}
 		if ((i=strtoword(lw, &d->ultpalabra))) {
			show_error(i, _("last word:"));
			exit(1);
		}
	}

	if (wordlen(&d->ultpalabra) < wordlen(&d->primpalabra)) {
		fprintf(stderr, _("the last word is shorter than the first!\n"));
		exit(1);
	}
	if (!restore) {
		if (charsetl < 2) {
			fprintf(stderr, _("the charset must have at least 2 characters\n"));
			exit(1);
		}
		memset(d->cant, 0, sizeof(d->cant));
		d->cant[W_FREE] = MAX_WORK;
		d->idx[W_FREE] = 0;
		d->idx[W_WORKING] = d->idx[W_REASSIGN] = -1;
		for (i = 0; i < MAX_WORK; i++) {
			d->packets[i].estado = W_FREE;	
			d->packets[i].sig = (i == MAX_WORK-1 ? -1 : i+1);
		}
		if (wordlen(&d->primpalabra) == 0) {
			fprintf(stderr, _("the length of the first word must be > 0\n"));
			exit(1);
		}
		if (wordlen(&d->ultpalabra) == 0) {
			fprintf(stderr, _("the length of the last word must be > 0\n"));
			exit(1);
		}
#ifdef DEBUGGING
		printf("first_word: [%s] last_word: [%s]\n",
			wordtostr(&d->primpalabra, &s[0]),
			wordtostr(&d->ultpalabra, &s1[0]));
#endif
		memcpy(&d->palabra, &d->primpalabra, sizeof(struct word));
	} else
		if (wordcmp(d->palabra, d->ultpalabra) >= 0) {
			printf(_("finished searching\n"));
			exit(0);
		}
}

void opt_init_djs(int argc, char **argv)
{
	int i;

	flags = 0;
	while ((i=getopt(argc, argv, "eHhn")) != -1)
		switch (i) {
			case 'e':
				flags |= FLAG_ELAPSED;
				break;
			case 'H':
				flags |= FLAG_HUMAN;
				break;
			case 'n':
				flags |= FLAG_NUM;
				break;
			case 'h':
				printf("Distributed John\n");
				printf("Copyright (C) 2001-2003 Luis Parravicini\n\n");
				printf(_("usage: djs [options]\n"));
				printf(_("-e : elapsed time of each packet\n"));
				printf(_("-H : show time of each packet in human readable form\n"));
				printf(_("-n : dont resolve names\n"));
				printf(_("-h : this help message\n"));
				exit(0);
			case ':': exit(1);
		}
}

/* put charset in shared memory */
int mkcharsetsh()
{
	char *aux;

	if ((aux=newshmbuff(getIPCCharsetKey(), &shmid[1], charsetl, 1)) == NULL) 
		return 1;
	memcpy(aux, charset, charsetl);
	free(charset);
	charset = aux;

	return 0;
}

/* creates the shared memory region and registers the cleanup function. */
int init_shmem(int create)
{
	memset(shmid, -1, sizeof(shmid));
	if ((d=(struct data *)newshmbuff(getIPCKey(), &shmid[0],
	sizeof(struct data), create)) == NULL)
		return 1;
	if (!create) {
		free(charset);
		if ((charset=newshmbuff(getIPCCharsetKey(), &shmid[1],
		charsetl, create)) == NULL)
			return 2;
	}
	if (create) atexit(salida2);

	return 0;
}

void printPacket(struct trabajo p, char state)
{
	char s1[MAX_WORD], s2[MAX_WORD];
	char aux[21], stime[20];
	time_t now = time(NULL), time;
	struct hostent *st;
	int i, j;

	if (flags & FLAG_NUM || (st=gethostbyaddr((char *)&p.client.sin_addr,
	sizeof(p.client.sin_addr), AF_INET)) == NULL)
		sprintf(aux, "%s:%d", inet_ntoa(p.client.sin_addr),
			htons(p.client.sin_port));
	else {
		i = 1; j = htons(p.client.sin_port);
		while (j >= 10) { j /= 10; i++; }
		sprintf(strncpy(aux, st->h_name, 20-i) + MIN(20-i,
			strlen(st->h_name)), ":%d", htons(p.client.sin_port));
	}
	time = (flags & FLAG_ELAPSED ? now - p.pedido : p.pedido);
	if (flags & FLAG_HUMAN) {
		if (flags & FLAG_ELAPSED)
			if (time / 86400 > 9)
				sprintf(stime, "*:%02lu:%02lu:%02lu",
					(unsigned long) time % 86400 / 3600,
					(unsigned long) time % 3600 / 60,
					(unsigned long) time % 60);
			else
				sprintf(stime, "%lu:%02lu:%02lu:%02lu",
					(unsigned long) time / 86400,
					(unsigned long) time % 86400 / 3600,
					(unsigned long) time % 3600 / 60,
					(unsigned long) time % 60);
		else
			strftime(stime, sizeof(stime), "%Y-%m-%d %T",
				gmtime(&time));
	} else
		sprintf(stime, "%-10lu", (unsigned long) time);
	printf("%s %-21s %-15s %-15s %-8d %c\n", stime, aux,
		wordtostr(&p.palabra1, &s1[0]), wordtostr(&p.palabra2, &s2[0]),
		p.cant, state);
}

int djs(int argc, char **argv)
{
	int i;
	char s1[MAX_WORD];

	init_i18n();
	opt_init_djs(argc, argv);
	if (init_shmem(0)) {
		fprintf(stderr, _("Is djohnd running?\n"));
		return 1;
	}

	printf(_("First word = [%s]\t"), wordtostr(&d->primpalabra, &s1[0]));
	printf(_("Last word = [%s]\t"), wordtostr(&d->ultpalabra, &s1[0]));
	printf(_("Current word = [%s]\n\n"), wordtostr(&d->palabra, &s1[0]));
	printf(_("State\t\tPackets\n"));
	printf(_("FREE\t\t%d\nWORKING\t\t%d\nREASSIGN\t%d\n"), d->cant[W_FREE],
		d->cant[W_WORKING], d->cant[W_REASSIGN]);

	printf(_("\nTime       "));
	if (flags & FLAG_HUMAN && !(flags & FLAG_ELAPSED))
		printf(_("         "));
	printf(_("Client                First word      Last word       Words    State\n"));
	i = d->idx[W_WORKING];
	while (i != -1) {
		printPacket(d->packets[i], 'W');
		i = d->packets[i].sig;
	}

	i = d->idx[W_REASSIGN];
	while (i != -1) {
		printPacket(d->packets[i], 'R');
		i = d->packets[i].sig;
	}
	printf("\n");

	return 0;
}

void get_work()
{
	int i;

	if (verbose)
		printf("%s %s:%d\tGET_WORK\t", ahora(),
			inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));
	if (d->cant[W_REASSIGN] > 0) {
		i = d->idx[W_REASSIGN];
		d->cant[W_REASSIGN]--;
		d->cant[W_WORKING]++;
		sendwork(server, sa.sin_addr.s_addr,
			creatework(sa, W_REASSIGN));
		if (verbose) printf(" reassign %d\n", i);
		return;
	}
	if (fin) {
		if (verbose) printf("no more packets\n");
		sleep_client(server, sa);
		return;
	}
	if (d->cant[W_FREE] > 0) {
		i = d->idx[W_FREE];
		d->cant[W_FREE]--;
		d->cant[W_WORKING]++;
		sendwork(server, sa.sin_addr.s_addr, creatework(sa, W_FREE));
	  	if (verbose)
			printf(" new %d\n", i);
	} else
		if (verbose) printf("full\n");
}

void finish_work(char *buf)
{
	int i,j;
	time_t hora;
	struct finishwork *fw;

	if (verbose)
		printf("%s %s:%d\tFINISH_WORK\t", ahora(),
			inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));
	hora = time(NULL);
	fw = (struct finishwork *)buf;
	i = d->idx[W_WORKING];
	j = -1;
	while (i != -1 && d->packets[i].id != fw->unit_id) {
		if (d->packets[i].client.sin_addr.s_addr == sa.sin_addr.s_addr
		&& d->packets[i].client.sin_port == sa.sin_port) {
			i = -2;
			break;
		}
		j = i;
		i = d->packets[i].sig;
	}
	if (i == -2) {
		fprintf(stderr, _("correct unit_id (%ld) but from a different"\
			"client (recvedfrom=%s:%d"), fw->unit_id,
			inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));
		fprintf(stderr, _(", owner=%s:%d)\n"),
			inet_ntoa(d->packets[i].client.sin_addr),
			ntohs(d->packets[i].client.sin_port));
	} else
	if (i == -1)
		fprintf(stderr, _("bogus unit_id! (%ld)\n"), fw->unit_id);
	else {
		hora -= d->packets[i].pedido;
		if (verbose)
			printf("%ld\tdone. got %d/%d in %ld:%02ld:%02ld\n",
				fw->unit_id, fw->cracked, d->packets[i].cant,
				(long) hora/3600,
				(long) (hora/60) % 60,
				(long) hora % 60);
		savepasswd(fw);
		d->cant[W_FREE]++;
		d->cant[W_WORKING]--;
		if (d->idx[W_WORKING] == i)
			d->idx[W_WORKING] = d->packets[i].sig;
		if (j != -1)
			d->packets[j].sig = d->packets[i].sig;
	    d->packets[i].estado = W_FREE;
	    d->packets[i].sig = d->idx[W_FREE];
	    d->idx[W_FREE] = i;
	}
}

void get_charset()
{
	if (verbose)
		printf("%s %s:%d\tGET_CHARSET\t", ahora(),
			inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));
	sendcharset(server, sa);
	if (verbose) printf("\n");
}

/* waits x msecs for data in the socket. */
int data_ready(int sock)
{
	fd_set rfds;
	struct timeval tv;
	int r;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);
	tv.tv_sec = 0;
	tv.tv_usec = 50000;

	if ((r=select(sock+1, &rfds, NULL, NULL, &tv)) == -1) {
		perror("data_ready");
		exit(1);
	}
	return (r == 1);
}

int djohnd(int argc, char *argv[])
{
	int leido;
	int tam;
	char buf[61440];
	time_t serial, aux_time;
	struct finishwork *fw;

	init_i18n();
	last_id = time(NULL);
	verbose = 0;
	if (init_shmem(1)) return 1;
	opt_init_djohnd(argc, argv);
	mkcharsetsh();
#ifdef DEBUGGING
	dumpvars();
#endif
	setSerial(serial);
	printf(_("Starting on %s"), ctime(&serial));
	printf(_("using serial id %ld\n"), (long) serial);
	/* para que es SIGQUIT??? */
	if (signal(SIGQUIT, salida) == SIG_ERR)
		fprintf(stderr, _("can't set signal handler for SIGQUIT\n"));
	if (signal(SIGTERM, salida) == SIG_ERR)
		fprintf(stderr, _("can't set signal handler for SIGTERM\n"));
	if (signal(SIGINT, salida) == SIG_ERR)
		fprintf(stderr, _("can't set signal handler for SIGINT\n"));

	if ((server=mkserver(&bindto, SERVER_PORT)) == -1) return 1;

	aux_time = time(NULL);
	do {
#ifdef DEBUGGING
		dumpvars();
#endif
		/* wait for a client request and save the state if needed */
		do {
			if (aux_time + SAVE_DELAY < time(NULL)) {
				savestate();
				aux_time = time(NULL);
			}
			expireunits();
		} while (!data_ready(server));

		tam = sizeof(sa);
		if ((leido=recvfrom(server, &buf, sizeof(buf), 0,
		(struct sockaddr*)&sa, &tam)) == -1) {
			fprintf(stderr, _("%s djohnd.main: couldn't read "\
				"dgram: "), ahora());
			perror("");
			continue;
	  	}
		if (leido < 1) {
			fprintf(stderr, _("%s %d bytes read from %s:%d (?)\n"),
				ahora(), leido, inet_ntoa(sa.sin_addr),
				ntohs(sa.sin_port));
			continue;
		}
		fw = (struct finishwork *)buf;
		if ((fw->action != FINISH_WORK &&
		leido != packetsize(fw->action)) ||
		(fw->action == FINISH_WORK && leido != fw->size)) {
			fprintf(stderr, _("%s not all the datagram from "\
				"client %s:%d could be read (read=%d/action"\
				"=%d)\n"), ahora(), inet_ntoa(sa.sin_addr),
				ntohs(sa.sin_port), leido, fw->action);
			continue;
		}
		switch (buf[0]) {
			case GET_WORK:
				get_work(buf);
				break;
			case FINISH_WORK:
				finish_work(buf);
				break;
			case GET_CHARSET:
				get_charset(buf);
				break;
			default:
				fprintf(stderr, _("%s %s:%d\tunknown "\
					"action\t%d\n"), ahora(),
					inet_ntoa(sa.sin_addr),
					ntohs(sa.sin_port), buf[0]);
		}
	} while (1);
}

/* stolen from john main */
int main(int argc, char *argv[])
{
	char *name;

	if (!argv[0])
		name = "";
	else
	if ((name = strrchr(argv[0], '/')))
		name++;
	else
		name = argv[0];

	if (!strcmp(name, "djohnd"))
		return djohnd(argc, argv);

	if (!strcmp(name, "djs"))
		return djs(argc, argv);

	fprintf(stderr, "Sorry, I can't find myself\n");
	return 1;
}
