/*
 * VMS patch 5 for John the Ripper version 1.6.32
 * (C) 2002 Jean-loup Gailly  http://gailly.net
 * license: GPL <http://www.gnu.org/>
 *
 * This file is based on code from John the Ripper,
 *   Copyright (c) 1996-2002 by Solar Designer.
 * Original hash algorithm by Shawn Clifford in February 1993.
 * Code from VMSCRACK  by Davide Casale.
 * Code optimizations by Mario Ambrogetti in June 1994.
 * Code fixes with the help of Terence Lee.
 */

#include <ctype.h>

#include "formats.h"
#include "params.h"
#include "VMS_std.h"

#define FORMAT_LABEL		"vms" /* MUST be lowercase, see john.c */
#define FORMAT_NAME		"OpenVMS"
#define ALGORITHM_NAME          "Purdy"

#define VMS_MAGIC "$VMS"

#define BENCHMARK_COMMENT	""
#define BENCHMARK_LENGTH	-1

#define BINARY_KEY_SIZE         sizeof(qword)
#define SALT_SIZE               sizeof(salt_t)

#define MIN_KEYS_PER_CRYPT	1
#define MAX_KEYS_PER_CRYPT	1

/* convert little-endian hex strings to integers */
#define hex2byte(string) (((atoi16[ARCH_INDEX((string)[0])])<<4) | \
                            atoi16[ARCH_INDEX((string)[1])])
#define hex2word(string)  ((hex2byte((string)+2)<<8)  | hex2byte(string))
#define hex2dword(string) ((hex2word((string)+4)<<16) | hex2word(string))

#define PLAINTEXT_LENGTH	32

/* username (32 chars max) must be included in salt for john because it is
 * used in the hash computation. The ciphertext is thus encoded (for john) as:
 * $VMSn$salthashusername
 *   n is '1' to '3'
 *   salt and hash are hex encoded on 4 and 16 bytes respectively.
 * USER_SIZE is the size of the user name, with space for an extra null byte.
 * username must be stripped of spaces and converted to upper case
 * by the unuaf program.
 */
#define USER_SIZE    (32 + 1)

typedef struct {
    word  salt; /* stored in cracking host format (not always little-endian) */
    byte  method;
    char  username[USER_SIZE];
} salt_t;

static qword crypt_key;
/* set by crypt_all, stored in cracking host format */

static salt_t saved_salt;   /* set by set_salt */

static char saved_plain[PLAINTEXT_LENGTH + 1]; /* set by set_key */

static struct fmt_tests tests[] = {
        {"$VMS1$F00F584719BEFDAAFB51TEST",        "TEST"},
        {"$VMS2$01100C95516F272EEEF5USERNAME",    "PASSWORD"},
        {"$VMS3$5FF5B04FCE64A83614AAJOHN$RIPPER", "JOHN$THE_RIPPER"},
        {"$VMS3$0000804F93CBB67389B1X",           ""},
        {NULL}
};

/* ============================================================================
 * Checks if an ASCII ciphertext is valid for this format. Returns zero for
 * invalid ciphertexts, or a number of parts the ciphertext should be split
 * into (up to 9, will usually be 1).
 * ciphertext is encoded as: $VMSn$salthashusername
 *   n is '1' to '3'
 *   salt and hash are hex encoded on 4 and 16 bytes respectively.
 */
static int valid(char *ciphertext)
{
    int n;
    char *p;

    if (!ciphertext || strncasecmp(ciphertext, VMS_MAGIC, 4)) {
        return 0;
    }
    /* check ciphertext is of the form: $VMSn$salthashusername
     *                          offset: 0     6   10  26
     * unuaf ensure that all the remaining checks will pass, but since the
     * checks cost very little we make them anyway.
     */
    n = strlen(ciphertext);
    if (n <= 6 + 4 + 16 || n >= 6 + 4 + 16 + USER_SIZE) return 0;

    n = ciphertext[4] - '0';
    if (n != UAIC_PURDY && n != UAIC_PURDY_V && n != UAIC_PURDY_S) {
	return 0;
    }
    if (ciphertext[5] != '$') return 0;

    /* check that salt and hash are in hex: */
    for (n = 6; n < 6 + 4 + 16; n++) {
	if (!isxdigit(ciphertext[n])) return 0;
    }
    /* check that username is valid for VMS and has no spaces: */
    p = ciphertext + 26;
    while (*p) {
        char c = *p;
	if (!isalnum(c) && c != '_' && c != '$') return 0;
	p++;
    }
    return 1;
}

/* ============================================================================
 * Converts an ASCII ciphertext to binary, possibly using the salt.
 * ciphertext is of the form: $VMSn$salthashusername
 *                    offset: 0     6   10  26
 * IN assertion: valid(ciphertext) == 1
 * The 8-byte binary hash is stored in cracking host format.
 */
static void *binary(char *ciphertext)
{
    static qword hash;

    if (valid(ciphertext) != 1) error();
    /* This check is not strictly needed but doesn't cost much since
     * this function is called only when loading the password file
     * and just once to verify a correct guess.
     */

    hash.d_low = hex2dword(ciphertext+10);
    hash.d_high = hex2dword(ciphertext+18);

    return (void *)&hash;
}

/* ============================================================================
 * Converts an ASCII salt to its internal representation
 */
static void *salt(char *ciphertext)
{
    static salt_t bin_salt;
    char *p;

    if (valid(ciphertext) != 1) error();
    /* This check is not strictly needed but doesn't cost much since
     * this function is called only when loading the password file
     */

    bin_salt.method = ciphertext[4] - '0';
    bin_salt.salt = hex2word(ciphertext+6);

    strnzcpy(bin_salt.username, ciphertext+26, USER_SIZE);

    /* Force upper case. Normally done by unuaf already but again not costly */
    p = bin_salt.username;
    while (*p) {
	*p = toupper(*p);
	p++;
    }
    return (void *)&bin_salt;
}

/* ============================================================================
 * These functions calculate a hash out of a binary ciphertext. To be used
 * for hash table initialization. One of the three should be used depending
 * on the hash table size.
*/
static int binary_hash_0(void *binary)
{
    return ((qword *)binary)->d_low & 0xF;
}

static int binary_hash_1(void *binary)
{
    return ((qword *)binary)->d_low & 0xFF;
}

static int binary_hash_2(void *binary)
{
    return ((qword *)binary)->d_low & 0xFFF;
}

/* ============================================================================
 * Calculates a hash out of a salt (given in internal representation).
 */
static int salt_hash(void *salt)
{
    return ((salt_t *)salt)->salt & (SALT_HASH_SIZE-1);
}

/* ============================================================================
 * Sets a salt for the crypt_all() method
 */
static void set_salt(void *salt)
{
    saved_salt = *(salt_t *)salt;
}

/* ============================================================================
 * Sets a plaintext, with index from 0 to fmt_params.max_keys_per_crypt - 1
 * Note: for efficiency, we assume that spaces have already been
 * removed and that the key uses only valid characters.
 * Since it doesn't cost much (we have to copy the string anyway)
 * we force the password to upper case also.
 */
static void set_key(char *key, int index)
{
    int count = PLAINTEXT_LENGTH+1;
    char *dest = saved_plain;

    while (--count) {
	char c = *key++;
	*dest++ = c >= 'a' && c <= 'z' ? c + 'A' - 'a' : c;
	if (c == '\0') break;
    }
    *dest = 0;
}

/* ============================================================================
 * Returns a plaintext previously set with set_key()
 */
static char *get_key(int index)
{
    return saved_plain;
}

/* ============================================================================
 * Calculates the ciphertexts for given salt and plaintexts.
 * The output (crypt_key) is stored in cracking host format.
 */
static void crypt_all(int count)
{
    VMS_lgihpwd(&crypt_key,
		saved_plain,
		saved_salt.method,
		saved_salt.salt,
		saved_salt.username);
}

/* ============================================================================
 * These functions calculate a hash out of a ciphertext that has just been
 * generated with the crypt_all() method.
 */
static int get_hash_0(int index)
{
    return crypt_key.d_low & 0xF;
}

static int get_hash_1(int index)
{
    return crypt_key.d_low & 0xFF;
}

static int get_hash_2(int index)
{
    return crypt_key.d_low & 0xFFF;
}

/* ============================================================================
 * Compares a given ciphertext against all the crypt_all() method outputs and
 * returns zero if no matches detected. A non-zero return value means that
 * there might be matches, and more checks are needed.
 * To simplify the code on big-endian systems, we check all 8 bytes of the key.
 */
static int cmp_all(void *binary, int index)
{
    return ((qword*)binary)->d_low  == crypt_key.d_low &&
           ((qword*)binary)->d_high == crypt_key.d_high;
}

/* ============================================================================
 * Compares an ASCII ciphertext against a particular crypt_all() output
 */
static int cmp_exact(char *source, int index)
{
    qword *hash = binary(source);

    return hash->d_low  == crypt_key.d_low &&
           hash->d_high == crypt_key.d_high;
}

/* ============================================================================
 * description of the VMS format
 */
struct fmt_main fmt_VMS = {
  {
    FORMAT_LABEL,
    FORMAT_NAME,
    ALGORITHM_NAME,
    BENCHMARK_COMMENT,
    BENCHMARK_LENGTH,
    PLAINTEXT_LENGTH,
    BINARY_KEY_SIZE,
    SALT_SIZE,
    MIN_KEYS_PER_CRYPT,
    MAX_KEYS_PER_CRYPT,
    FMT_8_BIT, /* upper case only */
    tests
  }, {
    fmt_default_init,
    valid,
    fmt_default_split,
    binary,
    salt,
    {
	binary_hash_0,
	binary_hash_1,
	binary_hash_2
    },
    salt_hash,
    set_salt,
    set_key,
    get_key,
    fmt_default_clear_keys, /* comment this line for john 1.6 */
    crypt_all, 
    {
	get_hash_0,
	get_hash_1,
	get_hash_2
    },
    cmp_all,
    cmp_all,
    cmp_exact
  }
};
