
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_notify.c
**  Date:    July 1, 2003
**
**  Author:  Brian Wotring
**  Purpose: notify admin of scan logs, initially this will be via email.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_log.h"
#include "md_config.h"
#include "md_compare.h"
#include "md_notify.h"

#include "common.h"
#include "logging.h"


extern OSI_MANAGEMENT_CONFIG *config;

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>

#include <openssl/ssl.h>
#include <openssl/err.h>

#ifdef WIN32

#include <winsock.h>
#include <errno.h>
#include <io.h>

#define EWOULDBLOCK WSAEWOULDBLOCK

#else   /* WIN32 */

#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/time.h>

#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>

#endif  /* WIN32 */

#include <fcntl.h>

#include "utilities.h"
#include "socketapi.h"

#define OSIRIS_SMTP_PORT        25

#define OSIRIS_SMTP_CMD_DATA    "DATA\r\n"
#define OSIRIS_SMTP_CMD_QUIT    "QUIT\r\n"

#define OSIRIS_SMTP_CMD_BUFSZ   2048
#define MAX_SMTP_LINE   128
#define LOGFILE_PERMISSIONS 0600

#define NOTIFY_SCHEDULE_FAIL_FORMAT  "The scheduler was unable to initiate the following scan:\r\n\r\nhost: %s\r\nscheduled scan time: %s\r\n\r\nThe scheduler produced the following error message:\r\n\r\n%s"

#define NOTIFY_SESSION_REKEY_FORMAT "The host, \"%s\" seems to have lost its \
resident session key.\r\nThis key is lost when the agent is restarted \
or the host is rebooted.\
\r\n\r\nThis host is configured to generate an alert upon detection of \
this event.\r\nTo change this, adjust the notification settings for \
this host.\r\n\r\n"

#define NOTIFY_UPDATE_URL_FORMAT "If these changes are approved, visit the \
URL below to set the latest scan database to become the trusted database. \
Alternately, you can login to the management console and set the trusted \
database to (%s).  \
If these notifications persist, you may need to modify the scan configuration \
for this host and/or set some comparison filters.\
\r\n\r\n    <https://%s:%d/?action=v&host=%s&base_db=%s&log=%s>\r\n\r\n"

#define NOTIFY_TEST_MESSAGE "This is a test notification message sent by the \
Osiris management console.\r\n"

/* 
 * Read a response from the SMTP server
 */

static int
read_smtp_response( int sock, char *expect )
{
    int i, pos;
    char buf[MAX_SMTP_LINE];

    /* not expecting a response, just succeed */

    if (expect == NULL)
    {
        return 0;
    }

    pos = 0;

    while (pos < MAX_SMTP_LINE)
    {
        if ((i = osi_read(sock, &buf[pos], MAX_SMTP_LINE - pos)) <= 0)
        {
            break;
        }

        while (i > 0)
        {
            if (buf[pos++] == '\n')
            {
                buf[pos] = '\0';

                if (strncmp(expect, &buf[0], strlen(expect)) == 0)
                {
                    /* server response matched the expected response */
                    return 0;
                }

                else
                {
                    log_error( LOG_ID_NOTIFY_ERROR, NULL,
                          "unrecognized SMTP response: (%s) expected: (%s)",
                           &buf[0], expect);

                    return 1;
                }
            }

            i--;
        }
    }

    return 1;
}

/*
 * Write a command string to the SMTP server, checking it's response
 */
static int
send_smtp_command(int sock, char *cmd, char *rsp)
{
    /* write a command to the SMTP server */

    if ((osi_write(sock, cmd, strlen(cmd))) == 0)
    {
        log_error( LOG_ID_NOTIFY_ERROR, NULL,
                   "error sending command (%s) to SMTP server.", cmd  );
    }

    /* read the response from the SMTP server */
    return read_smtp_response(sock, rsp);
}

/*
 * Open and write the contents of 'logfile' to the given socket
 */
static int
send_smtp_file(int sock, char *logfile)
{
    FILE *fp = NULL;
    int len = 0;

    char *line;
    char buf[OSIRIS_SMTP_CMD_BUFSZ];

    if ((fp = osi_fopen( logfile, "r", LOGFILE_PERMISSIONS)) == NULL)
    {
        log_error( LOG_ID_NOTIFY_ERROR, NULL,
                   "notification failure: error opening log file (%s).",
                   logfile );
        return 1;
    }

    for(;;)
    {
        char *temp;
        line = fgets( buf, sizeof( buf ), fp );

        if( line == NULL)
        {
            break;
        }

	    /* now strip off any newline characters. */

	    if( ( temp = strrchr( buf, '\r' ) ) )
        {
            *temp = '\0';
        }

        if( ( temp = strrchr( buf, '\n' ) ) )
        {
            *temp = '\0';
        }

	    /* now add our own. */

	    osi_strlcat( buf, "\r\n", sizeof( buf ) );
        len = strlen( buf );

        /* write the buffer to the SMTP server */

        if ((osi_write(sock, buf, len)) == 0)
        {
            log_error( LOG_ID_NOTIFY_ERROR, NULL,   
                       "unable to send data to SMTP server.");
            break;
        }
    }

    fclose(fp);
    return 0;
}

static int disconnect_from_mail_server( int sock )
{
    int result = 0;

    result = send_smtp_command( sock, OSIRIS_SMTP_CMD_QUIT, "250" );
    osi_close_socket( sock );

    return result;
}

static int connect_to_mail_server( const char *email, OSI_ERROR *error )
{
    int sock = 0;

    int smtp_port = config->notify_smtp_port;
    char *smtp_host = config->notify_smtp_host;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];
    OSI_ERROR err;

    int notify_email = 1;
    int admin_email  = 1;

    /* no error passed in, use a local one. */

    if( error == NULL )
    {
        error = &err;
    }

    if ( ( email == NULL ) || ( strlen( email ) == 0 ) )
    {
        notify_email = 0;
    }

    if ( ( config->admin_email == NULL ) ||
         ( strlen( config->admin_email ) == 0 ) )
    {
        admin_email = 0;
    } 

    if ( notify_email == 0 && admin_email == 0 )
    {
        osi_strlcpy( error->message,
                     "notification failure: no admin or notify address set!!.",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );
        return -1;
    }

    if( smtp_host == NULL )
    {
        osi_strlcpy( error->message,
                     "notification failure: SMTP server not set",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );

        return -1;
    }

    if( smtp_port == 0 )
    {
        /* fallback to the default SMTP port */

        smtp_port = OSIRIS_SMTP_PORT;
    }

    /* connect to the SMTP server */

    if( ( sock = osi_connect_to_host_on_port( smtp_host, smtp_port ) ) == 0 )
    {
        osi_snprintf( error->message, sizeof( error->message ),
                      "notification failure: unable to connect to (%s:%d).",
                      smtp_host, smtp_port );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );
        return -1; 
    }

    /* read the SMTP servers initial greeting */

    if( read_smtp_response( sock, "220" ) )
    {
        osi_strlcpy( error->message,
                     "notification failure: unfriendly SMTP server.",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );

        osi_close_socket( sock );
        return FALSE;
    }

    do
    {
        /* HELO */
        osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ, "HELO %s\r\n", smtp_host);

        if (send_smtp_command(sock, buf, "250"))
        {
            break;
        }

        if ( notify_email )
        {
            /* MAIL - sender uses same addresss as recipient */
            osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ,
                         "MAIL FROM: <%s>\r\n", email);

            if (send_smtp_command(sock, buf, "250"))
            {
                break;
            }

            /* RCPT */
            osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ,
                         "RCPT TO: <%s>\r\n", email);

            if (send_smtp_command(sock, buf, "250"))
            {
                break;
            }
        }

        /* always send to admin. */

        if ( admin_email )
        {
            if ( !notify_email )
            {            
                osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ,
                              "MAIL FROM: <%s>\r\n", config->admin_email );

                if ( send_smtp_command( sock, buf, "250" ) )
                {
                    break;
                }
            }

            osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ, "RCPT TO: <%s>\r\n",
                          config->admin_email );

            if ( send_smtp_command( sock, buf, "250" ) )
            {
                break;
            }
        }

        /* DATA */
        if( send_smtp_command( sock, OSIRIS_SMTP_CMD_DATA, "354" ) )
        {
            break;
        }

    } while( 0 );

    return sock;
}

void md_notify_session_rekey( const char *host, const char *email )
{
    int sock = 0;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];
    char this_host[MAX_HOSTNAME_LENGTH];

    const char *from_email = email;

    if( host == NULL )
    {
        log_error( LOG_ID_NOTIFY_ERROR, "unknown", "null host" );
        return;
    }

    if ( email && strlen( email ) == 0 )
    {
        return;
    }

    if( ( sock = connect_to_mail_server( email, NULL ) ) < 0 )
    {
        return;
    }

    if( config->http_host[0] && strcmp( config->http_host, "" ) != 0 )
    {
        osi_snprintf( this_host, sizeof(this_host), "%s", config->http_host );
    }

    else
    {
        gethostname( this_host, sizeof(this_host) );
    }

    osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ,
                  /* The following lines are where the default
                  ** email header is constructed from. This may
                  ** need to be changed to an actual network account
                  ** to pass communication validity checks in
                  ** high-security environments.
                  */
                  "To: %s\r\n"
                  "From: \"Osiris Host Integrity System\" <osirismd@%s>\r\n"
                  "Date: %s\r\n"
                  "Subject: [osiris rekey][host: %s] session rekey.\r\n\r\n",
                  from_email, this_host,
                  osi_time_to_string(osi_get_time()), host );

    send_smtp_command( sock, buf, NULL );

    /* send a pretty message body. */

    osi_snprintf( buf, sizeof( buf ), NOTIFY_SESSION_REKEY_FORMAT, host );

    osi_write( sock, buf, strlen( buf ) );
    send_smtp_command( sock, "\r\n.\r\n", NULL );
    disconnect_from_mail_server( sock );
    return;
} 

void md_notify_scan_failed( const char *host,
                                  const char *email,
                                  unsigned long scheduled_time,
                                  const char *message )
{
    int sock = 0;
    char *schedule;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];
    char this_host[MAX_HOSTNAME_LENGTH];

    const char *from_email = email;

    if( host == NULL )
    {
        log_error( LOG_ID_NOTIFY_ERROR, "unknown", "null host" );
        return;
    }

    if ( email && strlen( email ) == 0 )
    {
        return;
    }

    /* if we were given a scheduled time, we get it as a string. */

    if( scheduled_time > 0 )
    {
        schedule = osi_time_to_string( scheduled_time );
    }

    else
    {
        schedule = "<unknown>";
    }

    if( ( sock = connect_to_mail_server( email, NULL ) ) < 0 )
    {
        return;
    }

    if( config->http_host[0] && strcmp( config->http_host, "" ) != 0 )
    {
        osi_snprintf( this_host, sizeof(this_host), "%s", config->http_host );
    }             
                  
    else          
    {             
        gethostname( this_host, sizeof(this_host) );
    }

    osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ,
                  /* The following lines are where the default
                  ** email header is constructed from. This may
                  ** need to be changed to an actual network account
                  ** to pass communication validity checks in
                  ** high-security environments.
                  */
                  "To: %s\r\n"
                  "From: \"Osiris Host Integrity System\" <osirismd@%s>\r\n"
                  "Date: %s\r\n"
"Subject: [osiris scheduler][host: %s] failed to start the scheduled scan.\r\n\r\n",
                  from_email, this_host,
                  osi_time_to_string(osi_get_time()), host );

    send_smtp_command( sock, buf, NULL );

    /* send a pretty message body. */
       
    osi_snprintf( buf, sizeof( buf ), NOTIFY_SCHEDULE_FAIL_FORMAT,
                  host, schedule, message );

    osi_write( sock, buf, strlen( buf ) );
    send_smtp_command( sock, "\r\n.\r\n", NULL );
    disconnect_from_mail_server( sock );    

    return;
}

void md_notify( const char *email, OSI_DB_CTX *context,osi_bool show_url )
{
    int sock = 0;
    int diff_count = context->diff_count;
    int smtp_port = config->notify_smtp_port;

    char *smtp_host = config->notify_smtp_host;
    char *logfile  = context->filepath;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];    /* KISS !malloc */
    char host[MAX_HOSTNAME_LENGTH];

    const char *from_email = email;

    if (logfile == NULL)
    {
        log_error( LOG_ID_NOTIFY_ERROR, context->host,
                   "notification failure: log file not set." );
        return;
    }

    if ( email && strlen( email ) == 0 )
    {
        return;
    }

    if ( ( sock = connect_to_mail_server( email, NULL ) ) < 0  )
    {
        return;
    }

   log_info( LOG_ID_NOTIFY_INFO, context->host,
            "notifying %s using %s:%d, sending contents of log file: %s, "
            "containing %d entries.",
            from_email, smtp_host, smtp_port,logfile, diff_count);

    /* BODY */

    if( config->http_host[0] && strcmp( config->http_host, "" ) != 0 )
    {
        osi_snprintf( host, sizeof(host), "%s", config->http_host );
    }             
                  
    else          
    {             
        gethostname( host, sizeof(host) );
    }

    osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ,
                  /* The following lines are where the default
                  ** email header is constructed from. This may
                  ** need to be changed to an actual network account
                  ** to pass communication validity checks in
                  ** high-security environments.
                  */
                  "To: %s\r\n"
                  "From: \"Osiris Host Integrity System\" <osirismd@%s>\r\n"
                  "Date: %s\r\n"
                  "Subject: [osiris log][host: %s][%d changes]\r\n\r\n",
                  from_email, host, osi_time_to_string( osi_get_time() ),
                  context->host, diff_count );

    send_smtp_command( sock, buf, NULL );

    /* before sending the file, we prepend a URL that enables */
    /* the recipient to approve of these changes by updating  */
    /* the trusted database via https.                        */

    if( ( config->http_port > 0 ) && ( show_url ) )
    {
        char filename[MAX_PATH_LENGTH] = "";

        /* get filename, strip path. */

        osi_get_file_from_path( logfile, filename, sizeof( filename ) );

        if( diff_count > 0 )
        {
            osi_snprintf( buf, sizeof( buf ), NOTIFY_UPDATE_URL_FORMAT,
                          context->db2_name, host, config->http_port,
                          context->host, context->db2_name, filename );

            if ((osi_write(sock, buf, strlen(buf))) == 0)
            {
                log_error( LOG_ID_NOTIFY_ERROR, NULL,
                  "notification failure: unable to send data to SMTP server." );
                return;
            } 
        }
    }

	osi_write( sock, "\r\n\r\n", 4 );
        
    /* send the log file contents. */
 
    send_smtp_file( sock, logfile );
    send_smtp_command(sock, "\r\n.\r\n", NULL);

    disconnect_from_mail_server( sock );
    return;
}

osi_bool md_notify_send_test( const char *email, OSI_ERROR *error )
{
    int sock = 0;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];
    char this_host[MAX_HOSTNAME_LENGTH];

    const char *from_email = email;

    if( error == NULL )
    {
        return FALSE;
    }

    if ( email && strlen( email ) == 0 )
    {
        return FALSE;
    }

    memset( error, 0, sizeof( OSI_ERROR ) );

    if( ( sock = connect_to_mail_server( email, error ) ) < 0 )
    {
        return FALSE;
    }

    if( config->http_host[0] && strcmp( config->http_host, "" ) != 0 )
    {
        osi_snprintf( this_host, sizeof(this_host), "%s", config->http_host );
    }             
                  
    else          
    {             
        gethostname( this_host, sizeof(this_host) );
    }

    osi_snprintf( buf, OSIRIS_SMTP_CMD_BUFSZ,
                  /* The following lines are where the default
                  ** email header is constructed from. This may
                  ** need to be changed to an actual network account
                  ** to pass communication validity checks in
                  ** high-security environments.
                  */
                  "To: %s\r\n"
                  "From: \"Osiris Host Integrity System\" <osirismd@%s>\r\n"
                  "Date: %s\r\n"
                  "Subject: [osiris test][notification system test]\r\n\r\n",
                  from_email, this_host, osi_time_to_string( osi_get_time() ) );

     send_smtp_command( sock, buf, NULL );

    /* send a pretty message body. */

    osi_strlcpy( buf, NOTIFY_TEST_MESSAGE, sizeof( buf ) );
    osi_write( sock, buf, strlen( buf ) );
    send_smtp_command( sock, "\r\n.\r\n", NULL );

    disconnect_from_mail_server( sock );
    return TRUE;
}
