
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_ports.c
**  Date:    January 15, 2005
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for detecting open network ports.
**
******************************************************************************/

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"


static const char *MODULE_NAME = "mod_ports";

#ifdef WIN32

#include "windows.h"
#include "psapi.h"
#include "winsock.h"
#include "iprtrmib.h"
#include "tlhelp32.h"
#include "iphlpapi.h"


typedef struct
{
  DWORD   dwState;        
  DWORD   dwLocalAddr;    
  DWORD   dwLocalPort;    
  DWORD   dwRemoteAddr;   
  DWORD   dwRemotePort;   
  DWORD	  dwProcessId;

} MIB_TCPEXROW, *PMIB_TCPEXROW;


typedef struct
{
	DWORD			dwNumEntries;
	MIB_TCPEXROW	table[ANY_SIZE];

} MIB_TCPEXTABLE, *PMIB_TCPEXTABLE;



typedef struct
{
  DWORD   dwLocalAddr;    
  DWORD   dwLocalPort;    
  DWORD	  dwProcessId;

} MIB_UDPEXROW, *PMIB_UDPEXROW;


typedef struct
{
	DWORD			dwNumEntries;
	MIB_UDPEXROW	table[ANY_SIZE];

} MIB_UDPEXTABLE, *PMIB_UDPEXTABLE;


DWORD( WINAPI *pAllocateAndGetTcpExTableFromStack )
(
  PMIB_TCPEXTABLE *pTcpTable,  
  BOOL bOrder,               
  HANDLE heap,
  DWORD zero,
  DWORD flags
);

DWORD( WINAPI *pAllocateAndGetUdpExTableFromStack )
(
  PMIB_UDPEXTABLE *pTcpTable,  
  BOOL bOrder,           
  HANDLE heap,
  DWORD zero,
  DWORD flags
);


static PMIB_TCPEXTABLE tcpTable = NULL;
static PMIB_UDPEXTABLE udpTable = NULL;


BOOLEAN XPApisArePresent( VOID )
{
	pAllocateAndGetTcpExTableFromStack = (PVOID)GetProcAddress( 
            LoadLibrary( "iphlpapi.dll"),
			        	 "AllocateAndGetTcpExTableFromStack" );

	if( !pAllocateAndGetTcpExTableFromStack )
    {
        return FALSE;
    }

	pAllocateAndGetUdpExTableFromStack = (PVOID)GetProcAddress(
            LoadLibrary( "iphlpapi.dll"),
				         "AllocateAndGetUdpExTableFromStack" );

	if( !pAllocateAndGetUdpExTableFromStack )
    {
        return FALSE;
    }

	return TRUE;
}

char * getProcessFilename( ULONG pid )
{
    const C_MAX_NBR = 256 ;
    DWORD dwaProcessIds [C_MAX_NBR];
    DWORD dwNbrProcesses;
    DWORD dwNbrModules;

    HANDLE hProcess;
    HMODULE hmaModules [C_MAX_NBR];

    char cFilename[MAX_PATH];
    char *pReturn;

    pReturn = "SYSTEM";

    int iSuccess = EnumProcesses( dwaProcessIds, sizeof( dwaProcessIds ),
                                  &dwNbrProcesses);

    dwNbrProcesses = ( dwNbrProcesses / sizeof( dwaProcessIds[0] ) );

    if ( iSuccess )
    {
        unsigned int i;

        for ( i = 0; i < dwNbrProcesses; i++ )
        {
            if ( pid == dwaProcessIds [i] )
            {
                hProcess = OpenProcess( PROCESS_QUERY_INFORMATION |
                                        PROCESS_VM_READ, FALSE,
                                        dwaProcessIds[i] );

                EnumProcessModules( hProcess, hmaModules, sizeof( hmaModules ),
                                    &dwNbrModules);

                dwNbrModules = ( dwNbrModules / sizeof (hmaModules [0] ) );

                if ( GetModuleFileNameEx( hProcess, hmaModules [0], cFilename,
                                          sizeof (cFilename) ) )
                {
                    pReturn = cFilename;
                }

                CloseHandle (hProcess);
            }
        }
    }

    return pReturn;
}


PCHAR ip_to_name( UINT ipaddr, PCHAR name, int namelen )
{
	unsigned int ip = htonl( ipaddr );

	osi_snprintf( name, namelen, "%d.%d.%d.%d",
			(ip >> 24) & 0xFF,
			(ip >> 16) & 0xFF,
			(ip >> 8) & 0xFF,
			(ip ) & 0xFF);

	return name;
}


void process_windows_ports( SCANNER *scanner )
{
	DWORD		error, dwSize;
	HANDLE		hProcessSnap;


	BOOLEAN		libsPresent;
	DWORD		i;
	CHAR		processName[255];
	CHAR		localname[255], remotename[255];
	CHAR		remoteport[255], localport[255];
	CHAR		localaddr[255], remoteaddr[255];

    SCAN_RECORD_TEXT_1 record;

	if( GetVersion() >= 0x80000000 )
    {

		return;
	}

	libsPresent = XPApisArePresent();

	if ( !libsPresent )
	{
		return;
	}

    /* GET Tcp table if not already. */

	if ( tcpTable == NULL )
	{
		error = pAllocateAndGetTcpExTableFromStack( &tcpTable, TRUE,
                                                    GetProcessHeap(), 2, 2 );
		if( error )
        {
            log_error( "error reading TCP table (%d)", error );
		    return;
		}
	}

    /* Get UDP table if not already. */

	if ( udpTable == NULL )
	{
	  error = pAllocateAndGetUdpExTableFromStack( &udpTable, TRUE,
                                                  GetProcessHeap(), 2, 2 );
	    if( error )
        {
		    log_error( "error reading UDP table (%d)", error );
		    return;
		}
	}

     /* write out TCP records.

	 format for records is:
	
     key:     TCP:<port>
	 record:  TCP:<port>;<process_name>;<process_id>;<local_addr>;<remote_addr>

     */	

	for( i = 0; i < tcpTable->dwNumEntries; i++ )
    {

		if( tcpTable->table[i].dwState == MIB_TCP_STATE_LISTEN )
		{
            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                          "TCP:%d", htons(tcpTable->table[i].dwLocalPort) );

            osi_snprintf( record.data, sizeof( record.data ),
              		      "TCP:%d;exe=%s;pid=%d;local=%s;remote=%s",
                           htons(tcpTable->table[i].dwLocalPort),
                           getProcessFilename( tcpTable->table[i].dwProcessId ),
                    		                   tcpTable->table[i].dwProcessId ,
                    		         ip_to_name( tcpTable->table[i].dwLocalAddr,
                                                 localname, 255),

                   		   ip_to_name( tcpTable->table[i].dwRemoteAddr,
                                       remotename, 255 ) );

            send_scan_data( scanner, (SCAN_RECORD *)&record );

	    }
	}

    /* generate records for UDP

        key:       UDP:<port>
        record:    UDP:<port>;exe=<name>;pid=<pid>

    */

    for( i = 0; i < udpTable->dwNumEntries; i++ )
    {
        initialize_scan_record( (SCAN_RECORD *)&record,
                                 SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        osi_snprintf( record.name, sizeof( record.name ),
                      "UDP:%d", htons(udpTable->table[i].dwLocalPort) );


        osi_snprintf( record.data, sizeof( record.data ),
              	      "UDP:%d;exe=%s;pid=%d",
                       htons(udpTable->table[i].dwLocalPort),
              		   getProcessFilename( udpTable->table[i].dwProcessId ),
                 		                   udpTable->table[i].dwProcessId );

        send_scan_data( scanner, (SCAN_RECORD *)&record );

    }
}



#endif /* WIN32 */

void mod_ports( SCANNER *scanner )
{
#ifdef WIN32
    process_windows_ports( scanner );
#endif /* WIN32 */
}

