
/*****************************************************************************
**
**  Copyright (C) 2001  - the shmoo group -
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
*****************************************************************************/

/*****************************************************************************
**
**    The Shmoo Group (TSG)
**
**    File:      configuration.c
**    Author:    Brian Wotring
**
**    Date:      June 22, 2001.
**    Project:   osiris
**
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <ctype.h>
#include <string.h>

#include <openssl/md5.h>

#include "utilities.h"
#include "status.h"

#include "list.h"
#include "string_list.h"

#include "scan_record.h"

#include "filter.h"
#include "configuration.h"

static struct keywords config_keywords[] =
{
    { "name", CONFIG_KEYWORD_NAME },
    { "recursive", CONFIG_KEYWORD_RECURSIVE },
    { "followlinks", CONFIG_KEYWORD_FOLLOWLINKS },
    { "hash", CONFIG_KEYWORD_HASH },
    { "includeall", CONFIG_KEYWORD_INCLUDEALL },
    { "excludeall", CONFIG_KEYWORD_EXCLUDEALL },
    { "#", CONFIG_KEYWORD_COMMENT },
    { "<directory", CONFIG_KEYWORD_DIRECTORY_START },
    { "</directory>", CONFIG_KEYWORD_DIRECTORY_END },
    { "<modules>", CONFIG_KEYWORD_SYSTEM_START },
    { "</modules>", CONFIG_KEYWORD_SYSTEM_END },
    { "<system>", CONFIG_KEYWORD_SYSTEM_START },
    { "</system>", CONFIG_KEYWORD_SYSTEM_END },
    { "include", CONFIG_KEYWORD_INCLUDE },
    { "exclude", CONFIG_KEYWORD_EXCLUDE },
    { "noentry", CONFIG_KEYWORD_NOENTRY },
    { "param", CONFIG_KEYWORD_PARAM },
    { NULL, 0 }
};


static struct keywords boolean_keywords[] =
{
    { "1", TRUE },
    { "y", TRUE },
    { "yes", TRUE },
    { "true", TRUE },
    { "0", FALSE },
    { "n", FALSE },
    { "no", FALSE },
    { "false", FALSE },
    { NULL, 0 }
};

static struct keywords action_keywords[] =
{
    { "include", ACTION_TYPE_INCLUDE },
    { "includeall", ACTION_TYPE_INCLUDE_ALL },
    { "exclude", ACTION_TYPE_EXCLUDE },
    { "excludeall", ACTION_TYPE_EXCLUDE_ALL },
    { "noentry", ACTION_TYPE_NO_ENTRY },
    { NULL, 0 }
};

static struct keywords hash_keywords[] =
{
    { "md5", HASH_MD5 },
    { "sha", HASH_SHA },
    { "ripemd", HASH_RIPEMD },
    { NULL, 0 }
};


static char * valid_hash_values[] =
{
    "md5",
    "sha",
    "ripemd",
    NULL
};


char * valid_boolean_values[] =
{
    "1",
    "y",
    "yes",
    "true",
    "0",
    "n",
    "no",
    "false",
    NULL
};


/*****************************************************************************
**
**    Function: osi_config_new
**
**    Purpose:  create a new config module with the specified
**              file name as the name of the config file.  here
**              we create the global block and add it to the
**              block list.
**
******************************************************************************/

OSI_SCAN_CONFIG * osi_config_new()
{
    OSI_SCAN_CONFIG *new_config;
    block *global_block;

    new_config = osi_malloc( sizeof( OSI_SCAN_CONFIG ) );
    memset( new_config, 0, sizeof( OSI_SCAN_CONFIG ) );

    global_block              = create_block();

    new_config->data          = string_list_new();
    new_config->errors        = string_list_new();
    new_config->state	      = CONFIG_STATE_NULL;

    new_config->blocks        = list_new();
    new_config->modules       = list_new();

    new_config->block_names   = string_list_new();

    new_config->warning_count = 0;
    new_config->error_count   = 0;

    new_config->print_errors_on_verify = FALSE;
    new_config->store_errors_on_verify = FALSE;

    /* we create a default global block here. */

    list_add( new_config->blocks, global_block );

    return new_config;
}


/*****************************************************************************
**
**    Function: osi_config_destroy
**
**    Purpose:  destroy the config file data, and destroy the
**              list of blocks.
**
******************************************************************************/

void osi_config_destroy( OSI_SCAN_CONFIG *cfg )
{
    if ( cfg == NULL )
    {
        return;
    }

    /* destroy config file data. */

    string_list_destroy( cfg->data );
    string_list_destroy( cfg->errors );

    /* destroy the block names list. */

    string_list_destroy( cfg->block_names );

    /* get first block, then go through and */
    /* destroy all blocks, then destroy the */
    /* block list itself.                   */

    list_destroy_with_function( cfg->blocks, (void (*)(void *))&destroy_block );
    list_destroy_with_function( cfg->modules,(void (*)(void *))&destroy_module);

    /* now destroy the config structure itself. */

    osi_free( cfg );
}

/*****************************************************************************
**
**    Function: osi_config_receive_first
**
**    Purpose:  receive the first portion of data for this pushed config.
**              Add data to the data store to set state to receiving.
**
*****************************************************************************/

osi_bool osi_config_receive_first( OSI_SCAN_CONFIG *cfg, unsigned char *data,
                                   int data_length )
{
    if ( cfg != NULL )
    {
        /* destroy data if not destroyed already, then create a new list */
        /* and set state to receiving.  then add incoming data.          */

        string_list_destroy( cfg->data );
        string_list_destroy( cfg->errors );

        cfg->data = string_list_new();
        cfg->errors = string_list_new();

        cfg->state = CONFIG_STATE_RECEIVING;
        return osi_config_receive( cfg, data, data_length );
    }

    return FALSE;
}

/******************************************************************************
**
**    Function: osi_config_receive
**
**    Purpose:  receive a portion of data for this pushed config.  Add data
**		        to the data store.
**
*****************************************************************************/

osi_bool osi_config_receive( OSI_SCAN_CONFIG *cfg, unsigned char *data,
                             int data_length )
{
    if ( ( cfg != NULL ) && ( data != NULL ) && ( data_length > 0 ) )
    {
        data = trim_white_space( data );
        string_list_add_item( cfg->data, data );

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
**
**    Function: osi_config_receive_last
**
**    Purpose:  receive the last portion of data for this pushed config.
**              Add data to the data store and perform evaluation.  Set
**              state accordingly.
**
******************************************************************************/

osi_bool osi_config_receive_last( OSI_SCAN_CONFIG *cfg, unsigned char *data,
                                  int data_length )
{
    osi_bool result = FALSE;

    if ( cfg != NULL )
    {
        osi_config_receive( cfg, data, data_length );

        if ( osi_config_verify( cfg ) )
        {
            result = osi_config_parse( cfg );

            /* if we have a valid config, generate the id. */

            if ( result )
            {
                osi_config_generate_id( cfg );
            }
        }
    }

    return result;
}

/*****************************************************************************
**
**    Function: osi_config_verify
**
**    Purpose:  open the config file, read in the data, then
**              verify both the syntax and the semantis.
**
******************************************************************************/

osi_bool osi_config_verify( OSI_SCAN_CONFIG *cfg )
{
    if ( cfg != NULL )
    {
        cfg->error_count = 0;
        cfg->warning_count = 0;
    }

    if ( osi_config_verify_syntax( cfg ) == FALSE )
    {
        cfg->state = CONFIG_STATE_INVALID;
        return FALSE;
    }

    if ( osi_config_verify_semantics( cfg ) == FALSE )
    {
        cfg->state = CONFIG_STATE_INVALID;
        return FALSE;
    }

    cfg->state = CONFIG_STATE_VALID;
    return TRUE;
}

/****************************************************************************
**
**    Function: osi_config_get_global_block
**
**    Purpose:  peer into the block list structure and return the
**              first block in the list, which is always the global one.
**
*****************************************************************************/

block * osi_config_get_global_block( OSI_SCAN_CONFIG *cfg )
{
    if ( ( cfg != NULL ) && ( cfg->blocks != NULL ) &&
        ( cfg->blocks->next != NULL ) )
    {
        return (block *)( cfg->blocks->next->data );
    }

    return NULL;
}

/*****************************************************************************
**
**    Function: osi_config_get_first_block
**
**    Purpose:  peer into the block list structure and return the
**              first block after the <Global> block.
**
******************************************************************************/

block * osi_config_get_first_block( OSI_SCAN_CONFIG *cfg )
{
    block *global = osi_config_get_global_block( cfg );

    if ( global != NULL )
    {
        return osi_config_get_next_block( cfg, global );
    }

    return NULL;
}

/*****************************************************************************
**
**    Function: osi_config_get_next_block
**
**    Purpose:  peer into the block list structure and return the
**              first block after the specified block.
**
******************************************************************************/

block * osi_config_get_next_block( OSI_SCAN_CONFIG *cfg, block *current_block )
{
    osi_list head;

    if ( ( cfg != NULL ) && ( current_block != NULL ) )
    {
        head = list_get_first_node( cfg->blocks );

        while( head )
        {
            block *b = (block *)head->data;

            if ( ( b == current_block ) && ( head->next != NULL ) )
            {
                return ( (block *)head->next->data );
            }

            head = head->next;
        }
    }

    return NULL;
}

/*****************************************************************************
**
**    Function: new_block_from_block
**
**    Purpose:  allocate for a new block, but inherit the attributes
**              from the parent, everything except the action list.
**
*****************************************************************************/

block *new_block_from_block( block *parent_block )
{
    /* create a blank config block. */

    block *config_block = create_block();

    /* now copy all the options from the parent block. */

    config_block->recursive    = parent_block->recursive;
    config_block->follow_links = parent_block->follow_links;
    config_block->hash         = parent_block->hash;

    /* we don't copy the actions....*/

    return config_block;
}


/*****************************************************************************
**
**    Function: create_block
**
**    Purpose:  default allocation for a block, initialized to
**              nothing.
**
*****************************************************************************/

block *create_block()
{
    struct block *config_block = osi_malloc( sizeof( struct block ) );
    memset( config_block->directory, 0, sizeof( config_block->directory ) );

    /* new block options according to our specified defaults. */

    config_block->recursive     = FALSE;
    config_block->follow_links  = FALSE;

    config_block->hash          = HASH_DEFAULT;

    config_block->actions       = list_new();
    config_block->no_entry_list = string_list_new();

    return config_block;
}

/*****************************************************************************
**
**    Function: create_module
**
**    Purpose:  default allocation for a module, initialized to
**              nothing.
**
*****************************************************************************/

module *create_module()
{
    struct module *m = osi_malloc( sizeof( struct module ) );

    if ( m == NULL )
    {
        return NULL;
    }

    memset( m, 0, sizeof( struct module ) );
    m->params = list_new();

    return m;
}

/*****************************************************************************
**
**    Function: create_param
**
**    Purpose:  default allocation for a module parameter, initialized to
**              nothing.
**
*****************************************************************************/

param *create_param()
{
    struct param *p = osi_malloc( sizeof( struct param ) );

    if ( p == NULL )
    {
        return NULL;
    }

    memset( p, 0, sizeof( struct param ) );
    return p;
}


/*****************************************************************************
**
**    Function: create_action
**
**    Purpose:  default allocation for an action, initialize to nothing.
**
*****************************************************************************/

action *create_action()
{
    struct action *config_action = osi_malloc( sizeof( struct action ) );

    memset( config_action, 0, sizeof( struct action ) );
    memset( config_action->action_filter.argument, 0,
            sizeof( config_action->action_filter.argument ) );
    config_action->type = 0;

    return config_action;
}


/*****************************************************************************
**
**    Function: destroy_block
**
**    Purpose:  free the specified block, destroy the dynamic data
**              found in a block include the noentry and action lists.
**
*****************************************************************************/

void destroy_block( block *config_block )
{
    if ( config_block != NULL )
    {
        list_destroy_with_function( config_block->actions,
                                    (void (*)(void *))&destroy_action );

        string_list_destroy( config_block->no_entry_list );
        osi_free( config_block );
    }
}

/*****************************************************************************
**
**    Function: destroy_module
**
*****************************************************************************/

void destroy_module( module *config_module )
{
    if ( config_module != NULL )
    {
        list_destroy_with_function( config_module->params,
                                    (void (*)(void *))&destroy_param );

        osi_free( config_module );
    }
}

/*****************************************************************************
**
**    Function: destroy_param
**
*****************************************************************************/

void destroy_param( param *module_param )
{
    if ( module_param != NULL )
    {
        osi_free( module_param );
    }
}


/*****************************************************************************
**
**    Function: destroy_action
**
**    Purpose:  free the specified action.
**
****************************************************************************/

void destroy_action( action *config_action )
{
    if ( config_action != NULL )
    {
        osi_free( config_action );
    }
}

osi_bool get_module_parameter( OSI_SCAN_CONFIG *cfg,  char *line )
{
    char *temp = NULL;
    char *trimmed = NULL;

    char word[MAX_LINE_LENGTH] = "";

    param *p = NULL;
    module *m = NULL;

    if ( line == NULL )
    {
        return FALSE;
    }

    p = create_param();

    /* get module name. */

    temp = get_token( line, word, sizeof( word ) );
    trimmed = trim_quotes( word );

    if ( ( m = get_module_with_name( cfg, trimmed ) ) == NULL )
    {
        goto exit_error;
    }

    /* get param name and value. */

    temp = get_token( temp, word, sizeof( word ) );
    trimmed = trim_quotes( word );
    osi_strlcpy( p->name, trimmed, sizeof( p->name ) );

    temp = get_token( temp, word, sizeof( word ) );
    trimmed = trim_quotes( word );
    osi_strlcpy( p->value, trimmed, sizeof( p->value ) );

    /* add param to module's list. */

    list_add( m->params, p );
    return TRUE;

exit_error:

    destroy_param( p );
    return FALSE;
}


/*****************************************************************************
**
**    Function: get_keyword_type
**
**    Purpose:  determine the type of keyword this is based upon
**              our lists.
**
*****************************************************************************/

int get_keyword_type( char *string )
{
    char word[MAX_LINE_LENGTH] = "";
    int index;

    if ( string == NULL )
    {
        return CONFIG_KEYWORD_NULL;
    }

    get_token( string, word, sizeof( word ) );
    lowercase_string( word );

    for( index = 0; config_keywords[index].word != NULL; index++ )
    {
        if ( strncmp( word, config_keywords[index].word,
             strlen( config_keywords[index].word ) ) == 0 )
        {
            return ( config_keywords[index].type );
        }
    }

    return CONFIG_KEYWORD_UNKNOWN;
}


/*****************************************************************************
**
**    Function: verify_single_value
**
**    Purpose:  determine if the value is invalid based upon the
**              valid values list given.
**
******************************************************************************/

char * verify_single_value( char *value, char *valid_values[] )
{
    int index;

    if ( valid_values == NULL || value == NULL )
    {
        return NULL;
    }

    lowercase_string( value );

    for( index = 0; valid_values[index] != NULL; index++ )
    {
        if ( strcmp( value, valid_values[index] ) == 0 )
        {
            return NULL;
        }
    }

    return value;
}

osi_bool verify_param_values( string_list *modules, char *line )
{
    char *temp = NULL;
    char *module_name = NULL;

    char word[MAX_LINE_LENGTH] = "";

    if ( ( line == NULL ) || ( modules == NULL ) )
    {
        return FALSE;
    }

    /* module name */

    word[0] = '\0';
    temp = get_token( line, word, sizeof( word ) );

    if ( word[0] == '\0' )
    {
        return FALSE;
    }

    /* make sure we have that module! */

    module_name = trim_quotes( word );

    if ( string_list_contains_item( modules, module_name, FALSE ) == FALSE )
    {
        return FALSE;
    }

    /* param name. */

    word[0] = '\0';
    temp = get_token( temp, word, sizeof( word ) );

    if ( word[0] == '\0' )
    {
        return FALSE;
    }

    /* param value. */

    word[0] = '\0';
    temp = get_token( temp, word, sizeof( word ) );

    if ( word[0] == '\0' )
    {
        return FALSE;
    }

    return TRUE;
}


/*****************************************************************************
**
**    Function: print_config_error
**
**    Purpose:  display pretty error information includeing
**              line number nad error message.
**
*****************************************************************************/

void print_config_error( OSI_SCAN_CONFIG *cfg, int line_number,
                         char *line, char *error, ... )
{
    va_list args;
    char buf[MAX_LINE_LENGTH] = "";
    char msg[MAX_LINE_LENGTH] = "";

    if ( cfg->print_errors_on_verify == FALSE &&
         cfg->store_errors_on_verify == FALSE )
    {
        return;
    }

    va_start( args, error );
    osi_vsnprintf( msg, sizeof(msg), error, args );
    va_end( args );

    osi_snprintf( buf, sizeof(buf), "\nline %d: ==> %s\n%s\n",
                  line_number, line, msg );

    if ( cfg->print_errors_on_verify )
    {
        fprintf( stderr, "%s\n", buf );
    }

    if ( cfg->store_errors_on_verify )
    {
        string_list_add_item( cfg->errors, buf );
    }
}


/*****************************************************************************
**
**    Function: get_name_value
**
**    Purpose:  copy as much of the remainder of this line into the name
**              for this config, it's a pretty name, not required.
**
*****************************************************************************/

int get_name_value( OSI_SCAN_CONFIG *cfg, char *line )
{
    if ( ( line != NULL ) && ( cfg != NULL ) )
    {
        osi_strlcpy( cfg->name, line, sizeof( cfg->name ) );
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
**
**    Function: get_boolean_value
**
**    Purpose:  examine keyword based upon our list of valid booleans
**              and return TRUE or FALSE appropriatly.
**
*****************************************************************************/

int get_boolean_value( char *keyword, osi_bool *value )
{
    int index;

    if ( keyword == NULL || value == NULL )
    {
        return 0;
    }

    lowercase_string( keyword );

    for( index = 0; boolean_keywords[index].word != NULL; index++ )
    {
        if ( strcmp( keyword, boolean_keywords[index].word ) == 0 )
        {
            *value = boolean_keywords[index].type;
            return 1;
        }
    }

    return 0;
}


/****************************************************************************
**
**    Function: get_hash_value
**
**    Purpose:  determine the hash type of the string given, e.g. md5
**              and return the constant value associated with that algorithm.
**
******************************************************************************/

int get_hash_value( char *keyword, unsigned int *value )
{
    int index;

    if ( keyword == NULL || value == NULL )
    {
        return 0;
    }

    lowercase_string( keyword );

    for( index = 0; hash_keywords[index].word != NULL; index++ )
    {
        if ( strcmp( keyword, hash_keywords[index].word ) == 0 )
        {
            *value = hash_keywords[index].type;
            return 1;
        }
    }

    return 0;
}

/******************************************************************************
**
**    Function: get_action_name_from_type
**
**    Purpose:  return string pretty name of checksum algorithm, based upon
**      the constant type given.
**
******************************************************************************/

char * get_action_name_from_type( int type )
{
    int index;
    char *name = "";

    for( index = 0; action_keywords[index].word != NULL; index++ )
    {
        if ( type == action_keywords[index].type )
        {
            name = action_keywords[index].word;
            return name;
        }
    }

    return name;
}

/******************************************************************************
**
**    Function: get_hash_name_from_type
**
**    Purpose:  return string pretty name of checksum algorithm, based upon
**		the constant type given.
**
******************************************************************************/

char * get_hash_name_from_type( int type )
{
    int index;
    char *name = "";
    
    for( index = 0; hash_keywords[index].word != NULL; index++ )
    {
        if ( type == hash_keywords[index].type )
        {
            name = hash_keywords[index].word;
            return name;
        }
    }
    
    return name;
}


/******************************************************************************
**
**    Function: get_block_with_name
**
**    Purpose:  get the block in the configuration with the specified
**              name.
**
******************************************************************************/

block * get_block_with_name( OSI_SCAN_CONFIG *cfg, char *name )
{
    osi_list head;

    if ( cfg != NULL && name != NULL )
    {
        head = list_get_first_node( cfg->blocks );

        while( head )
        {
            block *b = (block *)head->data;

            if   ( b && ( strcmp( b->directory, name ) == 0 ) )
            {
                return ( (block *)head->data );
            }

            head = head->next;
        }
    }

    return NULL;
}

/******************************************************************************
**
**    Function: get_module_with_name
**
**    Purpose:  get the module in the configuration with the specified
**              name.
**
******************************************************************************/

module * get_module_with_name( OSI_SCAN_CONFIG *cfg, char *name )
{
    osi_list head;

    if ( ( cfg == NULL ) || ( name == NULL ) )
    {
        return NULL;
    }

    head = list_get_first_node( cfg->modules );

    while( head )
    {
        module *m = (module *)head->data;

        if ( ( m != NULL ) && ( strcmp( m->name, name ) == 0 ) )
        {
            return ( (module *)head->data );
        }

        head = head->next;
    }

    return NULL;
}


/******************************************************************************
**
**    Function: get_action_from_block_with_filter
**
**    Purpose:  get the action in the block with the specified
**              filter type.
**
******************************************************************************/

action * get_action_from_block_with_filter( block *config_block,
                                            int filter_type,
                                            char *filter_argument )
{
    osi_list head;

    if ( config_block != NULL )
    {
        head = list_get_first_node( config_block->actions );

        while( head )
        {
            action *a = (action *)head->data;

            if ( a && ( a->action_filter.type == filter_type ) )
            {
                /* filter argument is null, so there shouldn't be */
                /* multiple, this is it.                          */

                if ( filter_argument == NULL )
                {
                    return a;
                }

                /* otherwise, there is an argument and we must make   */
                /* sure we find the correct action inside this block. */

                else if ( strcmp( filter_argument,
                                 a->action_filter.argument ) == 0 )
                {
                    return a;
                }
            }

            head = head->next;
        }
    }

    return NULL;
}




/* verify basic semantics.  check for the following: */
/*                                                   */
/*   - unseen directives                             */
/*                                                   */
/*     ExcludeAll                                    */
/*     Include user( "bob" )                         */
/*                                                   */
/*   - duplicate directives                          */
/*                                                   */
/*     Options Recursive,FollowLinks                 */
/*     Options Recursive                             */
/*                                                   */
/*   - multiple blocks for the same directory.       */
/*                                                   */
/*   - unecessary directives                         */
/*                                                   */
/*     IncludeAll                                    */
/*     ...                                           */
/*     <directory /usr>                              */
/*         Include file( "some_file" )               */
/*     </directory>                                  */
/*                                                   */
/* since these are not fatal errors, we just up the  */
/* warning count to give any app the chance to warn  */
/* user before continuing.                           */

osi_bool osi_config_verify_semantics( OSI_SCAN_CONFIG *cfg )
{
    return TRUE;
}

/******************************************************************************
**
**    Function: osi_config_verify_syntax
**
**    Purpose:  examine all of the config file data to make
**              certain there are no syntactical errors in
**              the file, return FALSE upon failure and print
**              any errors with detailed messages
**
******************************************************************************/

osi_bool osi_config_verify_syntax( OSI_SCAN_CONFIG *cfg )
{
    unsigned int index;

    int line_type;
    int line_number;
    int filter_type;

    char *line;
    char *complete_line;
    char *attributes;

    char keyword[MAX_LINE_LENGTH] = "";
    osi_bool inside_directory_block   = FALSE;
    osi_bool inside_system_block      = FALSE;

    string_list *modules;

    if ( cfg == NULL )
    {
        return FALSE;
    }

    modules = string_list_new();

    for( index = 0; index < cfg->data->size; index++ )
    {
        line          = cfg->data->list[index];
        complete_line = line;

        line_type   = get_keyword_type( line );
        line        = get_token( line, keyword, sizeof( keyword ) );
        line_number = ( index + 1 );

        if ( cfg->error_count > CONFIG_MAX_ERRORS )
        {
            string_list_destroy( modules );
            return FALSE;
        }

        switch( line_type )
        {
            /* skip over empty lines and comments. */

            case CONFIG_KEYWORD_NULL:
            case CONFIG_KEYWORD_COMMENT:

                break;

            case CONFIG_KEYWORD_DIRECTORY_START:

                if ( inside_directory_block ||
                    inside_system_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "misplaced <Directory> statement." );
                    cfg->error_count++;
                    break;
                }

                inside_directory_block = TRUE;

                if ( line == NULL || strcmp( line, ">" ) == 0 )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "no directory specified." );
                    cfg->error_count++;
                    break;
                }

                break;

            case CONFIG_KEYWORD_DIRECTORY_END:

                if ( !inside_directory_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "misplaced </Directory> statement." );
                    cfg->error_count++;
                    break;
                }

                inside_directory_block = FALSE;
                break;

            case CONFIG_KEYWORD_SYSTEM_START:

                if ( inside_directory_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "misplaced <System> statement." );
                    cfg->error_count++;
                    break;
                }

                inside_system_block = TRUE;
                break;

            case CONFIG_KEYWORD_SYSTEM_END:

                if ( !inside_system_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "misplaced </System> statement." );
                    cfg->error_count++;
                    break;
                }

                inside_system_block = FALSE;
                break;

            case CONFIG_KEYWORD_PARAM:

                if ( !inside_system_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                                       "param must be inside <System> block." );

                    cfg->error_count++;
                }

                if ( verify_param_values( modules, line ) == FALSE )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "invalid module parameter." );

                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_RECURSIVE:
            case CONFIG_KEYWORD_FOLLOWLINKS:

                if ( line == NULL )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "no boolean value specified." );
                    cfg->error_count++;
                    break;
                }

                if ( verify_single_value( line, valid_boolean_values ) != NULL )
                {
                    print_config_error( cfg, line_number, complete_line,
                                    "unrecognized boolean value: %s", line );
                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_HASH:

                if ( line == NULL )
                {
                    print_config_error( cfg, line_number, complete_line,
                                        "no hash algorithm specified." );
                    cfg->error_count++;
                    break;
                }

                if ( verify_single_value( line, valid_hash_values ) != NULL )
                {
                    print_config_error( cfg, line_number, complete_line,
                                    "unrecognized hash algorithm: %s", line );
                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_NAME:
                break;

            case CONFIG_KEYWORD_INCLUDEALL:
                break;

            case CONFIG_KEYWORD_EXCLUDEALL:
                break;

            case CONFIG_KEYWORD_INCLUDE:

                if ( inside_system_block )
                {
                    if ( line == NULL )
                    {
                        print_config_error( cfg, line_number, complete_line,
                                            "no system item specified." );
                        cfg->error_count++;
                    }

                    else
                    {
                        string_list_add_item( modules, line );
                    }

                    break;
                }

                else if ( inside_directory_block )
                {
                    if ( line == NULL )
                    {
                        print_config_error( cfg, line_number, complete_line,
                                            "no filter specified." );
                        cfg->error_count++;
                        break;
                    }

                    /* verify filter name. */

                    filter_type = get_filter_type( line );

                    if ( filter_type == FILTER_TYPE_UNKNOWN )
                    {
                        print_config_error( cfg, line_number, complete_line,
                                            "invalid filter: %s", line );
                        cfg->error_count++;
                        break;
                    }

                    /* verify filter value, and any attribute list. */

                    if ( filter_is_dynamic( filter_type ) )
                    {
                        attributes = get_filter_value( line, keyword );

                        if ( attributes == NULL )
                        {
                            print_config_error( cfg,line_number, complete_line, 
                                                "invalid filter: %s", line );
                            cfg->error_count++;
                            break;
                        }

                        break;
                    }

                }

                else
                {
                    print_config_error( cfg, line_number, complete_line,
                         "'Include' directive not allowed in <Global> block." );
                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_EXCLUDE:

                if ( inside_directory_block )
                {
                    if ( line == NULL )
                    {
                        print_config_error( cfg, line_number, complete_line,
                                            "no filter specified." );
                        cfg->error_count++;
                        break;
                    }

                    /* verify filter name. */

                    filter_type = get_filter_type( line );

                    if ( filter_type == FILTER_TYPE_UNKNOWN )
                    {
                        print_config_error( cfg, line_number, line,
                                            "unrecognized filter." );
                        cfg->error_count++;
                        break;
                    }

                    /* if dynamic, verify it's value/syntax. */

                    if ( filter_is_dynamic( filter_type ) &&
                        get_filter_value( line, keyword ) == NULL )
                    {
                        print_config_error( cfg, line_number, complete_line,
                                            "invalid filter: %s", line );
                        cfg->error_count++;
                        break;
                    }
                }

                else
                {
                    print_config_error( cfg, line_number, complete_line,
                        "'Exclude' directive not allowed in <Global> block." );
                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_NOENTRY:

                if ( !inside_directory_block )
                {
                    print_config_error( cfg, line_number, complete_line,
                        "'NoEntry' directive not allowed in <Global> block." );
                    cfg->error_count++;
                }

                break;

            case CONFIG_KEYWORD_UNKNOWN:
            default:

                print_config_error( cfg, line_number, complete_line,
                                    "unrecognized directive: %s", keyword );
                cfg->error_count++;
                break;

        } /* end switch */

    } /* end for loop */

    string_list_destroy( modules );
    return ( cfg->error_count == 0 ) ? TRUE : FALSE;
}

/******************************************************************************
**
**    Function: osi_config_parse
**
**    Purpose:  parse the global data first, then we parse the block
**              data.  we do this because the blocks are to inherit
**              from the <Global> block.
**
******************************************************************************/

osi_bool osi_config_parse( OSI_SCAN_CONFIG *cfg )
{
    if ( osi_config_parse_globals( cfg ) == FALSE )
    {
        return FALSE;
    }

    if ( osi_config_parse_non_globals( cfg ) == FALSE )
    {
        return FALSE;
    }

    return TRUE;
}


/******************************************************************************
**
**    Function: osi_config_parse_globals
**
**    Purpose:  parse the global block data into the specified config module.
**
******************************************************************************/

osi_bool osi_config_parse_globals( OSI_SCAN_CONFIG *cfg )
{
    unsigned int index;
    int line_type;

    char *line;
    char *complete_line;

    char keyword[MAX_LINE_LENGTH] = "";
    action *temp_action;

    block *global_block;

    osi_bool inside_directory_block = FALSE;
    osi_bool inside_system_block = FALSE;

    if ( osi_config_verify( cfg ) == FALSE )
    {
        return FALSE;
    }

    /* create the global block, and make it the */
    /* current block.  there should always be   */
    /* a <Global> block.                        */

    global_block  = cfg->blocks->next->data;

    for( index = 0; index < cfg->data->size; index++ )
    {
        line          = cfg->data->list[index];
        complete_line = line;

        line_type     = get_keyword_type( line );
        line          = get_token( line, keyword, sizeof( keyword ) );

        switch( line_type )
        {
            /* skip over empty lines and comments. */

            case CONFIG_KEYWORD_NULL:
            case CONFIG_KEYWORD_COMMENT:

                break;

            case CONFIG_KEYWORD_DIRECTORY_START:

                inside_directory_block = TRUE;
                break;

            case CONFIG_KEYWORD_DIRECTORY_END:

                inside_directory_block = FALSE;
                break;

            case CONFIG_KEYWORD_SYSTEM_START:
                inside_system_block = TRUE;
                break;
            
            case CONFIG_KEYWORD_SYSTEM_END:
                inside_system_block = FALSE;
                break;

            case CONFIG_KEYWORD_RECURSIVE:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_boolean_value( keyword, &global_block->recursive );

                break;

            case CONFIG_KEYWORD_FOLLOWLINKS:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_boolean_value( keyword, &global_block->follow_links );

                break;

            case CONFIG_KEYWORD_HASH:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_hash_value( keyword, &global_block->hash );

                break;

            case CONFIG_KEYWORD_NAME:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                get_name_value( cfg, line );

            case CONFIG_KEYWORD_INCLUDEALL:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_INCLUDE_ALL;

                /* add action to block's list of actions. */

                list_add( global_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_EXCLUDEALL:

                if ( inside_directory_block || inside_system_block )
                {
                    break;
                }

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_EXCLUDE_ALL;

                list_add( global_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_INCLUDE:
                break;

            case CONFIG_KEYWORD_EXCLUDE:
                break;

            case CONFIG_KEYWORD_PARAM:
                break;

            case CONFIG_KEYWORD_NOENTRY:
                break;

            case CONFIG_KEYWORD_UNKNOWN:
            default:

                /* this should NEVER happen if config data was verified. */
                break;

        } /* end switch */

    } /* end for loop */

    return TRUE;
}

/******************************************************************************
**
**    Function: osi_config_parse_non_globals
**
**    Purpose:  parse all of the blocks, inheriting any missing portions
**              from the <Global> block.  store blocks into the specified
**              config module.
**
******************************************************************************/

osi_bool osi_config_parse_non_globals( OSI_SCAN_CONFIG *cfg )
{
    unsigned int index;
    int line_type;
    int line_number;

    osi_bool inside_directory_block = FALSE;
    osi_bool inside_system_block    = FALSE;

    osi_list action_list;

    char *line;
    char *complete_line;

    char keyword[MAX_LINE_LENGTH] = "";
    char temp[MAX_PATH_LENGTH] = "";

    action *temp_action;
    module *temp_module;

    block *current_block;
    block *global_block;

    char path_separator[2] = { PATH_SEPARATOR, '\0' };

    if ( osi_config_verify( cfg ) == FALSE )
    {
        return FALSE;
    }

    /* get the global block, and make it the    */
    /* current block.  there should always be   */
    /* a <Global> block.                        */

    global_block  = cfg->blocks->next->data;
    current_block = global_block;

    /* note default attribute list for the   */
    /* actions with none specified.  we have */
    /* to be careful here in case there is   */
    /* no global attributes list present.    */

    action_list = global_block->actions->next;

    if ( action_list != NULL && action_list->data != NULL )
    {
        temp_action = action_list->data;
    }

    for( index = 0; index < cfg->data->size; index++ )
    {
        line          = cfg->data->list[index];
        complete_line = line;

        line_number   = ( index + 1 );

        line_type     = get_keyword_type( line );
        line          = get_token( line, keyword, sizeof( keyword ) );

        switch( line_type )
        {
            /* skip over empty lines and comments. */

            case CONFIG_KEYWORD_NULL:
            case CONFIG_KEYWORD_COMMENT:

                break;

            case CONFIG_KEYWORD_DIRECTORY_START:

                inside_directory_block = TRUE;

                /* create a new block, add it to the list.  it */
                /* is now the current block.                   */

                current_block = new_block_from_block( global_block );
                list_add( cfg->blocks, current_block );

                /* acquire the directory path and remove the   */
                /* closing ">" if there is one.                */

                if ( line != NULL )
                {
                    osi_strlcpy( current_block->directory, line,
                                 sizeof( current_block->directory ) );

                    line = strrchr( current_block->directory, '>' );

                    if ( line != NULL )
                    {
                        *line = '\0';
                    }
                }

                /* add to the block directory names list. */

                string_list_add_item( cfg->block_names,
                                      current_block->directory );

                break;

            case CONFIG_KEYWORD_DIRECTORY_END:

                current_block = global_block;
                inside_directory_block  = FALSE;

                break;

            case CONFIG_KEYWORD_SYSTEM_START:
                
                inside_system_block = TRUE;
                break;

            case CONFIG_KEYWORD_SYSTEM_END:

                inside_system_block = FALSE;
                break;

            case CONFIG_KEYWORD_PARAM:

                if ( !inside_system_block )
                {
                    break;
                }

                get_module_parameter( cfg, line );
                break;

            case CONFIG_KEYWORD_RECURSIVE:

                if ( !inside_directory_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_boolean_value( keyword, &current_block->recursive );

                break;

            case CONFIG_KEYWORD_FOLLOWLINKS:

                if ( !inside_directory_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_boolean_value( keyword, &current_block->follow_links );

                break;

            case CONFIG_KEYWORD_HASH:

                if ( !inside_directory_block )
                {
                    break;
                }

                get_token( line, keyword, sizeof( keyword ) );
                get_hash_value( keyword, &current_block->hash );

                break;

            case CONFIG_KEYWORD_INCLUDEALL:

                if ( !inside_directory_block )
                {
                    break;
                }

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_INCLUDE_ALL;

                /* add action to block's list of actions. */

                list_add( current_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_EXCLUDEALL:

                if ( !inside_directory_block )
                {
                     break;
                }

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_EXCLUDE_ALL;

                list_add( current_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_INCLUDE:

                if ( inside_system_block )
                {
                    get_token( line, keyword, sizeof( keyword ) );
                    temp_module = create_module();

                    osi_strlcpy( temp_module->name,
                                 keyword, sizeof( temp_module->name ) );

                    list_add( cfg->modules, temp_module );

                    break;
                }

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_INCLUDE;

                /* set the filter type. */

                temp_action->action_filter.type = get_filter_type( line );

                /* get argument if there is one, setup filter, then */
                /* get the attributes.                              */

                if ( filter_is_dynamic( temp_action->action_filter.type ) )
                {
                    line = get_filter_value( line,
                                 temp_action->action_filter.argument );
                }

                else
                {
                    line = strstr( line, " " );
                }

                setup_filter_function( &temp_action->action_filter,
                                       temp_action->action_filter.type );

                /* add action to block's list of actions. */

                list_add( current_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_EXCLUDE:

                temp_action = create_action();
                temp_action->type = ACTION_TYPE_EXCLUDE;

                /* set the filter type. */

                temp_action->action_filter.type = get_filter_type( line );

                /* get argument if there is one, then setup filter function. */

                if ( filter_is_dynamic( temp_action->action_filter.type ) )
                {
                    line = get_filter_value( line,
                                temp_action->action_filter.argument );
                }

                setup_filter_function( &temp_action->action_filter,
                                       temp_action->action_filter.type );

                /* add action to block's list of actions. */

                list_add( current_block->actions, temp_action );

                break;

            case CONFIG_KEYWORD_NOENTRY:

                osi_strlcpy( temp, current_block->directory, sizeof( temp ) );

                if ( ( strlen(temp) > 0 ) &&
                     ( temp[strlen(temp)-1] != PATH_SEPARATOR ) )
                {
                    osi_strlcat( temp, path_separator, sizeof( temp ) );
                }

                osi_strlcat( temp, line, sizeof( temp ) );

                string_list_add_item( current_block->no_entry_list, temp );

                break;

            case CONFIG_KEYWORD_UNKNOWN:
            default:

                /* this should NEVER happen if config data was verified. */

                break;

        } /* end switch */

    } /* end for loop */

        return TRUE;
}

/******************************************************************************
**
**    Function: osi_config_generate_id
**
**    Purpose:  open and read data from the specified file into the passed in
**		        configuration structure.
**
******************************************************************************/

osi_bool osi_config_generate_id( OSI_SCAN_CONFIG *cfg )
{
    MD5_CTX context;
    unsigned char digest[MD5_DIGEST_LENGTH];

    osi_bool result = FALSE;
    unsigned int index;

    char config_hash[MAX_CONFIG_ID_LENGTH];

    if ( cfg != NULL )
    {
        MD5_Init( &context );

        /* go through each line in config data. */

        for( index = 0; ( index < cfg->data->size ); index++ )
        {
            char *line = cfg->data->list[index];

            /* ignore comments and empty lines. */

            if ( ( line != NULL ) && ( line[0] != 0 ) )
            {
                unsigned int position             = 0;
                unsigned int config_line_position = 0;

                char config_line[MAX_LINE_LENGTH];

                /* go through line copying only printable characters. */

                for( position = 0; ( position < strlen( line ) ); position++ )
                {
                    if ( isgraph( line[position] ) )
                    {
                        config_line[config_line_position] = line[position];
                        config_line_position++;
                    }
                }

                config_line[config_line_position] = '\0';

                /* now add to the MD5 context. */

                if ( config_line_position > 0 )
                {
                    MD5_Update( &context, config_line,
                                (unsigned long)config_line_position );
                }
            }
        }

        MD5_Final( &( digest[0] ), &context );
        
        /* we use only the last four bytes of the digest for an */
        /* identifier for configs, not the entire checksum.     */

        memset( config_hash, 0, sizeof( config_hash ) );

        osi_snprintf( &config_hash[0], 3, "%02x", digest[12] );
        osi_snprintf( &config_hash[2], 3, "%02x", digest[13] );
        osi_snprintf( &config_hash[4], 3, "%02x", digest[14] );
        osi_snprintf( &config_hash[6], 3, "%02x", digest[15] );

        config_hash[8] = '\0';

        osi_strlcpy( cfg->id, config_hash, sizeof( config_hash ) );
        result = TRUE;
    }

    return result;
}

/******************************************************************************
**
**    Function: osi_config_write_to_file
**
**    Purpose:  open and write data to the specified file from the passed in
**              configuration structure.
**
******************************************************************************/

osi_bool osi_config_write_to_file( OSI_SCAN_CONFIG *cfg, const char *file_path )
{
    int index;
    char *line;

    FILE *config_file = NULL;

    if ( ( cfg == NULL ) || ( file_path == NULL )  )
    {
        return FALSE;
    }

    if ( ( cfg->data == NULL ) || ( cfg->data->size == 0 ) )
    {
        return FALSE;
    }

    config_file = osi_fopen( file_path, "w+", 0600 );

    if ( config_file == NULL )
    {
        return FALSE;
    }

    for( index = 0; (unsigned int)index < cfg->data->size; index++ )
    {
        line = cfg->data->list[index];

        if ( line != NULL )
        {
            fwrite( line, sizeof( char ), strlen( line ), config_file );
            fwrite( "\n", sizeof( char ), 1, config_file );
        }

        else
        {
            fwrite( "\n", sizeof( char ), 1, config_file );
        }
    }

    fclose( config_file );
    return TRUE;
}



/******************************************************************************
**
**    Function: osi_config_read_from_file
**
**    Purpose:  open and read data from the specified file into the passed in
**		        configuration structure.
**
******************************************************************************/

osi_bool osi_config_read_from_file( OSI_SCAN_CONFIG *cfg,const char *file_path )
{
    FILE *file;
    string_list *lines;

    char *line;
    char temp_line[MAX_LINE_LENGTH] = "";

    unsigned int index;
    osi_bool result = FALSE;

    if ( ( cfg == NULL ) || ( file_path == NULL ) )
    {
        return FALSE;
    }

    file = osi_fopen( file_path, "r", 0 );

    if ( file == NULL )
    {
        return FALSE;
    }

    lines = string_list_new();

    /* create string list of lines from file. */

    for (;;)
    {
        line = fgets( temp_line, sizeof( temp_line ), file );

        if ( line == NULL)
        {
            break;
        }

        line = trim_white_space( line );

        if ( line != NULL )
        {
            string_list_add_item( lines, line );
        }
    }

    if ( lines->size > 0 )
    {
        /* initialize config by sending first line. */

        osi_config_receive_first( cfg, lines->list[0],
                                  strlen( lines->list[0] ) );

        /* now populate the config structure with rest of lines. */

        for( index = 1; index < lines->size; index++ )
        {
            if ( ( index + 1 ) != lines->size )
            {
                osi_config_receive( cfg, lines->list[index],
                                    strlen( lines->list[index] ) );
            }

            else
            {
                osi_config_receive_last( cfg, lines->list[index],
                                         strlen( lines->list[index] ) );
            }
        }

        result = TRUE;
    }

    string_list_destroy( lines );
    fclose( file );

    return result;
}


/******************************************************************************
**
**    Function: osi_config_dump
**
**    Purpose:  debug routine that is a very raw dump of the config data.
**              this is a bit old and unused since the implementation of
**              the debug macros.
**
******************************************************************************/

void osi_config_dump( OSI_SCAN_CONFIG *cfg )
{
    node *temp_node = NULL;

    if ( cfg == NULL )
    {
        return;
    }

    fprintf( stderr,"\n\n---------------- " );
    fprintf( stderr, "[ config (%s) ] ----------------\n\n",
           ( ( strlen( cfg->name ) > 0) ? cfg->name : "<no name>" ) );

    fprintf( stderr, "FILE:  " );
    fprintf( stderr, "LINES: " );

    if ( cfg->data != NULL )
    {
        fprintf( stderr, "%d\n", cfg->data->size );
    }

    else
    {
        fprintf( stderr, "<unknown>\n" );
    }

    fprintf( stderr, "\nERRORS:   %d\nWARNINGS: %d\n\n",
             cfg->error_count, cfg->warning_count );

    /* dump all blocks in config. */

    if ( cfg->blocks )
    {
        temp_node = cfg->blocks->next;
    }

    fprintf( stderr, "( GLOBAL BLOCK )\n" );
    osi_config_dump_block( temp_node->data );
    temp_node = temp_node->next;

    while( temp_node != NULL )
    {
        osi_config_dump_block( temp_node->data );
        temp_node = temp_node->next;
    }

    /* dump modules. */

    if ( cfg->modules )
    {
        temp_node = cfg->modules->next;
    }

    fprintf( stderr, "( MODULES )\n\n" );

    if( temp_node )
    {
        osi_config_dump_module( temp_node->data );
        temp_node = temp_node->next;
    }

    while( temp_node != NULL )
    {
        osi_config_dump_module( temp_node->data );
        temp_node = temp_node->next;
    }
}

void osi_config_dump_module( module *mod )
{
    node *n;

    if ( mod == NULL )
    {
        return;
    }

    fprintf( stderr, "\n====================\n\n" );
    fprintf( stderr, "name: %s\n", mod->name );

    n = list_get_first_node( mod->params );

    while( n )
    {
        param *p = (param *)n->data;

        if ( p )
        {
            fprintf( stderr, "  ==> param: %s; %s\n", p->name, p->value );
        }

        n = n->next;
    }
}

/******************************************************************************
**
**    Function: osi_config_dump_block
**
**    Purpose:  raw dump of config block for debug purposes.
**
******************************************************************************/

void osi_config_dump_block( block *config_block )
{
    node *temp_node = NULL;

    if ( config_block == NULL )
    {
        return;
    }

    fprintf( stderr, "\n====================\n\n" );

    fprintf( stderr, "DIRECTORY:  %s\n", config_block->directory );
    fprintf( stderr, "RECURSIVE:  %d\n", config_block->recursive );
    fprintf( stderr, "FOLLOWLINKS:%d\n", config_block->follow_links );
    fprintf( stderr, "HASH:       %d\n", config_block->hash );

    fprintf( stderr, "NO ENTRY LIST: \n" );
    string_list_dump( config_block->no_entry_list );

    if ( config_block->actions )
    {
        temp_node = config_block->actions->next;
    }

    while( temp_node != NULL )
    {
        osi_config_dump_action( temp_node->data );
        temp_node = temp_node->next;
    }

    fprintf( stderr, "\n" );
}

/******************************************************************************
**
**    Function: osi_config_dump_action
**
**    Purpose:  raw dump of action for debugging purposes.
**
******************************************************************************/

void osi_config_dump_action( action *config_action )
{
    fprintf( stderr,
             "[action %d] type(%d) filter type(%d) filter argument(%s)\n",
              (int)config_action, (int)config_action->type,
              (int)config_action->action_filter.type,
              config_action->action_filter.argument );
}

