
/*****************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    cmp_filter.c
**  Date:    September 17, 2003
**
**  Author:  Brian Wotring
**
**  Purpose: structure representing a comparison filter object
**           used to filter out deltas from the logs.
**
*****************************************************************************/

#include "libosirism.h"
#include <string.h>

static struct n_keywords cmp_keywords[] =
{
    { "csum", ATTR_CHECKSUM },
    { "device", ATTR_DEVICE },
    { "inode", ATTR_INODE },
    { "perm", ATTR_PERMISSIONS },
    { "links", ATTR_LINKS },
    { "uid", ATTR_UID },
    { "gid", ATTR_GID },
    { "mtime", ATTR_MTIME },
    { "atime", ATTR_ATIME },
    { "ctime", ATTR_CTIME },
    { "dtype", ATTR_DEVICE_TYPE },
    { "bytes", ATTR_BYTES },
    { "blocks", ATTR_BLOCKS },
    { "bsize", ATTR_BLOCK_SIZE },
    { "osid", ATTR_OWNER_SID },
    { "gsid", ATTR_GROUP_SID },
    { "fileattr", ATTR_WIN32_FILE },
    { "new", ATTR_NEW },
    { "missing", ATTR_MISSING },
    { NULL, 0 }
};

static struct n_keywords cmp_descriptions[] =
{
    { "checksum", ATTR_CHECKSUM },
    { "device number", ATTR_DEVICE },
    { "inode number", ATTR_INODE },
    { "permissions (mode)", ATTR_PERMISSIONS },
    { "number of hard links", ATTR_LINKS },
    { "user ID", ATTR_UID },
    { "group ID", ATTR_GID },
    { "last modification time", ATTR_MTIME },
    { "last access time", ATTR_ATIME },
    { "last change time", ATTR_CTIME },
    { "device type", ATTR_DEVICE_TYPE },
    { "number of bytes", ATTR_BYTES },
    { "number of blocks", ATTR_BLOCKS },
    { "block size", ATTR_BLOCK_SIZE },
    { "owner SID", ATTR_OWNER_SID },
    { "group SID", ATTR_GROUP_SID },
    { "windows file attributes", ATTR_WIN32_FILE },
    { "not in trusted database", ATTR_NEW },
    { "not present in latest scan", ATTR_MISSING },
    { NULL, 0 }
};


unsigned long mask_from_cmp_keywords( const char *keywords )
{
    int index   = 0;
    int valid   = 0;

    unsigned long mask = 0;

    char *token = NULL;
    char temp_values[MAX_LINE_LENGTH] = "";

    if( keywords == NULL )
    {
        return 0;
    }

    /* strtok will ruin the string we were given */
    /* so we copy the values data.               */

    osi_strlcpy( temp_values, keywords, ( strlen( keywords ) + 1 ) );

    /* only one value, don't seperate. */

    token = strtok( temp_values, "," );
    token = trim_white_space( token );

    while( token )
    {
        valid = 0;
        lowercase_string( token );

        for( index = 0; cmp_keywords[index].word != NULL; index++ )
        {
            if( strcmp( token, cmp_keywords[index].word ) == 0 )
            {
                mask |= cmp_keywords[index].type;
            }
        }

        token = strtok( NULL, "," );
        token = trim_white_space( token );
    }

    return mask;
}

void print_cmp_filter( OSI_CMP_FILTER *filter )
{
    if( filter == NULL )
    {
        return;
    }

    fprintf( stdout, "exclude anything matching: \"%s\"", filter->exclude );
}

void print_cmp_attribute_mask( osi_uint64 attr_list )
{
    int count = 0;
    int index;

    unsigned int attr = (unsigned int)attr_list;

    for( index = 0; cmp_keywords[index].word != NULL; index++ )
    {
        if( attr & cmp_keywords[index].type )
        {
            if( count > 0 )
            {
                fprintf( stdout, "," );
            }

            fprintf( stdout, "%s",  cmp_keywords[index].word );
            count++;
        }
    }
}

char * osi_get_name_for_cmp_attribute( osi_uint64 attr )
{
    int index;
    char *name = "?";

    for( index = 0; cmp_keywords[index].word != NULL; index++ )
    {
        if( attr == cmp_keywords[index].type )
        {
            name = cmp_keywords[index].word;
            break;
        }
    }

    return name;
}

void print_valid_cmp_filter_attributes()
{
    int index;

    fprintf( stdout, "\n" );

    for( index = 0; cmp_keywords[index].word != NULL; index++ )
    {
        fprintf( stdout, "  %10s - %s\n",
                 cmp_keywords[index].word,
                 cmp_descriptions[index].word );
    }
}

unsigned int pack_cmp_filter( OSI_CMP_FILTER *filter, char *buffer,
                                 int buffer_size )
{
    char *temp = buffer;

    if( ( filter == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return 0;
    }

    memcpy( temp, filter->exclude, sizeof( filter->exclude) );
    temp += sizeof( filter->exclude);

    return ( temp - buffer );
}

void unpack_cmp_filter( OSI_CMP_FILTER *filter, char *buffer,
                        int buffer_size )
{
    char *temp = buffer;

    if( ( filter == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return;
    }

    memcpy( filter->exclude, temp, sizeof( filter->exclude ) );
}

