
/*****************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    host_config.c
**  Date:    August 18, 2002
**
**  Author:  Brian Wotring
**  Purpose: common host configuration module to read and parse host
**           configuration from a flat file.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"

static struct h_keywords host_config_keywords[] =
{
    { "host", HOST_CONFIG_KEYWORD_HOSTNAME },
    { "description", HOST_CONFIG_KEYWORD_DESCRIPTION },
    { "type", HOST_CONFIG_KEYWORD_TYPE },
    { "enabled", HOST_CONFIG_KEYWORD_ENABLED },
    { "notify_enabled", HOST_CONFIG_KEYWORD_NOTIFY_ENABLED },
    { "notify_flags", HOST_CONFIG_KEYWORD_NOTIFY_FLAGS },
    { "notify_email", HOST_CONFIG_KEYWORD_NOTIFY_EMAIL },
    { "session_key", HOST_CONFIG_KEYWORD_SESSION_KEY },
    { "base_db", HOST_CONFIG_KEYWORD_BASE_DB },
    { "log_to_file", HOST_CONFIG_KEYWORD_FILE_LOG },
    { "schedule_start", HOST_CONFIG_KEYWORD_SCHEDULE_START },
    { "schedule_period", HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD },
    { "db_flags", HOST_CONFIG_KEYWORD_DB_FLAGS },
    { "port", HOST_CONFIG_KEYWORD_PORT },
    { "config", HOST_CONFIG_KEYWORD_CONFIG },
    { NULL, 0 }
};

static struct h_keywords h_boolean_keywords[] =
{
    { "1", TRUE },
    { "y", TRUE },
    { "yes", TRUE },
    { "true", TRUE },
    { "0", FALSE },
    { "n", FALSE },
    { "no", FALSE },
    { "false", FALSE },
    { NULL, 0 }
};


OSI_HOST_CONFIG * osi_host_config_new()
{
    OSI_HOST_CONFIG *config = osi_malloc( sizeof( OSI_HOST_CONFIG ) );
    
    if( config != NULL )
    {
        memset( config, 0, sizeof( OSI_HOST_CONFIG ) );
        config->data = string_list_new();
    }
    
    return config;
}

void osi_host_config_destroy( OSI_HOST_CONFIG *config )
{
    if( config != NULL )
    {
        /* destroy the string list containig raw data. */
        
        string_list_destroy( config->data );
        
        /* now destroy the config structure itself. */
        
        osi_free( config );
    }
}

OSI_HOST_CONFIG * osi_host_config_new_from_file( const char *filepath )
{
    OSI_HOST_CONFIG *config = osi_host_config_new();
    
    if( osi_host_config_read_file( filepath, config ) )
    {
        return config;
    }
    
    osi_host_config_destroy( config );
    return NULL;
}

osi_bool osi_host_config_receive_first( OSI_HOST_CONFIG *config,
                                        unsigned char *data, int data_length )
{
    if( config != NULL )
    {
        /* destroy data if not destroyed already, then create a new list */
        /* and set state to receiving.  then add incoming data.          */

        string_list_destroy( config->data );
        config->data = string_list_new();

        return osi_host_config_receive( config, data, data_length );
    }

    return FALSE;
}

osi_bool osi_host_config_receive( OSI_HOST_CONFIG *config, unsigned char *data,
                                  int data_length )
{
    if( ( config != NULL ) && ( data != NULL ) && ( data_length > 0 ) )
    {
        data = trim_white_space( data );
        
        string_list_add_item( config->data, data );
        parse_host_config_line( data, config );

        return TRUE;
    }

    return FALSE;
}

osi_bool osi_host_config_receive_last( OSI_HOST_CONFIG *config,
                                       unsigned char *data, int data_length )
{
    osi_bool result = FALSE;

    if( config != NULL )
    {
        osi_host_config_receive( config, data, data_length );
    }

    return result;
}

osi_bool osi_host_config_read_file( const char *filepath,
                                    OSI_HOST_CONFIG *config )
{
    osi_bool result = FALSE;
    FILE *file      = NULL;

    char *line;
    char temp_line[MAX_HOST_CONFIG_LINE_LENGTH] = "";
        
    if( ( filepath != NULL ) && ( config != NULL ) )
    {
        /* clobber current data. */

        string_list_destroy( config->data );
        config->data = string_list_new();
        
        if( ( file = osi_fopen( filepath, "r",
                                HOST_CONFIG_PERMISSIONS ) ) != NULL )
        {
            for(;;)
            {
                line = fgets( temp_line, sizeof( temp_line ), file );

                if( line  == NULL )
                {
                    break;
                }

                line = trim_white_space( line );

                if( line != NULL )
                {
                    string_list_add_item( config->data, line );
                    parse_host_config_line( line, config );
                }
            }

            fclose( file );
            result = TRUE;
        }
    }

    return result;
}

osi_bool osi_host_config_write_file( const char *filepath,
                                     OSI_HOST_CONFIG *config )
{
    osi_bool result = FALSE;
    FILE *file = NULL;

    if( ( filepath != NULL ) && ( config != NULL ) )
    {
        if( ( config->data != NULL ) && ( config->data->size > 0 ) )
        {
            int index = 0;
            file = osi_fopen( filepath, "w+", HOST_CONFIG_PERMISSIONS );

            if( file != NULL )
            {
                result = TRUE;
                
                for( index = 0;
                     (unsigned int)index < ( config->data->size );
                     index++ )
                {
                    char *line = (config->data->list)[index];
                    
                    if( line != NULL )
                    {
                        fwrite( line, sizeof( char ), strlen( line ), file );
                    }
                }
                
                fclose( file );
            }
        }
    }
    
    return result;
}

osi_bool osi_host_brief_read_file( const char *filepath,
                                   OSI_HOST_BRIEF *host_brief )
{
    osi_bool result = FALSE;
    FILE *file  = NULL;

    char *line;
    char temp_line[MAX_HOST_CONFIG_LINE_LENGTH] = "";
        
    if( ( filepath != NULL ) && ( host_brief != NULL ) )
    {
        /* clobber host brief data so we have a true representation */
        /* of the file.                                             */

        memset( host_brief, 0, sizeof( OSI_HOST_BRIEF ) );
        file = osi_fopen( filepath, "r", HOST_CONFIG_PERMISSIONS );

        if( file != NULL )
        {
            for(;;)
            {
                line = fgets( temp_line, sizeof( temp_line ), file );

                if( line == NULL )
                {
                    break;
                }

                line = trim_white_space( line );

                if( line != NULL )
                {
                    parse_host_brief_line( line, host_brief );
                }
            }

            fclose( file );
            result = TRUE;
        }
    }

    return result;
}

osi_bool osi_host_brief_write_file( const char *filepath,
                                    OSI_HOST_BRIEF *host_brief )
{
    osi_bool result = FALSE;
    FILE *file  = NULL;

    char *line;
    char temp_line[MAX_HOST_CONFIG_LINE_LENGTH] = "";

    char buffer[30];

    if( ( filepath != NULL ) && ( host_brief != NULL ) )
    {
        int index;
        string_list *lines = string_list_new();
            
        /* first, we put our host brief values into the string list. */
        
        for( index = 0; host_config_keywords[index].word; index++ )
        {
            osi_strlcpy( temp_line, host_config_keywords[index].word,
                         sizeof( temp_line ) );

            osi_strlcat( temp_line, " = ", sizeof( temp_line ) );

            switch( host_config_keywords[index].type )
            {
                case HOST_CONFIG_KEYWORD_HOSTNAME:
                    osi_strlcat( temp_line, host_brief->host,
                                 sizeof( temp_line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_DESCRIPTION:
                    osi_strlcat( temp_line, host_brief->description,
                                 sizeof( temp_line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_TYPE:
                    osi_strlcat( temp_line,
                                 get_name_from_host_type( 
                                            (int)host_brief->type ),
                                            sizeof( temp_line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_ENABLED:
                
                    if( host_brief->enabled == 1 )
                    {
                        osi_strlcat( temp_line, "yes", sizeof( temp_line ) );
                    }
    
                    else
                    {
                        osi_strlcat( temp_line, "no", sizeof( temp_line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_FILE_LOG:

                    if( host_brief->file_log_enabled == 1 )
                    {
                        osi_strlcat( temp_line, "yes", sizeof( temp_line ) );
                    }

                    else
                    {
                        osi_strlcat( temp_line, "no", sizeof( temp_line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_ENABLED:

                    if( host_brief->notify_enabled == 1 )
                    {
                        osi_strlcat( temp_line, "yes", sizeof( temp_line ) );
                    }

                    else
                    {
                        osi_strlcat( temp_line, "no", sizeof( temp_line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_FLAGS:


                    osi_snprintf( buffer, sizeof(buffer), "%llu",
                                  host_brief->notify_flags );

                    osi_strlcat( temp_line, buffer, sizeof( temp_line ) );

                    break;

                case HOST_CONFIG_KEYWORD_DB_FLAGS:

                    osi_snprintf( buffer, sizeof(buffer), "%llu",
                                  host_brief->db_flags );

                    osi_strlcat( temp_line, buffer, sizeof( temp_line ) );

                    break;

                case HOST_CONFIG_KEYWORD_PORT:

                    osi_snprintf( buffer, sizeof(buffer), "%llu",
                                  host_brief->port );

                    osi_strlcat( temp_line, buffer, sizeof( temp_line ) );

                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_EMAIL:
                    osi_strlcat( temp_line, host_brief->notify_email,
                                 sizeof( temp_line ) );
                    break;

                case HOST_CONFIG_KEYWORD_SCHEDULE_START:

                    /* never write zero, it prevents the scheduler */
                    /* from working correctly.                     */

                    if ( host_brief->schedule_start == 0 )
                    {
                        host_brief->schedule_start = (osi_get_time() - 61);
                    }

                    osi_snprintf( buffer, sizeof(buffer), "%llu",
                                  host_brief->schedule_start );

                    osi_strlcat( temp_line, buffer, sizeof( temp_line ) );
                    
                    break;

                case HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD:

                    osi_snprintf( buffer, sizeof(buffer), "%llu",
                                  host_brief->schedule_period );

                    osi_strlcat( temp_line, buffer, sizeof( temp_line ) );

                    break;

                case HOST_CONFIG_KEYWORD_SESSION_KEY:

                    osi_strlcat( temp_line, host_brief->session_key,
                                 sizeof( temp_line ) );
                    break;

                case HOST_CONFIG_KEYWORD_BASE_DB:

                    osi_strlcat( temp_line, host_brief->base_db,
                                 sizeof( temp_line ) );
                    break;

                case HOST_CONFIG_KEYWORD_CONFIG:

                    osi_strlcat( temp_line, host_brief->config,
                                 sizeof( temp_line ) );
                    break;

                default:
                    break;
            }
            
            osi_strlcat( temp_line, "\n", sizeof( temp_line ) );
            string_list_add_item( lines, temp_line );
        }

        /* now read in the other lines, not host brief items, if the file */
        /* didn't exist, this will just get ignored.                      */     
        file = osi_fopen( filepath, "r", HOST_CONFIG_PERMISSIONS );

        if( file != NULL )
        {
            for(;;)
            {
                line = fgets( temp_line, sizeof( temp_line ), file );

                if( line == NULL )
                {
                    break;
                }

                line = trim_white_space( line );

                if( line != NULL )
                {
                    char save_line[MAX_HOST_CONFIG_LINE_LENGTH];
                    int type = get_host_config_line_type( line );
                    
                    switch( type )
                    {
                        case HOST_CONFIG_KEYWORD_HOSTNAME:                     
                        case HOST_CONFIG_KEYWORD_DESCRIPTION:
                        case HOST_CONFIG_KEYWORD_TYPE:
                        case HOST_CONFIG_KEYWORD_ENABLED:
                        case HOST_CONFIG_KEYWORD_SESSION_KEY:
                        case HOST_CONFIG_KEYWORD_BASE_DB:
                        case HOST_CONFIG_KEYWORD_SCHEDULE_START:
                        case HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD:
                        case HOST_CONFIG_KEYWORD_FILE_LOG:
                        case HOST_CONFIG_KEYWORD_NOTIFY_ENABLED:
                        case HOST_CONFIG_KEYWORD_NOTIFY_FLAGS:
                        case HOST_CONFIG_KEYWORD_NOTIFY_EMAIL:
                        case HOST_CONFIG_KEYWORD_DB_FLAGS:
                        case HOST_CONFIG_KEYWORD_PORT:
                        case HOST_CONFIG_KEYWORD_CONFIG:

                            break;
                        
                        default:

                       /* if this is not a host brief attribute, then we     */
                       /* put it back into the file, otherwise it's ignored. */
                            
                        osi_strlcpy( save_line, line, sizeof( save_line ) );
                        osi_strlcat( save_line, "\n", sizeof( save_line ) );
                        string_list_add_item( lines, save_line );
                    
                        break;
                    }
                }
            }
    
            fclose( file );
        }

        /* now, we clober the file with the new string list data */
        /* we have created.                                      */

        file = osi_fopen( filepath, "w+", HOST_CONFIG_PERMISSIONS );
            
        if( file == NULL )
        {
            string_list_destroy( lines );
            return FALSE;
        }
        
        /* write each line in our string list to the file. */
        
        for( index = 0; (unsigned int)index < ( lines->size ); index++ )
        {
            line = (lines->list)[index];
            
            if( line != NULL )
            {
                fwrite( line, sizeof( char ), strlen( line ), file );
            }
        }
    
        string_list_destroy( lines );
        fclose( file );
        
        result = TRUE;
    }

    return result;
}


osi_bool osi_host_config_update_data( OSI_HOST_CONFIG *config )
{
    osi_bool result = FALSE;
    
    if( config != NULL )
    {
        int index;
        char line[MAX_HOST_CONFIG_LINE_LENGTH];

        char buffer[30];
        
        /* destroy the old data. */
        
        string_list_destroy( config->data );
        config->data = string_list_new();
        
        result = TRUE;
        
        /* put config data into the string list. */
        
        for( index = 0; host_config_keywords[index].word; index++ )
        {
            osi_strlcpy( line, host_config_keywords[index].word,
                         sizeof( line ) );

            osi_strlcat( line, " = ", sizeof( line ) );

            switch( host_config_keywords[index].type )
            {
                case HOST_CONFIG_KEYWORD_HOSTNAME:
                    osi_strlcat( line, config->hostname, sizeof( line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_DESCRIPTION:
                    osi_strlcat( line, config->description, sizeof( line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_TYPE:
                    osi_strlcat( line, get_name_from_host_type( config->type ),
                                 sizeof( line ) );
                    break;
                    
                case HOST_CONFIG_KEYWORD_ENABLED:
                
                    if( config->enabled == 1 )
                    {
                        osi_strlcat( line, "yes", sizeof( line ) );
                    }
    
                    else
                    {
                        osi_strlcat( line, "no", sizeof( line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_FILE_LOG:

                    if( config->file_log_enabled == 1 )
                    {
                        osi_strlcat( line, "yes", sizeof( line ) );
                    }

                    else
                    {
                        osi_strlcat( line, "no", sizeof( line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_ENABLED:

                    if( config->notify_enabled == 1 )
                    {
                        osi_strlcat( line, "yes", sizeof( line ) );
                    }

                    else
                    {
                        osi_strlcat( line, "no", sizeof( line ) );
                    }

                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_FLAGS:

                    osi_snprintf( buffer, sizeof(buffer), "%lu",
                                  config->notify_flags );
                    break;

                case HOST_CONFIG_KEYWORD_NOTIFY_EMAIL:
                    osi_strlcat( line, config->notify_email, sizeof( line ) );
                    break;

                case HOST_CONFIG_KEYWORD_DB_FLAGS:

                    osi_snprintf( buffer, sizeof(buffer), "%lu",
                                  config->db_flags );
                    break;

                case HOST_CONFIG_KEYWORD_PORT:

                    osi_snprintf( buffer, sizeof(buffer), "%lu",
                                  config->port );
                    break;

                case HOST_CONFIG_KEYWORD_SCHEDULE_START:

                    osi_snprintf( buffer, sizeof(buffer), "%lu",
                                  config->schedule_start );
                    break; 

                case HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD:

                    osi_snprintf( buffer, sizeof(buffer), "%lu", 
                                  config->schedule_period );
                    break;
                    
                case HOST_CONFIG_KEYWORD_SESSION_KEY:
                    osi_strlcat( line, config->session_key, sizeof( line ) );
                    break;

                case HOST_CONFIG_KEYWORD_BASE_DB:
                    osi_strlcat( line, config->base_db, sizeof( line ) );
                    break;

                case HOST_CONFIG_KEYWORD_CONFIG:
                    osi_strlcat( line, config->config, sizeof( line ) );
                    break;
                
                default:
                    break;
            }
            
            osi_strlcat( line, "\n", sizeof( line ) );
            string_list_add_item( config->data, line );
        }
    }
    
    return result;
}

char *get_name_from_host_type( int type )
{
    char *name = OSI_HOST_TYPE_NAME_UNKNOWN;
    
    if( type > 0 )
    {
        switch( type )
        {
            case OSI_HOST_TYPE_GENERIC:
                name = OSI_HOST_TYPE_NAME_GENERIC;
                break;
            
            case OSI_HOST_TYPE_ROUTER:
                name = OSI_HOST_TYPE_NAME_ROUTER;
                break;
                
            default:
                break;
        }
    }
    
    return name;
}


osi_bool parse_host_config_line( char *line, OSI_HOST_CONFIG *config )
{
    osi_bool result = FALSE;
    char value[MAX_HOST_CONFIG_KEY_LENGTH] = "";
    
    if( ( line != NULL ) && ( config != NULL ) )
    {
        int line_type = get_host_config_line_type( line );

        switch( line_type )
        {
            case HOST_CONFIG_KEYWORD_DESCRIPTION:

                if( get_host_config_line_value( line,
                                                value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->description,
                                 value, sizeof( config->description ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_ENABLED:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        config->enabled = 1;
                    }

                    else
                    {
                        config->enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_FILE_LOG:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        config->file_log_enabled = 1;
                    }

                    else
                    {
                        config->file_log_enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_ENABLED:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        config->notify_enabled = 1;
                    }

                    else
                    {
                        config->notify_enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_FLAGS:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    config->notify_flags = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_EMAIL:

                if( get_host_config_line_value( line,
                                                value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->notify_email,
                                 value, sizeof( config->notify_email ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_DB_FLAGS:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    config->db_flags = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_PORT:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    config->port = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_SCHEDULE_START:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    config->schedule_start = atol( value );
                }
                
                break;

            case HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    config->schedule_period = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_HOSTNAME:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->hostname,
                                 value, sizeof( config->hostname ) );
                }

                break;
                
            case HOST_CONFIG_KEYWORD_TYPE:
            
                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    lowercase_string( value );
                    
                    if( strcmp( value, OSI_HOST_TYPE_NAME_ROUTER ) == 0 )
                    {
                        config->type = OSI_HOST_TYPE_ROUTER;
                    }
                    
                    else
                    {
                        config->type = OSI_HOST_TYPE_GENERIC;
                    }
                }
                
                break;

            case HOST_CONFIG_KEYWORD_SESSION_KEY:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->session_key,
                                 value, sizeof( config->session_key ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_BASE_DB:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->base_db,
                                 value, sizeof( config->base_db ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_CONFIG:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( config->config,
                                 value, sizeof( config->config ) );
                }

                break;
                
            case HOST_CONFIG_KEYWORD_COMMENT:
            
                /* suck up the rest of the line, do nothing with it. */

                get_host_config_line_value( line, value, sizeof( value ) );
                break;

            case HOST_CONFIG_KEYWORD_UNKNOWN:
            default:
                break;
        }
    }

    return result;
}

osi_bool parse_host_brief_line( char *line, OSI_HOST_BRIEF *host_brief )
{
    osi_bool result = FALSE;
    char value[MAX_HOST_CONFIG_KEY_LENGTH] = "";
    
    if( ( line != NULL ) && ( host_brief != NULL ) )
    {
        int line_type = get_host_config_line_type( line );

        switch( line_type )
        {
            case HOST_CONFIG_KEYWORD_DESCRIPTION:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->description,
                                 value, sizeof( host_brief->description ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_ENABLED:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        host_brief->enabled = 1;
                    }

                    else
                    {
                        host_brief->enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_FILE_LOG:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        host_brief->file_log_enabled = 1;
                    }

                    else
                    {
                        host_brief->file_log_enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_ENABLED:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    if( translate_boolean_value( value ) == TRUE )
                    {
                        host_brief->notify_enabled = 1;
                    }

                    else
                    {
                        host_brief->notify_enabled = 0;
                    }
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_FLAGS:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    host_brief->notify_flags = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_NOTIFY_EMAIL:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->notify_email,
                                 value, sizeof( host_brief->notify_email ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_DB_FLAGS:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    host_brief->db_flags = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_PORT:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    host_brief->port = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_SCHEDULE_START:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    host_brief->schedule_start = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_SCHEDULE_PERIOD:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    host_brief->schedule_period = atol( value );
                }

                break;

            case HOST_CONFIG_KEYWORD_HOSTNAME:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->host,
                                 value, sizeof( host_brief->host ) );
                }

                break;
                
            case HOST_CONFIG_KEYWORD_TYPE:
            
                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    lowercase_string( value );
                    
                    if( strcmp( value, OSI_HOST_TYPE_NAME_ROUTER ) == 0 )
                    {
                        host_brief->type = OSI_HOST_TYPE_ROUTER;
                    }
                    
                    else
                    {
                        host_brief->type = OSI_HOST_TYPE_GENERIC;
                    }
                }
                
                break;

            case HOST_CONFIG_KEYWORD_SESSION_KEY:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->session_key,
                                 value, sizeof( host_brief->session_key ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_BASE_DB:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->base_db,
                                 value, sizeof( host_brief->base_db ) );
                }

                break;

            case HOST_CONFIG_KEYWORD_CONFIG:

                if( get_host_config_line_value( line, value, sizeof( value ) ) )
                {
                    osi_strlcpy( host_brief->config,
                                 value, sizeof( host_brief->config) );
                }

                break;

            case HOST_CONFIG_KEYWORD_COMMENT:
            
                /* suck up the rest of the line, do nothing with it. */
                get_host_config_line_value( line, value, sizeof( value ) );
                break;

            case HOST_CONFIG_KEYWORD_UNKNOWN:
            default:
                break;
        }
    }

    return result;
}


/*****************************************************************************
**
**    Function: get_keyword_type
**
**    Purpose:  determine the type of keyword this is based upon
**              our lists.
**
******************************************************************************/

int get_host_config_line_type( char *string )
{
    int type = HOST_CONFIG_KEYWORD_UNKNOWN;

    int index;
    char word[MAX_HOST_CONFIG_KEY_LENGTH] = "";

    if( string != NULL )
    {
        get_string_up_to_token( string, word, sizeof( word ),
                                HOST_CONFIG_KEY_VALUE_SEPARATOR );

        lowercase_string( word );

        for( index = 0; host_config_keywords[index].word != NULL; index++ )
        {
            if( strncmp( word, host_config_keywords[index].word,
                         strlen( host_config_keywords[index].word ) ) == 0 )
            {
                type = ( host_config_keywords[index].type );
                break;
            }
            
            if( word[0] == HOST_CONFIG_COMMENT_LINE )
            {
                type = HOST_CONFIG_KEYWORD_COMMENT;
                break;
            }
        }
    }

    return type;
}

osi_bool get_host_config_line_value( char *line, char *value, int value_size )
{
    osi_bool result = FALSE;
    char *start = NULL;

    if( ( line != NULL ) && ( value != NULL ) )
    {
        start = (char *)strchr( line, HOST_CONFIG_KEY_VALUE_SEPARATOR );

        if( start != NULL )
        {
            start++;
            start = trim_white_space( start );

            if( ( start != NULL ) && ( (*start) != '\0' ) )
            {
                osi_strlcpy( value, start, value_size );

                if( strlen( value ) > 0 )
                {
                    result = TRUE;
                }
            }
        }
    }

    return result;
}

/*****************************************************************************
**
**    Function: get_boolean_value
**
**    Purpose:  examine keyword based upon our list of valid booleans
**              and return TRUE or FALSE based upon what the string represents.
**
*****************************************************************************/

osi_bool translate_boolean_value( char *keyword )
{
    int index;
    osi_bool result = FALSE;

    if( keyword != NULL )
    {
        lowercase_string( keyword );

        for( index = 0; h_boolean_keywords[index].word != NULL; index++ )
        {
            if( strcmp( keyword, h_boolean_keywords[index].word ) == 0 )
            {
                result = h_boolean_keywords[index].type;
                break;
            }
        }
    }

    return result;
}

