
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    logging.c
**  Date:    February 17, 2002
**
**  Author:  Brian Wotring
**  Purpose: error handling routines
**
******************************************************************************/

#include "libosiris.h"
#include "logging.h"


#ifdef WIN32
#include "windows_service.h"
#else
#include <sys/syslog.h>
#endif

#define MAX_LOG_MESSAGE_LENGTH 1024

extern osi_bool run_as_daemon;
extern const int SYSLOG_FACILITY;
extern const int quiet_mode;

#ifdef WIN32
extern HANDLE event_source;
#endif

void log_error( const char *message, ... )
{
    va_list ap;

#ifdef WIN32
    LPCSTR  lpszStrings[1];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    if ( quiet_mode )
    {
        return;
    }

    /* write the header. */

    osi_strlcpy( header, "[err] ", sizeof( header ) );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_ERROR_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings, /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( SYSLOG_FACILITY | LOG_ERR ), header );
#endif
}

void log_info( const char *message, ... )
{
    va_list ap;

#ifdef WIN32
    LPCSTR  lpszStrings[2];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    if ( quiet_mode )
    {
        return;
    }

    /* write the header. */

    osi_strlcpy( header, "[info] ", sizeof(header) );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_INFORMATION_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings,          /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( SYSLOG_FACILITY | LOG_INFO ), header );
#endif
}

void log_warning( const char *message, ... )
{
    va_list ap;

#ifdef WIN32
    LPCSTR  lpszStrings[2];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    if ( quiet_mode )
    {
        return;
    }

    /* write the header. */

    osi_strlcpy( header, "[warning] ", sizeof( header ) );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_WARNING_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings,          /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( SYSLOG_FACILITY | LOG_WARNING ), header );
#endif
}


