
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_ports.c
**  Date:    January 15, 2005
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for detecting open network ports.
**
******************************************************************************/

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"


static const char *MODULE_NAME = "mod_ports";

#ifdef WIN32

#include "windows.h"
#include "psapi.h"
#include "winsock.h"
#include "iprtrmib.h"
#include "tlhelp32.h"
#include "iphlpapi.h"


typedef struct
{
  DWORD   dwState;        
  DWORD   dwLocalAddr;    
  DWORD   dwLocalPort;    
  DWORD   dwRemoteAddr;   
  DWORD   dwRemotePort;   
  DWORD	  dwProcessId;

} MIB_TCPEXROW, *PMIB_TCPEXROW;


typedef struct
{
	DWORD			dwNumEntries;
	MIB_TCPEXROW	table[ANY_SIZE];

} MIB_TCPEXTABLE, *PMIB_TCPEXTABLE;



typedef struct
{
  DWORD   dwLocalAddr;    
  DWORD   dwLocalPort;    
  DWORD	  dwProcessId;

} MIB_UDPEXROW, *PMIB_UDPEXROW;


typedef struct
{
	DWORD			dwNumEntries;
	MIB_UDPEXROW	table[ANY_SIZE];

} MIB_UDPEXTABLE, *PMIB_UDPEXTABLE;


DWORD( WINAPI *pAllocateAndGetTcpExTableFromStack )
(
  PMIB_TCPEXTABLE *pTcpTable,  
  BOOL bOrder,               
  HANDLE heap,
  DWORD zero,
  DWORD flags
);

DWORD( WINAPI *pAllocateAndGetUdpExTableFromStack )
(
  PMIB_UDPEXTABLE *pTcpTable,  
  BOOL bOrder,           
  HANDLE heap,
  DWORD zero,
  DWORD flags
);


static PMIB_TCPEXTABLE tcpTable = NULL;
static PMIB_UDPEXTABLE udpTable = NULL;


BOOLEAN XPApisArePresent( VOID )
{
	pAllocateAndGetTcpExTableFromStack = (PVOID)GetProcAddress( 
            LoadLibrary( "iphlpapi.dll"),
			        	 "AllocateAndGetTcpExTableFromStack" );

	if( !pAllocateAndGetTcpExTableFromStack )
    {
        return FALSE;
    }

	pAllocateAndGetUdpExTableFromStack = (PVOID)GetProcAddress(
            LoadLibrary( "iphlpapi.dll"),
				         "AllocateAndGetUdpExTableFromStack" );

	if( !pAllocateAndGetUdpExTableFromStack )
    {
        return FALSE;
    }

	return TRUE;
}

char * getProcessFilename( ULONG pid )
{
    const C_MAX_NBR = 256 ;
    DWORD dwaProcessIds [C_MAX_NBR];
    DWORD dwNbrProcesses;
    DWORD dwNbrModules;

    HANDLE hProcess;
    HMODULE hmaModules [C_MAX_NBR];

    char cFilename[MAX_PATH];
    char *pReturn;

    pReturn = "SYSTEM";

    int iSuccess = EnumProcesses( dwaProcessIds, sizeof( dwaProcessIds ),
                                  &dwNbrProcesses);

    dwNbrProcesses = ( dwNbrProcesses / sizeof( dwaProcessIds[0] ) );

    if ( iSuccess )
    {
        unsigned int i;

        for ( i = 0; i < dwNbrProcesses; i++ )
        {
            if ( pid == dwaProcessIds [i] )
            {
                hProcess = OpenProcess( PROCESS_QUERY_INFORMATION |
                                        PROCESS_VM_READ, FALSE,
                                        dwaProcessIds[i] );

                EnumProcessModules( hProcess, hmaModules, sizeof( hmaModules ),
                                    &dwNbrModules);

                dwNbrModules = ( dwNbrModules / sizeof (hmaModules [0] ) );

                if ( GetModuleFileNameEx( hProcess, hmaModules [0], cFilename,
                                          sizeof (cFilename) ) )
                {
                    pReturn = cFilename;
                }

                CloseHandle (hProcess);
            }
        }
    }

    return pReturn;
}


PCHAR ip_to_name( UINT ipaddr, PCHAR name, int namelen )
{
	unsigned int ip = htonl( ipaddr );

	osi_snprintf( name, namelen, "%d.%d.%d.%d",
			(ip >> 24) & 0xFF,
			(ip >> 16) & 0xFF,
			(ip >> 8) & 0xFF,
			(ip ) & 0xFF);

	return name;
}


void process_ports_windows( SCANNER *scanner )
{
	DWORD		error, dwSize;
	HANDLE		hProcessSnap;


	BOOLEAN		libsPresent;
	DWORD		i;
	CHAR		processName[255];
	CHAR		localname[255], remotename[255];
	CHAR		remoteport[255], localport[255];
	CHAR		localaddr[255], remoteaddr[255];

    SCAN_RECORD_TEXT_1 record;

	if( GetVersion() >= 0x80000000 )
    {

		return;
	}

	libsPresent = XPApisArePresent();

	if ( !libsPresent )
	{
		return;
	}

    /* GET Tcp table if not already. */

	if ( tcpTable == NULL )
	{
		error = pAllocateAndGetTcpExTableFromStack( &tcpTable, TRUE,
                                                    GetProcessHeap(), 2, 2 );
		if( error )
        {
            log_error( "error reading TCP table (%d)", error );
		    return;
		}
	}

    /* Get UDP table if not already. */

	if ( udpTable == NULL )
	{
	  error = pAllocateAndGetUdpExTableFromStack( &udpTable, TRUE,
                                                  GetProcessHeap(), 2, 2 );
	    if( error )
        {
		    log_error( "error reading UDP table (%d)", error );
		    return;
		}
	}

     /* write out TCP records.

	 format for records is:
	
     key:     TCP:<local_addr:port>
	 record:  TCP:<port>;<process_name>;<process_id>;<local_addr>;<remote_addr>

     */	

	for( i = 0; i < tcpTable->dwNumEntries; i++ )
    {

		if( tcpTable->table[i].dwState == MIB_TCP_STATE_LISTEN )
		{
            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                          "TCP:%s:%d",
                    ip_to_name( tcpTable->table[i].dwLocalAddr, localname, 255),
                         htons(tcpTable->table[i].dwLocalPort) );

            osi_snprintf( record.data, sizeof( record.data ),
              		      "TCP:%d;exe=%s;pid=%d;local=%s;remote=%s",
                           htons(tcpTable->table[i].dwLocalPort),
                           getProcessFilename( tcpTable->table[i].dwProcessId ),
                    		                   tcpTable->table[i].dwProcessId ,
                    		         ip_to_name( tcpTable->table[i].dwLocalAddr,
                                                 localname, 255),

                   		   ip_to_name( tcpTable->table[i].dwRemoteAddr,
                                       remotename, 255 ) );

            send_scan_data( scanner, (SCAN_RECORD *)&record );

	    }
	}

    /* generate records for UDP

        key:       UDP:<port>
        record:    UDP:<port>;exe=<name>;pid=<pid>

    */

    for( i = 0; i < udpTable->dwNumEntries; i++ )
    {
        initialize_scan_record( (SCAN_RECORD *)&record,
                                 SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        osi_snprintf( record.name, sizeof( record.name ),
                      "UDP:%d", htons(udpTable->table[i].dwLocalPort) );


        osi_snprintf( record.data, sizeof( record.data ),
              	      "UDP:%d;exe=%s;pid=%d",
                       htons(udpTable->table[i].dwLocalPort),
              		   getProcessFilename( udpTable->table[i].dwProcessId ),
                 		                   udpTable->table[i].dwProcessId );

        send_scan_data( scanner, (SCAN_RECORD *)&record );

    }
}



#endif /* WIN32 */

#if defined(SYSTEM_LINUX)

#define TCP_FILE  "/proc/net/tcp"
#define TCP6_FILE "/proc/net/tcp6"
#define UDP_FILE  "/proc/net/udp"

void process_proc_file( SCANNER *scanner, const char *filename )
{
    FILE *file;
    char line[1024];
    char *data;
    char *protocol = "TCP";

    SCAN_RECORD_TEXT_1 record;

    if ( ( scanner == NULL ) || ( filename == NULL ) )
    {
        return;
    }

    if( ( file = osi_fopen( filename, "r", 0 ) ) == NULL )
    {
        log_error( "unable to open proc file (%s).", filename );
        return;
    }

    /* determine protocol. */

    if ( filename == UDP_FILE )
    {
        protocol = "UDP";
    }

    /* skip first line, it is the header. */

    fgets( line, sizeof(line), file );

    /* read other lines. */

    while( ( data = fgets( line, sizeof( line ), file ) ) != NULL )
    {
        char crap[255];
        char local_addr[255];
        char local_port[16];
        char remote_addr[255];
        char state[5];

        char *temp = NULL;
        int port = 0;

        /* file format is: id local remote state */

        temp = get_token( line, crap, sizeof(crap) );

        /* get local address and port. */

        temp = get_string_up_to_token( temp, local_addr,
                                       sizeof(local_addr), ':' );

        temp = get_token( temp, local_port, sizeof(local_port) );

        /* get remote address. */

        temp = get_string_up_to_token( temp, remote_addr,
                                       sizeof(remote_addr), ':' );

        temp = get_token( temp, crap, sizeof(crap) );

        /* get socket state. */

        temp = get_token( temp, state, sizeof(state) );

        /* if TCP, and not in LISTEN state, we bail. */

        if ( filename != UDP_FILE && strcmp( state, "0A" ) != 0 )
        {
            continue; 
        }

        /* if UDP, and not open, we bail. */

        if ( filename == UDP_FILE && strcmp( state, "07" ) != 0 )
        {
            continue;
        }

        /* if local is either IPV4 or IPV6 127.0.0.1, bail. */

        if ( ( strcmp( local_addr, "0100007F" ) == 0 ) ||
           ( strcmp( local_addr, "00000000000000000000000001000000" ) == 0 ) )
        {
            continue;
        }

        /* convert hex into readable. */

        sscanf( local_port, "%X", &port );

        if ( atoi( local_addr ) == 0 )
        {
            strcpy( local_addr, "0" );
        }        

        if ( atoi( remote_addr ) == 0 )
        {
            strcpy( remote_addr, "0" );
        }

        /* populate and send scan record. */

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        osi_snprintf( record.name, sizeof( record.name ), "%s:%s:%d",
                      protocol, local_addr, port );

        osi_snprintf( record.data, sizeof( record.data ),
                      "%s:%d;local=%s;remote=%s",
                      protocol, port,local_addr,remote_addr );

        send_scan_data( scanner, (SCAN_RECORD *)&record );
        
    }

    fclose(file);
}

void process_ports_linux( SCANNER *scanner )
{
    process_proc_file( scanner, TCP_FILE );
    process_proc_file( scanner, TCP6_FILE );
    process_proc_file( scanner, UDP_FILE );
}

#endif

void mod_ports( SCANNER *scanner )
{
#ifdef WIN32
    process_ports_windows( scanner );
#endif

#if defined(SYSTEM_LINUX)
    process_ports_linux( scanner );
#endif 
}

