
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_notify.c
**  Date:    July 1, 2003
**
**  Author:  Brian Wotring
**  Purpose: notify admin of scan logs, initially this will be via email.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_log.h"
#include "md_config.h"
#include "md_compare.h"
#include "md_notify.h"
#include "md_utilities.h"

#include "common.h"
#include "logging.h"

extern OSI_MANAGEMENT_CONFIG *config;

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>

#include <openssl/ssl.h>
#include <openssl/err.h>

#ifdef WIN32

#include <winsock.h>
#include <errno.h>
#include <io.h>

#define EWOULDBLOCK WSAEWOULDBLOCK

#else   /* WIN32 */

#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/time.h>

#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>

#endif  /* WIN32 */

#include <fcntl.h>

#include "utilities.h"
#include "socketapi.h"

#define OSIRIS_SMTP_PORT        25

#define OSIRIS_SMTP_CMD_DATA    "DATA\r\n"
#define OSIRIS_SMTP_CMD_QUIT    "QUIT\r\n"

#define OSIRIS_SMTP_CMD_BUFSZ   2048
#define MAX_SMTP_LINE   128
#define LOGFILE_PERMISSIONS 0600

#define NOTIFY_SCHEDULE_FAIL_FORMAT  "The scheduler was unable to initiate the following scan:\r\n\r\nhost: %s\r\nscheduled scan time: %s\r\n\r\nThe scheduler produced the following error message:\r\n\r\n%s"

#define NOTIFY_UPDATE_URL_FORMAT "If these changes are approved, visit the \
URL below to set the latest scan database to become the trusted database. \
Alternately, you can login to the management console and set the trusted \
database to (%s).  \
If these notifications persist, you may need to modify the scan configuration \
for this host and/or set some comparison filters.\
\r\n\r\n    <https://%s:%d/?action=v&host=%s&base_db=%s&log=%s>\r\n\r\n"

#define NOTIFY_SESSION_REKEY_FORMAT "The host, \"%s\" seems to have lost its \
resident session key.\r\nThis key is lost when the agent is restarted \
or the host is rebooted.\
\r\n\r\nThis host is configured to generate an alert upon detection of \
this event.\r\nTo change this, adjust the notification settings for \
this host.\r\n\r\n"

#define NOTIFY_TEST_MESSAGE "This is a test notification message sent by the \
Osiris management console.\r\n"

/* 
 * Read a response from the SMTP server
 */

int read_smtp_response( int sock, char *expect )
{
    int i, pos;
    char buf[MAX_SMTP_LINE];

    /* not expecting a response, just succeed */

    if (expect == NULL)
    {
        return 0;
    }

    pos = 0;

    while (pos < MAX_SMTP_LINE)
    {
        if ((i = osi_read(sock, &buf[pos], MAX_SMTP_LINE - pos)) <= 0)
        {
            break;
        }

        while (i > 0)
        {
            if (buf[pos++] == '\n')
            {
                buf[pos] = '\0';

                if (strncmp(expect, &buf[0], strlen(expect)) == 0)
                {
                    /* server response matched the expected response */
                    return 0;
                }

                else
                {
                    log_error( LOG_ID_NOTIFY_ERROR, NULL,
                          "unrecognized SMTP response: (%s) expected: (%s)",
                           &buf[0], expect);

                    return 1;
                }
            }

            i--;
        }
    }

    return 1;
}

/*
 * Write a command string to the SMTP server, checking it's response
 */

int send_smtp_command(int sock, char *cmd, char *rsp)
{
    /* write a command to the SMTP server */

    if ((osi_write(sock, cmd, strlen(cmd))) == 0)
    {
        log_error( LOG_ID_NOTIFY_ERROR, NULL,
                   "error sending command (%s) to SMTP server.", cmd  );
    }

    /* read the response from the SMTP server */
    return read_smtp_response(sock, rsp);
}

int disconnect_from_mail_server( int sock )
{
    int result = 0;

    result = send_smtp_command( sock, OSIRIS_SMTP_CMD_QUIT, "250" );
    osi_close_socket( sock );

    return result;
}

int connect_to_mail_server( const char *email, OSI_ERROR *error )
{
    int sock = 0;

    int smtp_port = config->notify_smtp_port;
    char *smtp_host = config->notify_smtp_host;

    char buf[OSIRIS_SMTP_CMD_BUFSZ];
    OSI_ERROR err;

    int notify_email = 1;

    /* no error passed in, use a local one. */

    if( error == NULL )
    {
        error = &err;
    }

    if ( ( email == NULL ) || ( strlen( email ) == 0 ) )
    {
        notify_email = 0;
    }

    if ( notify_email == 0 )
    {
        osi_strlcpy( error->message,
                     "notification failure: no notify address set!!.",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );
        return -1;
    }

    if( smtp_host == NULL )
    {
        osi_strlcpy( error->message,
                     "notification failure: SMTP server not set",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );

        return -1;
    }

    if( smtp_port == 0 )
    {
        /* fallback to the default SMTP port */

        smtp_port = OSIRIS_SMTP_PORT;
    }

    /* connect to the SMTP server */

    if( ( sock = osi_connect_to_host_on_port( smtp_host, smtp_port ) ) == 0 )
    {
        osi_snprintf( error->message, sizeof( error->message ),
                      "notification failure: unable to connect to (%s:%d).",
                      smtp_host, smtp_port );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );
        return -1; 
    }

    /* read the SMTP servers initial greeting */

    if( read_smtp_response( sock, "220" ) )
    {
        osi_strlcpy( error->message,
                     "notification failure: unfriendly SMTP server.",
                     sizeof( error->message ) );

        log_error( LOG_ID_NOTIFY_ERROR, NULL, error->message );

        osi_close_socket( sock );
        return FALSE;
    }

    do
    {
        /* HELO */
        osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ, "HELO %s\r\n", smtp_host);

        if (send_smtp_command(sock, buf, "250"))
        {
            break;
        }

        if ( notify_email )
        {
            /* MAIL - sender uses same addresss as recipient */
            osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ,
                         "MAIL FROM: <%s>\r\n", email);

            if (send_smtp_command(sock, buf, "250"))
            {
                break;
            }

            /* RCPT */
            osi_snprintf(buf, OSIRIS_SMTP_CMD_BUFSZ,
                         "RCPT TO: <%s>\r\n", email);

            if (send_smtp_command(sock, buf, "250"))
            {
                break;
            }
        }

        /* DATA */

        if( send_smtp_command( sock, OSIRIS_SMTP_CMD_DATA, "354" ) )
        {
            break;
        }

    } while( 0 );

    return sock;
}

char * get_current_date_header()
{
    static char buf[40];
#ifdef WIN32
    static char tz[20];
#endif

    time_t curtime;
    struct tm *loc;

    curtime = osi_get_time();
    loc = localtime( &curtime );

    /* based on RFC 2822 it should be:
    3.3. Date and Time Specification
    ...
    date-time       =       [ day-of-week "," ] date FWS time [CFWS]
    ...
    */

    /* WIN32 strftime doesn't give TZ offset. */

#ifdef WIN32
    get_tz_bias( tz, sizeof(tz) );
    strftime( buf, sizeof(buf) ,"%a, %e %b %Y %H:%M:%S", loc );
    osi_strlcat( buf, " ", sizeof(buf) );
    osi_strlcat( buf, tz, sizeof(buf) );
#else
    strftime( buf, sizeof(buf) ,"%a, %e %b %Y %H:%M:%S %z", loc );
#endif
    buf[39] = '\0';

    return buf;
}


static int write_log_file_to_socket( int sock, char *logfile )
{
    FILE *fp = NULL;
    int len = 0;

    char *line;
    char buf[OSIRIS_SMTP_CMD_BUFSZ];

    if ( ( fp = osi_fopen( logfile, "r", LOGFILE_PERMISSIONS ) ) == NULL )
    {
        log_error( LOG_ID_NOTIFY_ERROR, NULL,
                   "notification failure: error opening log file (%s).",
                   logfile );
        return 1;
    }

    for(;;)
    {
        char *temp;
        line = fgets( buf, sizeof( buf ), fp );

        if( line == NULL)
        {
            break;
        }

        /* now strip off any newline characters. */

        if( ( temp = strrchr( buf, '\r' ) ) )
        {
            *temp = '\0';
        }

        if( ( temp = strrchr( buf, '\n' ) ) )
        {
            *temp = '\0';
        }

        /* now add our own. */

        osi_strlcat( buf, "\r\n", sizeof( buf ) );
        len = strlen( buf );

        /* write the buffer to the SMTP server */

        if ((osi_write(sock, buf, len)) == 0)
        {
            log_error( LOG_ID_NOTIFY_ERROR, NULL,
                       "unable to send data to SMTP server.");
            break;
        }
    }

    fclose(fp);
    return 0;
}


char * get_notify_type_name( int notify_type )
{
    char * type = NULL;

    switch( notify_type )
    {
        case NOTIFY_TYPE_REPORT:
            type = "report";
            break;

        case NOTIFY_TYPE_LOG:
            type = "log";
            break;

        case NOTIFY_TYPE_REKEY:
            type = "rekey";
            break;

        case NOTIFY_TYPE_TEST:
            type = "test";
            break;

        case NOTIFY_TYPE_SCHEDULE_FAIL:
            type = "schedule_fail";
            break;

        default:
            type = "unkonwn";
    }

    return type;
}


#ifndef WIN32
void md_notify_send_notification_to_app( MD_NOTIFY_CTX *ctx )
{
    int commpipe[2];
    char buf[256];

    launch_log_application( commpipe );

    if( ( ctx == NULL ) || ( commpipe[1] <= 0 ) )
    {
        log_error( LOG_ID_NOTIFY_ERROR, NULL,
                   "unable to launch notify app comm pipe." );
        return;
    }

    /* we have an open pipe to the application, send our payloads. */

    osi_snprintf( buf, sizeof(buf), "host: %s\ntype: %s\n", ctx->host,
                  get_notify_type_name( ctx->type ) );

    osi_write( commpipe[1], buf, strlen(buf) );

    if ( ctx->type == NOTIFY_TYPE_REPORT )
    {
        write_log_file_to_socket( commpipe[1], ctx->filename );
    }

    else
    {
        osi_write( commpipe[1], ctx->payload, strlen( ctx->payload ) );
    }

    close_log_application( commpipe );
}
#endif


void md_notify_send_email( MD_NOTIFY_CTX *ctx )
{
    int sock;
    char buf[2048];

    if ( ctx == NULL )
    {
        return;
    }

    /* first, we create the stupid headers. */

    osi_snprintf( ctx->headers, sizeof(ctx->headers),
                  "To: %s\r\n"
                  "From: \"Osiris Host Integrity System\" <osirismd@%s>\r\n"
                  "Date: %s\r\n",
              ctx->email, ctx->hostname, get_current_date_header() );

    switch( ctx->type )
    {

        case NOTIFY_TYPE_REPORT:
        case NOTIFY_TYPE_LOG:
        {
            osi_snprintf( buf, sizeof(buf),
                  "Subject: [osiris log][host: %s][%d changes]\r\n\r\n",
                  ctx->host, ctx->diff_count );

            osi_strlcat( ctx->headers, buf, sizeof( ctx->headers ) );
        }

            break;

        case NOTIFY_TYPE_REKEY:

            osi_snprintf( buf, sizeof(buf),
                  "Subject: [osiris rekey][host: %s] session rekey.\r\n\r\n",
                  ctx->host );

            osi_strlcat( ctx->headers, buf, sizeof( ctx->headers ) );
            break;

        case NOTIFY_TYPE_SCHEDULE_FAIL:

            osi_snprintf( buf, sizeof(buf),
                "Subject: [osiris scheduler][host: %s] failed to scan.\r\n\r\n",
                ctx->host );

            osi_strlcat( ctx->headers, buf, sizeof( ctx->headers ) );
            break;

        case NOTIFY_TYPE_TEST:

            osi_strlcat( ctx->headers,
                  "Subject: [osiris test][notification system test]\r\n\r\n",
                  sizeof( ctx->headers ) );
            break;

        default:
            osi_strlcat( ctx->headers, "\r\n", sizeof( ctx->headers ) );
            break;
    }

    /* now, we send the mail. */

    if( ( sock = connect_to_mail_server( ctx->email, &(ctx->error) ) ) < 0 )
    {
        return;
    }

    send_smtp_command( sock, ctx->headers, NULL );

    /* send our payload, and the file if we have them. */

    if ( ctx->payload[0] )
    {
        osi_write( sock, ctx->payload, strlen( ctx->payload ) );
    }

    if ( ctx->filename )
    {
        write_log_file_to_socket( sock, ctx->filename );
    }

    send_smtp_command( sock, "\r\n.\r\n", NULL );
    disconnect_from_mail_server( sock );
}

void md_create_payload( MD_NOTIFY_CTX *ctx )
{
    if ( ctx == NULL )
    {
        return;
    }

    /* setup our hostname. */

    if( config->http_host[0] && strcmp( config->http_host, "" ) != 0 )
    {
        osi_strlcpy( ctx->hostname, config->http_host, sizeof(ctx->hostname) );
    }

    else
    {
        gethostname( ctx->hostname, sizeof(ctx->hostname) );
    }


    switch( ctx->type )
    {
        case NOTIFY_TYPE_LOG:
        case NOTIFY_TYPE_REPORT:

            if( ( config->http_port > 0 ) && ( ctx->show_url ) )
            {
                char filename[MAX_PATH_LENGTH] = "";

                /* get filename, strip path. */

                osi_get_file_from_path( ctx->filename, filename,
                                        sizeof( filename ) );

                osi_snprintf( ctx->payload, sizeof( ctx->payload ),
                              NOTIFY_UPDATE_URL_FORMAT,
                              ctx->db_name, ctx->hostname, config->http_port,
                              ctx->host, ctx->db_name, filename );
            }

            break;

        case NOTIFY_TYPE_REKEY:

            osi_snprintf( ctx->payload, sizeof( ctx->payload ),
                          NOTIFY_SESSION_REKEY_FORMAT, ctx->host );

            break;

        case NOTIFY_TYPE_SCHEDULE_FAIL:
        {
            char *now = osi_time_to_string( osi_get_time() );

            osi_snprintf( ctx->payload, sizeof( ctx->payload ),
                          NOTIFY_SCHEDULE_FAIL_FORMAT, ctx->host,
                          now, ctx->message );
        }
            break;

        case NOTIFY_TYPE_TEST:

            osi_strlcpy( ctx->payload, NOTIFY_TEST_MESSAGE,
                         sizeof( ctx->payload ) );

            break;

        default:
            break;
    }
}


/* right now, the only notification methods are email, or pipe to */
/* an application, so we check for that information in the ctx.   */

void md_notify( MD_NOTIFY_CTX *ctx )
{
    if ( ctx == NULL )
    {
        return;
    }
        
    md_create_payload( ctx );
    
    /* we have an email target, use it. */
    
    if ( ( ctx->email ) && ( strlen( ctx->email ) > 0 ) )
    {
        md_notify_send_email( ctx );
    }

    /* we have an application, pipe our data to it. */

#ifndef WIN32
    if ( ( config->notify_app ) && ( strlen( config->notify_app ) > 0 ) )
    {
        md_notify_send_notification_to_app( ctx );
    }
#endif
}

void md_notify_init( MD_NOTIFY_CTX *ctx, int type )
{
    if ( ctx == NULL )
    {
        return;
    }

    memset( ctx, 0, sizeof(MD_NOTIFY_CTX) );
    ctx->type = type;
}


