
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit http://osiris.shmoo.com for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    printdb.c
**  Date:    June 11, 2002
**
**  Author:  Brian Wotring
**  Purpose: standalone osiris databse cli utility.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libfileapi.h"

char db_path[MAX_PATH_LENGTH] = "";

osi_bool print_all_records = FALSE;
osi_bool print_all_errors  = FALSE;

/******************************************************************************
**
**    Function: print_usage
**
**    Purpose:  print usage statement.
**
******************************************************************************/

void print_usage()
{
    fprintf( stderr, "osiris db command line utility (v0.1).\n" );
    fprintf( stdout, "usage: printdb [-a] [-e] <database>\n\n" );
}

/******************************************************************************
**
**    Function: parse_arguments
**
**    Purpose:  read command line arguments.
**
******************************************************************************/

void parse_arguments( int argument_count, char *argument_list[] )
{
    int count;
    
    if( argument_count == 1 )
    {
        print_usage();
        exit(-1);
    }
    
    /* if only given a db name, run with that. */
    
    if( argument_count == 2 )
    {
        osi_strlcpy( db_path, argument_list[1], sizeof( db_path ) );
        return;
    }
    
    /* otherwise, look for options. */
    
    for( count = 1; count < ( argument_count - 1 ); count++ )
    {
        if( ( strcmp( argument_list[count], "-e" ) == 0 ||
              strcmp( argument_list[count], "--print-errors" ) == 0 ) )
        {
            print_all_errors = TRUE;
            continue;
        }
        
        if( ( strcmp( argument_list[count], "-a" ) == 0 ||
              strcmp( argument_list[count], "--all-records" ) == 0 ) )
        {
            print_all_records = TRUE;
            continue;
        }
                    
        else
        {
            print_usage();
            exit( -1 );
        }
    }
    
    /* get db name. */

    osi_strlcpy( db_path, argument_list[(argument_count-1)],
                 sizeof( db_path ) );
}

void print_database_records( OSI_DB *db )
{
    int result;
    
    int record_size;
    int record_type;
    
    SCAN_RECORD *record;
        
    record_type = osi_scan_db_get_record_type( db );
    record_size = MAX_SCAN_RECORD_LENGTH;
    
    if( record_size <= 0 )
    {
        return;
    }
    
    record = osi_malloc( record_size );
    result = osi_scan_db_get_first_record( db, record, record_size );
    
    while( result == OSI_DB_OK )
    {
        print_scan_record( record );
        result = osi_scan_db_get_next_record( db, record, record_size );
    }

    /* print system records. */

    fprintf( stdout, "\n[ MODULES ]\n\n" );
    result = osi_scan_db_get_first_system( db, record, record_size );
    
    while( result == OSI_DB_OK )
    {
        print_scan_record( record );
        result = osi_scan_db_get_next_system( db, record, record_size );
    }
    
    osi_free( record );
    fprintf( stdout, "\n" );
}


int main( int argument_count, char *argument_list[] )
{    
    OSI_DB db;
    int result;

    memset( &db, 0, sizeof( db ) );
    parse_arguments( argument_count, argument_list );
    
    if( osi_file_is_regular_file( db_path ) == FALSE )
    {
        fprintf( stderr, "printdb: error, file (%s) is not a regular file.\n\n",                 db_path );

        exit(-1);
    }
    
    osi_strlcpy( db.path, db_path, sizeof( db.path ) );
   
    result = osi_scan_db_open( &db, OSI_DB_READ );
 
    if( result != OSI_DB_OK )
    {
        fprintf( stderr, "error: %s\n", osi_get_name_for_error( result ) );
        exit(-1);
    }

    print_database_header( &db );
    
    /* now walk through the database. */
    
    if( print_all_records )
    {
        print_database_records( &db );
    }
    
    if( print_all_records || print_all_errors )
    {
        print_database_errors( &db );
    }
    
    osi_scan_db_close( &db );
    return 0;
}


