
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_users.c
**  Date:    January 1, 2004
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for reading user file information.
**
******************************************************************************/

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"

#ifdef HAVE_GETPWENT
#include <pwd.h>
#endif

#define USER_FILE  "/etc/passwd"

static const char *MODULE_NAME = "mod_users";

#ifdef WIN32

#include <lm.h>

void process_windows_users( SCANNER *scanner )
{
    LPUSER_INFO_2 pBuf = NULL;
    LPUSER_INFO_2 pTmpBuf;
    DWORD dwLevel = 2;
    DWORD dwPrefMaxLen = -1;
    DWORD dwEntriesRead = 0;
    DWORD dwTotalEntries = 0;
    DWORD dwResumeHandle = 0;
    DWORD i;
    DWORD dwTotalCount = 0;
    NET_API_STATUS nStatus;

    SCAN_RECORD_TEXT_1 record;

    do 
    {
        nStatus = NetUserEnum( NULL,
                               dwLevel,
                               FILTER_NORMAL_ACCOUNT, // global users
                               (LPBYTE*)&pBuf,
                               dwPrefMaxLen,
                               &dwEntriesRead,
                               &dwTotalEntries,
                               &dwResumeHandle);

        if( ( nStatus == NERR_Success) || ( nStatus == ERROR_MORE_DATA ) )
        {
            if( ( pTmpBuf = pBuf ) != NULL )
            {
                for( i = 0; (i < dwEntriesRead); i++ )
                {
                    if( pTmpBuf == NULL )
                    {
                        log_error( "error reading user entry." );
                        break;
                    }

                    initialize_scan_record( (SCAN_RECORD *)&record,
                                            SCAN_RECORD_TYPE_TEXT_1 );

                    osi_strlcpy( record.module_name,
                                 MODULE_NAME, sizeof( record.module_name ) );

                    _snprintf( record.name, sizeof( record.name ),
                               "user:%ls", pTmpBuf->usri2_name );


                    _snprintf( record.data, sizeof( record.data ),
                               "name:%ls;priv:%d;home:%ls;flags:%d;auth:%d;",
                                pTmpBuf->usri2_name,
                                pTmpBuf->usri2_priv,
                                pTmpBuf->usri2_home_dir,
                                pTmpBuf->usri2_flags,
                                pTmpBuf->usri2_auth_flags );

                    osi_strlcat( record.data, "groups:", sizeof(record.data) );

                    /* now read the groups this user is a member of. */

                    {
                        LPGROUP_USERS_INFO_0 pBuf2 = NULL;
                        LPGROUP_USERS_INFO_0 pTmpBuf2 = NULL;
                        DWORD num_read = 0;
                        DWORD num_groups = 0;
                        unsigned int j;

                        if( NetUserGetLocalGroups( NULL, pTmpBuf->usri2_name,
                                                   0, LG_INCLUDE_INDIRECT,
                                                   (LPBYTE *)&pBuf2,-1,
                                      &num_read,&num_groups ) != NERR_Success )
                        {
                            continue;
                        }
                
                        if( ( pTmpBuf2 = pBuf2 ) != NULL )
                        {
                            for( j = 0; ( j < num_read ); j++ )
                            {
                                char buf[1024];

                                if( pTmpBuf2 == NULL )
                                {
                                    break;
                                }

                                _snprintf( buf, sizeof(buf),
                                           "%ls", pTmpBuf2->grui0_name );

                                osi_strlcat( record.data, buf,
                                             sizeof(record.data) );

                                if( (j+1) != num_read )
                                {
                                    osi_strlcat( record.data, ",",
                                                 sizeof( record.data ) );
                                }

                                pTmpBuf2++;
              
                            }

                        }

                        if( pBuf2 != NULL )
                        {
                            NetApiBufferFree( pBuf2 );
                            pBuf2 = NULL;
                        }
                    }

                    send_scan_data( scanner, (SCAN_RECORD *)&record );

                    pTmpBuf++;
                    dwTotalCount++;
                }
            }
        }
         
        else
	    {
            log_error( "error retrieving users." ); 
	    }

        if( pBuf != NULL )
        {
            NetApiBufferFree( pBuf );
            pBuf = NULL;
        }

    } while( nStatus == ERROR_MORE_DATA );

    if( pBuf != NULL )
    {
        NetApiBufferFree(pBuf);
    }
}

#else

void process_local_etc_passwd_file( SCANNER *scanner )
{
    char temp_line[MAX_TEXT_RECORD_DATA_LENGTH];

    FILE *user_file;
    SCAN_RECORD_TEXT_1 record;

    /* open the local user database file for read only. */

#ifdef USE_PRIVSEP
    user_file = rootpriv_fopen( USER_FILE );
#else
    user_file = fopen( USER_FILE, "r" );
#endif

    if( user_file == NULL )
    {
        log_error( "unable to open local user file (%s).", USER_FILE );
        return;
    }
   
    /* now read each uncommented line */ 

    for(;;)
    {
        char *line;
        char *username_end;

        line = fgets( temp_line, sizeof( temp_line ), user_file );

        if( line == NULL)
        {
            break;
        }

        line = trim_white_space( line );

        /* skip commented and empty lines. */
    
        if( ( line == NULL ) || ( line[0] == '#' ) )
        {
            continue;
        }

        /* locate the username, this is the first item in the colon list. */

        if( ( username_end = strchr( line, ':' ) ) == NULL )
        {
            continue;
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        /* user the username as a key/path for this record. */

        (*username_end) = '\0';
        osi_strlcpy( record.name, "user:", sizeof( record.name ) );
        osi_strlcat( record.name, line, sizeof( record.name ) );
        (*username_end) = ':'; 

        /* now copy in the entire passwd entry into the data portion. */
        /* and send this record on its way.                           */

        osi_strlcpy( record.data, line, sizeof( record.data ) );
        send_scan_data( scanner, (SCAN_RECORD *)&record );
    }    

    fclose( user_file );
}

void process_passwd_entries( SCANNER *scanner )
{
    struct passwd *passwd;
    SCAN_RECORD_TEXT_1 record;

    passwd = getpwent();

    while( passwd != NULL )
    {
        if( passwd->pw_name != NULL )
        {
            char buf[10];

            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            /* user the username as a key/path for this record. */

            osi_strlcpy( record.name, "user:", sizeof(record.name) );
            osi_strlcat( record.name, passwd->pw_name, sizeof(record.name) );

            /* now assemble the passwd entry string. */

            osi_strlcpy( record.data, passwd->pw_name, sizeof(record.data) );
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_strlcat( record.data, passwd->pw_passwd, sizeof(record.data) );
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_snprintf( buf, sizeof(buf), "%d", passwd->pw_uid );
            osi_strlcat( record.data, buf, sizeof(record.data));
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_snprintf( buf, sizeof(buf), "%d", passwd->pw_gid );
            osi_strlcat( record.data, buf, sizeof(record.data));
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_strlcat( record.data, passwd->pw_gecos, sizeof(record.data));
            osi_strlcat( record.data, ":", sizeof(record.data));
            osi_strlcat( record.data, passwd->pw_dir, sizeof(record.data));
            osi_strlcat( record.data, ":", sizeof(record.data));
            osi_strlcat( record.data, passwd->pw_shell, sizeof(record.data));

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }
        
        passwd = getpwent();

    } /* while passwd != NULL */
}

#endif /* WIN32 */

void mod_users( SCANNER *scanner )
{
#ifdef WIN32
    process_windows_users( scanner );
#else

#ifdef HAVE_GETPWENT
    process_passwd_entries( scanner );
#else
    process_local_etc_passwd_file( scanner );
#endif
#endif /* WIN32 */
}

