
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_http.c
**  Date:    July 17, 2003
**
**  Author:  Brian Wotring
**  Purpose: http management routines.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "version.h"

#include "md_hosts.h"
#include "md_compare.h"
#include "md_database.h"
#include "md_http.h"
#include "md_auth.h"
#include "md_log.h"

#include "common.h"
#include "logging.h"

#define HTTP_PROTOCOL         " HTTP/"   /* need the leading space. */ 

#define HTTP_ACTION_FIELD     "action="
#define HTTP_HOST_FIELD       "host="
#define HTTP_BASE_DB_FIELD    "base_db=" 
#define HTTP_LOG_FIELD        "log="

#define HTTP_AUTH_HEADER                "Authorization: Basic "
#define MAX_HTTP_AUTH_PASSWORD_LENGTH   128 

#define RESPONSE_FORMAT "HTTP/1.1 200\r\nContent-Length: %d\r\n\
Content-Type: text/html\r\n\r\n%s"

#define RESPONSE_SUCCESS_FORMAT "<HTML><BODY><FONT FACE='arial' SIZE='2'>\
<STRONG>Success</STRONG><BR><BR>Host: %s<BR>Trusted database changed \
to: %s<BR><BR><I>Osiris Host Integrity System version: %s</I><BR></FONT>\
</BODY></HTML>"

#define HTTP_AUTH_REQUEST "HTTP/1.1 401 Unauthorized\r\nConnection: close\r\nContent-Length: 71\r\nContent-Type: text/html\r\nWWW-Authenticate: Basic realm=\"osiris\"\r\nExpires: Sat, Jan 01 2000 01:01:01 GMT\r\n\r\n<HTML><HEAD><META HTTP-EQUIV=\"Pragma\" CONTENT=\"no-cache\"></HEAD></HTML>"

#define HTTP_VIEW_LOG_HEADER_FORMAT "<HTML><BODY><FONT FACE='arial' SIZE='2'>\
<STRONG>Changes for host: %s</STRONG><BR><BR>\
The following is a snippet of the latest scan log for this host.</FONT> \
<FORM METHOD='GET' ACTION='/'><PRE>\
<INPUT TYPE='HIDDEN' NAME='action' VALUE='u'></INPUT>\
<INPUT TYPE='HIDDEN' NAME='host' VALUE='%s'></INPUT>\
<INPUT TYPE='HIDDEN' NAME='base_db' VALUE='%s'></INPUT>"

#define HTTP_VIEW_LOG_FOOTER_FORMAT "</PRE><BR><HR>\
<FONT FACE='arial' SIZE='2'>Accepting this form will \
cause the trusted database for host: %s to be updated<BR> to \
database: (%s). <BR><BR>\
<I>Alternatively, you can login to the management console \
and update the database manually.</I><BR><BR> \
<INPUT TYPE=SUBMIT VALUE=Accept><BR><BR></FORM>\
</BODY></HTML>"


extern char current_user[MAX_AUTH_USERNAME_LENGTH];

/* the HTTP server only really supports on operation currently.  If */
/* it is ever expanded, this function will need to be broken down.  */

void process_http_message( SSL *ssl, const char *http_message )
{
    int index;

    char request[MAX_MESSAGE_DATA_SIZE];
    char response[MAX_MESSAGE_DATA_SIZE];

    char *protocol = NULL;
    char *action = NULL;
    char *host = NULL;
    char *base_db = NULL;
    char *logname = NULL;

    if( ( ssl == NULL ) || ( http_message == NULL ) )
    {
        return;
    }

    /* if this request does not have an authorization field. */
    /* we ask for it.                                        */

    if( strstr( http_message, HTTP_AUTH_HEADER ) == NULL )
    {
        osi_ssl_write_bytes( ssl, HTTP_AUTH_REQUEST,
                             strlen( HTTP_AUTH_REQUEST ) );
        return;
    }

    /* we have an auth header; first, we must authorize this user. */

    if( http_request_is_authorized( http_message ) == FALSE )
    {
        osi_strlcpy( response, "authorization failure.",
                     sizeof( response ) );

        goto exit_error;    
    }

    /* make a copy. */

    osi_strlcpy( request, http_message, sizeof( request ) );

    /* some sanity; find the protocol section. if not there, we bail. */

    protocol = strstr( request, HTTP_PROTOCOL );

    if( protocol == NULL )
    {
        osi_strlcpy( response, "invalid http request stream.",
                     sizeof( response ) );

        goto exit_error;
    }

    /* locate the stuff we care about. */

    action = strstr( request, HTTP_ACTION_FIELD );
    host = strstr( request, HTTP_HOST_FIELD );
    base_db = strstr( request, HTTP_BASE_DB_FIELD );
    logname = strstr( request, HTTP_LOG_FIELD );

    /* not our request, ignore it.  However, we log this on high */

    if( ( host == NULL ) || ( action == NULL ) )
    {
        osi_strlcpy( response, "invalid request.", sizeof( response ) );
        goto exit_error;
    }

    /* cap the string before the protocol section. */

    (*protocol) = '\0';

    /* replace all of the '&' chars with NULLs. */

    for( index = 0; request[index] != '\0'; index++ )
    {
        if( request[index] == '&' )
        {
            request[index] = '\0';
        }
    }

    action += strlen( HTTP_ACTION_FIELD );
    host += strlen( HTTP_HOST_FIELD );

    if( base_db != NULL )
    {
        base_db += strlen( HTTP_BASE_DB_FIELD );
    }

    if( logname != NULL )
    {
        logname += strlen( HTTP_LOG_FIELD );
    }

    /* find out what kind of message this is.  Either a  */
    /* request for the latest log, or an update.         */

    if( action[0] == 'u' )
    {
        process_http_update_host( ssl, host, base_db );
    }

    else
    {
        process_http_view_host( ssl, host, base_db, logname );
    }

    return;
    
exit_error:

    log_error( LOG_ID_HTTP_ERROR, NULL, response );
    send_http_response( ssl, response );
}

void process_http_update_host( SSL *ssl, const char *host, const char *base_db )
{
    OSI_HOST_BRIEF *host_brief;
    char response[MAX_MESSAGE_DATA_SIZE];

    if( ssl == NULL )
    {
        return;
    }
   
    if( ( host == NULL ) || ( base_db == NULL ) )
    {
        osi_strlcpy( response, "the format of this request is not valid.",
                     sizeof( response ) );

        goto exit_error;
    }

    /* update the trusted database for this host, or try to. */

    host_brief = osi_read_host_brief( host );

    if( host_brief == NULL )
    {
        osi_strlcpy( response, 
                     "unable to set trusted database, host doesn't exist.",
                     sizeof( response ) );

        goto exit_error;
    }

    if( osi_host_database_exists( host, base_db ) == FALSE )
    {
        osi_snprintf( response, sizeof( response ),
               "unable to set trusted database, database (%s) doesn't exist.",
                base_db );

        goto exit_error;
    }

    /* success !! */

    if( osi_host_brief_set_base_db( host_brief, (char *)base_db ) )
    {
        osi_snprintf( response, sizeof( response ), RESPONSE_SUCCESS_FORMAT,
                      host, base_db, OSIRIS_VERSION );
            
        log_info( LOG_ID_DB_TRUSTED_SET, host,
                  "[%s] trusted database set to: %s",current_user, base_db );

        send_http_response( ssl, response ); 

        return;
    }

    else
    {
        osi_snprintf( response, sizeof( response ),
                      "unable to set trusted database to: %s.",
                       base_db );
    }

exit_error:

    log_error( LOG_ID_HTTP_ERROR, host, response );
    send_http_response( ssl, response );
}

void process_http_view_host( SSL *ssl, const char *host, const char *base_db,
                             const char *logname )
{
    int index;
    int content_length = 0;

    string_list *log_data = NULL;

    char buffer[MAX_MESSAGE_DATA_SIZE];
    char header[MAX_MESSAGE_DATA_SIZE];
    char footer[MAX_MESSAGE_DATA_SIZE];

    if( ssl == NULL )
    {
        return;
    }

    /* first, we read in the log data. */

    log_data = osi_host_get_log_data( host, logname );
   
    if( log_data == NULL )
    {
        osi_strlcpy( buffer,
      "the requested log file does not exist or cannot be viewed at this time.",
                     sizeof( buffer ) );

        send_http_response( ssl, buffer );
        return;
    } 

    osi_snprintf( header, sizeof( header ),
                  HTTP_VIEW_LOG_HEADER_FORMAT, host, host, base_db );

    osi_snprintf( footer, sizeof( footer ),
                  HTTP_VIEW_LOG_FOOTER_FORMAT, host, base_db );

    /* content length is size of header, footer, and the total size of */
    /* the string list containing the log data.                        */

    content_length = ( strlen( header ) +
                       strlen( footer ) +
                       string_list_get_total_length( log_data ) +
                       log_data->size );  /* for newlines. */

    osi_snprintf( buffer, sizeof( buffer ),
                  RESPONSE_FORMAT, content_length, header );

    /* write HTTP header including the our header. */

    osi_ssl_write_bytes( ssl, buffer, strlen( buffer ) );

    /* write out the log data. */

    for( index = 0; (unsigned int)index < log_data->size; index++ )
    {
        char *line = (log_data->list)[index];
        osi_ssl_write_bytes( ssl, line, strlen( line ) );
        osi_ssl_write_bytes( ssl, "\n", 1 );
    } 

    /* write out the footer. */

    osi_ssl_write_bytes( ssl, footer, strlen( footer ) );
    string_list_destroy( log_data );
}


void send_http_response( SSL *ssl, const char *response )
{
    int content_length = 0;
    char data[MAX_MESSAGE_DATA_SIZE];

    if( ( ssl == NULL ) || ( response == NULL ) )
    {
        return;
    }

    content_length = strlen( response );

    osi_snprintf( data, sizeof( data ),
                  RESPONSE_FORMAT, content_length, response );

    osi_ssl_write_bytes( ssl, data, strlen( data ) );
}

osi_bool http_request_is_authorized( const char *http_message )
{
    char *auth = NULL;
    char *temp;

    char *user;
    char *pass;

    char password[MAX_HTTP_AUTH_PASSWORD_LENGTH];
    char decoded[MAX_HTTP_AUTH_PASSWORD_LENGTH];

    OSI_AUTH_CONTEXT auth_context;

    memset( &auth_context, 0, sizeof( auth_context ) );
    auth = strstr( http_message, HTTP_AUTH_HEADER ); 

    if( auth == NULL )
    {
        return FALSE;
    }

    auth += strlen( HTTP_AUTH_HEADER );

    if( auth == NULL )
    {
        return FALSE;
    }

    osi_strlcpy( password, auth, sizeof( password ) );

    /* now replace the first '\r' and '\n' characters with NULLs. */

    if( ( temp = strchr( password, '\r' ) ) != NULL )
    {
        (*temp) = '\0';
    }

    if( ( temp = strchr( password, '\n' ) ) != NULL )
    {
        (*temp) = '\0';
    }

    /* now we base64 decode this thing. It must end in a newline  */

    osi_strlcat( password, "\n", sizeof( password ) );
    decode_base64_string( password, decoded, sizeof( decoded ) );

    if( strlen( decoded ) == 0 )
    {
        return FALSE;
    }

    user = strtok( decoded, ":" );
    pass = strtok( NULL, ":" );

    if( ( user == NULL ) )
    {
        return FALSE;
    }

    /* now authenticate against our user database. */

    osi_strlcpy( auth_context.auth_user, user,
                 sizeof( auth_context.auth_user ) );    

    if( pass != NULL )
    {
        osi_strlcpy( auth_context.auth_pass, pass,
                     sizeof( auth_context.auth_pass ) );
    }

    return ( md_auth_authenticate( &auth_context ) );
}

void decode_base64_string( char *source, char *dest, int dest_size )
{
    BIO *bio_b64, *bio_s, *mem;

    bio_b64 = BIO_new((BIO_METHOD*) BIO_f_base64());
    bio_s = BIO_new((BIO_METHOD*)BIO_f_buffer());

    mem = BIO_new(BIO_s_mem());
    BIO_puts(mem, source);

    bio_b64 = BIO_push(bio_b64, mem);
    bio_s = BIO_push(bio_s, bio_b64);

    memset(dest,0,dest_size);
    BIO_read(bio_s,dest,dest_size);

    BIO_free_all( bio_s );
}

