
/******************************************************************************
**
**  Copyright (C) 2006 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_compare.c
**  Date:    Jan 17, 2003
**
**  Author:  Brian Wotring
**  Purpose: handle database comparisons.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_log.h"
#include "md_compare.h"
#include "md_database.h"
#include "md_notify.h"
#include "md_filter.h"
#include "md_utilities.h"

#include "common.h"
#include "logging.h"

extern char root_path[MAX_PATH_LENGTH];
extern OSI_MANAGEMENT_CONFIG *config;
extern int quiet_mode;

OSI_DB_CTX * osi_db_ctx_new()
{
    OSI_DB_CTX *context = osi_malloc( sizeof( OSI_DB_CTX ) );

    if( context != NULL )
    {
        memset( context, 0, sizeof( OSI_DB_CTX ) );

        /* setup the defaults for a compare context. */

        context->host[0]      = '\0';
        context->filepath[0]  = '\0';

        context->file_handle  = NULL;

        context->compare_mask = CMP_MASK_DEFAULT;

        if ( quiet_mode )
        {
            context->use_system_log = FALSE;
        }

        else
        {
            context->use_system_log     = TRUE;
        }

        context->include_missing    = TRUE;
        context->include_new        = TRUE;
        context->include_system     = TRUE;

        context->diff_count          = 0;

        context->checksum_diff_count = 0;
        context->perm_diff_count     = 0;
        context->suid_diff_count     = 0;
        context->root_diff_count     = 0;
        context->new_count           = 0;
        context->missing_count       = 0;
    }

    return context;
}

void osi_db_ctx_destroy( OSI_DB_CTX *context )
{
    if( context != NULL )
    {
        if( context->file_handle != NULL )
        {
            fclose( context->file_handle );
            context->file_handle = NULL;
        }

        /* don't bother deleting the temporary log file. */

        osi_free( context );
    }
}

void osi_db_ctx_preflight( OSI_DB_CTX *context )
{
    if( context == NULL )
    {
        return;
    }

    /* if we have a filename to log to, we open it for write.  */

    if( context->filepath[0] != '\0' )
    {
        context->file_handle = osi_fopen( context->filepath, "w+", 
                                          OSI_LOG_DEFAULT_PERMS );
        if( context->file_handle )
        {
            print_log_file_heading( context );
        }
    }
}

void osi_db_ctx_postflight( OSI_DB_CTX *context )
{
    if( context == NULL )
    {
        return;
    }

    /* if we have an open file handle, we close it. */

    if( context->file_handle != NULL )
    {
        print_log_file_footer( context );

        fclose( context->file_handle );
        context->file_handle = NULL;
    }
}

void osi_db_ctx_set_use_system_log( OSI_DB_CTX *context, osi_bool use )
{
    if( context != NULL )
    {
        context->use_system_log = use;
    }
}

void osi_db_ctx_set_host( OSI_DB_CTX *context, const char *host )
{
    if( context != NULL )
    {
        osi_strlcpy( context->host, host, sizeof( context->host ) );
    }
}

void osi_db_ctx_set_log_file( OSI_DB_CTX *context, const char *filepath )
{
    if( ( context != NULL ) && ( filepath != NULL ) )
    {
        osi_strlcpy( context->filepath, filepath, sizeof( context->filepath ) );
    }
}

osi_bool md_compare_record_db( OSI_DB_CTX *context, OSI_DB *db1, OSI_DB *db2 )
{
    int rc;
    int result;
    int record_size;

    SCAN_RECORD *r1;
    SCAN_RECORD *r2;

    if( ( context == NULL ) || ( db1 == NULL ) || ( db2 == NULL ) )
    {
        return FALSE;
    }

    record_size = MAX_SCAN_RECORD_LENGTH;

    /* create temporary objects for holding one from each db. */ 

    r1 = osi_malloc( record_size );
    r2 = osi_malloc( record_size );

    result = osi_scan_db_get_first_record( db1, r1, record_size );

    /* look for changes and missing records.  If we had ordered */
    /* lists we could do this in one pass.  instead, we have to */
    /* do a second pass if we are checking for new records.     */

    while( result == OSI_DB_OK )
    {
        char *name = get_scan_record_path( r1 );
        result = osi_scan_db_get_record_with_name( db2, r2, record_size, name );

        if( result == OSI_DB_OK )
        {
            /* compare record attributes, then see if changes are filtered. */

            md_compare_db_records( context, r1, r2 );

            if( context->changed_mask )
            {
                record_file_differences( context, r1, r2 );
            }
        }

        if ( ( result == OSI_ERROR_DB_ITEM_NOT_FOUND ) &&
             ( context->include_missing ) )
        {
            rc = log_cmp_message( LOG_ID_CMP_FILE_MISSING, context,
                                 "[%s][missing][%s]",
                                 context->host, name );

            if ( rc > 0 )
            {
                context->diff_count++;
                context->missing_count++;
            }
        }

        result = osi_scan_db_get_next_record( db1, r1, record_size );
    }

    /* now look for new records. */

    if( context->include_new )
    {
        result = osi_scan_db_get_first_record( db2, r2, record_size );

        while( result == OSI_DB_OK )
        {
            char *name = get_scan_record_path( r2 );
            result = osi_scan_db_get_record_with_name( db1, r1,
                                                       record_size, name );

            if( result == OSI_ERROR_DB_ITEM_NOT_FOUND )
            {
                rc = log_cmp_message( LOG_ID_CMP_FILE_NEW, context,
                                     "[%s][new][%s]", context->host, name );

                if ( rc > 0 )
                {
                    context->diff_count++;
                    context->new_count++;
                }
            }

            result = osi_scan_db_get_next_record( db2, r2, record_size );
        }
    }

    osi_free( r1 );
    osi_free( r2 );

    return TRUE;
}

osi_bool md_compare_system_db( OSI_DB_CTX *context, OSI_DB *db1, OSI_DB *db2 )
{
    int rc;
    int result;
    int record_size;

    SCAN_RECORD *r1;
    SCAN_RECORD *r2;

    record_size = MAX_SCAN_RECORD_LENGTH;

    if( ( context == NULL ) || ( db1 == NULL ) || ( db2 == NULL ) )
    {
        return FALSE;
    }

    /* create temporary objects for holding one from each db. */

    r1 = osi_malloc( record_size );
    r2 = osi_malloc( record_size );

    /* look for altered and missing entries. */

    result = osi_scan_db_get_first_system( db1, r1, record_size );

    while( result == OSI_DB_OK )
    {
        char *name = get_scan_record_path( r1 );
        result = osi_scan_db_get_system_with_name( db2, r2,
                                                       record_size, name );
        if( result == OSI_DB_OK )
        {
            /* compare system data entries. */

            char *r1_data = get_scan_record_system_data( r1 );
            char *r2_data = get_scan_record_system_data( r2 );

            if( ( r1_data != NULL ) && ( r2_data != NULL ) )
            {
                if( strcmp( r1_data, r2_data ) != 0 )
                {
                    SCAN_RECORD_TEXT_1 *rec = (SCAN_RECORD_TEXT_1 *)r1;

                    rc = log_cmp_message( LOG_ID_CMP_GENERIC_DIFF, context,
                                     "[%s][cmp][%s][%s][%s][%s]",
                                     context->host, rec->module_name,
                                     rec->name, r1_data, r2_data );

                    if ( rc > 0 )
                    {
                        context->diff_count++;
                    }
                }
            }
        }

        else if( result == OSI_ERROR_DB_ITEM_NOT_FOUND )
        {
            SCAN_RECORD_TEXT_1 *rec = (SCAN_RECORD_TEXT_1 *)r1;

            rc = log_cmp_message( LOG_ID_CMP_GENERIC_MISSING, context,
                             "[%s][missing][%s][%s][%s]",
                             context->host, rec->module_name,
                             rec->name, rec->data );

            if ( rc > 0 )
            {
                context->diff_count++;
                context->missing_count++;
            }
        }

        result = osi_scan_db_get_next_system( db1, r1, record_size );
    }

    /* now look for new entries. */

    result = osi_scan_db_get_first_system( db2, r2, record_size );

    while( result == OSI_DB_OK )
    {
        char *name = get_scan_record_path( r2 );

        result = osi_scan_db_get_system_with_name( db1, r1,
                                                   record_size, name );

        if( result == OSI_ERROR_DB_ITEM_NOT_FOUND )
        {
            SCAN_RECORD_TEXT_1 *rec = (SCAN_RECORD_TEXT_1 *)r2;

            rc = log_cmp_message( LOG_ID_CMP_GENERIC_NEW, context,
                             "[%s][new][%s][%s][%s]",
                             context->host, rec->module_name,
                             rec->name, rec->data );

            if ( rc > 0 )
            {
                context->diff_count++;
                context->new_count++;
            }
        }

        result = osi_scan_db_get_next_system( db2, r2, record_size );
    }

    osi_free( r1 );
    osi_free( r2 );

    return TRUE;
}


/* generic comparison routine.  the first is the base db, the second */
/* is the untrusted db (in most cases).                              */

osi_bool md_compare_db( OSI_DB_CTX *context, OSI_DB *db1, OSI_DB *db2 )
{
    int r1_type;
    int r2_type;

    OSI_DB_HEADER db1_header;
    OSI_DB_HEADER db2_header;

    if( ( db1 == NULL ) || ( !db1->open ) ||
        ( db2 == NULL ) || ( !db2->open ) )
    {
        return FALSE;
    }

    r1_type = osi_scan_db_get_record_type( db1 );
    r2_type = osi_scan_db_get_record_type( db2 );

    /* some sanity */

    if( r1_type != r2_type )
    {
        return FALSE;
    }

    osi_host_read_database_header_from_path( &db1_header, db1->path );
    osi_host_read_database_header_from_path( &db2_header, db2->path );

    /* copy in db names, then setup context. */

    osi_strlcpy( context->db1_name, db1_header.name,
                 sizeof( context->db1_name ) );

    osi_strlcpy( context->db2_name, db2_header.name,
                 sizeof( context->db2_name ) );

    osi_snprintf( context->config_name, sizeof( context->config_name ),
                  "%s (%s)", db2_header.config_name, db2_header.config_id );

    osi_db_ctx_preflight( context );

    log_info( LOG_ID_CMP_BEGIN, context->host, "compare begin (%s,%s)",
              db1_header.name, db2_header.name );

    /* compare databases with file records. */

    md_compare_record_db( context, db1, db2 );


    /* compare databases with system records (modules) */

    if( context->include_system )
    {
        md_compare_system_db( context, db1, db2 );
    }

    log_info( LOG_ID_CMP_END, context->host, "compare end (%s,%s)",
              db1_header.name, db2_header.name );

    osi_db_ctx_postflight( context );
    return TRUE;
}


osi_bool md_compare_db_records( OSI_DB_CTX *context,
                                 SCAN_RECORD *record1,
                                 SCAN_RECORD *record2 )
{
    osi_uint16 type;
    OSI_ATTR_MASK mask;

    /* sanity check */

    if( record1 == NULL || record2 == NULL || context == NULL )
    {
        return FALSE;
    }

    context->changed_mask = 0;

    /* we kind of have to have the same record types or this will */
    /* be a mess of a comparison.                                 */

    if( record1->type != record2->type )
    {
        return FALSE;
    }

    type = record1->type;
    mask = context->compare_mask;

    if( (int)type == SCAN_RECORD_TYPE_UNIX_1 )
    {
        SCAN_RECORD_UNIX_1 *r1 = (SCAN_RECORD_UNIX_1 *)record1;
        SCAN_RECORD_UNIX_1 *r2 = (SCAN_RECORD_UNIX_1 *)record2;

        if( ( mask & ATTR_CHECKSUM ) && 
            ( r1->checksum_algorithm == r2->checksum_algorithm ) &&
              strcmp( (const char *)r1->checksum,
                      (const char *)r2->checksum ) != 0 )
        {
            context->changed_mask |= ATTR_CHECKSUM;
        }

        if( ( mask & ATTR_DEVICE ) && ( r1->device != r2->device ) )
        { 
            context->changed_mask |= ATTR_DEVICE;
        }

        if( ( mask & ATTR_INODE ) && ( r1->inode != r2->inode ) )
        {
            context->changed_mask |= ATTR_INODE;
        }

        if( ( mask & ATTR_PERMISSIONS ) && 
            ( r1->permissions != r2->permissions ) )
        {
            context->changed_mask |= ATTR_PERMISSIONS;
        }

        if( ( mask & ATTR_LINKS ) & ( r1->links != r2->links ) )
        {
            context->changed_mask |= ATTR_LINKS;
        }

        if( ( mask & ATTR_UID ) && ( r1->uid != r2->uid ) )
        {
            context->changed_mask |= ATTR_UID;
        }

        if( ( mask & ATTR_GID ) && (  r1->gid != r2->gid ) )
        {
            context->changed_mask |= ATTR_GID;
        }

        if( ( mask & ATTR_MTIME ) && ( r1->mtime != r2->mtime ) )
        {
            context->changed_mask |= ATTR_MTIME;
        }

        if( ( mask & ATTR_ATIME ) && ( r1->atime != r2->atime ) )
        {
            context->changed_mask |= ATTR_ATIME;
        }

        if( ( mask & ATTR_CTIME ) && ( r1->ctime != r2->ctime ) )
        {
            context->changed_mask |= ATTR_CTIME;
        }

        if( ( mask & ATTR_DEVICE_TYPE ) && 
            ( r1->device_type != r2->device_type ) )
        {
            context->changed_mask |= ATTR_DEVICE_TYPE;
        }

        if( ( mask & ATTR_BYTES ) && ( r1->bytes != r2->bytes ) )
        {
            context->changed_mask |= ATTR_BYTES;
        }

        if( ( mask & ATTR_BLOCKS ) && (  r1->blocks != r2->blocks ) )
        {
            context->changed_mask |= ATTR_BLOCKS;
        }

        if( ( mask & ATTR_BLOCK_SIZE ) && 
            (  r1->block_size != r2->block_size ) )
        {
            context->changed_mask |= ATTR_BLOCK_SIZE;
        }
    }

    else if( (int)type == SCAN_RECORD_TYPE_UNIX_2 )
    {

    }

    else if( (int)type == SCAN_RECORD_TYPE_WIN32_1 )
    {

    }

    else if( (int)type == SCAN_RECORD_TYPE_WINNT_1 )
    {
        SCAN_RECORD_WINNT_1 *r1 = (SCAN_RECORD_WINNT_1 *)record1;
        SCAN_RECORD_WINNT_1 *r2 = (SCAN_RECORD_WINNT_1 *)record2;

        if( ( mask & ATTR_CHECKSUM ) &&
            ( r1->checksum_algorithm == r2->checksum_algorithm ) &&
              strcmp( (const char *)r1->checksum,
                      (const char *)r2->checksum ) != 0 )
        {
            context->changed_mask |= ATTR_CHECKSUM;
        }

        if( ( mask & ATTR_DEVICE ) && ( r1->device != r2->device ) )
        {
            context->changed_mask |= ATTR_DEVICE;
        }

        if( ( mask & ATTR_BYTES ) && ( r1->bytes != r2->bytes ) )
        {
            context->changed_mask |= ATTR_BYTES;
        }

        if( ( mask & ATTR_MTIME ) && ( r1->mtime != r2->mtime ) )
        {
            context->changed_mask |= ATTR_MTIME;
        }

        if( ( mask & ATTR_ATIME ) && ( r1->atime != r2->atime ) )
        {
            context->changed_mask |= ATTR_ATIME;
        }

        if( ( mask & ATTR_CTIME ) && ( r1->ctime != r2->ctime ) )
        {
            context->changed_mask |= ATTR_CTIME;
        }

        if( ( mask & ATTR_OWNER_SID ) &&
            ( strcmp( r1->owner_sid, r2->owner_sid ) != 0 ) )
        {
            context->changed_mask |= ATTR_OWNER_SID;
        }

        if( ( mask & ATTR_GROUP_SID ) &&
            ( strcmp( r1->group_sid, r2->group_sid ) != 0 ) )
        {
            context->changed_mask |= ATTR_GROUP_SID;
        }

        if( ( mask & ATTR_WIN32_FILE ) &&
            ( r1->attributes != r2->attributes ) )
        {
            context->changed_mask |= ATTR_WIN32_FILE;
        } 

        /* END WIN32 compare. */
    } 

    return TRUE;
}


void print_log_file_heading( OSI_DB_CTX *context )
{
    char buf[512];
    char *temp;

    time_t current_time;

    if( ( context == NULL ) && ( context->file_handle == NULL ) )
    {
        return;
    }

    current_time = osi_get_time();

    /* write out current time. */

    if( current_time > 0 )
    {
        char time_string[32];

        osi_strlcpy( time_string, ctime( &current_time ),
                     sizeof( time_string ) );

        time_string[24] = '\0';
        osi_snprintf( buf, sizeof( buf ), "     compare time: %s\n",
                      time_string );
        fwrite( buf, 1, strlen( buf ), context->file_handle );
    }

    /* write out hostname. */

    osi_snprintf( buf, sizeof( buf ),
                  "             host: %s\n", context->host );
    fwrite( buf, 1, strlen( buf ), context->file_handle );

    /* write out name of scan config used. */

    osi_snprintf( buf, sizeof( buf ),
                  "      scan config: %s\n", context->config_name );
    fwrite( buf, 1, strlen( buf ), context->file_handle );

    if( (temp = strrchr( context->filepath, PATH_SEPARATOR ) ) != NULL )
    {
        temp++;

        if( strcmp( temp, TEMP_SCAN_LOG_FILE_NAME ) == 0 )
        {
            osi_snprintf( buf, sizeof( buf ), "         log file: %s\n",
                          "no log file generated, see system log." );
        }

        else
        {
            osi_snprintf( buf, sizeof( buf ), "         log file: %s\n", temp );
        }

        fwrite( buf, 1, strlen( buf ), context->file_handle );
    }

    /* write out the db names. */

    osi_snprintf( buf, sizeof( buf ),
                 "    base database: %s\n compare database: %s\n\n",
                  context->db1_name, context->db2_name );

    fwrite( buf, 1, strlen( buf ), context->file_handle );
}

void print_log_file_footer( OSI_DB_CTX *context )
{
    char buffer[2048] = "";

#define COMPARE_STATS_FORMAT "\n\nChange Statistics:\n----------\
------------------------\n\n        checksums: %lu\n\
       SUID files: %lu\n root-owned files: %lu\n file permissions: %lu\n\
              new: %lu\n          missing: %lu\n\ntotal differences: %lu\n\n"

    if( ( context == NULL ) && ( context->file_handle == NULL ) )
    {
        return;
    }

    osi_snprintf( buffer, sizeof( buffer ), COMPARE_STATS_FORMAT,
                  context->checksum_diff_count, context->suid_diff_count,
                  context->root_diff_count, context->perm_diff_count,
                  context->new_count, context->missing_count,
                  context->diff_count );

    fwrite( buffer, 1, strlen( buffer ), context->file_handle );
}

void compare_db_with_default_for_host( OSI_DB *db, OSI_HOST *host )
{
    int result;

    OSI_DB base_db;
    OSI_DB_CTX *context;

    if( ( db == NULL ) || ( host == NULL ) )
    {
        return;
    }

    memset( &base_db, 0, sizeof( base_db ) );
    result = osi_host_open_default_db( &base_db, host );

    if( result != OSI_DB_OK )
    {
        log_error( LOG_ID_DB_OPEN_ERROR, host->name, "opening base database." );
        return;
    }

    /* make sure we aren't tring to compare a db against itself. */

    if( strcmp( base_db.path, db->path ) == 0 )
    {
        return;
    }

    context = osi_db_ctx_new();

    if( context )
    {
        char path[MAX_PATH_LENGTH];

        osi_db_ctx_set_host( context, host->name );
    
        /* if file logging is enabled, we create a new log file name. */
 
        if( host->file_log_enabled )
        {
            
            generate_log_file_name( host->name, path, sizeof( path ) );
        }

        else
        {
            osi_set_path_to_log( host->name, TEMP_SCAN_LOG_FILE_NAME,
                                 path, sizeof( path ) );
        }

        osi_db_ctx_set_log_file( context, path );
        md_compare_db( context, &base_db, db );

        /* if this host has notifications enabled, we kick off the    */
        /* notifications routine.  If the diff count is zero, we only */
        /* notify if the notify_flags ALWYAS flag is set.             */

        if( host->notify_enabled )
        {
            if( ( host->notify_flags & OSI_NOTIFY_SCAN_ALWAYS ) ||
                ( context->diff_count > 0 ) )
            {
                char *email = config->notify_email;
                MD_NOTIFY_CTX ctx;

                if( strlen( host->notify_email ) > 0 )
                {
                    email = host->notify_email;
                }

                md_notify_init( &ctx, NOTIFY_TYPE_REPORT );            
                ctx.host = host->name;
                ctx.email = email;
                ctx.filename = context->filepath;
                ctx.diff_count = context->diff_count;
                ctx.db_name = context->db2_name;
                ctx.show_url = !( host->db_flags & OSI_DB_AUTOACCEPT );

                md_notify( &ctx );
            }
        }

        osi_db_ctx_destroy( context );
    }

    osi_scan_db_close( &base_db );

	if( ( context->diff_count > 0 ) && ( host->db_flags & OSI_DB_AUTOACCEPT ) )
	{
        int rc;

        char name [15];
        OSI_HOST_BRIEF host_brief;
	
        rc = osi_get_file_from_path( db->path, name, sizeof(name) );
	
		if( rc != OSI_FILE_OK )
		{
            log_error( LOG_ID_DB_AUTOACCEPT_ERROR, host->name,
					   "unable to autoaccept changes: unable to get db name." );
			return;
		}

        osi_host_brief_from_host( host, &host_brief );

        if( osi_host_brief_set_base_db( &host_brief, name ) )
        {
            log_info( LOG_ID_DB_AUTOACCEPT, host->name,
                       "autoaccept changes: new baseline db is (%s)", name );
        }

        else
        {
            log_error( LOG_ID_DB_AUTOACCEPT_ERROR, host->name,
                       "unable to autoaccept changes: unable to get db name." );
        }
	}
}

void record_file_differences( OSI_DB_CTX *context, SCAN_RECORD *record1, 
                              SCAN_RECORD *record2 )

{
    osi_uint16 type;
    unsigned long pre_diff_count;
    unsigned int print_mask; 

    if( record1 == NULL || record2 == NULL || context == NULL )
    {
        return;
    }

    type = record1->type;
    pre_diff_count = context->diff_count;
    print_mask     = context->changed_mask;

    if( (int)type == SCAN_RECORD_TYPE_UNIX_1 )
    {
        int rc = 0;

        SCAN_RECORD_UNIX_1 *r1 = (SCAN_RECORD_UNIX_1 *)record1;
        SCAN_RECORD_UNIX_1 *r2 = (SCAN_RECORD_UNIX_1 *)record2;

        if( print_mask & ATTR_CHECKSUM )
        {
            rc = log_cmp_message( LOG_ID_CMP_CHECKSUM, context,
                           "[%s][cmp][%s][checksum][%s][%s]",
                           context->host, r1->path, r1->checksum,r2->checksum );

            if ( rc > 0 )
            {
                context->diff_count++;
                context->checksum_diff_count++;
            }
        }

        if( print_mask & ATTR_DEVICE )
        { 
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->device );
            osi_snprintf( right, sizeof(right), "%llu", r2->device );

            rc = log_cmp_message( LOG_ID_CMP_DEVICE, context,
                             "[%s][cmp][%s][device][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_DEVICE, context,
                             "[%s][cmp][%s][device][%llu][%llu]",
                             context->host, r1->path, r1->device, r2->device );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_INODE )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->inode );
            osi_snprintf( right, sizeof(right), "%llu", r2->inode );

            rc = log_cmp_message( LOG_ID_CMP_INODE, context,
                             "[%s][cmp][%s][inode][%s][%s]",
                             context->host, r1->path, left, right );

            #else

            
            rc = log_cmp_message( LOG_ID_CMP_INODE, context,
                             "[%s][cmp][%s][inode][%llu][%llu]",
                              context->host,r1->path, r1->inode, r2->inode );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_PERMISSIONS )
        {
            rc = log_cmp_message( LOG_ID_CMP_PERM, context,
                             "[%s][cmp][%s][perm][%s][%s]", context->host,
                             r1->path, r1->permissions_string, 
                             r2->permissions_string );

            if ( rc > 0 )
            {
                context->diff_count++;
                context->perm_diff_count++;
            }
        }

        if( print_mask & ATTR_LINKS )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->links );
            osi_snprintf( right, sizeof(right), "%llu", r2->links );

            rc = log_cmp_message( LOG_ID_CMP_SYMLINKS, context,
                             "[%s][cmp][%s][links][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_SYMLINKS, context,
                             "[%s][cmp][%s][links][%llu][%llu]",
                             context->host,
                             r1->path, r1->links, r2->links );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            } 
        }

        if( print_mask & ATTR_UID )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->uid );
            osi_snprintf( right, sizeof(right), "%llu", r2->uid );

            rc = log_cmp_message( LOG_ID_CMP_UID, context,
                             "[%s][cmp][%s][uid][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_UID, context,
                             "[%s][cmp][%s][uid][%llu][%llu]",
                             context->host, r1->path, r1->uid, r2->uid );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_GID )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->gid );
            osi_snprintf( right, sizeof(right), "%llu", r2->gid );

            rc = log_cmp_message( LOG_ID_CMP_GID, context,
                             "[%s][cmp][%s][gid][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_GID, context,
                             "[%s][cmp][%s][gid][%llu][%llu]",
                             context->host, r1->path, r1->gid, r2->gid );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_MTIME )
        {
            char left_mtime[25];
            char right_mtime[25];

            osi_strlcpy( left_mtime, osi_time_to_string( (long)r1->mtime ),
                         sizeof( left_mtime ) );

            osi_strlcpy( right_mtime, osi_time_to_string( (long)r2->mtime ),
                         sizeof( right_mtime ) );

            rc = log_cmp_message( LOG_ID_CMP_MTIME, context,
                             "[%s][cmp][%s][mtime][%s][%s]",
                             context->host, r1->path, left_mtime, right_mtime );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_ATIME )
        {
            char left_atime[25];
            char right_atime[25];

            osi_strlcpy( left_atime, osi_time_to_string( (long)r1->atime ),
                         sizeof( left_atime ) );

            osi_strlcpy( right_atime, osi_time_to_string( (long)r2->atime ),
                         sizeof( right_atime ) );

            rc = log_cmp_message( LOG_ID_CMP_ATIME, context,
                             "[%s][cmp][%s][atime][%s][%s]",
                             context->host, r1->path, left_atime, right_atime );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_CTIME )
        {
            char left_ctime[25];
            char right_ctime[25];

            osi_strlcpy( left_ctime, osi_time_to_string( (long)r1->ctime ),
                         sizeof( left_ctime ) );

            osi_strlcpy( right_ctime, osi_time_to_string( (long)r2->ctime ),
                         sizeof( right_ctime ) );

            rc = log_cmp_message( LOG_ID_CMP_CTIME, context,
                             "[%s][cmp][%s][ctime][%s][%s]",
                             context->host, r1->path, left_ctime, right_ctime );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_DEVICE_TYPE ) 
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->device_type );
            osi_snprintf( right, sizeof(right), "%llu", r2->device_type );

            rc = log_cmp_message( LOG_ID_CMP_DEVICETYPE, context,
                             "[%s][cmp][%s][devicetype][%s][%s]",
                             context->host, r1->path, left, right );
            #else

           rc = log_cmp_message( LOG_ID_CMP_DEVICETYPE, context, 
                           "[%s][cmp][%s][devicetype][%llu][%llu]",
                            context->host, r1->path,
                            r1->device_type, r2->device_type );
            #endif
            
            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_BYTES )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->bytes );
            osi_snprintf( right, sizeof(right), "%llu", r2->bytes );

            rc = log_cmp_message( LOG_ID_CMP_BYTES, context,
                             "[%s][cmp][%s][bytes][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_BYTES, context,
                             "[%s][cmp][%s][bytes][%llu][%llu]",
                             context->host, r1->path, r1->bytes, r2->bytes );
            #endif
    
            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_BLOCKS )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->blocks );
            osi_snprintf( right, sizeof(right), "%llu", r2->blocks );

            rc = log_cmp_message( LOG_ID_CMP_BLOCKS, context,
                             "[%s][cmp][%s][blocks][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_BLOCKS, context,
                             "[%s][cmp][%s][blocks][%llu][%llu]",
                             context->host, r1->path, r1->blocks, r2->blocks );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_BLOCK_SIZE ) 
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->block_size );
            osi_snprintf( right, sizeof(right), "%llu", r2->block_size );

            rc = log_cmp_message( LOG_ID_CMP_BLOCKSIZE, context,
                             "[%s][cmp][%s][blocksize][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_BLOCKSIZE, context,
                             "[%s][cmp][%s][blocksize][%llu][%llu]",
                             context->host, r1->path,
                             r1->block_size, r2->block_size );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        /* if we had a change, look to update some of our stats. */

        if( pre_diff_count != context->diff_count )
        {
            /* look for root owned changes. */ 

            if( ( strcmp( r1->user, "root" ) == 0 ) ||
                ( strcmp( r2->user, "root" ) == 0 ) )
            {
                context->root_diff_count++;
            }

            /* look for suid changes. */

            if( ( r1->permissions_string[3] == 's' ) ||
                ( r1->permissions_string[3] == 'S' ) ||
                ( r2->permissions_string[3] == 's' ) ||
                ( r2->permissions_string[3] == 'S' ) )
            {
                context->suid_diff_count++;
            }
        }
    }

    else if( (int)type == SCAN_RECORD_TYPE_UNIX_2 )
    {

    }

    else if( (int)type == SCAN_RECORD_TYPE_WIN32_1 )
    {

    }

    else if( (int)type == SCAN_RECORD_TYPE_WINNT_1 )
    {
        int rc = 0;

        SCAN_RECORD_WINNT_1 *r1 = (SCAN_RECORD_WINNT_1 *)record1;
        SCAN_RECORD_WINNT_1 *r2 = (SCAN_RECORD_WINNT_1 *)record2;

        if( print_mask & ATTR_CHECKSUM ) 
        {
            rc = log_cmp_message( LOG_ID_CMP_CHECKSUM, context,
                             "[%s][cmp][%s][checksum][%s][%s]",
                             context->host, r1->path,
                             r1->checksum, r2->checksum );
            
            if ( rc > 0 )
            {
                context->diff_count++;
                context->checksum_diff_count++;
            }
        }

        if( print_mask & ATTR_DEVICE ) 
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->device );
            osi_snprintf( right, sizeof(right), "%llu", r2->device );

            rc = log_cmp_message( LOG_ID_CMP_DEVICE, context,
                             "[%s][cmp][%s][device][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_DEVICE, context,
                             "[%s][cmp][%s][device][%llu][%llu]",
                             context->host, r1->path, 
                             r1->device, r2->device );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_BYTES )
        {
            #ifdef WIN32

            char left[32];
            char right[32];

            osi_snprintf( left, sizeof(left), "%llu", r1->bytes );
            osi_snprintf( right, sizeof(right), "%llu", r2->bytes );

            rc = log_cmp_message( LOG_ID_CMP_BYTES, context,
                             "[%s][cmp][%s][bytes][%s][%s]",
                             context->host, r1->path, left, right );
            #else

            rc = log_cmp_message( LOG_ID_CMP_BYTES, context,
                             "[%s][cmp][%s][bytes][%llu][%llu]",
                             context->host, r1->path,
                             r1->bytes, r2->bytes );
            #endif

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_MTIME )
        {
            char left_time[25];
            char right_time[25];

            osi_strlcpy( left_time, osi_time_to_string( (long)r1->mtime ),
                         sizeof( left_time ) );

            osi_strlcpy( right_time, osi_time_to_string( (long)r2->mtime ),
                         sizeof( right_time ) );


            rc = log_cmp_message( LOG_ID_CMP_MTIME, context,
                             "[%s][cmp][%s][mtime][%s][%s]",
                             context->host, r1->path, left_time, right_time );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_ATIME )
        {
            char left_time[25];
            char right_time[25];

            osi_strlcpy( left_time, osi_time_to_string( (long)r1->atime ),
                         sizeof( left_time ) );

            osi_strlcpy( right_time, osi_time_to_string( (long)r2->atime ),
                         sizeof( right_time ) );

            rc = log_cmp_message( LOG_ID_CMP_ATIME, context,
                             "[%s][cmp][%s][atime][%s][%s]",
                             context->host, r1->path, left_time, right_time );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_CTIME )
        {
            char left_time[25];
            char right_time[25];

            osi_strlcpy( left_time, osi_time_to_string( (long)r1->ctime ),
                         sizeof( left_time ) );

            osi_strlcpy( right_time, osi_time_to_string( (long)r2->ctime ),
                         sizeof( right_time ) );

            rc = log_cmp_message( LOG_ID_CMP_CTIME, context,
                             "[%s][cmp][%s][ctime][%s][%s]",
                             context->host, r1->path, left_time, right_time );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_OWNER_SID )
        {
            rc = log_cmp_message( LOG_ID_CMP_OWNER_SID, context,
                      "[%s][cmp][%s][owner_sid][%s:%s][%s:%s]",
                      context->host, r1->path,
                      r1->owner, r1->owner_sid,
                      r2->owner, r2->owner_sid );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_GROUP_SID )
        {
            rc = log_cmp_message( LOG_ID_CMP_GROUP_SID, context,
                      "[%s][cmp][%s][group_sid][%s:%s][%s:%s]",
                      context->host, r1->path,
                      r1->group, r1->group_sid,
                      r2->group, r2->group_sid );

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        }

        if( print_mask & ATTR_WIN32_FILE )
        {
            char buffer[1024];

            get_win32_file_attr_delta_string( r1->attributes, r2->attributes,
                                              buffer, sizeof( buffer ) );

            rc = log_cmp_message( LOG_ID_CMP_WIN_FILE_ATTR, context,
                             "[%s][cmp][%s][file_attributes][%s]",
                             context->host, r1->path, buffer ); 

            if ( rc > 0 )
            {
                context->diff_count++;
            }
        } 

        /* END WIN32 compare. */

        /* if we had a change, look to update some of our stats. */

        if( pre_diff_count != context->diff_count )
        {
            /* look for root owned changes. */

            if( ( strcmp( r1->owner, "Administrator" ) == 0 ) ||
                ( strcmp( r2->owner, "Administrator" ) == 0 ) )
            {
                context->root_diff_count++;
            } 
        }
    }   
}

