/*
 *  ncpmount.c
 *
 *  Copyright (C) 1995, 1997 by Volker Lendecke
 *
 *  1/20/96 - Steven N. Hirsch (hirsch@emba.uvm.edu)
 *
 *  If the ncpfs support is not loaded and we are using kerneld to
 *  autoload modules, then we don't want to do it here.  I added
 *  a conditional which leaves out the test and load code.
 *
 *  Even if we _do_ want ncpmount to load the module, passing a
 *  fully-qualified pathname to modprobe causes it to bypass a 
 *  path search.  This may lead to ncpfs.o not being found on
 *  some systems.
 * 
 *  11/28/98 - Wolfram Pienkoss <wp@bsz.shk.th.schule.de>
 *  NLS added. Thanks Petr Vandrovec for the idea of charset select
 *  and many hints.
 */
/*
 *    Modified to compile in the Pandora project [jitsu@nmrc.org]
 */

#define  NCPFS_VERSION "2.2.0.12 Pandora"

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include "../ncp/include/ext/socket.h"
#include <linux/limits.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/types.h>
extern pid_t waitpid(pid_t, int *, int);
#include <sys/errno.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/mount.h>
#include <mntent.h>
#include "../ncp/include/kernel/ipx.h"
#include <sys/ioctl.h>
#include <utmp.h>
#include <syslog.h>
#include "../ncp/include/kernel/ncp.h"
#include "../ncp/include/kernel/ncp_fs.h"
#include "ncpmount.h"
/* NCP handling from NCPFS project by Volker Lendke */
#include "../ncp/include/ncplib.h"
#include "../ncp/include/com_err.h"
#include "../ncp/include/ncplib_err.h"
#include "../ncp/include/ndslib.h"

#define MULTIVERSION_MOUNT 1
#include <sys/utsname.h>

#ifndef NR_OPEN
/* getrlimit... */
#include <linux/limits.h>
#endif
/* common Pandora GUI code base */
#include "../common/pangui.h"

#ifndef MS_MANDLOCK
#define MS_MANDLOCK	64
#endif
#ifndef MS_NOATIME
#define MS_NOATIME	1024
#endif
#ifndef MS_NODIRATIME
#define MS_NODIRATIME	2048
#endif

extern char **environ;
static char *progname;
static char mount_point[MAXPATHLEN + 1];

/******************************* GUI CODE ***********************************/

static int do_it_bab(int argc, char *argv[]);

static MOBJECT username,mountpoint,passwordhash,servername,contextname;
static MOBJECT optionlist,panmount_shell;

char *pan_argv[256]; /* max of 256 args */

void CancelPanMount(MOBJECT p, void *s, void *v)
{
 MShellDestroy(p);
}

void SelectMountPoint(MOBJECT p, void *s, void *v)
{
 char path[MAXPATHLEN];

 path[0]='\0';
 if (MFileSelection("Select an empty directory as the mount point, then click Ok and Close.", "", NULL, path, True))
  {
   MEditChangeText(mountpoint,0,strlen(path),path,MAXPATHLEN);
   MEditSetCursorPos(mountpoint,0);
  }
}

void ValidatePanMount(MOBJECT p, void *s, void *v)
{
 int pan_argc=0;
 pid_t pid;
 int status;
 char *pointer,optionstr[257];

 if ((pid =fork()) < 0)
  {
   MSListAddItem(rlist,"Unsufficient resource",NULL);
   MShellDestroy(p);
   return;
  }
 else if (pid == 0)
  {
   /* child */
   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*9);
   strcpy(pan_argv[pan_argc],"./pan-on"); pan_argc++;

   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*3);
   strcpy(pan_argv[pan_argc],"-S"); pan_argc++;
   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                     (strlen(MEditValue(servername))+1));
   strcpy(pan_argv[pan_argc],MEditValue(servername)); pan_argc++;

   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*3);
   strcpy(pan_argv[pan_argc],"-U"); pan_argc++;
   if (!strcmp(MEditValue(contextname),""))
    {
     pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                      (strlen(MEditValue(username))+1));
     strcpy(pan_argv[pan_argc],MEditValue(username));
    }
   else
    {
     pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                      (strlen(MEditValue(username))+
                                       strlen(MEditValue(contextname))+
                                       2));
     strcpy(pan_argv[pan_argc],MEditValue(username));
     strcat(pan_argv[pan_argc],".");
     strcat(pan_argv[pan_argc],MEditValue(contextname));
    }
   pan_argc++;

   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*3);
   strcpy(pan_argv[pan_argc],"-P"); pan_argc++;
   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                     (strlen(MEditValue(passwordhash))+1));
   strcpy(pan_argv[pan_argc],MEditValue(passwordhash)); pan_argc++;

   pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                     (strlen(MEditValue(mountpoint))+1));
   strcpy(pan_argv[pan_argc],MEditValue(mountpoint)); pan_argc++;

   strncpy(optionstr,MEditValue(optionlist),256);
   pointer=strtok(optionstr," ");
   while (pointer&&(pan_argc<255))
    {                /*** HAHA YOU WON'T OVERFLOW THAT !!! ***/
     pan_argv[pan_argc]=(char *)malloc(sizeof(char)*
                                       (strlen(pointer)+1));
     strcpy(pan_argv[pan_argc],pointer);
     pointer=strtok(NULL," ");
     pan_argc++;
    };
    
   pan_argv[pan_argc]=NULL;

   do_it_bab(pan_argc,pan_argv);

   exit(0);
  }
 else
  {
   /* parent */
   while (waitpid(pid, &status, 0) < 0)
    {
     if (errno != EINTR)
      {
       break;
      }
    }
   MShellDestroy(p);
   MSListAddItem(rlist,"Done. List the directory to check result.",NULL);
  }
}

void SelectPanMount(MOBJECT o, void *o_data, void *a_data)
{
 MOBJECT panmount_ok,panmount_cancel,mountpoint_sel;
 MOBJECT form_id;
 MOBJECT label;

 char user[129],password[33];
 int i;
 
 panmount_shell = MCreateShell("Mount server using hash", SF_MODAL|SF_NO_CLOSE);
 MObjectSetBackgroundRGB(panmount_shell, 56, 112, 112);

 form_id = MCreateRowForm(panmount_shell);
 MObjectSetBackgroundRGB(panmount_shell, 56, 112, 112);

 label = MCreateLabel(form_id, "User Name", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);

 sprintUnicodeName(user,pPassHack.userCN,128);
 username = MCreateClippedEdit(form_id, user, 128, 20, FIXED_SMALL);
 MRightAttach(username, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(username, 56, 112, 112);
 MObjectSetForegroundRGB(username, 192, 176, 160);
 MObjectSetResize(username, True, False);

 label = MCreateLabel(form_id, "Password Hash", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);

 for (i=0;i<16;i++) sprintf(password+i*2,"%02x",pPassHack.hash[i]);
 passwordhash = MCreateClippedEdit(form_id, password, 32, 32, FIXED_SMALL);
 MRightAttach(passwordhash, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(passwordhash, 56, 112, 112);
 MObjectSetForegroundRGB(passwordhash, 192, 176, 160);
 MObjectSetResize(passwordhash, True, False);
 MObjectSetUnsensitive(passwordhash);

 label = MCreateLabel(form_id, "Qualified Context (opt if bindery)", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);

 contextname = MCreateClippedEdit(form_id, "", 256, 20, FIXED_SMALL);
 MRightAttach(contextname, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(contextname, 56, 112, 112);
 MObjectSetForegroundRGB(contextname, 192, 176, 160);
 MObjectSetResize(contextname, True, False);

 label = MCreateLabel(form_id, "Server (requiered)", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);

 servername = MCreateClippedEdit(form_id, "", MAXNAMELEN, 20, FIXED_SMALL);
 MRightAttach(servername, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(servername, 56, 112, 112);
 MObjectSetForegroundRGB(servername, 192, 176, 160);
 MObjectSetResize(servername, True, False);

 label = MCreateLabel(form_id, "Options (ncpfs style)", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);

 optionlist = MCreateClippedEdit(form_id, "", 256, 20, FIXED_SMALL);
 MRightAttach(optionlist, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(optionlist, 56, 112, 112);
 MObjectSetForegroundRGB(optionlist, 192, 176, 160);
 MObjectSetResize(optionlist, True, False);

 label = MCreateLabel(form_id, "Mount point (requiered)", FIXED_SMALL);
 MRightAttach(label, F_ATT_NONE, 3, NULL);


 mountpoint = MCreateClippedEdit(form_id, "", MAXPATHLEN, 20, FIXED_SMALL);
 MRightAttach(mountpoint, F_ATT_NONE, 3, NULL);
 MObjectSetBackgroundRGB(mountpoint, 56, 112, 112);
 MObjectSetForegroundRGB(mountpoint, 192, 176, 160);
 MObjectSetResize(mountpoint, True, False);
 MObjectSetUnsensitive(mountpoint);

 mountpoint_sel = MCreatePButton(form_id, "Select...", FIXED_SMALL);
 MRightAttach(mountpoint_sel, F_ATT_NONE, 3, NULL);
 MPButtonSetCallback(mountpoint_sel, SelectMountPoint, 0L);


 form_id = MCreateColForm(panmount_shell);
 MObjectSetBackgroundRGB(form_id, 56, 112, 112);

 panmount_ok = MCreatePButton(form_id, "Ok", FIXED_SMALL);
 MPButtonSetCallback(panmount_ok, ValidatePanMount, 0L);
 panmount_cancel = MCreatePButton(form_id, "Cancel", FIXED_SMALL);
 MPButtonSetCallback(panmount_cancel, CancelPanMount, 0L);

 MShellRealize(panmount_shell);
}

void PanMount(MOBJECT o, void *o_data, void *a_data)
{

 if (!pPassHack.objectID)
  {
   MMessageDialog("", "o O Select a user first O o", "Ok", NULL);
   return;
  }
 SelectPanMount(o,o_data,a_data);
}

/*************************** end of GUI CODE ********************************/

/* Returns 0 if the filesystem is in the kernel after this routine
   completes */
static int
load_ncpfs(void)
{
	FILE *ffs;
	char s[1024];
	char *p, *p1;
	pid_t pid;
	int status;

	/* Check if ncpfs is in the kernel */
	ffs = fopen("/proc/filesystems", "r");

	if (ffs == NULL)
	{
		perror("Error: \"/proc/filesystems\" could not be read:");
		return -1;
	}
	p = NULL;
	while (!feof(ffs))
	{
		p1 = fgets(s, sizeof(s), ffs);
		if (p1)
		{
			p = strstr(s, "ncpfs");
			if (p)
			{
				break;
			}
		}
	}
	fclose(ffs);

	if (p)
	{
		return 0;
	}
	/* system() function without signal handling, from Stevens */

	if ((pid =fork()) < 0)
	{
		return 1;
	} else if (pid == 0)
	{
		/* child */
		execl("/sbin/modprobe", "modprobe", "ncpfs", NULL);
		_exit(127);	/* execl error */
	} else
	{
		/* parent */
		while (waitpid(pid, &status, 0) < 0)
		{
			if (errno != EINTR)
			{
				status = -1;
				break;
			}
		}
	}
	return status;
}

struct ncp_mount_data_independent {
	int		ncp_fd;
	int		wdog_fd;
	int		message_fd;
	uid_t		mounted_uid;
	struct sockaddr_ipx serv_addr;
	unsigned char	*server_name;
	unsigned char   *mount_point;
	const unsigned char   *mounted_vol;
	unsigned int	time_out;
	unsigned int	retry_count;
	struct {
	unsigned int	mount_soft:1;
	unsigned int    mount_intr:1;
	unsigned int	mount_strong:1;
	unsigned int	mount_no_os2:1;
	unsigned int	mount_no_nfs:1;
		      } flags;
	uid_t		uid;
	gid_t		gid;
	mode_t		file_mode;
	mode_t		dir_mode;
};

static int process_connection(const struct ncp_mount_data_independent* mnt);

int getmountver(void) {
	struct utsname name;
	int maj, mid;
	int ver;

	if (uname(&name)) {
		fprintf(stderr, "Cannot get kernel release\n");
                exit(0);
	}
	if (sscanf(name.release, "%d.%d", &maj, &mid) != 2) {
		fprintf(stderr, "Cannot convert kernel release \"%s\" to number\n", name.release);
                exit(0);
	}
	ver = maj*0x10000 + mid*0x100;
	if (ver < 0x20100) {
		return 2;
	}
	return 3;
}

/* Check whether user is allowed to mount on the specified mount point */
static int
mount_ok(struct stat *st)
{
	if (!S_ISDIR(st->st_mode))
	{
		errno = ENOTDIR;
		return -1;
	}
	if ((getuid() != 0)
	    && ((getuid() != st->st_uid)
		|| ((st->st_mode & S_IRWXU) != S_IRWXU)))
	{
		errno = EPERM;
		return -1;
	}
	return 0;
}

/*
 * This function changes the processes name as shown by the system.
 * Stolen from Marcin Dalecki's modald :-)
 */
static void inststr(char *dst[], int argc, const char *src)
{
	/* stolen from the source to perl 4.036 (assigning to $0) */
	char *ptr, *ptr2;
	int count;
	ptr = dst[0] + strlen(dst[0]);
	for (count = 1; count < argc; count++) {
		if (dst[count] == ptr + 1)
			ptr += strlen(++ptr);
	}
	if (environ[0] == ptr + 1) {
		for (count = 0; environ[count]; count++)
			if (environ[count] == ptr + 1)
				ptr += strlen(++ptr);
	}
	count = 0;
	for (ptr2 = dst[0]; ptr2 <= ptr; ptr2++) {
		*ptr2 = '\0';
		count++;
	}
	strncpy(dst[0], src, count);
	for (count = 1; count < argc; count++) {
		dst[count] = NULL;
	}
}

int ncp_mount_v2(const char* mount_name, unsigned long flags, const struct ncp_mount_data_independent* data) {
	struct ncp_mount_data_v2 datav2;

	if (data->serv_addr.sipx_family != AF_IPX) {
		errno = EPROTONOSUPPORT;
		return -1;
	}
	datav2.version    = NCP_MOUNT_VERSION_V2;
	datav2.ncp_fd     = data->ncp_fd;
	datav2.wdog_fd    = data->wdog_fd;
	datav2.message_fd = data->message_fd;
	datav2.mounted_uid = data->mounted_uid;
	memcpy(&datav2.serv_addr, &data->serv_addr, sizeof(datav2.serv_addr));
	strncpy(datav2.server_name, data->server_name, sizeof(datav2.server_name));
	strncpy(datav2.mount_point, data->mount_point, sizeof(datav2.mount_point));
	strncpy(datav2.mounted_vol, data->mounted_vol, sizeof(datav2.mounted_vol));
	datav2.time_out    = data->time_out;
	datav2.retry_count = data->retry_count;
	datav2.flags       = 0;
	datav2.flags      |= data->flags.mount_soft?NCP_MOUNT2_SOFT:0;
	datav2.flags      |= data->flags.mount_intr?NCP_MOUNT2_INTR:0;
	datav2.flags      |= data->flags.mount_strong?NCP_MOUNT2_STRONG:0;
	datav2.flags      |= data->flags.mount_no_os2?NCP_MOUNT2_NO_OS2:0;
	datav2.flags      |= data->flags.mount_no_nfs?NCP_MOUNT2_NO_NFS:0;
	datav2.uid         = data->uid;
	datav2.gid         = data->gid;
        datav2.file_mode   = data->file_mode;
        datav2.dir_mode    = data->dir_mode;
	return mount(mount_name, data->mount_point, "ncpfs", flags, (void*) &datav2);
}	

int ncp_mount_v3(const char* mount_name, unsigned long flags, const struct ncp_mount_data_independent* data, int argc, char* argv[]) {
	struct ncp_mount_data_v3 datav3;
	int err;

	datav3.version    = NCP_MOUNT_VERSION_V3;
	datav3.ncp_fd     = data->ncp_fd;
	datav3.mounted_uid = data->mounted_uid;
	strncpy(datav3.mounted_vol, data->mounted_vol, sizeof(datav3.mounted_vol));
	datav3.time_out    = data->time_out;
	datav3.retry_count = data->retry_count;
	datav3.flags       = 0;
	datav3.flags      |= data->flags.mount_soft?NCP_MOUNT3_SOFT:0;
	datav3.flags      |= data->flags.mount_intr?NCP_MOUNT3_INTR:0;
	datav3.flags      |= data->flags.mount_strong?NCP_MOUNT3_STRONG:0;
	datav3.flags      |= data->flags.mount_no_os2?NCP_MOUNT3_NO_OS2:0;
	datav3.flags      |= data->flags.mount_no_nfs?NCP_MOUNT3_NO_NFS:0;
	datav3.uid         = data->uid;
	datav3.gid         = data->gid;
        datav3.file_mode   = data->file_mode;
        datav3.dir_mode    = data->dir_mode;
	connect(datav3.ncp_fd, (const struct sockaddr *)&data->serv_addr, sizeof(data->serv_addr));
	datav3.wdog_pid = fork();
	if (datav3.wdog_pid < 0) {
		fprintf(stderr, "could not fork: %s\n", strerror(errno));
                exit(0);
	}
	if (datav3.wdog_pid == 0) {
		/* Child */
		inststr(argv, argc, "ncpd");
		process_connection(data);
			/* Should not return from process_connection */
                return(0);
	}
	err=mount(mount_name, data->mount_point, "ncpfs", flags, (void*) &datav3);
	if (err) {
		/* Mount unsuccesful so we have to kill daemon                */
		/* If mount success, kernel kills daemon (at least in 2.1.79) */
		kill(datav3.wdog_pid, SIGTERM);
	}
	return err;
}	

int
ncp_mount_specific(struct ncp_conn* conn, int pathNS, const unsigned char* NWpath, int pathlen) {
	int result;

#ifdef NCP_IOC_SETROOT
	{
		struct ncp_setroot_ioctl sr;

		if (pathlen == 1) {
			sr.volNumber = -1;
			sr.namespace = -1;
			sr.dirEntNum = 0;
		} else {
			struct nw_info_struct dirinfo;
	
			result = ncp_obtain_file_or_subdir_info2(conn, pathNS, NW_NS_DOS,
				0x8006, RIM_ALL, 0xFF, 0, 0, NWpath, pathlen, &dirinfo);
			if (result) {
				return -ENOENT;
			}
			if (!(dirinfo.attributes & htonl(0x10000000))) {
				return -ENOTDIR;
			}
			sr.volNumber = dirinfo.volNumber;
			sr.namespace = NW_NS_DOS;
			sr.dirEntNum = dirinfo.dirEntNum;
		}
		result = ioctl(conn->mount_fid, NCP_IOC_SETROOT, &sr);
		if (!result) {
			return 0;
		}
	}
#endif
	if ((pathlen != 1) && (*NWpath != 1)) {
		fprintf(stderr, "Remote directory is specified but "
#ifdef NCP_IOC_SETROOT
				"kernel"
#else
				"ncpmount"
#endif
				" does not support subdir mounts\n");
		return -ENOPKG;
	}
	if (ioctl(conn->mount_fid, NCP_IOC_CONN_LOGGED_IN, NULL) != 0) {
		return -errno;
	}
	return 0;
}

static int get_passwd(const char* file, const char* server, const char* user, const char** passwd) {
	FILE *f;
	char b[2048];

	if (*passwd) return 0;	/* you have password */
	if (!file) return 0;	/* you have not password file */
	if (!server) return 0;	/* something is wrong */
	if (!user) return 0;
	f = fopen(file, "r");
	if (!f) return -1;	/* NOENT */
	while (fgets(b, sizeof(b), f)) {
		char* s;
		char* u;
		char* p;
		int l;

		b[sizeof(b) - 1] = 0;
		l = strlen(b);
		while ((l > 0) && (b[l-1] == '\n')) l--;
		b[l] = 0;
		s = strtok(b, "/");
		if (!s) continue;	/* malformed line */
		u = strtok(NULL, ":");
		if (!u) continue;	/* malformed line */
		p = strtok(NULL, ":");
		if (!p) continue;
		if (*s && strcmp(server, s)) continue;
		if (strcmp(user, u)) continue;
		*passwd = strdup(p);
		fclose(f);
		return 0;
	}
	fclose(f);
	return 0;
}

struct ncp_mount_info {
	struct ncp_mount_data_independent mdata;
	struct ncp_nls_ioctl nls_cs;
	int		upcase_password;
	unsigned int	flags;
	const char*	server;
	const char*	user;
	const char*	password;
	const char*	server_name;
	const char*	passwd_file;
	int		allow_multiple_connections;
	int		sig_level;
	int		force_bindery_login;
	const char*	remote_path;
	unsigned int	pathlen;
	unsigned char	NWpath[512];
};

struct optinfo {
	char		shortopt;
	const char*	option;
#define FN_NOPARAM	1
#define FN_INT		2
#define FN_STRING	4
	int		flags;
	void		*proc;
	unsigned int	param;
};

static void opt_set_flags(struct ncp_mount_info* info, unsigned int param) {
	info->flags |= param;
}

static void opt_clear_flags(struct ncp_mount_info* info, unsigned int param) {
	info->flags &= ~param;
}

static void opt_remount(struct ncp_mount_info* info, unsigned int param) {
	fprintf(stderr, "Remounting not supported, sorry\n");
        exit(0);
}

static void opt_set_mount_soft(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_soft = param;
}

static void opt_set_mount_timeout(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 1) || (param > 900)) {
		fprintf(stderr, "Timeout must be between 1 and 900 secs inclusive\n");
                exit(0);
	}
	info->mdata.time_out = param;
}

static void opt_set_mount_retry(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 1) || (param > 0x10000)) {
		fprintf(stderr, "Retry count must be between 1 and 65536 inclusive\n");
                exit(0);
	}
	info->mdata.retry_count = param;
}

static void opt_set_mount_strong(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_strong = param;
}

static void opt_set_mount_no_nfs(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_no_nfs = param;
}

static void opt_set_mount_no_os2(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_no_os2 = param;
}

static void opt_set_mounted_uid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.mounted_uid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, "owner parameter `%s' is not valid decimal number\n", param);
                        exit(0);
		}
	} else {
		struct passwd *pwd = getpwnam(param);

		if (!pwd) {
			fprintf(stderr, "User `%s' does not exist on this machine\n", param);
                        exit(0);
		}
		info->mdata.mounted_uid = pwd->pw_uid;
	}
}

static void opt_set_uid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.uid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, "uid parameter `%s' is not valid decimal number\n", param);
                        exit(0);
                }
	} else {
		struct passwd *pwd = getpwnam(param);

		if (!pwd) {
			fprintf(stderr, "User `%s' does not exist on this machine\n", param);
                        exit(0);
		}
		info->mdata.uid = pwd->pw_uid;
	}
}

static void opt_set_gid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.gid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, "gid parameter `%s' is not valid decimal number\n", param);
                        exit(0);
		}
	} else {
		struct group* grp = getgrnam(param);

		if (!grp) {
			fprintf(stderr, "Group `%s' does not exist on this machine\n", param);
                        exit(0);
		}
		info->mdata.gid = grp->gr_gid;
	}
}

static void opt_set_passwd(struct ncp_mount_info* info, const char* param) {
	if (!param) param = "";
	
	if (strlen(param) >= sizeof(((const struct ncp_conn_spec*)param)->password)) {
		fprintf(stderr, "Specified password is too long\n");
                exit(0);
	}
	info->password = param;
}

static void opt_set_user(struct ncp_mount_info* info, const char* param) {
	info->user = param;
}

static void opt_set_server(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(((const struct ncp_conn_spec*)param)->server)) {
		fprintf(stderr, "Specified server name `%s' is too long\n", param);
                exit(0);
	}
	info->server = param;
}

static void opt_set_bindery(struct ncp_mount_info* info, unsigned int param) {
	info->force_bindery_login = param;
}

static void opt_set_server_name(struct ncp_mount_info* info, const char* param) {
	info->server_name = param;
}

static void opt_set_volume(struct ncp_mount_info* info, const char* param) {
	info->pathlen = ncp_path_to_NW_format(param, info->NWpath, sizeof(info->NWpath));
	info->remote_path = param;
	if (info->pathlen < 0) {
		fprintf(stderr, "Volume path `%s' is invalid: `%s'\n", param, strerror(-info->pathlen));
                exit(0);
	};
	if (info->pathlen == 1) {
		info->mdata.mounted_vol = "";
		info->remote_path = "/";
	} else if (info->NWpath[0] != 1) {
		info->mdata.mounted_vol = "dummy";
	} else if (strlen(param) > NCP_VOLNAME_LEN) {
		fprintf(stderr, "Volume name `%s' is too long\n", param);
                exit(0);
	} else {
		info->mdata.mounted_vol=param;
	}
}

static void opt_set_sig_level(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 0) || (param > 3)) {
		fprintf(stderr, "NCP signature level must be number between 0 and 3. You specified %u\n", param);
                exit(0);
	}
	info->sig_level = param;
}

static void opt_set_allow_multiple(struct ncp_mount_info* info, unsigned int param) {
	info->allow_multiple_connections = param;
}

static void opt_set_file_mode(struct ncp_mount_info* info, const char* param) {
	char* end;

	info->mdata.file_mode = strtoul(param, &end, 8);
	if (end && *end && !isspace(*end)) {
		fprintf(stderr, "File mode `%s' is not valid octal number\n", param);
                exit(0);
	}
}

static void opt_set_dir_mode(struct ncp_mount_info* info, const char* param) {
	char* end;

	info->mdata.dir_mode = strtoul(param, &end, 8);
	if (end && *end && !isspace(*end)) {
		fprintf(stderr, "Directory mode `%s' is not valid octal number\n", param);
                exit(0);
	}
}

static void opt_set_passwd_file(struct ncp_mount_info* info, const char* param) {
	info->passwd_file = param;
}

static void opt_set_iocharset(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(info->nls_cs.iocharset)) {
		fprintf(stderr, "I/O charset name `%s' is too long\n", param);
                exit(0);
	}
	strcpy(info->nls_cs.iocharset, param);
}

static void opt_set_codepage(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(info->nls_cs.codepage)) {
		fprintf(stderr, "Codepage name `%s' is too long\n", param);
                exit(0);
	}
	strcpy(info->nls_cs.codepage, param);
}

static void opt_set_upcase_passwd(struct ncp_mount_info* info, unsigned int param) {
	info->upcase_password = param;
}

static void opt_not_implemented(struct ncp_mount_info* info, unsigned int param) {
	/* noop */
}

static struct optinfo opts[] = {
	{0,   "ro",		FN_NOPARAM,	opt_set_flags,		MS_RDONLY},
	{0,   "rw",		FN_NOPARAM,	opt_clear_flags,	MS_RDONLY},
	{0,   "nosuid",		FN_NOPARAM,	opt_set_flags,		MS_NOSUID},
	{0,   "suid",		FN_NOPARAM,	opt_clear_flags,	MS_NOSUID},
	{0,   "nodev",		FN_NOPARAM,	opt_set_flags,		MS_NODEV},
	{0,   "dev",		FN_NOPARAM,	opt_clear_flags,	MS_NODEV},
	{0,   "noexec",		FN_NOPARAM,	opt_set_flags,		MS_NOEXEC},
	{0,   "exec",		FN_NOPARAM,	opt_clear_flags,	MS_NOEXEC},
	{0,   "sync",		FN_NOPARAM,	opt_set_flags,		MS_SYNCHRONOUS},
	{0,   "async",		FN_NOPARAM,	opt_clear_flags,	MS_SYNCHRONOUS},
	{0,   "mand",		FN_NOPARAM,	opt_set_flags,		MS_MANDLOCK},
	{0,   "nomand",		FN_NOPARAM,	opt_clear_flags,	MS_MANDLOCK},
	{0,   "noatime",	FN_NOPARAM,	opt_set_flags,		MS_NOATIME},
	{0,   "atime",		FN_NOPARAM,	opt_clear_flags,	MS_NOATIME},
	{0,   "nodiratime",	FN_NOPARAM,	opt_not_implemented,	MS_NODIRATIME},	/* does not exist in 2.0 kernel */
	{0,   "remount",	FN_NOPARAM,	opt_remount,		0},
	{0,   "soft",		FN_NOPARAM,	opt_set_mount_soft,	1},
	{0,   "hard",		FN_NOPARAM,	opt_set_mount_soft,	0},
	{'t', "timeo",		FN_INT,		opt_set_mount_timeout,	60},
	{'t', "timeout",	FN_INT,		opt_set_mount_timeout,	60},
	{'r', "retry",		FN_INT,		opt_set_mount_retry,	5},
	{'u', "uid",		FN_STRING,	opt_set_uid,		0},
	{'g', "gid",		FN_STRING,	opt_set_gid,		0},
	{'c', "owner",		FN_STRING,	opt_set_mounted_uid,	0},
	{0,   "nopasswd",	FN_NOPARAM,	opt_set_passwd,		0}, /* 'n' without mount.ncp */
	{'P', "passwd",		FN_STRING,	opt_set_passwd,		0},
	{0,   "passwdfile",	FN_STRING,	opt_set_passwd_file,	0},
	{'U', "user",		FN_STRING,	opt_set_user,		0},
	{'S', "server",		FN_STRING,	opt_set_server,		0},
	{'s', "strong",		FN_NOPARAM,	opt_set_mount_strong,	1},
	{0,   "nostrong",	FN_NOPARAM,	opt_set_mount_strong,	0},
	{0,   "nonfs",		FN_NOPARAM,	opt_set_mount_no_nfs,	1},
	{0,   "noos2",		FN_NOPARAM,	opt_set_mount_no_os2,	1},
	{0,   "nolong",		FN_NOPARAM,	opt_set_mount_no_os2,	1},
	{'b', "bindery",	FN_NOPARAM,	opt_set_bindery,	1},
	{'A', "ipserver",	FN_STRING,	opt_set_server_name,	0},
	{'V', "volume",		FN_STRING,	opt_set_volume,		0},
	{'i', "signature",	FN_INT,		opt_set_sig_level,	0},
	{'m', "multiple",	FN_NOPARAM,	opt_set_allow_multiple,	1},
	{'f', "mode",		FN_STRING,	opt_set_file_mode,	0},
	{'f', "filemode",	FN_STRING,	opt_set_file_mode,	0},
	{'d', "dirmode",	FN_STRING,	opt_set_dir_mode,	0},
	{'y', "iocharset",	FN_STRING,	opt_set_iocharset,	0},
	{'p', "codepage",	FN_STRING,	opt_set_codepage,	0},
	{'C', "noupcasepasswd",	FN_NOPARAM,	opt_set_upcase_passwd,	0},
	{0,   NULL,		0,		NULL,			0}
};

static void proc_option(struct ncp_mount_info* info, const char* opt, const char* param) {
	struct optinfo* optr;

	for (optr = opts; optr->option; optr++) {
		if (!strcmp(optr->option, opt) || ((opt[0] == optr->shortopt) && (opt[1] == 0))) {
			if (param) {
				if (optr->flags & FN_STRING)
                                       {
					((void (*)(struct ncp_mount_info*, const char*))(optr->proc))(info, param);
                                       }
				else if (optr->flags & FN_INT) {
					unsigned int i;
					char* end;

					i = strtoul(param, &end, 10);
					if (end && *end && !isspace(*end)) {
						fprintf(stderr, "Value `%s' for option `%s' is not a number\n", param, opt);
                                                exit(0);
                                        }
					((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, i);
				} else {
					fprintf(stderr, "Ignoring unneeded value for option `%s'\n", opt);
					((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, optr->param);
				}
			} else {
				if (!(optr->flags & FN_NOPARAM)) {
					fprintf(stderr, "Required parameter for option `%s' missing\n", opt);
                                        exit(0);
				}
				((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, optr->param);
			}
			return;
		}
	}
	fprintf(stderr, "Unknown option `%s', ignoring it\n", opt);
}

struct smntflags {
	unsigned int	flag;
	const char*	name;
		} mntflags[] = {
       {MS_NOEXEC,	"noexec"},
       {MS_NOSUID,	"nosuid"},
       {MS_NODEV,	"nodev"},
       {MS_SYNCHRONOUS,	"sync"},
       {MS_MANDLOCK,	"mand"},
       {MS_NOATIME,	"noatime"},
       {MS_NODIRATIME,	"nodiratime"},
       {0,		NULL}};

static int do_it_bab(int argc, char *argv[])
{
	struct ncp_mount_info info;
	struct stat st;
	char mount_name[256];

	int fd, result=-1;
	struct sockaddr_ipx addr;

	long err;

	int um;

	struct mntent ment;
	FILE *mtab;

	struct ncp_conn_spec *spec;

	struct ncp_conn *conn;

	int opt;

	struct sockaddr_in server_in;

	char *tmp_mount;

	int mount_protocol_version = -1;

	int mount_ncp = 0;
	int opt_n = 0;
	int opt_v = 0;

	progname = argv[0];

	memzero(info);
	memzero(spec);

	info.flags = MS_MGC_VAL;
	info.sig_level = -1;
	info.remote_path = "/";
	info.NWpath[0] = 0;
	info.pathlen = 1;

	info.mdata.mounted_uid = getuid();

	if (geteuid() != 0)
	{
		fprintf(stderr, "%s must be installed suid root\n", progname);
		return(-1);
	}
	info.mdata.uid = getuid();
	info.mdata.gid = getgid();
	um = umask(0);
	umask(um);
	info.mdata.file_mode = (S_IRWXU | S_IRWXG | S_IRWXO) & ~um;
	info.mdata.dir_mode = 0;
	info.mdata.flags.mount_soft = 1;
	info.mdata.flags.mount_intr = 0;
	info.mdata.flags.mount_strong = 0;
	info.mdata.flags.mount_no_os2 = 0;
	info.mdata.flags.mount_no_nfs = 0;
	info.mdata.time_out = 60;
	info.mdata.retry_count = 5;
	info.mdata.mounted_vol = "";

	info.upcase_password = 1;

	while ((opt = getopt(argc, argv, "CS:U:c:u:g:f:d:P:nh?vV:t:r:o:"
		"sN:y:p:bi:mA:"
		"2"
		"3"
	        )) != EOF)
	{
		char tmpopt[2];

		switch (opt)
		{
		case 'C':
		case 's':
		case 'b':
		case 'm':
			optarg = NULL;
		case 'S':
		case 'U':
		case 'c':
		case 'u':
		case 'g':
		case 'f':
		case 'd':
		case 'V':
		case 't':
		case 'r':
		case 'i':
		case 'A':
		case 'y':
		case 'p':
			tmpopt[0] = opt;
			tmpopt[1] = 0;
			proc_option(&info, tmpopt, optarg);
                        break;
		case 'P':
			if (!mount_ncp) opt_n = 0;	/* clear -n (nopasswd) */
			tmpopt[0] = opt;
			tmpopt[1] = 0;
			proc_option(&info, tmpopt, optarg);
			break;
		case 'n':
			opt_n = 1;	/* no passwd or no update /etc/fstab */
			break;
		case 'v':
			opt_v = 1;	/* version or verbose */
			break;	
		case '2':
			mount_protocol_version = 2;
			break;
		case '3':
			mount_protocol_version = 3;
			break;
		case 'N':
			{
				char *inp;
				char *out;
				char *dup;

                                dup = inp = strdup(optarg);

				if (!dup) {
					fprintf(stderr, "%s: out of memory\n", progname);
					return(-1);
				}
				while ((out = strtok(inp, ",;:"))!=NULL) {
					inp=NULL;
					if (!strcasecmp(out, "OS2"))
						info.mdata.flags.mount_no_os2=1;
					else if (!strcasecmp(out, "LONG"))
						info.mdata.flags.mount_no_os2=1;
					else if (!strcasecmp(out, "NFS"))
						info.mdata.flags.mount_no_nfs=1;
					else {
						fprintf(stderr, "Unknown namespace \"%s\"\n", out);
						return 128;
					}
				}
				free(dup);
			};
			break;
		case 'o':
			{
				char* dup;
				char* arg;

				arg = dup = strdup(optarg);	/* loose memory... but who cares */
				if (!dup) {
					fprintf(stderr, "%s: out of memory\n", progname);
					return(-1);
				}
				mount_ncp = 1;
				do {
					char* end;
					char* parm;

					arg += strspn(arg, " \t");
					if (!*arg) break;
					end = strchr(arg, ',');
					if (end) *end++ = 0;

					parm = strchr(arg, '=');
					if (parm) *parm++ = 0;

					proc_option(&info, arg, parm);
					arg = end;
				} while (arg);
			}
			break;
		default:
			return -1;
		}
	}
	
	if (opt_n && !mount_ncp) {
		opt_n = 0;
		opt_set_passwd(&info, "");
	}
	if (opt_v && !mount_ncp) {
		fprintf(stderr, "ncpfs version %s\n", NCPFS_VERSION);
		return(-1);
	}
	if (mount_ncp) {
		char* s;
		char* u;

		if (optind >= argc) {
			fprintf(stderr, "Required parameter missing\n");
			return(-1);
		}
		s = strdup(argv[optind++]);
		if (s) {
			u = strchr(s, '/');
			if (u) *u++ = 0;
		} else
			u = NULL;
		if (!info.server) info.server = s;
		if (!info.user) info.user = u;
	}

	if (mount_protocol_version < 0) {
		mount_protocol_version = getmountver();
	}
	if (optind != argc - 1)
	{
		return -1;
	}
	realpath(argv[optind], mount_point);

	if (stat(mount_point, &st) == -1)
	{
		fprintf(stderr, "could not find mount point %s: %s\n",
			mount_point, strerror(errno));
		return(-1);
	}
	if (mount_ok(&st) != 0)
	{
		fprintf(stderr, "cannot to mount on %s: %s\n",
			mount_point, strerror(errno));
		return(-1);
	}

	get_passwd(info.passwd_file, info.server, info.user, &info.password);
	
	if ((spec = ncp_find_conn_spec2(info.server, info.user, info.password, 1, info.mdata.uid, info.allow_multiple_connections, &err))
	    == NULL)
	{
		com_err(progname, err, "in find_conn_spec");
		return(-1);
	}
	if (info.upcase_password != 0)
	{
		str_upper(spec->password);
	}

	if (mount_protocol_version < 3) {
		/* Check if the ncpfs filesystem is in the kernel.  If not, attempt
		 * to load the ncpfs module */
		if (load_ncpfs() != 0)
		{
			fprintf(stderr, "Error: Unable to load ncpfs, exiting...\n");
			return(-1);
		}
	}

	info.mdata.server_name = spec->server;

	if (info.mdata.dir_mode == 0)
	{
		info.mdata.dir_mode = info.mdata.file_mode;
		if ((info.mdata.dir_mode & S_IRUSR) != 0)
			info.mdata.dir_mode |= S_IXUSR;
		if ((info.mdata.dir_mode & S_IRGRP) != 0)
			info.mdata.dir_mode |= S_IXGRP;
		if ((info.mdata.dir_mode & S_IROTH) != 0)
			info.mdata.dir_mode |= S_IXOTH;
	}
	if (info.server_name) {
		struct hostent* h;

		h = gethostbyname(info.server_name);
		if (!h) {
			fprintf(stderr, "Get host address `%s': ", info.server_name);
			herror(NULL);
			return 1;
		}
		if (h->h_addrtype != AF_INET) {
			fprintf(stderr, "Get host address `%s': Not AF_INET\n", info.server_name);
			return 1;
		}
		if (h->h_length != 4) {
			fprintf(stderr, "Get host address `%s': Bad address length\n", info.server_name);
			return 1;
		}
		server_in.sin_family = h->h_addrtype;
		memcpy(&server_in.sin_addr.s_addr, h->h_addr, 4);
		server_in.sin_port = htons(0x020C);
		memcpy(&info.mdata.serv_addr, &server_in, sizeof(server_in));
	} else 
	{
		if ((!info.allow_multiple_connections)&&
		    ((tmp_mount = ncp_find_permanent(spec)) != NULL))
		{
			fprintf(stderr,
				"You already have mounted server %s\nas user "
				"%s\non mount point %s\n", spec->server, spec->user,
				tmp_mount);
			return(-1);
		}
		if ((err = ncp_find_fileserver(spec->server, (struct sockaddr*)&info.mdata.serv_addr, sizeof(info.mdata.serv_addr))) != 0)
		{
			com_err("ncpmount", err, "when trying to find %s",
				spec->server);
			return(-1);
		}
	}

	if (info.mdata.serv_addr.sipx_family == AF_INET) {
		info.mdata.ncp_fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if (info.mdata.ncp_fd == -1) {
			com_err("ncpmount", err, "opening ncp_socket");
			return(-1);
		}
		info.mdata.wdog_fd = -1;
		info.mdata.message_fd = -1;
		memzero(addr);
		addr.sipx_family = AF_INET;
	} else 
	if (info.mdata.serv_addr.sipx_family == AF_IPX) {
		info.mdata.ncp_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.ncp_fd == -1)
		{
			com_err("ncpmount", err, "opening ncp_socket");
			return(-1);
		}
		info.mdata.wdog_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.wdog_fd == -1)
		{
			fprintf(stderr, "could not open wdog socket: %s\n",
				strerror(errno));
			return(-1);
		}
		memzero(addr);
		addr.sipx_family = AF_IPX;
		addr.sipx_type = NCP_PTYPE;
	} else
	{
		fprintf(stderr, "\nNo transport available\n");
		return(-1);
	}
	if (bind(info.mdata.ncp_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		fprintf(stderr, "\nbind: %s\n",
			strerror(errno));
		fprintf(stderr,
			"\nMaybe you want to use \n"
			"ipx_configure --auto_interface=on --auto_primary=on\n"
			"and try again after waiting a minute.\n\n");
		return(-1);
	}

	if (info.mdata.serv_addr.sipx_family == AF_IPX) {
		socklen_t addrlen;

		addrlen = sizeof(addr);

		if (getsockname(info.mdata.ncp_fd, (struct sockaddr *) &addr, &addrlen) == -1)
		{
			perror("getsockname ncp socket");
			close(info.mdata.ncp_fd);
			close(info.mdata.wdog_fd);
			return(-1);
		}
		addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);
	
		if (bind(info.mdata.wdog_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
		{
			fprintf(stderr, "bind(wdog_sock, ): %s\n",
				strerror(errno));
			return(-1);
		}

		info.mdata.message_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.message_fd == -1)
		{
			fprintf(stderr, "could not open message socket: %s\n",
				strerror(errno));
			return(-1);
		}
		addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);
	
		if (bind(info.mdata.message_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
		{
			fprintf(stderr, "bind(message_sock, ): %s\n",
				strerror(errno));
			return(-1);
		}
	}

	info.mdata.mount_point = mount_point;

	strcpy(mount_name, spec->server);
	strcat(mount_name, "/");
	strcat(mount_name, spec->user);

	switch (mount_protocol_version) {
		case 2:
			result = ncp_mount_v2(mount_name, info.flags, &info.mdata);
			break;
		case 3:
			result = ncp_mount_v3(mount_name, info.flags, &info.mdata, argc, argv);
			break;
		default:
			fprintf(stderr, "Unsupported mount protocol version %d\n", mount_protocol_version);
			return(-1);
	}
	if (result < 0)
	{
		com_err("ncpmount", errno, "in mount(2)");
		return(-1);
	}
	if ((err = ncp_open_mount(mount_point, &conn)) != 0)
	{
		com_err("ncpmount", err, "attempt to open mount point");
		umount(mount_point);
		return(-1);
	}

	if ((info.nls_cs.codepage[0] != 0) || (info.nls_cs.iocharset[0] != 0)) {
		int i;
		
		i = ioctl(conn->mount_fid, NCP_IOC_SETCHARSETS,  &info.nls_cs);
		if (i && (errno == EINVAL)) {
			struct ncp_nls_ioctl_old old_nls;
			char* p = info.nls_cs.codepage;
			
			/* skip `cp' */
			while (*p && !isdigit(*p)) p++;
			old_nls.codepage = strtoul(p, NULL, 0);
			strcpy(old_nls.iocharset, info.nls_cs.iocharset);
			i = ioctl(conn->mount_fid, NCP_IOC_SETCHARSETS_OLD, &old_nls);
		}
		if (i) {
			if (errno == EINVAL) {
				fprintf(stderr, "Your kernel does not support character mapping. You should upgrade to latest version.\n");
			} else {
				perror("Warning: Unable to load NLS charsets");
			}
		}
	}
	
	if (info.sig_level >= 0) {
		int err = 0;

		err = ncp_renegotiate_connparam(conn, conn->i.buffer_size, (info.sig_level > 1)? 2:0);
		if (conn->sign_wanted) {
			if (info.sig_level < 2) {
				err = ncp_renegotiate_connparam(conn, conn->i.buffer_size, 0);
			}
		} else {
			if (info.sig_level >= 2) {
				err = ncp_renegotiate_connparam(conn, conn->i.buffer_size, 2);
			}
		}
		if (err || ((info.sig_level == 0) && (conn->sign_wanted))
			|| ((info.sig_level == 3) && (!conn->sign_wanted))) {
			fprintf(stderr, "Unable to negotiate requested security level\n");
			umount(mount_point);
			return(-1);
		}
	}
	if ((!info.force_bindery_login) && (!nds_get_tree_name(conn, NULL, 0)))
	{
		if ((err = nds_login_auth(conn, spec->user, spec->password)))
		{
			if ((err != NCPL_ET_REQUEST_ERROR) || 
			    (conn->completion != NDS_GRACE_PERIOD)) {
				com_err("ncpmount", err, "in nds login");
				if (err == NCPL_ET_REQUEST_ERROR)
					fprintf(stderr, "NDS error code %d.\n", 
				                        conn->completion);
				fprintf(stderr, "Login denied.\n");
				ncp_close(conn);
				umount(mount_point);
				return(-1);
			}
	    		fprintf(stderr, "Your password has expired\n");
		}
	}
	else
	{
	if ((err = ncp_login_user(conn, spec->user, spec->password)) != 0)
	{
		struct nw_property p;
		struct ncp_prop_login_control *l
		= (struct ncp_prop_login_control *) &p;

		if (conn->completion != NCP_GRACE_PERIOD)
		{
			com_err("ncpmount", err, "in login");
			fprintf(stderr, "Login denied\n");
			ncp_close(conn);
			umount(mount_point);
			return(-1);
		}
		fprintf(stderr, "Your password has expired\n");

		if ((err = ncp_read_property_value(conn, NCP_BINDERY_USER,
						   spec->user, 1,
						   "LOGIN_CONTROL", &p)) == 0)
		{
			fprintf(stderr, "You have %d login attempts left\n",
				l->GraceLogins);
		}
	}
	}
	if ((err = ncp_mount_specific(conn, NW_NS_DOS, info.NWpath, info.pathlen)) != 0) 
	{
		fprintf(stderr, "Cannot access path \"%s\": %s\n", info.remote_path, strerror(-err));
		ncp_close(conn);
		umount(mount_point);
		return(-1);
	}
	ncp_close(conn);

	if (!opt_n) {
		struct smntflags* sf;
		char mnt_opts[80];
		char* p;

		ment.mnt_fsname = mount_name;
		ment.mnt_dir = mount_point;
		ment.mnt_type = "ncpfs";
		ment.mnt_opts = mnt_opts;
		ment.mnt_freq = 0;
		ment.mnt_passno = 0;

		p = mnt_opts;
		*p++ = 'r';
		*p++ = (info.flags & MS_RDONLY)?'o':'w';
		for (sf = mntflags; sf->flag; sf++) {
			if (info.flags & sf->flag) {
				*p++ = ',';
				strcpy(p, sf->name);
				p += strlen(p);
			}
		}
		*p = 0;

		if ((fd = open(MOUNTED "~", O_RDWR | O_CREAT | O_EXCL, 0600)) == -1)
		{
			fprintf(stderr, "Can't get " MOUNTED "~ lock file");
			return(-1);
		}
		close(fd);

		if ((mtab = setmntent(MOUNTED, "a+")) == NULL)
		{
			fprintf(stderr, "Can't open " MOUNTED);
			return(-1);
		}
		if (addmntent(mtab, &ment) == 1)
		{
			fprintf(stderr, "Can't write mount entry");
			return(-1);
		}
		if (fchmod(fileno(mtab), 0644) == -1)
		{
			fprintf(stderr, "Can't set perms on " MOUNTED);
			return(-1);
		}
		endmntent(mtab);

		if (unlink(MOUNTED "~") == -1)
		{
			fprintf(stderr, "Can't remove " MOUNTED "~");
			return(-1);
		}
	}
	return 0;
}

/* The following routines have been taken from util-linux-2.5's write.c */

/*
 * term_chk - check that a terminal exists, and get the message bit
 *     and the access time
 */
static int 
term_chk (char *tty, int *msgsokP, time_t * atimeP, int *showerror)
{
	struct stat s;
	char path[MAXPATHLEN];

	(void) sprintf(path, "/dev/%s", tty);
	if (stat(path, &s) < 0) {
		if (showerror)
			(void) fprintf(stderr,
			       "write: %s: %s\n", path, strerror(errno));
		return (1);
	}
	*msgsokP = (s.st_mode & (S_IWRITE >> 3)) != 0;	/* group write bit */
	*atimeP = s.st_atime;
	return (0);
}

/*
 * search_utmp - search utmp for the "best" terminal to write to
 *
 * Ignores terminals with messages disabled, and of the rest, returns
 * the one with the most recent access time.  Returns as value the number
 * of the user's terminals with messages enabled, or -1 if the user is
 * not logged in at all.
 *
 * Special case for writing to yourself - ignore the terminal you're
 * writing from, unless that's the only terminal with messages enabled.
 */
static int 
search_utmp (char *user, char *tty)
{
	struct utmp u;
	time_t bestatime, atime;
	int ufd, nloggedttys, nttys, msgsok, user_is_me;

	char atty[sizeof (u.ut_line) + 1];

	if ((ufd = open (_PATH_UTMP, O_RDONLY)) < 0)
	{
		perror ("utmp");
		return -1;
	}
	nloggedttys = nttys = 0;
	bestatime = 0;
	user_is_me = 0;
	while (read (ufd, (char *) &u, sizeof (u)) == sizeof (u))
		if (strncmp (user, u.ut_name, sizeof (u.ut_name)) == 0)
		{
			++nloggedttys;

			(void) strncpy (atty, u.ut_line, sizeof (u.ut_line));
			atty[sizeof (u.ut_line)] = '\0';

			if (term_chk (atty, &msgsok, &atime, 0))
			continue;		/* bad term? skip */
			if (!msgsok)
				continue;		/* skip ttys with msgs off */

			if (u.ut_type != USER_PROCESS)
				continue;		/* it's not a valid entry */

			++nttys;
			if (atime > bestatime)
			{
				bestatime = atime;
				(void) strcpy (tty, atty);
			}
      		}
 	(void) close (ufd);
 	if (nloggedttys == 0)
    	{
    		(void) fprintf (stderr, "write: %s is not logged in\n", user);
    		return -1;
    	}
  	return 0;
}

static void 
msg_received (void)
{
	struct ncp_conn *conn;
	char message[256];
	struct ncp_fs_info info;
	struct passwd *pwd;
	char tty[256];
	char tty_path[256];
	FILE *tty_file;
	FILE *mtab;
	struct mntent *mnt;
	long err;

	openlog ("nwmsg", LOG_PID, LOG_LPR);

	if ((err = ncp_open_mount(mount_point, &conn)) != 0) {
		return;
	}
	if (ncp_get_broadcast_message(conn, message) != 0) {
		ncp_close(conn);
		return;
	}
	if (strlen(message) == 0) {
		syslog(LOG_DEBUG, "no message");
		ncp_close(conn);
		return;
	}
	syslog(LOG_DEBUG, "message: %s", message);

	info.version = NCP_GET_FS_INFO_VERSION;
	if (ioctl(conn->mount_fid, NCP_IOC_GET_FS_INFO, &info) < 0) {
		ncp_close(conn);
		return;
	}
	ncp_close(conn);

	if ((pwd = getpwuid(info.mounted_uid)) == NULL) {
		fprintf(stderr, "%s: user %d not known\n",
			progname, info.mounted_uid);
		return;
	}
	if ((mtab = fopen(MOUNTED, "r")) == NULL) {
		fprintf(stderr, "%s: can't open %s\n",
			progname, MOUNTED);
		return;
	}
	while ((mnt = getmntent(mtab)) != NULL) {
		if (strcmp(mnt->mnt_dir, mount_point) == 0) {
			break;
		}
	}

	if (mnt == NULL) {
		syslog(LOG_DEBUG, "cannot find mtab entry\n");
	}
	if (search_utmp(pwd->pw_name, tty) != 0) {
		return;
	}
	sprintf(tty_path, "/dev/%s", tty);
	if ((tty_file = fopen(tty_path, "w")) == NULL) {
		fprintf(stderr, "%s: cannot open %s: %s\n",
			progname, tty_path, strerror(errno));
		return;
	}
	fprintf(tty_file, "\r\n\007\007\007Message from NetWare Server: %s\r\n",
		mnt->mnt_fsname);
	fprintf(tty_file, "%s\r\n", message);
	fclose(tty_file);
	fclose(mtab);
	return;
}

/* MSG_DONTWAIT defined and module can run only on 2.1.x kernel */
int recvfrom_notm(int fd, void* buf, size_t len, struct sockaddr* sender, socklen_t* addrlen) {
	int ret;
	int flg;

	flg = fcntl(fd, F_GETFL);
	if (flg == -1) return -1;
	fcntl(fd, F_SETFL, flg | O_NONBLOCK);
	ret = recvfrom(fd, buf, len, 0, sender, addrlen);
	fcntl(fd, F_SETFL, flg);
	return ret;
}

static void 
process_msg_packet (int msg_fd)
{
	struct sockaddr_ipx sender;
	socklen_t addrlen = sizeof(sender);
	char buf[1024];

	if (recvfrom_notm(msg_fd, buf, sizeof(buf), 
		     (struct sockaddr *) &sender, &addrlen) <= 0) {
		return;
	}
	msg_received();
}

static void 
process_wdog_packet (int wdog_fd)
{
	struct sockaddr_ipx sender;
	socklen_t addrlen = sizeof(sender);
	char buf[2];

	if (recvfrom_notm(wdog_fd, buf, sizeof(buf),
		  (struct sockaddr *) &sender, &addrlen) < (int)sizeof(buf)) {
		return;
	}
	if (buf[1] != '?') {
		return;
	}
	buf[1] = 'Y';
	sendto(wdog_fd, buf, 2, 0, (struct sockaddr *) &sender, addrlen);
}

static int 
process_connection (const struct ncp_mount_data_independent* mnt)
{
	int i;
	int result;
	int max;
	int wdog_fd = mnt->wdog_fd;
	int msg_fd = mnt->message_fd;

	chdir("/");
	setsid();
	
	if (wdog_fd == -1) {
		unsigned char buf[]={0x3E, 0x3E, 0, 0, 0, 0, 0, 0, 0, 'Y'};
		struct ncp_fs_info fsinfo;
		int fd, err;
		int ncp_fd = mnt->ncp_fd;

		for (i = 0; i < NR_OPEN; i++) {
			if (i == ncp_fd) continue;
			close(i);
		}
		/* we sleep here because of mount(2) must have enough time to run */
		sleep(180);	/* 3 min. */
		fd = open(mnt->mount_point, O_RDONLY);
		if (fd < 0) return 1;	/* Give up */
		fsinfo.version = NCP_GET_FS_INFO_VERSION;
		err = ioctl(fd, NCP_IOC_GET_FS_INFO, &fsinfo);
		close(fd);
		if (err) return 1;	/* Give up */
		buf[3] = buf[8] = fsinfo.connection&0xFF;
		buf[5] = buf[7] = (fsinfo.connection>>8)&0xFF;

		while (1) {
			send(ncp_fd, buf, sizeof(buf), 0);
			sleep(180);	/* 3 min. */
		}
		return 0;
	}

	for (i = 0; i < NR_OPEN; i++) {
		if ((i == wdog_fd) || (i == msg_fd)) {
			continue;
		}
		close(i);
	}

	max = (wdog_fd > msg_fd ? wdog_fd : msg_fd) + 1;

	while (1) {
		fd_set rd;

		FD_ZERO(&rd);
		FD_SET(wdog_fd, &rd);
		FD_SET(msg_fd, &rd);

		if ((result = select(max, &rd, NULL, NULL, NULL)) == -1) {
                            exit(0);
                        }
		if (FD_ISSET(wdog_fd, &rd)) {
			process_wdog_packet(wdog_fd);
		}
		if (FD_ISSET(msg_fd, &rd)) {
			process_msg_packet(msg_fd);
		}
	}
	return 0;
}
